﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Windows.Forms;
using System.Windows.Forms.Design;



namespace LayoutEditor.Controls
{
    using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters ;
    using Win32 = LECore.Win32;
    /// <summary>
    /// サムネイルボックスクラス。
    /// </summary>
    [ToolboxBitmap(typeof(PictureBox))]
    [Designer(typeof(ThumbnailBox.CustomDesigner))]
    public class ThumbnailBox : Control
    {
        private IThumbnail            _thumbnail = null;
        private ThumbnailBoxImageMode _imageMode = ThumbnailBoxImageMode.Color;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ThumbnailBox()
        {
            base.SetStyle(ControlStyles.Opaque, true);
            base.SetStyle(ControlStyles.DoubleBuffer, true);
            base.SetStyle(ControlStyles.Selectable, false);
            this.TabStop = false;
        }

        /// <summary>
        /// サムネイル。
        /// </summary>
        [DefaultValue(null)]
        [Description("サムネイル。")]
        public IThumbnail Thumbnail
        {
            get { return _thumbnail; }
            set
            {
                _thumbnail = value;
                Invalidate();
            }
        }

        /// <summary>
        /// イメージモード。
        /// </summary>
        [DefaultValue(ThumbnailBoxImageMode.Color)]
        [Description("イメージモード。")]
        public ThumbnailBoxImageMode ImageMode
        {
            get { return _imageMode;  }
            set { _imageMode = value; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override CreateParams CreateParams
        {
            get
            {
                CreateParams cp = base.CreateParams;
                cp.ExStyle |= Win32.WS_EX.WS_EX_STATICEDGE;
                return cp;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(64, 64); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            // 背景
            e.Graphics.Clear(SystemColors.Control);

            // 画像
            if (_thumbnail != null)
            {
                switch (_imageMode)
                {
                case ThumbnailBoxImageMode.Color: _thumbnail.DrawColorImage(e.Graphics, this.ClientRectangle, true); break;
                case ThumbnailBoxImageMode.Alpha: _thumbnail.DrawAlphaImage(e.Graphics, this.ClientRectangle, true); break;
                default: break;
                }
            }

            base.OnPaint(e);
        }

        #region CustomDesigner
        /// <summary>
        /// カスタムデザイナクラス。
        /// </summary>
        internal class CustomDesigner : ControlDesigner
        {
            /// <summary>
            /// オーバーライド。
            /// </summary>
            protected override void PostFilterProperties(IDictionary properties)
            {
                properties.Remove("AllowDrop");
                properties.Remove("BackColor");
                properties.Remove("BackgroundImage");
                properties.Remove("CausesValidation");
                properties.Remove("Font");
                properties.Remove("ForeColor");
                properties.Remove("ImeMode");
                properties.Remove("RightToLeft");
                properties.Remove("TabIndex");
                properties.Remove("TabStop");
                properties.Remove("Text");

                base.PostFilterProperties(properties);
            }
        }
        #endregion
    }

    /// <summary>
    /// サムネイルボックスイメージモード。
    /// </summary>
    public enum ThumbnailBoxImageMode
    {
        /// <summary>カラー。</summary>
        Color,
        /// <summary>アルファ。</summary>
        Alpha
    }
}
