﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using LayoutEditor.Controls;
using System.Drawing.Drawing2D;

namespace LayoutEditor.Controls
{
    using LECore.Structures;

    /// <summary>
    /// 文字カラーパネル
    /// </summary>
    public partial class TextColorPanel : UserControl
    {
        //----------------------------------------------------------
        //
        //----------------------------------------------------------
        public delegate void OnColorChange(FloatColor colT, FloatColor colB);
        public event OnColorChange ColorChange = null;

        //----------------------------------------------------------
        //
        //----------------------------------------------------------

        enum TargetColorKind
        {
            None,
            All,
            T,
            B,
        }

        //----------------------------------------------------------
        //
        //----------------------------------------------------------
        TargetColorKind _targetColorKind = TargetColorKind.None;
        FloatColor _colorT;
        FloatColor _colorB;
        private bool _firstSet = true;

        readonly ColorButton _colorButton = new ColorButton();
        readonly ColorPickerAdapter _colorPickerAdapter;
        readonly GraphicsPath _graphicsPath = new GraphicsPath();
        readonly Brush _brush = new HatchBrush(HatchStyle.LargeCheckerBoard, Color.White, Color.DarkGray);
        readonly Image _image;

        //----------------------------------------------------------
        //
        //----------------------------------------------------------

        /// <summary>
        /// 対象カラー
        /// </summary>
        FloatColor _TargetColor
        {
            get
            {
                switch(_targetColorKind)
                {
                    case TargetColorKind.All: return _CenterColor;
                    case TargetColorKind.T: return _colorT;
                    case TargetColorKind.B: return _colorB;
                    default: return FloatColor.Empty;
                }
            }

            set
            {
                switch (_targetColorKind)
                {
                    case TargetColorKind.All: _colorT = value; _colorB = value; break;
                    case TargetColorKind.T: _colorT = value; break;
                    case TargetColorKind.B: _colorB = value; break;
                    default: break;
                }

                AdjustLableColor_();
                UpdataImage_();

                if (this.ColorChange != null)
                {
                    this.ColorChange(_colorT, _colorB);
                }
            }
        }

        /// <summary>
        /// 中心の色
        /// </summary>
        FloatColor _CenterColor
        {
            get
            {
                return new FloatColor(
                            (_colorT.R + _colorB.R ) / 2,
                            (_colorT.G + _colorB.G ) / 2,
                            (_colorT.B + _colorB.B ) / 2,
                            (_colorT.A + _colorB.A ) / 2);
            }
        }

        //----------------------------------------------------------
        //
        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextColorPanel()
        {
            InitializeComponent();

            _lblT.Tag = TargetColorKind.T;
            _lblC.Tag = TargetColorKind.All;
            _lblB.Tag = TargetColorKind.B;

            _colorButton.EnableAlpha = true;
            _colorPickerAdapter = new ColorPickerAdapter(_colorButton);

            _colorPickerAdapter.ColorEdit += Event_ColorEditForm_ColorEdit;
            _colorPickerAdapter.ColorPickerClosed += Event_ColorPickerAdapter_ColorPickerClosed;

            _image = new Bitmap(this.Width, this.Height);

            // パネルに対するドロップイベントハンドラを設定
            foreach (var lbl in new[] {_lblT, _lblC, _lblB})
            {
                lbl.AllowDrop = true;
                lbl.DragEnter += this.Event_DragEnter;
                lbl.DragLeave += this.Event_DragLeave;
                lbl.DragDrop += this.Event_DragDrop;
            }
        }

        /// <summary>
        /// 色を設定します。
        /// </summary>
        public void Set(FloatColor colT, FloatColor colB)
        {
            bool bChanged = false;
            if (_firstSet ||
                (_colorT != colT))
            {
                _colorT = colT;
                bChanged |= true;
            }

            if (_firstSet ||
                (_colorB != colB))
            {
                _colorB = colB;
                bChanged |= true;
            }

            if (bChanged)
            {
                AdjustLableColor_();
                UpdataImage_();
                _firstSet = false;
            }
        }

        //----------------------------------------------------------
        //
        //----------------------------------------------------------

        /// <summary>
        /// カラーピッカー:色変更イベントハンドラ
        /// </summary>
        void Event_ColorEditForm_ColorEdit(object sender, ColorEditEventArgs e)
        {
            if (e.EditFixed)
            {
                _TargetColor = e.Color;
            }
        }

        /// <summary>
        /// カラーピッカー:閉じる
        /// </summary>
        void Event_ColorPickerAdapter_ColorPickerClosed( object sender, EventArgs e )
        {
            _targetColorKind = TargetColorKind.None;
            ColorPickerDialog.EndConnection(_colorPickerAdapter);
        }

        /// <summary>
        /// ラベルクリック
        /// </summary>
        private void Event_lblT_Click(object sender, EventArgs e)
        {
            _targetColorKind = (TargetColorKind)(sender as Label).Tag;
            ColorPickerDialog.EndConnection(_colorPickerAdapter);
            if (_targetColorKind != TargetColorKind.None)
            {
                // 色を設定して
                _colorButton.Color = _TargetColor;
                _colorPickerAdapter.NotifyUpdate();

                // 表示
                // クリックして、表示
                _colorButton.PerformClick();
            }
        }

        /// <summary>
        /// 画像更新
        /// </summary>
        private void UpdataImage_()
        {
            Graphics g = Graphics.FromImage(_image);

            Rectangle rect = new Rectangle(0, 0, _image.Width, _image.Height);
            _graphicsPath.Reset();
            _graphicsPath.AddRectangle(rect);

            // 格子を書く
            g.FillRectangle(_brush, rect);

            using (PathGradientBrush gb = new PathGradientBrush(_graphicsPath))
            {
                //パスグラデーションの中心の色を白にする
                gb.CenterColor = _CenterColor.ToSystemColor();

                //パス内の点に対応している色を指定する
                gb.SurroundColors = new Color[] { _colorT.ToSystemColor(),
                                                  _colorT.ToSystemColor(),
                                                  _colorB.ToSystemColor(),
                                                  _colorB.ToSystemColor() };

                //四角を描く
                g.FillRectangle(gb, rect);
            }

            _pnlColor.BackgroundImage = _image;
            _pnlColor.Invalidate();
        }

        /// <summary>
        /// ラベルカラー調整
        /// </summary>
        private void AdjustLableColor_()
        {
            if (_CenterColor.ToSystemColor().GetBrightness() <= 0.2f)
            {
                _lblT.ForeColor = Color.DimGray;
                _lblB.ForeColor = Color.DimGray;
                _lblC.ForeColor = Color.DimGray;
            }
            else
            {
                _lblT.ForeColor = SystemColors.ControlText;
                _lblB.ForeColor = SystemColors.ControlText;
                _lblC.ForeColor = SystemColors.ControlText;
            }
        }

        /// <summary>
        /// 色見本がドロップされた際の処理
        /// </summary>
        private void Event_DragDrop(object sender, DragEventArgs e)
        {
            Tuple<ColorData, ColorEditPanel.DragDropAction> data;
            if (ColorData.TryToGetColorDataFromDropItem(e, out data))
            {
                ColorData colorData = data.Item1;

                // 枠を消去
                var lbl = sender as Label;
                Debug.Assert(lbl != null);
                lbl.BorderStyle = BorderStyle.None;

                // ターゲットを指定して色をセット
                _targetColorKind = (TargetColorKind)lbl.Tag;
                FloatColor color = ColorEditPanel.getCopyColor(_TargetColor, data.Item1.Color, data.Item2);

                _TargetColor = color;
            }
        }

        /// <summary>
        /// 色見本のドラッグがパネルに重なった際の処理
        /// </summary>
        private void Event_DragEnter(object sender, DragEventArgs e)
        {
            if (ColorData.CheckDropItemHasColorDataWithAction(e))
            {
                e.Effect = e.AllowedEffect;

                // パネルにドラッグされていることを表す枠を表示
                var lbl = sender as Label;
                Debug.Assert(lbl != null);
                lbl.BorderStyle = BorderStyle.FixedSingle;
            }
        }

        /// <summary>
        /// 色見本のドラッグが外れた際の処理
        /// </summary>
        private void Event_DragLeave(object sender, EventArgs e)
        {
            // ドラッグされていることを表す枠を消去
            var lbl = sender as Label;
            Debug.Assert(lbl != null);
            lbl.BorderStyle = BorderStyle.None;
        }
    }
}
