﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Threading;
using System.Windows.Forms;
using LayoutEditor.Forms.ToolWindows.LayoutWindow;
using LayoutEditor.Structures.SerializableObject;
using LECore.Manipulator;
using LECore.Structures;
using LECore.Structures.Core;
using System.Diagnostics;

namespace LayoutEditor.Controls
{

    /// <summary>
    /// レンダラ対象です。
    /// <remarks>
    /// 試用直前にプログラムで生成して利用します。
    ///
    /// GetSubSceneFromNode_() で 入力として与えた、 List<ListViewItem> から ISubScene
    /// を取得して サムネイルを作り画像に設定します。
    /// </remarks>
    /// </summary>
    public class SubsceneThumbnailCreator : PictureBox
    {
        private readonly AppSetting.RendererType _rendererType;
        private readonly BackgroundWorker _backgroundWorker = new BackgroundWorker();
        private IRenderer _renderer = null;
        private readonly int _WaintMilliSec = 100;

        /// <summary>
        /// 処理中かどうか。
        /// </summary>
        public bool IsBusy
        {
            get { return _backgroundWorker.IsBusy; }
        }

        public bool AllowResize { get; set; } = false;

        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SubsceneThumbnailCreator(AppSetting.RendererType rendererType, AppSetting appSetting)
        {
            _rendererType = rendererType;

            if (_renderer == null)
            {
                _renderer = RendererFactory.Create(_rendererType);

                SetAppSetting(appSetting);
                SetStyle(RendererFactory.GetControlState(_rendererType), true);

                _renderer.Initialize(this);

                Disposed += (s, e) => {
                    if (_renderer != null)
                    {
                        _renderer.Dispose();
                        _renderer = null;
                    }
                };
            }

            _backgroundWorker.DoWork += new DoWorkEventHandler(DoRenderAll_);
            _backgroundWorker.WorkerSupportsCancellation = true;
        }

        public void SetAppSetting(AppSetting appSetting)
        {
            _renderer.LinearGammmaSRGBFetchEnabled = appSetting.ProjectSettings.DegammaTextureConversionEnabled;
            _renderer.LinearGammmaSRGBWriteEnabled = appSetting.ProjectSettings.SRGBWriteEnabled;
            _renderer.LinearGammmaColorParamatersEnabled = appSetting.ProjectSettings.DegammaParameterConversionEnabled && appSetting.ProjectSettings.SRGBWriteEnabled;
            _renderer.DepthWriteTestEnabled = appSetting.ProjectSettings.DepthTestEnabled;
        }

        //----------------------------------------------------------

        /// <summary>
        /// 非同期でキャプチャを開始します。
        /// </summary>
        public void CaptureAsync(IEnumerable<ListViewItem> items, Action onStartAction)
        {
            Debug.Assert(items.FirstOrDefault() == null || items.FirstOrDefault().ListView.LargeImageList != null);

            BackgroundWorker starter = new BackgroundWorker();
            starter.DoWork += (sender, e) =>
            {
                if (this.IsBusy)
                {
                    this.CancelAsync();
                }

                while (this.IsBusy)
                {
                    Thread.Sleep(_WaintMilliSec);
                }

                this.Invoke(new Action(() => onStartAction.Invoke()));

                _backgroundWorker.RunWorkerAsync(items);
            };
            starter.RunWorkerAsync();
        }

        /// <summary>
        ///
        /// </summary>
        virtual protected ISubScene GetSubSceneFromNode_(ListViewItem item)
        {
            return item.Tag as ISubScene;
        }

        //----------------------------------------------------------

        /// <summary>
        /// キャプチャイメージを取得します。
        /// </summary>
        /// <returns></returns>
        private Bitmap GetCaptureImage_(Action<IRenderer> renderingHandler)
        {
            DrawerForCapture drawerForCapture = (gc) =>
            {
                if (gc == null)
                {
                    gc = CreateGraphics();
                }

                renderingHandler.Invoke(_renderer);
            };

            return _renderer.Capture(drawerForCapture, Bounds, false);
        }

        /// <summary>
        /// 処理をキャンセルします。
        /// </summary>
        private void Chancel_()
        {
            if (_backgroundWorker.IsBusy && !_backgroundWorker.CancellationPending)
            {
                _backgroundWorker.CancelAsync();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void DoRenderAll_(object sender, DoWorkEventArgs e)
        {
            // リストの対応要素それぞれについて、サムネイルを生成して設定します。
            var listItems = e.Argument as IEnumerable<ListViewItem>;
            foreach (var item in listItems)
            {
                if (_backgroundWorker.CancellationPending)
                {
                    e.Cancel = true;
                    return;
                }

                var targetSubScene = GetSubSceneFromNode_(item);
                if (targetSubScene == null)
                {
                    return;
                }

                var partsSettings = item.Tag as IDerivativePartsSettings;
                this.Invoke(new Action(() =>
                {
                    Bitmap captureImage = this.GetCaptureImage_(
                        (renderer) => DoRender_(this, renderer, targetSubScene));
                    if(captureImage != null)
                    {
                        item.ListView.LargeImageList.Images.Add(captureImage);
                        item.ImageIndex = item.ListView.LargeImageList.Images.Count - 1;
                    }
                }));

                Thread.Sleep(_WaintMilliSec);
            }
        }

        public static RectangleF GetThubmnailBound(ISubScene subScene)
        {
            RectangleF rect = subScene.IPaneArray.Count() > 0 ?
            subScene.IPaneArray.FirstOrDefault().BoundingVolumeInWorld : RectangleF.Empty;
            foreach (var pane in subScene.IPaneArray)
            {
                rect = RectangleF.Union(rect, pane.BoundingVolumeInWorld);
            }

            RectangleF backGround = new RectangleF(
                -subScene.BackGround.ScreenSize.X / 2,
                -subScene.BackGround.ScreenSize.Y / 2,
                subScene.BackGround.ScreenSize.X,
                subScene.BackGround.ScreenSize.Y);

            if (rect.IntersectsWith(backGround))
            {
                rect = RectangleF.Intersect(rect, backGround);
            }

            if (rect != RectangleF.Empty)
            {
                // 少し拡大する
                rect.Inflate(rect.Width * 0.05f, rect.Height * 0.05f);
            }

            return rect;
        }

        /// <summary>
        ///
        /// </summary>
        private static void DoRender_(
            SubsceneThumbnailCreator rendererTarget,
            IRenderer renderer,
            ISubScene subScene)
        {
            // サブシーンのキャプチャ前にサブシーン中のキャプチャテクスチャの画像をすべて更新しておく。
            subScene.RequestRenderCaptureTexture(CaptureTextureUpdateMode.All);
            subScene.RenderCaptureTexture(renderer);

            // サムネイル用の解像度でキャプチャテクスチャのビットマップが更新される。
            // 次のプレビュー描画でサムネイル用のキャプチャテクスチャが使用されないように全キャプチャテクスチャ更新をリクエストしておく。
            subScene.RequestRenderCaptureTexture(CaptureTextureUpdateMode.All);

            var rect = GetThubmnailBound(subScene);
            if (rect != RectangleF.Empty)
            {
                PointF pCenter = LayoutWindowHelper.GetRectangleFCenter(rect);

                float magnify = Math.Min(
                    rendererTarget.Size.Width / rect.Width,
                    rendererTarget.Size.Height / rect.Height);

                renderer.SetViewTransform(
                    magnify,
                    magnify,
                    pCenter.X,
                    pCenter.Y,
                    -5.0f,
                    0.0f);

                DrawableOption drawableOption = new DrawableOption();
                drawableOption.EnableFlag(
                    DrawableOptionFlag.IgnoreBVDrawing |
                    DrawableOptionFlag.IgnoreDummyObjectDrawing |
                    DrawableOptionFlag.SyncTextureInitialization);

                subScene.Draw(renderer, drawableOption);
            }
        }

        public Bitmap GetCaptureImage(ISubScene targetSubScene)
        {
            Bitmap capturedBmp = GetCaptureImage_((renderer) =>
            {
                renderer.ClearBitmapTextureMap();
                DoRender_(this, renderer, targetSubScene);
            });

            return capturedBmp;
        }

        public void ClearBitmapTextureMap()
        {
            _renderer?.ClearBitmapTextureMap();
        }

        public bool IsDeviceLost
        {
            get
            {
                return _renderer?.IsDeviceLost ?? false;
            }
        }
    }
}
