﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;

namespace LayoutEditor.Controls
{
    using Forms.ToolWindows;
    using LECore.Structures.Core;
    using src.Controls;
    using System.Diagnostics;
    using App = LayoutEditor;
    /// <summary>
    /// 実数値エディットスライダクラス。
    /// </summary>
    [ToolboxBitmap(typeof(UserControl))]
    [DefaultProperty("Value")]
    [DefaultEvent("ValueEdit")]
    public sealed class FloatEditSlider : EditSlider
        , IAnimationMarkUsable
    {
        #region デザイナ変数
        private FloatTrackBar               trbValue;
        private System.Windows.Forms.Label labMin;
        private System.Windows.Forms.Label labMax;
        private src.Controls.AnimationMarkFloatTextBox tbxValue;
        #endregion

        #region デザイナコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.trbValue = new LayoutEditor.Controls.FloatTrackBar();
            this.labMin = new System.Windows.Forms.Label();
            this.labMax = new System.Windows.Forms.Label();
            this.tbxValue = new LayoutEditor.src.Controls.AnimationMarkFloatTextBox();
            ((System.ComponentModel.ISupportInitialize)(this.trbValue)).BeginInit();
            this.SuspendLayout();
            //
            // trbValue
            //
            this.trbValue.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Left)
            | System.Windows.Forms.AnchorStyles.Right)));
            this.trbValue.Location = new System.Drawing.Point(118, 19);
            this.trbValue.Name = "trbValue";
            this.trbValue.Size = new System.Drawing.Size(183, 24);
            this.trbValue.TabIndex = 3;
            //
            // labMin
            //
            this.labMin.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Left)));
            this.labMin.Location = new System.Drawing.Point(120, 3);
            this.labMin.Name = "labMin";
            this.labMin.Size = new System.Drawing.Size(64, 16);
            this.labMin.TabIndex = 1;
            this.labMin.Text = "min";
            //
            // labMax
            //
            this.labMax.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
            this.labMax.Location = new System.Drawing.Point(239, 3);
            this.labMax.Name = "labMax";
            this.labMax.Size = new System.Drawing.Size(64, 16);
            this.labMax.TabIndex = 2;
            this.labMax.Text = "max";
            this.labMax.TextAlign = System.Drawing.ContentAlignment.TopRight;
            //
            // tbxValue
            //
            this.tbxValue.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Left)));
            this.tbxValue.AnimationMarkSize = new System.Drawing.Size(5, 19);
            this.tbxValue.Location = new System.Drawing.Point(-3, 21);
            this.tbxValue.Name = "tbxValue";
            this.tbxValue.Size = new System.Drawing.Size(118, 23);
            this.tbxValue.TabIndex = 4;
            this.tbxValue.TextBoxAccessibleRole = System.Windows.Forms.AccessibleRole.Default;
            this.tbxValue.TextBoxAlign = System.Windows.Forms.HorizontalAlignment.Left;
            this.tbxValue.TextBoxFormat = "f6";
            this.tbxValue.TextBoxMaxValue = 99999F;
            this.tbxValue.TextBoxMinValue = -99999F;
            this.tbxValue.TextBoxSize = new System.Drawing.Size(100, 21);
            this.tbxValue.TextBoxValue = 0F;
            this.tbxValue.UseAnimationMark = false;
            //
            // FloatEditSlider
            //
            this.Controls.Add(this.tbxValue);
            this.Controls.Add(this.labMax);
            this.Controls.Add(this.labMin);
            this.Controls.Add(this.trbValue);
            this.Name = "FloatEditSlider";
            this.Size = new System.Drawing.Size(301, 45);
            ((System.ComponentModel.ISupportInitialize)(this.trbValue)).EndInit();
            this.ResumeLayout(false);

        }
        #endregion

        private string _labelFormat = FloatTextBox._defFormat;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FloatEditSlider()
        {
            InitializeComponent();

            // 初期設定
            trbValue.Minimum = tbxValue.TextBoxMinValue;
            trbValue.Maximum = tbxValue.TextBoxMaxValue;
            trbValue.SetValue(tbxValue.TextBoxValue);
            labMin.Text = tbxValue.TextBoxMinValue.ToString(_labelFormat);
            labMax.Text = tbxValue.TextBoxMaxValue.ToString(_labelFormat);

            // イベント登録
            tbxValue.TextBoxValueChanged += new EventHandler(Event_TbxValue_ValueChanged);
            trbValue.ScrollChanged += new ScrollChangedEventHandler(Event_TrbValue_ScrollChanged);
        }

        /// <summary>
        /// ViewManagerへのメッセージハンドラを設定します。
        /// </summary>
        public void SetViewManagerMessageHandler(Action<ViewManagerMessage> handler)
        {
            tbxValue.SetViewManagerMessageHandler(handler);
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(FloatTextBox._defMinValue)]
        [Description("最小値。")]
        public float MinValue
        {
            get { return tbxValue.TextBoxMinValue; }
            set
            {
                tbxValue.TextBoxMinValue = value;
                trbValue.Minimum = value;
                labMin.Text       = value.ToString(_labelFormat);
            }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(FloatTextBox._defMaxValue)]
        [Description("最大値。")]
        public float MaxValue
        {
            get { return tbxValue.TextBoxMaxValue; }
            set
            {
                tbxValue.TextBoxMaxValue = value;
                trbValue.Maximum = value;
                labMax.Text       = value.ToString(_labelFormat);
            }
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(FloatTextBox._defValue)]
        [Description("値。")]
        public float Value
        {
            get { return tbxValue.TextBoxValue; }
            set
            {
                tbxValue.TextBoxValue = value;
            }
        }

        /// <summary>
        /// テキストボックスサイズ。
        /// </summary>
        [DefaultValue(EditSlider._defTextBoxSize)]
        [Description("テキストボックスサイズ。")]
        public EditSliderTextBoxSize TextBoxSize
        {
            get { return _textBoxSize; }
            set
            {
                if (_textBoxSize != value)
                {
                    int delta = (int)value - (int)_textBoxSize;
                    _textBoxSize = value;

                    // レイアウト変更
                    tbxValue.TextBoxSize = new Size(
                            tbxValue.Width - (tbxValue.AnimationMarkSize.Width + tbxValue.SplitMargin) + delta,
                            tbxValue.TextBoxSize.Height);
                    trbValue.Width -= delta;
                    trbValue.Left  += delta;
                    labMin.Left    += delta;
                }
            }
        }

        /// <summary>
        /// テキストボックスマージン。
        /// </summary>
        [DefaultValue(EditSlider._defTextBoxMargin)]
        [Description("テキストボックスマージン。")]
        public EditSliderTextBoxMargin TextBoxMargin
        {
            get { return _textBoxMargin; }
            set
            {
                if (_textBoxMargin != value)
                {
                    int delta = (int)value - (int)_textBoxMargin;
                    _textBoxMargin = value;

                    // レイアウト変更
                    trbValue.Width -= delta;
                    trbValue.Left  += delta;
                    labMin.Left    += delta;
                }
            }
        }

        /// <summary>
        /// テキストボックスのテキスト整列。
        /// </summary>
        [DefaultValue(HorizontalAlignment.Left)]
        [Description("テキストボックスのテキスト整列。")]
        public HorizontalAlignment TextBoxTextAlign
        {
            get { return tbxValue.TextBoxAlign;  }
            set { tbxValue.TextBoxAlign = value; }
        }

        /// <summary>
        /// トラックバーの精度。
        /// </summary>
        [RefreshProperties(RefreshProperties.Repaint)]
        [DefaultValue(FloatTrackBar._defPrecision)]
        [Description("トラックバーの精度。")]
        public int TrackBarPrecision
        {
            get { return trbValue.Precision;  }
            set
            {
                if (trbValue.Precision != value)
                {
                    trbValue.Precision = value;
                    this.MaxValue = this.MaxValue;
                    this.MinValue = this.MinValue;
                }
            }
        }

        /// <summary>
        /// トラックバーの小変化量。
        /// </summary>
        [DefaultValue(FloatTrackBar._defSmallChange)]
        [Description("トラックバーの小変化量。")]
        public float TrackBarSmallChange
        {
            get { return trbValue.SmallChange;  }
            set { trbValue.SmallChange = value; }
        }

        /// <summary>
        /// トラックバーの大変化量。
        /// </summary>
        [DefaultValue(FloatTrackBar._defLargeChange)]
        [Description("トラックバーの大変化量。")]
        public float TrackBarLargeChange
        {
            get { return trbValue.LargeChange;  }
            set { trbValue.LargeChange = value; }
        }

        /// <summary>
        /// 範囲ラベルの可視。
        /// </summary>
        [DefaultValue(true)]
        [Description("範囲ラベルの可視。")]
        public bool ShowRangeLabel
        {
            get { return _showRangeLabel; }
            set
            {
                if (_showRangeLabel != value)
                {
                    _showRangeLabel = value;

                    labMin.Visible = _showRangeLabel;
                    labMax.Visible = _showRangeLabel;
                }
            }
        }

        /// <summary>
        /// 書式指定フォーマット。
        /// </summary>
        [DefaultValue(FloatTextBox._defFormat)]
        [Description("書式指定フォーマット。")]
        public string Format
        {
            get { return tbxValue.TextBoxFormat;  }
            set { tbxValue.TextBoxFormat = value; }
        }

        /// <summary>
        /// 範囲ラベル書式指定フォーマット。
        /// </summary>
        [DefaultValue(FloatTextBox._defFormat)]
        [Description("範囲ラベル書式指定フォーマット。")]
        public string RangeLabelFormat
        {
            get { return _labelFormat; }
            set
            {
                _labelFormat = value;

                labMin.Text = tbxValue.TextBoxMinValue.ToString(value);
                labMax.Text = tbxValue.TextBoxMaxValue.ToString(value);
            }
        }

        /// <summary>
        /// テキストボックスの背景色。
        /// </summary>
        [Description("テキストボックスの背景色。")]
        public Color TextBoxBackColor
        {
            get { return tbxValue.TextBoxBackColor; }
            set { tbxValue.TextBoxBackColor = value; }
        }

        /// <summary>
        /// Valueプロパティ設定。
        /// </summary>
        public void SetValue(float val)
        {
            tbxValue.SetValue(val);
            trbValue.SetValue(val);
        }

        /// <summary>
        /// アニメーションマーク
        /// </summary>
        public AnimationMark AnimationMarkInst
        {
            get { return tbxValue.AnimationMarkInst; }
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_TbxValue_ValueChanged(object sender, EventArgs e)
        {
            // スライダに通知
            trbValue.SetValue(tbxValue.TextBoxValue);

            // イベント発行
            OnValueEdit(new ValueEditEventArgs(true));
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_TrbValue_ScrollChanged(object sender, ScrollChangedEventArgs e)
        {
            // テキストボックスに通知
            tbxValue.SetValue(trbValue.Value);

            // イベント発行
            OnValueEdit(new ValueEditEventArgs(e.EndScroll));
        }

        /// <summary>
        /// コンテキストメニューを追加します。
        /// </summary>
        public void RegistContextMenu(string name, Image image, EventHandler handler)
        {
            tbxValue.RegistContextMenu(name, image, handler);
        }

        /// <summary>
        /// 操作対象のアトリビュートを設定します
        /// </summary>
        public void SetTargetAttribute(IAnmAttribute attr)
        {
            Debug.Assert(attr.NumSubAttribute == 0);

            tbxValue.SetTargetAttribute(attr);
        }

        /// <summary>
        /// アニメーションマークの色を更新します
        /// </summary>
        public void UpdateMarkColor()
        {
            tbxValue.UpdateMarkColor();
        }

        /// <summary>
        /// コンテキストメニューを更新します
        /// </summary>
        public void UpdateContextMenu()
        {
            tbxValue.UpdateContextMenu();
        }

        #region IAnimationMarkUsableメンバ

        /// <summary>
        /// アニメーションマークを利用するか
        /// </summary>
        public bool UseAnimationMark
        {
            get { return tbxValue.UseAnimationMark; }
            set
            {
                tbxValue.UseAnimationMark = value;
            }
        }

        /// <summary>
        /// アニメーションマークを更新します
        /// </summary>
        public void UpdateAnimationMark()
        {
            UpdateMarkColor();
            UpdateContextMenu();
        }

        #endregion
    }
}
