﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;

using LayoutEditor.Controls;
using LayoutEditor.Forms;

using LECore;
using LECore.Util;
using NW4R.ColorPicker;

namespace LayoutEditor.Controls
{
    using LECore.Structures;

    /// <summary>
    /// カラーピッカーダイアログクラス。
    /// </summary>
    public sealed class ColorPickerDialog : Form
    {
        #region デザイナ変数

        private LayoutEditor.Controls.SmallColorPickerControl ctlColorPicker;
        private System.Windows.Forms.Timer tmrDisconnect;
        private System.ComponentModel.IContainer components;
        #endregion

        #region デザイナコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(ColorPickerDialog));
            this.tmrDisconnect = new System.Windows.Forms.Timer(this.components);
            this.ctlColorPicker = new LayoutEditor.Controls.SmallColorPickerControl();
            this.SuspendLayout();
            //
            // tmrDisconnect
            //
            this.tmrDisconnect.Interval = 200;
            //
            // ctlColorPicker
            //
            resources.ApplyResources(this.ctlColorPicker, "ctlColorPicker");
            this.ctlColorPicker.Name = "ctlColorPicker";
            //
            // ColorPickerDialog
            //
            resources.ApplyResources(this, "$this");
            this.Controls.Add(this.ctlColorPicker);
            this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedToolWindow;
            this.MaximizeBox = false;
            this.MinimizeBox = false;
            this.Name = "ColorPickerDialog";
            this.ShowInTaskbar = false;
            this.FormClosing += new System.Windows.Forms.FormClosingEventHandler(this.Event_ColorPickerDialog_FormClosing);
            this.ResumeLayout(false);

        }
        #endregion

        // ダイアログインスタンス＆アダプタ
        private static ColorPickerDialog  _dialog  = null;
        private static ColorPickerAdapter _adapter = null;
        private static MessageReportDlg   _messageReportDlg = null;

        // プロファイル情報
        private static Point   _location    = Point.Empty;
        private static Color[] _customColos = new Color[16];

        static public event EventHandler DialogClose;

        #region アダプタ接続
        public static bool IsShowingDialog
        {
            get
            {
                if( _dialog != null )
                {
                    return _dialog.Visible;
                }
                return false;
            }
        }

        /// <summary>
        /// 接続開始。
        /// </summary>
        public static void StartConnection( ColorPickerAdapter adapter )
        {
            // ダイアログがなければ作成
            if (_dialog == null)
            {
                _dialog = new ColorPickerDialog();
            }

            // アダプタ設定
            SetAdapter(adapter);

            // 表示時にアダプタにロストフォーカス処理をさせない
            adapter.HandleLostFocus = false;

            // 表示
            _dialog.Show();

            _dialog.Activate();
            _dialog.Invalidate( true );
        }

        /// <summary>
        /// 接続切り替え。
        /// </summary>
        public static void ChangeConnection(ColorPickerAdapter adapter)
        {
            // ダイアログ表示中なら
            if( _dialog != null && _dialog.Visible )
            {
                // 接続アダプタを切り替える
                SetAdapter(adapter);
            }
        }

        /// <summary>
        /// 接続更新。
        /// </summary>
        public static void UpdateConnection(ColorPickerAdapter adapter)
        {
            // 接続中のアダプタなら
            //if( _dialog != null && _dialog.Visible )
            if (_dialog != null && _adapter == adapter)
            {
                // カラー更新
                _dialog.Color = adapter.Color;
            }
        }

        /// <summary>
        /// 接続終了。
        /// </summary>
        public static void EndConnection(ColorPickerAdapter adapter)
        {
            // 接続中のアダプタなら
            if (_dialog != null && _adapter == adapter)
            {
                // 接続解除開始
                if (!_dialog.tmrDisconnect.Enabled)
                {
                    _dialog.tmrDisconnect.Start();
                }
            }
        }

        /// <summary>
        /// アダプタ設定。
        /// </summary>
        private static void SetAdapter(ColorPickerAdapter adapter)
        {
            // アダプタ設定
            if (_adapter != adapter)
            {
                _adapter          = adapter;
            }

            _dialog.ColorMode = adapter.ColorMode;
            _dialog.Color = adapter.Color;

            // タイマ起動中なら停止
            if (_dialog.tmrDisconnect.Enabled)
            {
                _dialog.tmrDisconnect.Stop();
            }
        }
        #endregion

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private ColorPickerDialog()
        {
            InitializeComponent();

            // オーナー
            // this.Owner = TheApp.LayoutWindow;

            // 状態復帰
            if (_location != Point.Empty)
            {
                this.Location = _location;
                _customColos.CopyTo(ctlColorPicker.CustomColors, 0);
            }
            else
            {
                this.StartPosition = FormStartPosition.CenterParent;
            }

            // メッセージダイアログインスタンス生成
            _messageReportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, true);

            // イベント登録
            ctlColorPicker.ColorChanged += new ColorChangedEventHandler(Event_CtlColorPicker_ColorChanged);
            ctlColorPicker.HexColorError += new EventHandler(Event_CtlColorPicker_HexColorError);
            tmrDisconnect.Tick += new EventHandler(Event_TmrDisconnect_Tick);
        }

        /// <summary>
        /// カラー。
        /// </summary>
        private Color Color
        {
            get { return ctlColorPicker.Color; }
            set
            {
                if (ctlColorPicker.Color != value)
                {
                    ctlColorPicker.SetColor(value);
                    ctlColorPicker.Invalidate( true );
                }
            }
        }

        /// <summary>
        /// カラーモード。
        /// </summary>
        private ColorMode ColorMode
        {
            get { return ctlColorPicker.ColorMode;  }
            set { ctlColorPicker.ColorMode = value; }
        }

        /// <summary>
        /// 値モード。
        /// </summary>
        private ValueMode ValueMode
        {
            get { return ctlColorPicker.ValueMode;  }
            set { ctlColorPicker.ValueMode = value; }
        }


        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // TODO: 二回実行する理由が分からない、削除して問題なければ、削除する。
            // 既定の処理
            if (base.ProcessCmdKey(ref msg, keyData))
            {
                return true;
            }
           return base.ProcessCmdKey( ref msg, keyData );
            // メインフレームのショートカット処理
            // return TheApp.LayoutWindow.ProcessCommandKey(ref msg, keyData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessDialogKey(Keys keyData)
        {
            // Escキーで閉じる
            if (keyData == Keys.Escape)
            {
                // Close();
                CloseDialog_();
            }
            return base.ProcessDialogKey(keyData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
            case LECore.Win32.WM.WM_MOUSEACTIVATE:
                // アダプタのロストフォーカス処理で切断されないようにする
                // 1:WM_MOUSEACTIVATE (ColorPicker)
                // 2:WM_KILLFOCUS (ColorPickerAdapter)
                // の順にメッセージが届くはず
                if (_adapter != null)
                {
                    _adapter.HandleLostFocus = false;
                }
                break;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDeactivate(EventArgs e)
        {
            // Close後も呼ばれるのでインスタンスの有無を見て判断
            if (_dialog != null && this.CanFocus)
            {
                // 接続解除開始
                if (!tmrDisconnect.Enabled)
                {
                    tmrDisconnect.Start();
                }
            }

            base.OnDeactivate(e);
        }
        #endregion

        #region イベントハンドラ

        /// <summary>
        /// 閉じる途中ハンドラ。
        /// </summary>
        private void Event_ColorPickerDialog_FormClosing( object sender, FormClosingEventArgs e )
        {
            e.Cancel = true;
            CloseDialog_();
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_CtlColorPicker_ColorChanged(object sender, ColorChangedEventArgs e)
        {
            // アダプタにイベントを発行させる
            if (_adapter != null)
            {
                _adapter.InvokeColorEdit( new ColorEditEventArgs( new FloatColor(e.Color), e.EditFixed));
            }
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_CtlColorPicker_HexColorError(object sender, EventArgs e)
        {
            _messageReportDlg.Title = StringResMgr.Get("HEXCOLOR_ERROR_TITLE");
            _messageReportDlg.Message = StringResMgr.Get("HEXCOLOR_ERROR_MESSAGE");
            _messageReportDlg.ShowDialog();
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_TmrDisconnect_Tick(object sender, EventArgs e)
        {
            CloseDialog_();
        }

        /// <summary>
        /// ダイアログを閉じる
        /// 実際には、閉じずに非表示にします。
        /// </summary>
        void CloseDialog_()
        {
            // タイマー停止
            tmrDisconnect.Stop();

            //--------------------------------------
            // ダイアログ終了
            // 最後に選んだカラーモードを記録しておく。
            if (_adapter != null)
            {
                _adapter.ColorMode = _dialog.ColorMode;
            }

            _dialog.Visible = false;
            _adapter = null;


            _location = this.DesktopLocation;
            ctlColorPicker.CustomColors.CopyTo( _customColos, 0 );

            // イベント通知
            if( DialogClose != null )
            {
                DialogClose( this, null );
            }
        }
        #endregion


    }
}
