﻿using LayoutEditor.Forms.ToolWindows;
using LECore.Structures;
using LECore.Structures.Core;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace LayoutEditor.src.Controls
{
    /// <summary>
    /// コンテキストメニュー付きのラベルです。
    /// AnimationMarkとデータ連携することができます。
    /// </summary>
    public partial class BindAnimationLabel : Label
        , IAnimationMarkUsable
    {
        #region フィールド

        private Dictionary<string, ToolStripItem> _toolStripItemList = new Dictionary<string, ToolStripItem>();
        private AnimationMark _ownerPanel;
        private ContextMenuStrip contextMenuStrip1;

        private bool _useContextMenu = false;

        /// <summary>
        /// 必要なデザイナー変数です。
        /// </summary>
        private System.ComponentModel.IContainer components = null;

        #endregion

        #region プロパティ

        /// <summary>
        /// コンテキストメニューを表示するか。
        /// </summary>
        public bool UseContextMenu
        {
            get { return _useContextMenu; }
            set
            {
                _useContextMenu = value;
                ContextMenuStrip = _useContextMenu ? contextMenuStrip1 : null;
            }
        }

        #endregion

        #region IAnimationMarkUsableメンバ

        /// <summary>
        /// アニメーションマークを利用するか
        /// </summary>
        public bool UseAnimationMark
        {
            get { return UseContextMenu; }
            set { UseContextMenu = value; }
        }

        /// <summary>
        /// アニメーションマークを更新します
        /// </summary>
        public void UpdateAnimationMark() { }

        #endregion

        public BindAnimationLabel()
        {
            InitializeComponent();
        }

        /// <summary>
        /// 使用中のリソースをすべてクリーンアップします。
        /// </summary>
        /// <param name="disposing">マネージ リソースを破棄する場合は true を指定し、その他の場合は false を指定します。</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing && (components != null))
            {
                components.Dispose();
            }
            base.Dispose(disposing);
        }

        #region コンポーネント デザイナーで生成されたコード

        /// <summary>
        /// デザイナー サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディターで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            this.contextMenuStrip1 = new System.Windows.Forms.ContextMenuStrip(this.components);
            this.SuspendLayout();
            //
            // contextMenuStrip1
            //
            this.contextMenuStrip1.Name = "contextMenuStrip1";
            this.ResumeLayout(false);

        }

        #endregion

        /// <summary>
        /// AnimationMarkを関連付けます。
        /// </summary>
        public void BindAnimationMark(AnimationMark owner)
        {
            _ownerPanel = owner;
            BindContextMenuStrip(_ownerPanel.ContextMenuStripInst);
        }

        /// <summary>
        /// コンテキストメニューを関連付けます。
        /// </summary>
        private void BindContextMenuStrip(ContextMenuStrip menuStrip)
        {
            contextMenuStrip1 = menuStrip;
            ContextMenuStrip = contextMenuStrip1;

            UseContextMenu = true;
        }

        /// <summary>
        /// コンテキストメニューを登録します
        /// </summary>
        public void RegistContextMenu(string name, Image image, EventHandler handler)
        {
            ToolStripItem item = contextMenuStrip1.Items.Add(name, image, handler);

            if (item != null && name != "-")
            {
                _toolStripItemList.Add(name, item);
            }

            UseContextMenu = true;
        }

        #region オーバーライド

        protected override void OnMouseEnter(EventArgs e)
        {
            base.OnMouseEnter(e);

            if (_useContextMenu)
            {
                this.Cursor = AnimationMark.MouseOverCursor;
            }
        }

        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);

            this.Cursor = System.Windows.Forms.Cursors.Default;
        }

        #endregion

        #region イベント

        /// <summary>
        /// すべてのアニメーションマークのAddKeyを呼び出します。
        /// </summary>
        public static void Event_AddKeyAll(object sender, EventArgs e)
        {
            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;

            subScene.BeginMassiveModify();

            BindAnimationLabelArgs args = e as BindAnimationLabelArgs;
            foreach (AnimationMark mark in args.AnimationMarkArray)
            {
                mark.Event_AddKey(sender, e);
            }

            subScene.EndMassiveModify();
        }

        /// <summary>
        /// すべてのアニメーションマークのアトリビュートをカーブエディタに表示します。
        /// </summary>
        public static void Event_ShowCurveEditorAll(object sender, EventArgs e)
        {
            List<IAnmAttribute> targets = new List<IAnmAttribute>();
            BindAnimationLabelArgs args = e as BindAnimationLabelArgs;
            foreach (AnimationMark mark in args.AnimationMarkArray)
            {
                targets.AddRange(mark.Target);
            }

            var arg = new ViewManagerMessage(ViewManagerMessageKind.ShowCurveEditor, targets.ToArray());
            args.AnimationMarkArray[0].SendMessageToViewManager(arg);
        }

        #endregion

        /// <summary>
        /// イベントパラメータ型
        /// </summary>
        public sealed class BindAnimationLabelArgs : EventArgs
        {
            private AnimationMark[] _animationMarkArray;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public BindAnimationLabelArgs(AnimationMark[] animationMarkArray)
            {
                _animationMarkArray = animationMarkArray;
            }

            public AnimationMark[] AnimationMarkArray
            {
                get { return _animationMarkArray; }
            }
        }
    }
}
