﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Net;
using System.Runtime.InteropServices;
using System.Text;
using System.Linq;
using System.Threading;
using System.Windows.Forms;
using System.Collections.Generic;
using LECore.Structures;
using LECore.Structures.Core;
using System.Drawing;
using LECore.Save_Load;
using System.Reflection;

namespace LECore
{
    /// <summary>
    /// 外部プロセスのエラーを読む
    /// </summary>
    public class ProccesErrorReader
    {
        public readonly StreamReader stderr;
        public readonly EventWaitHandle waitHandle = new EventWaitHandle(false, EventResetMode.ManualReset);
        public volatile string errStr;

        /// <summary>
        /// 構築
        /// </summary>
        public ProccesErrorReader(StreamReader procStdError)
        {
            this.stderr = procStdError;
            ThreadPool.QueueUserWorkItem(new WaitCallback(GenerateStdErrString_), null);
        }

        /// <summary>
        ///
        /// </summary>
        public string ReadErrorMsg()
        {
            bool isErrMsg = this.waitHandle.WaitOne(5000, false);
            return isErrMsg ? this.errStr : "";
        }

        /// <summary>
        /// コンバータの標準エラー出力を文字列として取得。
        /// </summary>
        void GenerateStdErrString_(object state)
        {
            StringBuilder sb = new StringBuilder();
            try
            {
                string line;
                while (null != (line = this.stderr.ReadLine()))
                {
                    sb.AppendLine(line);
                }
            }
            catch (Exception ex)
            {
                // ログの書き出し失敗
                Debug.WriteLine(ex.ToString());
            }

            this.errStr = sb.ToString();

            if (!this.waitHandle.Set())
            {
                Debug.WriteLine("StdErr EventHandle fail \"Set()\"");
            }
        }
    }

    /// <summary>
    /// プレビューのためのバイナリファイル準備
    /// </summary>
    internal class ViewerBinaryCreator
    {
        /// <summary>
        /// 標準エラー出力取得スレッド用情報クラス
        /// </summary>
        class ReadStdErrState
        {
            public StreamReader stderr = null;
            public EventWaitHandle waitHandle = new EventWaitHandle(false, EventResetMode.ManualReset);
            public volatile string errStr = null;
        }

        ViewerPreviewParam _previewParam;

        // バイナリ生成メソッド
        private MethodInfo LayoutBinaryConverter_Program_Convert;

        // バイナリ生成ログ設定メソッド
        private MethodInfo LayoutBinaryConverter_Report_SetLogWriter;

        /// <summary>
        /// プレビューのためのバイナリデータをコンバータを起動して生成します。
        /// (ここにでしていた、タグ情報を利用したアニメーションバイナリ出力処理は利用されていなかったので削除しました。)
        /// </summary>
        public bool PrepareBinaryDataForPreview(ViewerPreviewParam param, IEnumerable<string> convertedImageFileNames)
        {
            try
            {
                _previewParam = param;

                DirectoryInfo binDir = new DirectoryInfo(_previewParam.GetBinaryOutputDirName());

                // テンポラリディレクトリが存在するなら一旦削除
                if (binDir.Exists)
                {
                    binDir.Delete(true);
                }
                binDir.Create();

                // 前処理
                PrepareBeforeBinaryConvert_();

                // 入出力ファイルの指定
                // 追加のオプションの指定
                // レイアウトの変換時にも、タグアニメーションを含む部品ペインの変換がおこなわれるので
                // タグ出力を指定します。
                // オプション文字列を 最小限外部からカスタイズできるようにする
                string args = string.Format("\"{0}\"\\ \"{1}\"\\", _previewParam.InputDir, binDir.FullName);

                string option = _previewParam.BinaryConverterOptionsString;
                if (!DoConvert_(args, option)) // コンバート失敗
                {
                    return false;
                }

                // フォントの処理
                PrepareBinaryFontForPreview_(convertedImageFileNames);

                // 背景画像の処理
                PrepareBackgroundImageForPreview_(_previewParam.BackgroundImagePath);

                return true;
            }
            catch
            {
                //// threadが終了したことを示す
                //_sPreViewerThread = null;
            }

            return false;
        }


        /// <summary>
        /// バイナリ変換前の処理
        /// </summary>
        void PrepareBeforeBinaryConvert_()
        {
            // TODO: PlatformDetail に依存しないようにする(現在は何もしていないので問題ない)
            LayoutEditorCore.PlatformDetail.PrepareBeforeBinaryConvert(
                _previewParam.Kind == ViewerPreviewParam.ViewerKind.PC, _previewParam.InputDir);
        }

        /// <summary>
        /// 自動フィルタフォントを準備する。
        /// </summary>
        void PrepareAutoFilteredFont_(IEnumerable<string> imageFontFileNames)
        {
            // バイナリフォルダにあるフォントを、自動フィルタフォントに置き換える
            string fontSourceDir = LECore.Util.TemporaryFileUtil.AutoFilteredFontDir;
            if (fontSourceDir != string.Empty && Directory.Exists(fontSourceDir))
            {
                // バイナリフォルダにあるすべてのフォントについて、自動フィルタフォントへの置き換えを試行する。
                foreach (var fontFile in Directory.GetFiles(_previewParam.GetBinaryOutputDirName(), "*.bffnt", SearchOption.AllDirectories))
                {
                    string filterdFontFileName = Path.GetFileNameWithoutExtension(fontFile) + AppConstants.FontFileAutoFilteredSuffix + AppConstants.BFFNTFileExt;
                    var filterdFonts = Directory.GetFiles(fontSourceDir, filterdFontFileName);
                    if (filterdFonts.Length > 0)
                    {
                        File.Copy(filterdFonts[0], fontFile, true);
                    }
                }

                // 変換済みのイメージフォントをコピーする
                foreach (var fileName in imageFontFileNames)
                {
                    var fonts = Directory.GetFiles(fontSourceDir, fileName);
                    if (fonts.Length > 0)
                    {
                        var destinationDirectory = Path.Combine(_previewParam.GetBinaryOutputDirName(), "font");
                        if (!Directory.Exists(destinationDirectory))
                        {
                            Directory.CreateDirectory(destinationDirectory);
                        }

                        var destination = Path.Combine(destinationDirectory, fileName);
                        File.Copy(fonts[0], destination, true);
                    }
                }
            }
        }

        /// <summary>
        /// プレビュー用のフォントコンバート
        /// </summary>
        void PrepareBinaryFontForPreview_(IEnumerable<string> convertedImageFontFileNames)
        {
            PrepareAutoFilteredFont_(convertedImageFontFileNames);

            // TODO: PlatformDetail に依存しないようにする(現在は何もしていないので問題ない)
            LayoutEditorCore.PlatformDetail.PrepareBinaryFontForPreview(
                _previewParam.Kind == ViewerPreviewParam.ViewerKind.PC, _previewParam.GetBinaryOutputDirName());
        }

        /// <summary>
        /// プレビュー用の背景画像のコンバート
        /// </summary>
        /// <param name="backgroundImagepath"></param>
        void PrepareBackgroundImageForPreview_(string backgroundImagepath)
        {
            var textureConverterExePath = Path.Combine(LayoutEditorCore.PlatformDetail.NwToolsRootPath, @"3dTools\3dTextureConverter.exe");

            // コンバートした背景画像を出力するために先に timg ディレクトリを作成します。
            string bgImageOutputDir = Path.Combine(_previewParam.GetBinaryOutputDirName(), "timg");
            if (!Directory.Exists(bgImageOutputDir))
            {
                Directory.CreateDirectory(bgImageOutputDir);
            }
            string error;
            var processResult = ProcessUtil.ProcessStart(
                textureConverterExePath,
                backgroundImagepath + " " + _previewParam.PlatformPreference.BgTextureConverterAdditionalArguments + " -o " + Path.Combine(_previewParam.GetBinaryOutputDirName(), @"timg\__BgImage.bntx"),
                null,
                null,
                out error,
                10);
        }

        /// <summary>
        /// コンバーターのパスを取得します。
        /// </summary>
        string GetConverterPath_(ViewerPreviewParam.ViewerKind kind)
        {
            // TODO: PlatformDetail に依存しないようにする(現在はユーザーがコンバータパスを書き換えない限り変更がないので問題ない)
            string path = kind == ViewerPreviewParam.ViewerKind.PC ? AppConstants.LayoutConverterForPCViewerPath : AppConstants.LayoutConverterPath;

            var prefPf = LayoutEditorCore.PlatformDetail as IPlatformPreferenceOwner;
            if (prefPf != null && prefPf.PlatformPreference != null)
            {
                path = Path.Combine(AppConstants.NwToolsRootPath, kind == ViewerPreviewParam.ViewerKind.PC ? prefPf.PlatformPreference.PCConverterPath : prefPf.PlatformPreference.TargetConverterPath);
            }

            return path;
        }

        /// <summary>
        /// コマンドラインの引数部分を分ける
        /// </summary>
        public static string[] SplitArguments(string commandLine)
        {
            if (commandLine == null)
            {
                return new string[0];
            }

            int argc;

            // コマンド部分が抜けているのでダミーのコマンド c をつけておく
            var argv = CommandLineToArgvW("c " + commandLine, out argc);

            var arguments = new string[argc - 1];
            try
            {
                // ダミーのコマンドは飛ばす
                for (int i = 1; i < argc; i++)
                {
                    // 文字列へのポインタを取得
                    var arg = Marshal.ReadIntPtr((IntPtr)argv, i * IntPtr.Size);

                    // string へ変換
                    arguments[i - 1] = Marshal.PtrToStringUni(arg);
                }
            }
            finally
            {
                // CommandLineToArgvW で確保されたメモリは呼び出し元が解放する
                Marshal.FreeHGlobal(argv);
            }

            return arguments;
        }

        [DllImport("shell32.dll", SetLastError = true)]
        private static extern IntPtr CommandLineToArgvW([MarshalAs(UnmanagedType.LPWStr)]string lpCmdLine, out int pNumArgs);

        /// <summary>
        /// 引数、オプションを指定してバイナリコンバータを実行します。
        /// コンバートに成功すれば true が返ります。
        /// </summary>
        bool DoConvert_(string argumentStr, string optionStr)
        {
            if (!_previewParam.ExecuteConverterAsProcess && LayoutBinaryConverter_Program_Convert == null)
            {
                try
                {
                    // 動的に取得せずにプロジェクトを参照を追加したほうがいいかもしれない
                    var m = Assembly.LoadFrom(GetConverterPath_(_previewParam.Kind));

                    var reportType = m.GetType("NW4F.LayoutBinaryConverter.Report");
                    LayoutBinaryConverter_Report_SetLogWriter = reportType.GetMethod("SetLogWriter");

                    var programType = m.GetType("NW4F.LayoutBinaryConverter.Program");
                    LayoutBinaryConverter_Program_Convert = programType.GetMethod("Convert");

                    if (LayoutBinaryConverter_Report_SetLogWriter == null || LayoutBinaryConverter_Program_Convert == null)
                    {
                        var msg = LECoreStringResMgr.Get("CONVERTER_ERROR_FAILSTARTUP1", GetConverterPath_(_previewParam.Kind));
                        ShowErrorDialog_(msg);
                        return false;
                    }
                }
                catch (Exception)
                {
                    var msg = LECoreStringResMgr.Get("CONVERTER_ERROR_FAILSTARTUP1", GetConverterPath_(_previewParam.Kind));
                    ShowErrorDialog_(msg);
                    return false;
                }
            }

            var args = SplitArguments(argumentStr + optionStr);

            var logFile = Path.Combine(Application.StartupPath, Path.GetFileNameWithoutExtension(GetConverterPath_(_previewParam.Kind)) + ".log");
            try
            {
                using (var errorWriter = new StringWriter())
                {
                    int result;
                    if (_previewParam.ExecuteConverterAsProcess)
                    {
                        // プロセスとして実行(デバッグ用)
                        var exePath = GetConverterPath_(_previewParam.Kind);
                        using (var outWriter = File.CreateText(logFile))
                        {
                            string error;
                            var processResult = ProcessUtil.ProcessStart(exePath, argumentStr + optionStr,
                                (s, e) => { outWriter.Write(e?.Data ?? ""); },
                                (s, e) => { outWriter.Write(e?.Data ?? ""); errorWriter.Write(e?.Data ?? ""); },
                                out error,
                                workingDirectory: Application.StartupPath);
                            if (processResult.HasValue)
                            {
                                result = processResult.Value;
                            }
                            else
                            {
                                result = 1;
                            }

                            if (!string.IsNullOrEmpty(error))
                            {
                                errorWriter.WriteLine(error);
                            }
                        }
                    }
                    else
                    {
                        // ロードしたアセンブリのメソッドとして実行
                        using (var outWriter = File.CreateText(logFile))
                        {
                            LayoutBinaryConverter_Report_SetLogWriter.Invoke(null, new object[] { outWriter, errorWriter });
                            result = (int)LayoutBinaryConverter_Program_Convert.Invoke(null, new object[] { args });
                        }
                    }

                    if (result != 0)
                    {
                        // -- CONVERTER_ERROR_FAILCONVERT "バイナリコンバートに失敗しました。 - ({0})\nログを次のファイルに出力しました。\n{1}\n\n{2}"
                        string errorMsg = errorWriter.ToString();
                        string msg = LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT", result, logFile, errorMsg);
                        ShowErrorDialog_(msg);
                        return false;
                    }
                }
            }
            catch (Exception e)
            {
                // -- CONVERTER_ERROR_FAILCONVERT "バイナリコンバートに失敗しました。\nログを次のファイルに出力しました。\n{1}\n\n{2}"
                string errorMsg = e.ToString();
                string msg = LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT2", logFile, errorMsg);
                ShowErrorDialog_(msg);
                return false;
            }

            return true;
        }

        /// <summary>
        /// エラーダイアログを表示します。
        /// </summary>
        static void ShowErrorDialog_(string message)
        {
            string title = LECoreStringResMgr.Get("PREVIEW_ERROR_DLG_TITLE");
            LayoutEditorCore.MsgReporter.OnShowMessageDialogAsync(title, message);
        }
    }
}
