﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;

namespace LECore.Structures
{
    /// <summary>
    /// 部品ペインプロパティのオプションです。
    /// </summary>
    [Flags]
    public enum PartsPropertyUsageOptions
    {
        None = 0x0,
        UsePanePosition = 0x1,
        UsePaneRotate = 0x2,
        UsePaneScale = 0x4,
        UsePaneSize = 0x8,
        UsePaneTransparency = 0x10,
        UsePaneVisibility = 0x20,
        UsePaneBasicUserData = 0x40,
        UsePaneExUserData = 0x80,
        UsePaneExParamater = 0x100,
        UsePaneExParamaterUseTextBoxText = 0x200,
        UsePaneExParamaterColorBlend = 0x400,
        UsePaneExParameterTexture = 0x800,
    };

    /// <summary>
    /// ヘルパークラスです。
    /// </summary>
    static public class PartsPropertyUsageOptionsHelper
    {
        /// <summary>
        /// カラー補間だけ上書きが可能なペインかどうか
        /// </summary>
        public static bool IsColorBlendOverwriteEnabled(PaneKind kind)
        {
            return kind == PaneKind.Textbox || kind == PaneKind.Picture || kind == PaneKind.Window;
        }

        /// <summary>
        /// テクスチャだけ上書きが可能なペインかどうか
        /// </summary>
        public static bool IsTextureOverwriteEnabled(PaneKind kind)
        {
            return kind == PaneKind.Picture || kind == PaneKind.Window;
        }

        /// <summary>
        /// ペイン拡張部分（基本情報以外の部分）が上書きされているかどうか
        /// </summary>
        public static bool IsPaneExDataOverwrote(PartsPropertyUsageOptions option)
        {
            return
                option.HasFlag(PartsPropertyUsageOptions.UsePaneExParamater) ||
                option.HasFlag(PartsPropertyUsageOptions.UsePaneExParamaterColorBlend) ||
                option.HasFlag(PartsPropertyUsageOptions.UsePaneExParameterTexture);
        }

        /// <summary>
        /// ペイン拡張部分（基本情報以外の部分）が限定的に上書き(テクスチャだけ上書きなど)されているかどうか
        /// </summary>
        public static bool IsPaneExDataPartlyOverwrote(PartsPropertyUsageOptions option)
        {
            return
                option.HasFlag(PartsPropertyUsageOptions.UsePaneExParamaterUseTextBoxText) ||
                option.HasFlag(PartsPropertyUsageOptions.UsePaneExParamaterColorBlend) ||
                option.HasFlag(PartsPropertyUsageOptions.UsePaneExParameterTexture);
        }

        /// <summary>
        /// ペインのテクスチャ関連パラメータが上書きされているかどうか
        /// </summary>
        public static bool IsPaneTextureDataOverwrote(PartsPropertyUsageOptions option)
        {
            // ペイン拡張部分（基本情報以外の部分）に上書きがなければ、テクスチャは上書きされない。
            if (!option.HasFlag(PartsPropertyUsageOptions.UsePaneExParamater))
            {
                return false;
            }
            // 限定的な上書き(テクスチャだけ上書きなど)がなければ、テクスチャは上書きされる。
            if (!IsPaneExDataPartlyOverwrote(option))
            {
                return true;
            }
            // テクスチャだけ上書きが有効なら、テクスチャは上書きされる。
            if (option.HasFlag(PartsPropertyUsageOptions.UsePaneExParameterTexture))
            {
                return true;
            }
            // それ以外のパターンでは、テクスチャは上書きされない。
            return false;
        }
    }

    /// <summary>
    /// IPartsLayout の ヘルパーです。
    /// </summary>
    public static class IPartsLayoutHelper
    {
        /// <summary>
        /// 部品ペインのテクスチャ上書きをしているペインを列挙します。
        /// </summary>
        public static IEnumerable<IPane> EnumeratePanesOverwritingTextures(this IPartsLayout partsLayout)
        {
            if (partsLayout == null)
            {
                yield break;
            }

            foreach (var partsPropaerty in partsLayout.PartsPropaerties)
            {
                if (partsPropaerty.Paramater != null && PartsPropertyUsageOptionsHelper.IsPaneTextureDataOverwrote(partsPropaerty.Option))
                {
                    yield return partsPropaerty.Paramater.OwnerPane;
                }
            }
        }

        /// <summary>
        /// 部品ペインの上書きに使っているテクスチャの名前を列挙します。
        /// </summary>
        public static IEnumerable<string> EnumerateTexturesForOverwrite(this IPartsLayout partsLayout)
        {
            foreach (var pane in IPartsLayoutHelper.EnumeratePanesOverwritingTextures(partsLayout))
            {
                foreach (var name in PaneHelper.EnumrateAllTextureName(pane))
                {
                    yield return name;
                }
            }
        }
    }

    /// <summary>
    /// 部品ペインが持つ部品レイアウト情報です。
    /// </summary>
    public interface IPartsLayout : IPaneExParamater
    {
        /// <summary>
        /// 部品レイアウト名です。
        /// </summary>
        string PartsLayoutName { get; }

        /// <summary>
        /// 上書きパラメータ列です。
        /// </summary>
        IEnumerable<IPartsPropaerty> PartsPropaerties { get; }

        /// <summary>
        /// 部品の倍率です。
        /// </summary>
        FVec3 Magnify { get; }

        /// <summary>
        /// 指定のサイズです。
        /// </summary>
        FVec3 SizeConstraint { get; }

        /// <summary>
        /// 部品サブシーンです。
        /// </summary>
        ISubScene PartsSubScene { get; }

        /// <summary>
        /// 部品の元サイズです。
        /// </summary>
        FVec3 InitialSize { get; }
    }

    /// <summary>
    /// 部品ペインの上書きプロパティです。
    /// </summary>
    public interface IPartsPropaerty
    {
        /// <summary>
        /// 種類です。
        /// </summary>
        PaneKind PaneKind { get; }

        /// <summary>
        /// 対象ペイン名です。
        /// </summary>
        string TargetName { get; }

        /// <summary>
        /// 上書きする実データです。
        /// </summary>
        IPaneExParamater Paramater { get; }

        /// <summary>
        /// 部品ペインプロパティのオプションです。
        /// </summary>
        PartsPropertyUsageOptions Option { get; }
    }
}
