﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Diagnostics;

namespace LECore.Util
{
    using System.Threading;

    public static class TemporaryFileUtil
    {
        private static TemporoaryFileManager ViewerFolder;
        private static TemporoaryFileManager ShaderCacheFolder;
        private static TemporoaryFileManager AutoFilteredFontFolder;
        private static TemporoaryFileManager AutoFilteredFontLinearFolder;
        private static TemporoaryFileManager TempImageFontFolder;
        private static TemporoaryFileManager FtxbCacheFolder;

        private static string tempRoot;

        // 複数起動時にテンポラリフォルダ操作の競合を防ぐ
        // 現在は LayoutEditor のメインウインドウは複数開かないが、プロセスを複数起動することは可能
        private static Mutex mutex = new Mutex(false, "Nintendo_LayoutEditor_TemporaryFile_Mutex");

        public static string ShaderCacheRoot { get { return ShaderCacheFolder.FolderPath; } }
        public static string AutoFilteredFontDir {  get { return AutoFilteredFontFolder.FolderPath; } }
        public static string AutoFilteredLinearFontDir { get { return AutoFilteredFontLinearFolder.FolderPath; } }
        public static string TempImageFontDir { get { return AutoFilteredFontLinearFolder.FolderPath; } }
        public static string FtxbCacheDir { get { return AutoFilteredFontLinearFolder.FolderPath; } }

        public static void Destroy()
        {
            try
            {
                mutex.WaitOne();
                ViewerFolder.Destroy(false);
                ShaderCacheFolder.Destroy(false);
                AutoFilteredFontFolder.Destroy(false);
                AutoFilteredFontLinearFolder.Destroy(false);
                TempImageFontFolder.Destroy(false);
                FtxbCacheFolder.Destroy(false);

                // Nintendo_LayoutEditor_Workフォルダが空のときは削除
                var entries = Directory.EnumerateFileSystemEntries(tempRoot);
                if (!entries.Any())
                {
                    try
                    {
                        Directory.Delete(tempRoot);
                    }
                    catch (Exception)
                    {
                    }
                }
            }
            finally
            {
                mutex.ReleaseMutex();
            }
        }

        public static void Initialize()
        {
            try
            {
                mutex.WaitOne();
                tempRoot = Path.Combine(Path.GetTempPath(), "Nintendo_LayoutEditor_Work");
                Directory.CreateDirectory(tempRoot);

                ViewerFolder = new TemporoaryFileManager(Path.Combine(tempRoot, @"Viewer"));
                ShaderCacheFolder = new TemporoaryFileManager(Path.Combine(tempRoot, @"ShaderCache"));
                AutoFilteredFontFolder = new TemporoaryFileManager(Path.Combine(tempRoot, @"AutoFilteredFont"));
                AutoFilteredFontLinearFolder = new TemporoaryFileManager(Path.Combine(tempRoot, @"AutoFilteredFontLinear"));
                TempImageFontFolder = new TemporoaryFileManager(Path.Combine(tempRoot, @"TempImageFont"));
                FtxbCacheFolder = new TemporoaryFileManager(Path.Combine(tempRoot, @"FtxbCache"));
            }
            finally
            {
                mutex.ReleaseMutex();
            }
        }

        /// <summary>
        /// 指定したパスが、一時ディレクトリ内か判定します。
        /// </summary>
        public static bool IsInTempDirectory(string path)
        {
            Debug.Assert(!string.IsNullOrEmpty(path));

            return Path.GetFullPath(path).StartsWith(tempRoot);
        }

        /// <summary>
        /// ビューア用一時ディレクトリを作成
        /// </summary>
        public static string MakeViewerTempDirectory()
        {
            return ViewerFolder.MakeTempDirectory();
        }

        /// <summary>
        /// ビューア用一時ディレクトリに含まれるディレクトリを削除
        /// </summary>
        public static void DeleteViewerTempDirectory(string path)
        {
            try
            {
                if (Directory.Exists(path))
                {
                    Directory.Delete(path, true);
                }
            }
            catch
            {
            }
        }

        private class TemporoaryFileManager
        {
            private static readonly object syncObject_ = new object();

            private readonly string tempRootDir_;
            private string tempDir_;
            private int nameCounter_;

            private FileStream lockFile_ = null;

            private const string LockFileName = "lock";

            public string FolderPath
            {
                get
                {
                    return tempDir_;
                }
            }

            public TemporoaryFileManager(string rootPath)
            {
                tempRootDir_ = rootPath;
                Destroy(false, false);
                using (var process = Process.GetCurrentProcess())
                {
                    tempDir_ = Path.Combine(tempRootDir_, process.Id.ToString());
                    Directory.CreateDirectory(tempDir_);
                    try
                    {
                        lockFile_ = File.Open(Path.Combine(tempDir_, LockFileName), FileMode.OpenOrCreate, FileAccess.Read, FileShare.None);
                    }
                    catch
                    {
                    }
                }
            }

            /// <summary>
            /// 指定したディレクトリがlockファイルでロックされてなければ削除する。
            /// </summary>
            /// <param name="dir"></param>
            /// <param name="flag"></param>
            private static void DeleteUnlockedDirectory(string dir, bool flag)
            {
                var lockpath = Path.Combine(dir, LockFileName);
                if (File.Exists(lockpath))
                {
                    try
                    {
                        var templock = File.Open(lockpath, FileMode.Open, FileAccess.Read, FileShare.None);
                        templock.Close();
                    }
                    catch (Exception)
                    {
                        return;
                    }
                }
                Directory.Delete(dir, flag);
            }

            public void Destroy(bool deleteNonDigit, bool deleteSelf = true)
            {
                try
                {
                    if (lockFile_ != null)
                    {
                        lockFile_.Close();
                    }

                    if (!Directory.Exists(this.tempRootDir_))
                    {
                        return;
                    }

                    foreach (var dir in Directory.EnumerateDirectories(this.tempRootDir_))
                    {
                        try
                        {
                            if (deleteNonDigit)
                            {
                                DeleteUnlockedDirectory(dir, true);
                            }
                            else
                            {
                                string name = Path.GetFileName(dir);
                                if (name.All(x => char.IsDigit(x)))
                                {
                                    DeleteUnlockedDirectory(dir, true);
                                }
                            }
                        }
                        catch
                        {
                        }
                    }

                    // フォルダが空のとき削除
                    if (deleteSelf && !Directory.EnumerateFileSystemEntries(this.tempRootDir_).Any())
                    {
                        try
                        {
                            Directory.Delete(this.tempRootDir_);
                        }
                        catch (Exception)
                        {
                        }
                    }
                }
                catch(Exception)
                {

                }
            }

            // テンポラリファイル名を作る
            public string MakeTemporaryFileName(string ext)
            {
                using (var process = Process.GetCurrentProcess())
                {
                    lock (syncObject_)
                    {
                        ++nameCounter_;

                        return
                            Path.Combine(
                                tempDir_,
                                nameCounter_.ToString() + (ext ?? string.Empty)
                            );
                    }
                }
            }

            // テンポラリディレクトリを作成
            public string MakeTempDirectory()
            {
                using (var process = Process.GetCurrentProcess())
                {
                    lock (syncObject_)
                    {
                        ++nameCounter_;

                        var directory = Path.Combine(tempDir_, nameCounter_.ToString());
                        Directory.CreateDirectory(
                            directory);
                        return directory;
                    }
                }
            }
        }
    }
}
