﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Diagnostics;
using System.Windows.Forms;

namespace LECore.Util
{
    using LECore.Structures;


    #region PointUtil
    /// <summary>
    /// ポイントユーティリティクラス。
    /// </summary>
    public sealed class PointUtil
    {
        /// <summary>
        /// ２点間の距離を取得。
        /// </summary>
        public static float GetDistance(PointF pt1, PointF pt2)
        {
            return (float)Math.Sqrt((pt2.X - pt1.X) * (pt2.X - pt1.X) + (pt2.Y - pt1.Y) * (pt2.Y - pt1.Y));
        }

        /// <summary>
        /// POINT型に変換。
        /// </summary>
        public static Win32.POINT ToPOINT(Point pt)
        {
            Win32.POINT result = new Win32.POINT();
            result.x = pt.X;
            result.y = pt.Y;
            return result;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private PointUtil() {}
    }
    #endregion

    #region RectangleUtil
    /// <summary>
    /// レクタングルユーティリティクラス。
    /// </summary>
    public sealed class RectangleUtil
    {
        /// <summary>
        /// 左位置を設定。
        /// </summary>
        public static void SetLeft(ref Rectangle rect, int val)
        {
            if (val > rect.Left) { rect.Width -= val - rect.Left; }
            else                 { rect.Width += rect.Left - val; }
            rect.X = val;
        }

        /// <summary>
        /// 左位置を設定。
        /// </summary>
        public static void SetLeft(ref RectangleF rect, float val)
        {
            if (val > rect.Left) { rect.Width -= val - rect.Left; }
            else                 { rect.Width += rect.Left - val; }
            rect.X = val;
        }

        /// <summary>
        /// 上位置を設定。
        /// </summary>
        public static void SetTop(ref Rectangle rect, int val)
        {
            if (val > rect.Top) { rect.Height -= val - rect.Top; }
            else                { rect.Height += rect.Top - val; }
            rect.Y = val;
        }

        /// <summary>
        /// 上位置を設定。
        /// </summary>
        public static void SetTop(ref RectangleF rect, float val)
        {
            if (val > rect.Top) { rect.Height -= val - rect.Top; }
            else                { rect.Height += rect.Top - val; }
            rect.Y = val;
        }

        /// <summary>
        /// 右位置を設定。
        /// </summary>
        public static void SetRight(ref Rectangle rect, int val)
        {
            if (val > rect.Right) { rect.Width += val - rect.Right; }
            else                  { rect.Width -= rect.Right - val; }
        }

        /// <summary>
        /// 右位置を設定。
        /// </summary>
        public static void SetRight(ref RectangleF rect, float val)
        {
            if (val > rect.Right) { rect.Width += val - rect.Right; }
            else                  { rect.Width -= rect.Right - val; }
        }

        /// <summary>
        /// 下位置を設定。
        /// </summary>
        public static void SetBottom(ref Rectangle rect, int val)
        {
            if (val > rect.Bottom) { rect.Height += val - rect.Bottom; }
            else                   { rect.Height -= rect.Bottom - val; }
        }

        /// <summary>
        /// 下位置を設定。
        /// </summary>
        public static void SetBottom(ref RectangleF rect, float val)
        {
            if (val > rect.Bottom) { rect.Height += val - rect.Bottom; }
            else                   { rect.Height -= rect.Bottom - val; }
        }

        /// <summary>
        /// 左位置を移動。
        /// </summary>
        public static void OffsetLeft(ref Rectangle rect, int delta)
        {
            rect.X     += delta;
            rect.Width -= delta;
        }

        /// <summary>
        /// 左位置を移動。
        /// </summary>
        public static void OffsetLeft(ref RectangleF rect, float delta)
        {
            rect.X     += delta;
            rect.Width -= delta;
        }

        /// <summary>
        /// 上位置を移動。
        /// </summary>
        public static void OffsetTop(ref Rectangle rect, int delta)
        {
            rect.Y      += delta;
            rect.Height -= delta;
        }

        /// <summary>
        /// 上位置を移動。
        /// </summary>
        public static void OffsetTop(ref RectangleF rect, float delta)
        {
            rect.Y      += delta;
            rect.Height -= delta;
        }

        /// <summary>
        /// 右位置を移動。
        /// </summary>
        public static void OffsetRight(ref Rectangle rect, int delta)
        {
            rect.Width += delta;
        }

        /// <summary>
        /// 右位置を移動。
        /// </summary>
        public static void OffsetRight(ref RectangleF rect, float delta)
        {
            rect.Width += delta;
        }

        /// <summary>
        /// 下位置を移動。
        /// </summary>
        public static void OffsetBottom(ref Rectangle rect, int delta)
        {
            rect.Height += delta;
        }

        /// <summary>
        /// 下位置を移動。
        /// </summary>
        public static void OffsetBottom(ref RectangleF rect, float delta)
        {
            rect.Height += delta;
        }

        /// <summary>
        /// 中心位置を取得。
        /// </summary>
        public static Point GetCenter(Rectangle rect)
        {
            return new Point(
                (rect.Left + rect.Right)  / 2,
                (rect.Top  + rect.Bottom) / 2
            );
        }

        /// <summary>
        /// 中心位置を取得。
        /// </summary>
        public static PointF GetCenter(RectangleF rect)
        {
            return new PointF(
                (rect.Left + rect.Right)  / 2.0f,
                (rect.Top  + rect.Bottom) / 2.0f
            );
        }

        /// <summary>
        /// 左上位置を取得。
        /// </summary>
        public static Point GetTopLeft(Rectangle rect)
        {
            return new Point(rect.Left, rect.Top);
        }

        /// <summary>
        /// 左上位置を取得。
        /// </summary>
        public static PointF GetTopLeft(RectangleF rect)
        {
            return new PointF(rect.Left, rect.Top);
        }

        /// <summary>
        /// 右上位置を取得。
        /// </summary>
        public static Point GetTopRight(Rectangle rect)
        {
            return new Point(rect.Right, rect.Top);
        }

        /// <summary>
        /// 右上位置を取得。
        /// </summary>
        public static PointF GetTopRight(RectangleF rect)
        {
            return new PointF(rect.Right, rect.Top);
        }

        /// <summary>
        /// 左下位置を取得。
        /// </summary>
        public static Point GetBottomLeft(Rectangle rect)
        {
            return new Point(rect.Left, rect.Bottom);
        }

        /// <summary>
        /// 左下位置を取得。
        /// </summary>
        public static PointF GetBottomLeft(RectangleF rect)
        {
            return new PointF(rect.Left, rect.Bottom);
        }

        /// <summary>
        /// 右下位置を取得。
        /// </summary>
        public static Point GetBottomRight(Rectangle rect)
        {
            return new Point(rect.Right, rect.Bottom);
        }

        /// <summary>
        /// 右下位置を取得。
        /// </summary>
        public static PointF GetBottomRight(RectangleF rect)
        {
            return new PointF(rect.Right, rect.Bottom);
        }

        /// <summary>
        /// ２点から作成。
        /// </summary>
        public static Rectangle FromPoints(Point pt1, Point pt2)
        {
            return new Rectangle(
                Math.Min(pt1.X, pt2.X),
                Math.Min(pt1.Y, pt2.Y),
                Math.Abs(pt2.X - pt1.X),
                Math.Abs(pt2.Y - pt1.Y)
            );
        }

        /// <summary>
        /// ２点から作成。
        /// </summary>
        public static RectangleF FromPoints(PointF pt1, PointF pt2)
        {
            return new RectangleF(
                Math.Min(pt1.X, pt2.X),
                Math.Min(pt1.Y, pt2.Y),
                Math.Abs(pt2.X - pt1.X),
                Math.Abs(pt2.Y - pt1.Y)
            );
        }

        /// <summary>
        /// RECT型に変換。
        /// </summary>
        public static Win32.RECT ToRECT(Rectangle rect)
        {
            Win32.RECT result = new Win32.RECT();
            result.left   = rect.Left;
            result.top    = rect.Top;
            result.right  = rect.Right;
            result.bottom = rect.Bottom;
            return result;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private RectangleUtil() {}
    }
    #endregion

    #region GraphicsUtil
    /// <summary>
    /// 描画ユーティリティクラス。
    /// </summary>
    public sealed class GraphicsUtil
    {
        /// <summary>
        /// 四角形を描画。
        /// </summary>
        public static void DrawRectangle(Graphics g, Pen pen, Rectangle rect)
        {
            g.DrawRectangle(pen, rect.X, rect.Y, rect.Width - 1, rect.Height - 1);
        }

        /// <summary>
        /// 四角形を描画。
        /// </summary>
        public static void DrawRectangle(Graphics g, Pen pen, RectangleF rect)
        {
            g.DrawRectangle(pen, rect.X, rect.Y, rect.Width - 1, rect.Height - 1);
        }

        /// <summary>
        /// 丸みのある四角形を描画。
        /// </summary>
        public static void DrawRoundishRectangle(Graphics g, Pen pen, RectangleF rect)
        {
            rect.Width  -= 1;
            rect.Height -= 1;

            g.DrawLine(pen, rect.Left  + 1, rect.Top,     rect.Right - 1, rect.Top);
            g.DrawLine(pen, rect.Right,     rect.Top + 1, rect.Right,     rect.Bottom - 1);
            g.DrawLine(pen, rect.Left,      rect.Top + 1, rect.Left,      rect.Bottom - 1);
            g.DrawLine(pen, rect.Left  + 1, rect.Bottom,  rect.Right - 1, rect.Bottom);
        }

        /// <summary>
        /// ３Ｄ四角形を描画。
        /// </summary>
        public static void Draw3DRectangle(Graphics g, Pen penLT, Pen penRB, RectangleF rect)
        {
            rect.Width  -= 1;
            rect.Height -= 1;

            g.DrawLine(penLT, rect.Left,  rect.Top,     rect.Right, rect.Top);
            g.DrawLine(penLT, rect.Left,  rect.Top,     rect.Left,  rect.Bottom - 1);
            g.DrawLine(penRB, rect.Right, rect.Top + 1, rect.Right, rect.Bottom);
            g.DrawLine(penRB, rect.Left,  rect.Bottom,  rect.Right, rect.Bottom);
        }

        /// <summary>
        /// フォーカス四角形を描画。
        /// </summary>
        public static void DrawFocusRectangle(Graphics g, Rectangle rect)
        {
            IntPtr     hdc = g.GetHdc();
            Win32.RECT rc  = RectangleUtil.ToRECT(rect);
            Win32.User32.DrawFocusRect(hdc, ref rc);
            g.ReleaseHdc(hdc);
        }

        /// <summary>
        /// サイズ変更グリップを描画。
        /// </summary>
        public static void DrawSizeGrip(Graphics g, Control control)
        {
            // Form.OnPaint()で実装されているグリップ描画と同じ方法
            Size szClient = control.ClientSize;
            ControlPaint.DrawSizeGrip(g, control.BackColor, szClient.Width - 0x10, szClient.Height - 0x10, 0x10, 0x10);
        }

        //---------------------------------------------------------------------
        // テキスト
        //---------------------------------------------------------------------
        /// <summary>
        /// 矩形領域にテキストを描画。
        /// </summary>
        public static void DrawText(Graphics g, string text, Font font, Color color, Rectangle rect, HorizontalAlignment alignment)
        {
            DrawText(g, text, font, color, rect, alignment, null);
        }

        /// <summary>
        /// 矩形領域にテキスト（＋イメージ）を描画。
        /// <remarks>
        /// 領域に収まるようにスケーリングして描画します。
        /// </remarks>
        /// </summary>
        public static void DrawText(Graphics g, string text, Font font, Color color, Rectangle rect, HorizontalAlignment alignment, Image image)
        {
            // イメージ
            if (image != null)
            {
                Rectangle rcImage = rect;
                rcImage.X += 2;
                rcImage.Y += (Math.Max(rect.Height - image.Height, 0)) / 2;
                double ratio = Math.Min(1.0, Math.Min((double)rect.Width / image.Width, (double)rect.Height / image.Height));
                rcImage.Size = new Size((int)Math.Round(ratio*image.Width), (int)Math.Round(ratio*image.Height));
                g.DrawImage(image, rcImage);

                // テキスト描画用にオフセット
                RectangleUtil.OffsetLeft(ref rect, rcImage.Right - rect.Left + 1);
            }

            // テキスト
            using (StringFormat sf = new StringFormat())
            {
                sf.LineAlignment = StringAlignment.Center;
                sf.Trimming = StringTrimming.None;
                sf.FormatFlags |= StringFormatFlags.NoWrap;

                switch (alignment)
                {
                    case HorizontalAlignment.Left: sf.Alignment = StringAlignment.Near; break;
                    case HorizontalAlignment.Right: sf.Alignment = StringAlignment.Far; break;
                    case HorizontalAlignment.Center: sf.Alignment = StringAlignment.Center; break;
                    default: break;
                }

                using (Brush brush = new SolidBrush(color))
                {
                    g.DrawString(text, font, brush, rect, sf);
                }
            }
        }

        /// <summary>
        /// 割合率（0.0-1.0）を要素（0-255）に変換。
        /// </summary>
        static byte ToByte_( float ratio )
        {
            if( ratio < 0.0f ) { return 0; }
            else if( ratio > 1.0f ) { return 255; }
            else { return (byte)( ratio * 255.0f ); }
        }

        /// <summary>
        /// 要素（0-255）を割合率（0.0-1.0）に変換。
        /// </summary>
        static float ToRatio_( byte byteVal )
        {
            return (float)byteVal / 255.0f;
        }

        /// <summary>
        /// 混合率を指定してブレンド。
        /// </summary>
        public static Color BlendColors( Color color1, Color color2, float ratio )
        {
            byte r = ToByte_( ToRatio_( color1.R ) * ( 1.0f - ratio ) + ToRatio_( color2.R ) * ratio );
            byte g = ToByte_( ToRatio_( color1.G ) * ( 1.0f - ratio ) + ToRatio_( color2.G ) * ratio );
            byte b = ToByte_( ToRatio_( color1.B ) * ( 1.0f - ratio ) + ToRatio_( color2.B ) * ratio );

            return Color.FromArgb( r, g, b );
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private GraphicsUtil() {}
    }
    #endregion

    #region MathUtil
    /// <summary>
    /// 数学ユーティリティクラス。
    /// </summary>
    public sealed class MathUtil
    {
        static readonly float InvPI  = (float)(1 / Math.PI);
        static readonly float Inv180 = (float)(1 / 180.0f);


        #region デバック文字列生成

        /// <summary>
        ///
        /// </summary>
        public static string ToString( PointF p )
        {
            return "[ " + p.X.ToString( "000.00" ) + ", " + p.Y.ToString( "000.00" ) + " ]";
        }

        /// <summary>
        ///
        /// </summary>
        public static string ToString( Matrix mtx )
        {
            return
                Environment.NewLine +
                mtx.Elements[0].ToString( "000.00" ) + ", " + mtx.Elements[1].ToString( "000.00" ) + Environment.NewLine +
                mtx.Elements[2].ToString( "000.00" ) + ", " + mtx.Elements[3].ToString( "000.00" ) + Environment.NewLine +
                mtx.Elements[4].ToString( "000.00" ) + ", " + mtx.Elements[5].ToString( "000.00" ) + Environment.NewLine;
        }
        #endregion
        /// <summary>
        /// 補間計算。
        /// </summary>
        public static float Interpolate(float min1, float max1, float val1, float min2, float max2)
        {
            return (max2 - min2) * ((val1 - min1) / (max1 - min1)) + min2;
        }

        /// <summary>
        /// 入れ替え。
        /// </summary>
        public static void Swap(ref int val1, ref int val2)
        {
            int temp = val1;
            val1 = val2;
            val2 = temp;
        }

        /// <summary>
        /// 入れ替え。
        /// </summary>
        public static void Swap(ref float val1, ref float val2)
        {
            float temp = val1;
            val1 = val2;
            val2 = temp;
        }

         /// <summary>
         /// 型変換
         /// </summary>
         /// <param name="pos"></param>
         /// <returns></returns>
        public static Point PointFToPoint( PointF pos )
        {
            return new Point( (int)pos.X, (int)pos.Y );
        }

        /// <summary>
        /// 型変換
        /// </summary>
        /// <param name="pos"></param>
        /// <returns></returns>
        public static PointF PointToPointF( Point pos )
        {
            return new PointF( pos.X, pos.Y );
        }

        /// <summary>
        /// 型変換
        /// </summary>
        public static Size SizeFToSize( SizeF size )
        {
            return new Size( (int)size.Width, (int)size.Height );
        }

        /// <summary>
        /// 型変換
        /// </summary>
        public static Rectangle RectangleFToRectangle( RectangleF rect )
        {
            return new Rectangle( PointFToPoint( rect.Location ), SizeFToSize( rect.Size ) );
        }

        /// <summary>
        /// RectangleF の中心を計算します。
        /// </summary>
        public static PointF RectangleCenter( RectangleF rect )
        {
            return new PointF(
                ( rect.Left + rect.Right ) * 0.5f,
                ( rect.Top + rect.Bottom ) * 0.5f );
        }


        /// <summary>
        /// ベクトルの長さ
        /// </summary>
        public static float GetVecLength( Point p )
        {
            return (float)Math.Sqrt( p.X * p.X + p.Y * p.Y );
        }

        /// <summary>
        /// ベクトルの長さ
        /// </summary>
        public static float GetVecLength( PointF p )
        {
            return (float)Math.Sqrt( p.X * p.X + p.Y * p.Y );
        }

        /// <summary>
        /// ベクトルの長さ
        /// </summary>
        public static float GetVecLength(FVec3 p)
        {
            return (float)Math.Sqrt(p.X * p.X + p.Y * p.Y + p.Z * p.Z);
        }

        /// <summary>
        /// ベクトルの正規化
        /// </summary>
        public static PointF NormalizeVec( Point p )
        {
            return NormalizeVec( PointToPointF( p ) );
        }

        /// <summary>
        /// ベクトルの正規化
        /// </summary>
        public static PointF NormalizeVec( PointF p )
        {
            float len = GetVecLength( p );
            if( len >= float.Epsilon )
            {
                return new PointF( p.X / len, p.Y / len );
            }
            else
            {
                return new PointF( p.X, p.Y );
            }
        }

        /// <summary>
        /// 2つのベクトルが等しいか判定します。
        /// </summary>
        public static bool CheckVecsSame( PointF v0, PointF v1 )
        {
            return v0.X == v1.X && v0.Y == v1.Y;
        }

        /// <summary>
        /// 2つのベクトルがほぼ等しいか判定します。
        /// </summary>
        public static bool CheckVecsNearlySame( PointF v0, PointF v1 )
        {
             PointF vD = SubVec( v0, v1 );

            return Math.Abs( vD.X ) < float.Epsilon && Math.Abs( vD.Y ) < float.Epsilon;
        }

         /// <summary>
         /// アークタンジェント：不正な入力時には、ゼロを返します。
         /// </summary>
        public static float SafeAcos( float cos )
        {
            cos = MathUtil.Clamp( cos, -1.0f, 1.0f );

            float rad = (float)Math.Acos( cos );

            return float.IsNaN( rad ) ? 0.0f : rad;
        }

        /// <summary>
        /// アークタンジェント：不正な入力時には、ゼロを返します。
        /// </summary>
        public static float SafeAtan( float tan )
        {
            Debug.Assert(
                !float.IsNaN( tan ) &&
                !float.IsNegativeInfinity( tan ) &&
                !float.IsPositiveInfinity( tan ) );

            float rad = (float)Math.Acos( tan );

            return float.IsNaN( rad ) ? 0.0f : rad;
        }

        /// <summary>
        /// p0 と p1 の角度を計算します。
        /// </summary>
        /// <returns></returns>
        public static float CalcAngleBtweenLines( PointF p0, PointF p1 )
        {
            if( CheckVecsNearlySame( p0, p1 ) )
            {
                return 0.0f;
            }

            float length1 = GetVecLength( p1 );

            // 計算できないときは、ゼロ度とする
            if( length1 < float.Epsilon )
            {
                return 0;
            }

            float cos = p1.X / GetVecLength( p1 );
            // line_pk_p0 と line_pk_p1 の角度を計算します。
            float diffRot = MathUtil.RadToDeg( SafeAcos( cos ) );

            // p0 と直交する直線を求めます。
            // p0 が X軸と平行である場合は、Y軸基底ベクタをvPenに使用します。
            PointF       vPen =
                ( Math.Abs( p0.Y ) > float.Epsilon ) ? new PointF( 1.0f, p0.X / -p0.Y ) : new PointF( 0.0f, Math.Sign( p0.X ) * 1.0f );


            // 直交する直線への投影成分の正負を調べます。
            // p0 の Y 成分の符号によって、判定結果を反転させる必要があるため、乗算しています。
            if( MathUtil.DotProduct( vPen, p1 ) < 0.0f )
            {
                diffRot = -diffRot;
            }

            return diffRot;
        }

        public static float CalcAngleBtweenLines2( PointF p0, PointF p1 )
        {
            if( CheckVecsNearlySame( p0, p1 ) )
            {
                return 0.0f;
            }

            // 計算できないときは、ゼロ度とする
            if( GetVecLength( p0 ) < float.Epsilon ||
                GetVecLength( p1 ) < float.Epsilon )
            {
                return 0;
            }

            float cos = MathUtil.DotProduct( p0, p1 ) / ( GetVecLength( p0 ) * GetVecLength( p1 ) );
            // line_pk_p0 と line_pk_p1 の角度を計算します。
            float diffRot = MathUtil.RadToDeg( SafeAcos( cos ) );

            // p0 と直交する直線を求めます。
            // p0 が X軸と平行である場合は、Y軸基底ベクタをvPenに使用します。
            PointF vPen =
                ( Math.Abs( p0.Y ) > float.Epsilon ) ? new PointF( 1.0f, p0.X / -p0.Y ) : new PointF( 0.0f, Math.Sign( p0.X ) * 1.0f );


            // 直交する直線への投影成分の正負を調べます。
            // p0 の Y 成分の符号によって、判定結果を反転させる必要があるため、乗算しています。
            if( MathUtil.DotProduct( vPen, p1 ) * Math.Sign( p0.Y ) < 0.0f )
            {
                diffRot = -diffRot;
            }

            return diffRot;
        }


        /// <summary>
        /// ベクトル加算
        /// </summary>
        public static Point AddVec( Point p0, Point p1 )
        {
            return new Point( p0.X + p1.X , p0.Y + p1.Y );
        }

        public static PointF AddVec( PointF p0, PointF p1 )
        {
            return new PointF( p0.X + p1.X , p0.Y + p1.Y );
        }

        public static Size AddVec( Size p0, Size p1 )
        {
            return new Size( p0.Width + p1.Width, p0.Height + p1.Height );
        }


        /// <summary>
        /// ベクトル減算
        /// </summary>
        public static Point SubVec( Point p0, Point p1 )
        {
            return new Point( p0.X - p1.X , p0.Y - p1.Y );
        }
        public static PointF SubVec( PointF p0, PointF p1 )
        {
            return new PointF( p0.X - p1.X , p0.Y - p1.Y );
        }
        public static Size SubVec( Size p0, Size p1 )
        {
            return new Size( p0.Width - p1.Width, p0.Height - p1.Height );
        }

        /// <summary>
        /// ベクトルのスケール
        /// </summary>
        public static PointF ScaleVec( PointF p0, float s )
        {
            return new PointF( p0.X * s, p0.Y * s);
        }

        #region 行列関連
        /// <summary>
        /// 行列が規定する空間の縦横費を取得します。(sy/sx)
        /// </summary>
        /// <param name="mtx"></param>
        public static float GetMtxAspectRatio( Matrix mtx )
        {
            return mtx.Elements[3] / mtx.Elements[0];
        }

        /// <summary>
        /// 位置ベクトルに対する行列変換
        /// </summary>
        /// <param name="mtx"></param>
        /// <param name="vec"></param>
        /// <returns></returns>
        public static PointF MtxTransformPoint( Matrix mtx, PointF point )
        {
            PointF[] temp = new PointF[1];
            temp[0] = point;
            mtx.TransformPoints( temp );
            return temp[0];
        }

        /// <summary>
        /// 方向ベクトルに対する行列変換
        /// </summary>
        /// <param name="mtx"></param>
        /// <param name="vec"></param>
        /// <returns></returns>
        public static PointF MtxTransformVec( Matrix mtx, PointF vec )
        {
            PointF[] temp = new PointF[1];
            temp[0] = vec;
            mtx.TransformVectors( temp );
            return temp[0];
        }

        /// <summary>
        /// 方向ベクトルに対する行列変換
        /// </summary>
        /// <param name="mtx"></param>
        /// <param name="vec"></param>
        /// <returns></returns>
        public static FVec3 MtxTransformVec( Matrix mtx, FVec3 vec )
        {
            PointF    result = MtxTransformVec( mtx, new PointF( vec.X, vec.Y ) );
            return new FVec3( result.X, result.Y, vec.Z );
        }

        /// <summary>
        /// ある点を変換原点とした、スケール変換行列を計算します。
        /// </summary>
        public static Matrix GetScaleMtxAt( PointF vS, PointF vO )
        {
            Matrix mtxS = new Matrix();
            Matrix mtxIT = new Matrix();

            mtxS.Translate( vO.X, vO.Y );
            mtxS.Scale( vS.X, vS.Y );
            mtxIT.Translate( -vO.X, -vO.Y );

            mtxS.Multiply( mtxIT );

            return mtxS;
        }

        /// <summary>
        /// ある点を変換原点とした、スケール変換行列を計算します。
        /// </summary>
        public static Matrix GetScaleMtxAt(double vSx, double vSy, double vOx, double vOy)
        {
            Matrix mtxS = new Matrix();

            // 誤差を抑えるために double のまま計算する。
            mtxS.Translate((float)(-vSx * vOx + vOx), (float)(-vSy * vOy + vOy));

            mtxS.Scale((float)vSx, (float)vSy);
            return mtxS;
        }
        #endregion

        /// <summary>
        /// 内積計算
        /// </summary>
        public static float DotProduct( Point p0, Point p1 )
        {
            return DotProduct( PointToPointF( p0 ), PointToPointF( p1 ) );
        }

        /// <summary>
        /// 内積計算
        /// </summary>
        public static float DotProduct( PointF p0, PointF p1 )
        {
            return p0.X * p1.X + p0.Y * p1.Y;
        }

        /// <summary>
        /// 内積計算
        /// </summary>
        public static float DotProduct(FVec3 p0, FVec3 p1)
        {
            return p0.X * p1.X + p0.Y * p1.Y + p0.Z * p1.Z;
        }

        /// <summary>
        /// 外積計算
        /// </summary>
        public static PointF CrossProduct(PointF p0, PointF p1)
        {
            return new PointF(p0.X * p1.Y, - p0.Y * p1.X);
        }

        /// <summary>
        /// 外積計算
        /// </summary>
        public static FVec3 CrossProduct(FVec3 p0, FVec3 p1)
        {
            return new FVec3(
                p0.Y * p1.Z - p0.Z * p1.Y,
                p0.Z * p1.X - p0.X * p1.Z,
                p0.X * p1.Y - p0.Y * p1.Z);
        }

        /// <summary>
        /// ラジアン => 度 変換
        /// </summary>
        public static float RadToDeg( float radian )
        {
            return radian * 180.0f * InvPI;
        }

        /// <summary>
        /// 度 => ラジアン 変換
        /// </summary>
        public static float DegToRad( float deg )
        {
            return (float)(deg * Inv180 * Math.PI);
        }

        /// <summary>
        /// ラジアン => 度 変換
        /// </summary>
        public static FVec3 RadToDeg(FVec3 radVec)
        {
            return new FVec3(RadToDeg(radVec.X), RadToDeg(radVec.Y), RadToDeg(radVec.Z));
        }

        /// <summary>
        /// 指定した値の、値よりも小さい境界値に整列した値を取得します。
        /// </summary>
        /// <param name="src"></param>
        /// <param name="boundary"></param>
        /// <returns></returns>
        public static float AlignValueOnTheLowerBoundary( float src, float boundary )
        {
            // 値が正ならば...
            if( src > 0 )
            {
                return src - ( src % boundary);
            }
            else
            {
                return src - ( boundary - Math.Abs((src % boundary)) );
            }
        }

        /// <summary>
        /// 指定した値の、値よりも大きい境界値に整列した値を取得します。
        /// </summary>
        /// <param name="src"></param>
        /// <param name="boundary"></param>
        /// <returns></returns>
        public static float AlignValueOnTheHigherBoundary( float src, float boundary )
        {
            // 値が正ならば...
            if( src > 0 )
            {
                return src + ( boundary - Math.Abs((src % boundary)) );
            }
            else
            {
                return src + ( Math.Abs((src % boundary)) );
            }
        }

        /// <summary>
        /// サイズが正の値を持つ、矩形を生成して返します。
        /// </summary>
        /// <returns></returns>
        public static RectangleF MakePositiveSizeRectangle( RectangleF rect )
        {
            RectangleF       newRect = rect;

            if( rect.Width < 0.0f )
            {
                newRect.X       = rect.X + rect.Width;
                newRect.Width   = -rect.Width;
            }

            if( rect.Height < 0.0f )
            {
                newRect.Y       = rect.Y + rect.Height;
                newRect.Height  = -rect.Height;
            }

            return newRect;
        }

        /// <summary>
        /// 値をmin, max でクランプした値を返します。
        /// </summary>
        public static float Clamp( float val, float min, float max )
        {
            if( val <= min )
            {
                val = min;
            }

            if( val >= max )
            {
                val = max;
            }
            return val;
        }

        /// <summary>
        /// int 用
        /// 残念ながら、generics を利用したものは、定義できない。
        /// </summary>
        public static int Clamp( int val, int min, int max )
        {
            if( val <= min )
            {
                val = min;
            }

            if( val >= max )
            {
                val = max;
            }
            return val;
        }

        /// <summary>
        /// フロート値が指定値に近いか判定します。
        /// </summary>
        public static bool CheckValueNear(
            float val,
            float threshold,
            float delta )
        {
            Debug.Assert( Math.Abs( delta ) >= 0.0f );

            return Math.Abs( val - threshold ) < delta;
        }

       /// <summary>
        /// コンストラクタ。
        /// </summary>
        private MathUtil() {}
    }
    #endregion
}
