﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Text.RegularExpressions;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Linq;

namespace LECore.Structures
{
    using NW4C.Tga;
    using LECore.Structures.Nsrif.Attributes;

    internal delegate void OnTextureMgrUpdateHandler();

    /// <summary>
    /// テクスチャマネージャ変更通知を受け取るクラスが実装するインタフェースです。
    /// </summary>
    internal interface ITextureMgrUpdateListener
    {
        /// <summary>
        /// デリゲータ OnTextureMgrUpdateHandler を取得します。
        /// </summary>
        OnTextureMgrUpdateHandler OnTextureMgrUpdateHandler { get;}
    }

    /// <summary>
    /// テクスチャマネージャの外部インタフェース
    /// </summary>
    public interface ITextureMgr
    {
        // メンバ参照
        IEnumerable<ITextureImage> ITextureImageSet { get; }

        // 自分自身の属しているサブシーン
        ISubScene ISubScene { get; }

        // 検索
        ITextureImage FindITextureImageByName( string textureName );

        // 読み取り専用か？
        bool CheckReadOnlyByName(string textureName);
    }

    /// <summary>
    /// シーン中のテクスチャリソースを管理するモジュール。
    /// 内部に複数のテクスチャサブセットを管理しています。
    ///
    /// 過去のバージョンではシングルトンクラスでした。
    /// 現在は、サブシーン(Rlytファイル)ごとにインスタンスが生成されます。
    /// </summary>
    internal class TextureMgr :
        ITextureMgr,
        IDisposable
    {
        #region フィールド
        static bool _bUpdating = false;
        static bool _bEventCalceled = false;

        ISubScene _ownerSubScene = null;

        List<ITextureImage> _textureSet = new List<ITextureImage>();
        List<ITextureImage> _readOnlyTextureSet = new List<ITextureImage>();
        // TextureSubSet
        ArrayList _textureSubsets = new ArrayList();

        // メンバテクスチャ更新イベントハンドラ
        readonly EventHandler _memberTextureChangedHandler;
        #endregion フィールド

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextureMgr(ISubScene subScene)
        {
            // メンバテクスチャ更新イベントハンドラを初期化します。
            _memberTextureChangedHandler = new EventHandler( Event_TexImage_OnChanged );

            _ownerSubScene = subScene;
        }

        /// <summary>
        /// マネージャ更新イベントを通知します。
        /// </summary>
        void NotifyTextureMgrUpdateEvent_()
        {
            // 更新中ではなく...
            if (!_bUpdating)
            {
                // 通知ハンドラが設定されていれば...
                if (OnTextureMgrUpdate != null)
                {
                    // 変更の通知
                    OnTextureMgrUpdate();
                }
            }
            else
            {
                _bEventCalceled = true;
            }
        }

        /// <summary>
        /// 更新イベントを強制的に起こす
        /// </summary>
        public void RaiseChangeEvent()
        {
            NotifyTextureMgrUpdateEvent_();
        }

        /// <summary>
        /// マネージャの内容を更新開始します。
        /// EndUpdate()が呼ばれるまで、メッセージの通知が抑制されます。
        /// </summary>
        public void BeginUpdate()
        {
            Debug.Assert( _bUpdating == false );
            _bUpdating = true;
            _bEventCalceled = false;
        }

        /// <summary>
        /// マネージャの内容を更新終了します。
        /// メッセージの通知が行われます。
        /// </summary>
        public void EndUpdate()
        {
            Debug.Assert( _bUpdating == true );
            _bUpdating = false;

            // イベント通知がキャンセルされた場合はイベント通知を行う。
            if (_bEventCalceled)
            {
                NotifyTextureMgrUpdateEvent_();
                _bEventCalceled = false;
            }
        }

        /// <summary>
        /// 新しいテクスチャサブセットを作成します。
        ///
        /// 同等の内容を持つサブセットが存在した場合は、新規作成を行わず、
        /// 既存のサブセットを返します。
        /// </summary>
        public TextureSubSet MakeNewTextureSubSet( string[] textureNameSet )
        {
            // 既存のサブセットを調査し、
            // 等値なサブセットが発見されたのならばそれを返します。
            foreach( TextureSubSet textureSubSet in _textureSubsets )
            {
                if( textureSubSet.IsSameSet( textureNameSet ) )
                {
                    return textureSubSet;
                }
            }

            // 新規作成し登録します。
            TextureSubSet newSubSet = new TextureSubSet();
            newSubSet.AddTextures( textureNameSet );
            _textureSubsets.Add( newSubSet );
            return newSubSet;
        }


        /// <summary>
        ///
        /// </summary>
        string GetFileNameWithExt_( string fileName )
        {
            if( Path.GetExtension( fileName ) == string.Empty )
            {
                fileName = fileName + ".tga";
            }
            return fileName;
        }

        /// <summary>
        /// テクスチャを新規に登録します。
        /// </summary>
        void RegisterNewTexture_( ITextureImage texImage )
        {
            Debug.Assert( texImage != null );
            Debug.Assert( !_textureSet.Contains( texImage ) );

            // 変更イベントハンドラの設定。
            texImage.OnChanged += _memberTextureChangedHandler;
            // 登録
            _textureSet.Add( texImage );

            // 変更の通知
            NotifyTextureMgrUpdateEvent_();
        }

        /// <summary>
        /// ファイルから、TextureImage インスタンスを生成します。
        /// 同一ファイルに対する、呼び出しがあった場合には、
        /// 以前の結果が返されます。
        /// </summary>
        public ITextureImage RegisterTextureImageFromFile( string filePath)
        {
            bool registered;
            return RegisterTextureImageFromFile(filePath, out registered);
        }

        /// <summary>
        /// ファイルから、TextureImage インスタンスを生成します。
        /// 同一ファイルに対する、呼び出しがあった場合には、
        /// 以前の結果が返されます。
        /// registered は新規に登録されたかを示します。
        /// </summary>
        public ITextureImage RegisterTextureImageFromFile( string filePath, out bool registered)
        {
            Debug.Assert( _bUpdating == true );

            // filePath = GetFileNameWithExt_( filePath );
            string fileName = Path.GetFileNameWithoutExtension( filePath );

            // 以前に登録されていなければ...
            ITextureImage texImage = FindTextureImageByName( fileName );

            registered = false;
            if( texImage == null )
            {
                texImage = TextureImageFactory.BuildTextureImageFromTga(this, filePath );

                // 読み込みに失敗してもインスタンスはできるはず
                Debug.Assert(texImage != null);

                if( texImage != null )
                {
                    // 登録します。
                    RegisterNewTexture_( texImage );

                    registered = true;
                }
            }

            return texImage;
        }

        /// <summary>
        /// 多数のファイルの登録
        /// </summary>
        public ITextureImage[]
            RegisterITextureImageSetFromFile( string[] filePathSet)
        {
            List<ITextureImage> textureImageSet = new List<ITextureImage>();

            BeginUpdate();
            foreach( string filePath in filePathSet )
            {
                bool registered;
                var image = RegisterTextureImageFromFile(filePath, out registered);
                if (registered)
                {
                    textureImageSet.Add(image);
                }
            }
            EndUpdate();

            return textureImageSet.ToArray();
        }

        /// <summary>
        /// ファイルから、TextureImage インスタンスを生成します。
        /// 生成結果の、マネージャへの登録をしません。、
        /// </summary>
        public ITextureImage CreateITextureImageFromFileWithoutRegistering( string filePath )
        {
            Debug.Assert( _bUpdating == true );
            ITextureImage texImage = null;
            if( File.Exists( filePath ) )
            {
                texImage = TextureImageFactory.BuildTextureImageFromTga(this, filePath);
            }
            return texImage;
        }

        /// <summary>
        /// 使用方法を設定してキャプチャテクスチャを作成します。使用方法に応じて指定した名前に接尾辞が付加されます。
        /// </summary>
        public ITextureImage RegisterCaptureTexture(IPane pane, string textureNameBase, bool isHidingFromList, CaptureTextureUsage usage)
        {
            string extendedName = textureNameBase + CaptureTexture.GetNameSuffix(usage);

            // 以前に登録されていなければ...
            ITextureImage texImage = FindTextureImageByName(extendedName);
            if (texImage == null)
            {
                CaptureTexture newTex = new CaptureTexture(pane, isHidingFromList, usage);
                if (newTex != null)
                {
                    newTex.LastReferencedPaneName = textureNameBase;
                    // 登録します。
                    RegisterNewTexture_(newTex);
                }
                texImage = newTex;
            }

            return texImage;
        }

        /// <summary>
        /// </summary>
        public ITextureImage RegisterCaptureTexture(IPane pane, string textureName, bool isHidingFromList)
        {
            return RegisterCaptureTexture(pane, textureName, isHidingFromList, CaptureTextureUsage.Normal);
        }

        /// <summary>
        /// テクスチャを削除します。
        /// </summary>
        public void RemoveTexture( ITextureImage texImage )
        {
            Debug.Assert( _bUpdating == true );
            Debug.Assert( texImage != null );
            Debug.Assert( _textureSet.Contains( texImage ) );

            // 変更イベントハンドラの設定。
            texImage.OnChanged -= _memberTextureChangedHandler;

            SetReadOnlyState(texImage, false);
            _textureSet.Remove( texImage );

            if (texImage.SourceType == LECoreInterface.TextureSourceType.File)
            {
                TextureImageFactory.DisposeTexture(this, texImage);
            }

            // 変更の通知
            NotifyTextureMgrUpdateEvent_();
        }

        /// <summary>
        /// テクスチャが読み取り専用か調べます。
        /// </summary>
        public bool CheckReadOnlyByName(string textureName)
        {
            var tex = FindTextureImageByName(textureName);
            return (tex == null) ? false : _readOnlyTextureSet.Contains(tex);
        }

        /// <summary>
        /// テクスチャの読みとり専用状態を設定します。
        /// <remarks>
        /// ユーザのUI 操作によって実行されないメソッドなの
        /// で、UIへの変更通知(NotifyTextureMgrUpdateEvent_() )をしていません。
        /// </remarks>
        /// </summary>
        public void SetReadOnlyState(ITextureImage texture, bool isReadonly)
        {
            if (!this._textureSet.Contains(texture))
            {
                return;
            }

            switch (texture.SourceType)
            {
                case LECoreInterface.TextureSourceType.File:
                    {
                        TextureImage tex = texture as TextureImage;
                        tex.PixelFmtIsFixed = isReadonly;
                    }
                    break;
                case LECoreInterface.TextureSourceType.Dynamic:
                    {
                        CaptureTexture tex = texture as CaptureTexture;
                        tex.PixelFmtIsFixed = isReadonly;
                    }
                    break;

            }

            if (isReadonly)
            {
                if (!_readOnlyTextureSet.Contains(texture)) { _readOnlyTextureSet.Add(texture); }
            }
            else
            {
                _readOnlyTextureSet.Remove(texture);
            }
        }

        /// <summary>
        /// テクスチャの画像データを最新の状態に更新します。
        /// </summary>
        public bool UpdateTextureImage( ITextureImage iTexImg )
        {
            return UpdateTextureImage( iTexImg, iTexImg.FilePath );
        }

        /// <summary>
        /// テクスチャの画像データを最新の状態に更新します。
        /// </summary>
        public bool UpdateTextureImage( ITextureImage iTexImg, string newFilePath )
        {
            Debug.Assert( _bUpdating == true );
            bool bUpdated = false;
            if( this._textureSet.Contains( iTexImg ) )
            {
                bUpdated = TextureImageFactory.UpdateTextureImage(iTexImg, newFilePath );
                if( bUpdated )
                {
                    NotifyTextureMgrUpdateEvent_();
                }
            }
            return bUpdated;
        }

        /// <summary>
        /// メンバ  TextureImage の更新イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="args"></param>
        void Event_TexImage_OnChanged(object sender, EventArgs args)
        {
            // メンバからの通知であるか確認します。
            Debug.Assert(this._textureSet.Contains(sender as ITextureImage));

            // 変更の通知
            NotifyTextureMgrUpdateEvent_();
        }


        /// <summary>
        /// テクスチャ名、もしくは、テクスチャファイルパスから、
        /// テクスチャ画像を検索します。
        ///
        /// ！！変更 => 拡張子無しのテクスチャ名からのみ、
        /// 検索するように変更になりました。本関数を利用するコードは、
        /// 適切に引数を加工する必要があります。
        ///
        /// 発見できなかった場合は、nullが返ります。
        ///
        /// </summary>
        public ITextureImage FindTextureImageByName( string textureName )
        {
            string textureNameWithoutExt = textureName;

            foreach( ITextureImage textureImage in _textureSet )
            {
                // 拡張子無しの、テクスチャ名でのみ判定するようにしました。
                if (textureImage.IsSameName(textureNameWithoutExt))
                {
                    // 同一名称テクスチャについて、パスが同一か判定します。
                    // 異なる場合は、警告を表示する必要があります。

                    return textureImage;
                }
            }

            return null;
        }


        #region ITextureMgr メンバ

        public event OnTextureMgrUpdateHandler OnTextureMgrUpdate;

        public IEnumerable<ITextureImage> ITextureImageSet
        {
            get { return _textureSet; }
        }

        public ISubScene ISubScene
        {
            get { return _ownerSubScene; }
        }

        public ITextureImage RegisterITextureImageFromFile( string filePath )
        {
            return RegisterTextureImageFromFile( filePath );
        }

        public ITextureImage FindITextureImageByName( string textureName )
        {
            return FindTextureImageByName( textureName );
        }

        #endregion ITextureMgr メンバ

        #region IDisposable メンバ

        /// <summary>
        /// リソースを破棄します。
        /// </summary>
        public void Dispose()
        {
            ITextureImage[] texSet = _textureSet.ToArray();
            BeginUpdate();
            foreach (ITextureImage tex in texSet)
            {
                RemoveTexture(tex);
            }
            EndUpdate();
        }

        #endregion
    }

    /// <summary>
    /// ヘルパクラス
    /// </summary>
    static public class TextureMgrHelper
    {
        /// <summary>
        /// フォント登録を試行します。
        /// </summary>
        internal static void TryRegisterTextureImage(this ITextureMgr mgr, ITextureImage texture)
        {
            if (texture != null && mgr.FindITextureImageByName(texture.Name) == null)
            {
                (mgr as TextureMgr).BeginUpdate();

                switch (texture.SourceType)
                {
                    case LECoreInterface.TextureSourceType.File:
                        ITextureImage newTexture = (mgr as TextureMgr).RegisterITextureImageFromFile(texture.FilePath);
                        newTexture.PixelFmt = texture.PixelFmt;
                        break;
                    case LECoreInterface.TextureSourceType.Dynamic:
                        // キャプチャテクスチャはそれぞれ所属しているレイアウトのテクスチャマネージャのみで管理されているためコピーは不要です。
                        break;
                }

                (mgr as TextureMgr).EndUpdate();
            }
        }

        /// <summary>
        /// 部品ペインの持つテクスチャマネージャの、テクスチャ状態を同期します。
        /// </summary>
        internal static void SyncTextureImageNoEvent(this ITextureMgr mgr, ITextureImage texture)
        {
            if (texture == null)
            {
                return;
            }

            ITextureImage newTexture = mgr.FindITextureImageByName(texture.Name);
            if (newTexture == null)
            {
                switch (texture.SourceType)
                {
                    case LECoreInterface.TextureSourceType.File:
                        newTexture = (mgr as TextureMgr).RegisterITextureImageFromFile(texture.FilePath);
                        newTexture.PixelFmt = texture.PixelFmt;
                        break;
                    case LECoreInterface.TextureSourceType.Dynamic:
                        // キャプチャテクスチャはそれぞれ所属しているレイアウトのテクスチャマネージャのみで管理されているためコピーは不要です。
                        break;
                }
            }
            else
            {
                newTexture.PixelFmt = texture.PixelFmt;
            }
        }

        /// <summary>
        /// 渡されたテクスチャマネージャからキャプチャテクスチャの配列を取得します。
        /// </summary>
        /// <param name="mgr"></param>
        /// <returns></returns>
        public static ITextureImage[] GetCaptureTextureArray(ITextureMgr mgr)
        {
            return mgr.ITextureImageSet.Where(tex => tex.SourceType == LECoreInterface.TextureSourceType.Dynamic).ToArray();
        }

        /// <summary>
        /// 渡されたテクスチャマネージャから指定された名前のキャプチャテクスチャを検索します。
        /// </summary>
        /// <param name="mgr"></param>
        /// <param name="name"></param>
        /// <returns></returns>
        internal static ITextureImage FindCaptureTextureByName(ITextureMgr mgr, string name)
        {
            IEnumerable<ITextureImage> textures = mgr.ITextureImageSet.Where(tex => tex.SourceType == LECoreInterface.TextureSourceType.Dynamic && tex.Name == name);

            if (textures.Count() == 1)
            {
                return textures.First();
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// 一時的に UI で使用するキャプチャテクスチャを作成します。作成されたインスタンスは内部で管理されません。
        /// </summary>
        /// <param name="pane"></param>
        /// <param name="width"></param>
        /// <param name="height"></param>
        /// <returns></returns>
        public static ITextureImage CreateTemporaryCaptureTexture(IPane pane, int width, int height)
        {
            return new CaptureTexture(pane, width, height, false);
        }

        /// <summary>
        /// ペインと使用方法から自動生成されるキャプチャテクスチャの名前を作成します。
        /// </summary>
        /// <param name="paneName">ペインの名前です</param>
        /// <param name="usage">キャプチャテクスチャの使用方法です</param>
        /// <returns></returns>
        public static string CreateAutoGeneratedCaptureTextureName(string paneName, CaptureTextureUsage usage)
        {
            return paneName + CaptureTexture.GetNameSuffix(usage);
        }
    }
}
