﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;
using System.Text.RegularExpressions;

namespace LECore.Structures
{

    using System.Collections.Generic;
    using System.Drawing.Drawing2D;
    using System.Threading;
    using Core;
    using LECore.Structures.Nsrif.Attributes;
    using LECore.Util;
    using NWFont = NW4R.Font;
    using System.IO;

    /// <summary>
    /// TextBox のアニメーションアトリビュート
    /// </summary>
    internal class TextBoxAttribute
        : AnmAttribute
    {
        static readonly AnmAttrDescripter[]  SubAttrDescs =
        {
            new  AnmAttrDescripter( AttributeType.ByteRGBA4 , "TopColor", null, RGBAColor.White ),
            new  AnmAttrDescripter( AttributeType.ByteRGBA4 , "BottomColor", null, RGBAColor.White ),
            new  AnmAttrDescripter( AttributeType.ByteRGBA4 , "ShadowBlackColor", null, RGBAColor.Black ),
            new  AnmAttrDescripter( AttributeType.ByteRGBA4 , "ShadowWhiteColor", null, RGBAColor.White ),
            new  AnmAttrDescripter( AttributeType.Float , "CharTransformTimeOffset", null, 0.0f ),
            new  AnmAttrDescripter( AttributeType.FloatVec2, "CharTransformScale",  null, new FVec2(1.0f, 1.0f), Core.AnmAttrDescripter.AnimationTypeLock.False, true),
            new  AnmAttrDescripter( AttributeType.FloatVec3, "CharTransformTrans",  null, null, Core.AnmAttrDescripter.AnimationTypeLock.False, true),
            new  AnmAttrDescripter( AttributeType.FloatVec3, "CharTransformRotate",  null, null, Core.AnmAttrDescripter.AnimationTypeLock.False, true),
            new  AnmAttrDescripter( AttributeType.ByteRGBA4 , "CharTransformTopColor", null, RGBAColor.White, Core.AnmAttrDescripter.AnimationTypeLock.False, true ),
            new  AnmAttrDescripter( AttributeType.ByteRGBA4 , "CharTransformBottomColor", null, RGBAColor.White, Core.AnmAttrDescripter.AnimationTypeLock.False, true ),
            new  AnmAttrDescripter( AttributeType.Float , "CharTransformTimeWidth", null, 100.0f ),
        };

        static readonly AnmAttrDescripter  SelfAttrDescs =
            new AnmAttrDescripter( AttributeType.Combined, "TextBox", null, null );

        #region ------------- プロパティ -------------

        public AnmAttribute TopColorAnmAttr
        {
            get{ return FindAttributeByIdx( 0 );}
        }

        public AnmAttribute BottomColorAnmAttr
        {
            get{ return FindAttributeByIdx( 1 );}
        }

        public RGBAColor TopColor
        {
            get{ return TopColorAnmAttr.GetAsRBGA();}
            set{ TopColorAnmAttr.SetValue( value );}
        }

        public RGBAColor BottomColor
        {
            get{ return BottomColorAnmAttr.GetAsRBGA();}
            set{ BottomColorAnmAttr.SetValue( value );}
        }

        /// <summary>
        /// 影黒カラー補間
        /// </summary>
        public AnmAttribute ShadowBlackColorAttr
        {
            get { return FindAttributeByIdx(2); }
        }

        /// <summary>
        /// 影白カラー補間
        /// </summary>
        public AnmAttribute ShadowWhiteColorAttr
        {
            get { return FindAttributeByIdx(3); }
        }

        /// <summary>
        ///
        /// </summary>
        public AnmAttribute PerCharTimeOffsetAnmAttr
        {
            get { return FindAttributeByIdx(4); }
        }

        /// <summary>
        ///
        /// </summary>
        public AnmAttribute PerCharTransformSAttr
        {
            get { return FindAttributeByIdx(5); }
        }

        /// <summary>
        ///
        /// </summary>
        public AnmAttribute PerCharTransformTAttr
        {
            get { return FindAttributeByIdx(6); }
        }

        /// <summary>
        ///
        /// </summary>
        public AnmAttribute PerCharTransformRAttr
        {
            get { return FindAttributeByIdx(7); }
        }

        /// <summary>
        ///
        /// </summary>
        public AnmAttribute PerCharTransformTopColorAttr
        {
            get { return FindAttributeByIdx(8); }
        }

        /// <summary>
        ///
        /// </summary>
        public AnmAttribute PerCharTransformBottomColorAttr
        {
            get { return FindAttributeByIdx(9); }
        }

        /// <summary>
        ///
        /// </summary>
        public AnmAttribute PerCharTimeWidthAnmAttr
        {
            get { return FindAttributeByIdx(10); }
        }

        #endregion ------------- プロパティ -------------

        readonly private TimeReleyProvider _timeOffsetEventReleyProvider = new TimeReleyProvider();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextBoxAttribute( LEDataNode owner )
            : base( owner, SubAttrDescs, SelfAttrDescs )
        {
            // PerCharTimeOffsetAnmAttr が変更されたら、_timeOffsetEventReleyProvider を使って時間変更イベントとして通知する。
            this.PerCharTimeOffsetAnmAttr.OnModify += TimeOffsetAnmAttr_OnModify_;

            // PerCharTransform は PerCharTimeOffsetAnmAttr の値をカスタム時間イベントソースとするように設定する。
            {
                foreach (AnmAttribute curveAttr in this.PerCharTransformSAttr.EnumCurveAttribute())
                {
                    curveAttr.InitializeCustomTimeSource(_timeOffsetEventReleyProvider);
                    curveAttr.OnModify += TextBoxAttribute_OnModify_;
                }

                foreach (AnmAttribute curveAttr in this.PerCharTransformTAttr.EnumCurveAttribute())
                {
                    curveAttr.InitializeCustomTimeSource(_timeOffsetEventReleyProvider);
                    curveAttr.OnModify += TextBoxAttribute_OnModify_;
                }

                foreach (AnmAttribute curveAttr in this.PerCharTransformRAttr.EnumCurveAttribute())
                {
                    curveAttr.InitializeCustomTimeSource(_timeOffsetEventReleyProvider);
                    curveAttr.OnModify += TextBoxAttribute_OnModify_;
                }

                foreach (AnmAttribute curveAttr in this.PerCharTransformTopColorAttr.EnumCurveAttribute())
                {
                    curveAttr.InitializeCustomTimeSource(_timeOffsetEventReleyProvider);
                    curveAttr.OnModify += TextBoxAttribute_OnModify_;
                }

                foreach (AnmAttribute curveAttr in this.PerCharTransformBottomColorAttr.EnumCurveAttribute())
                {
                    curveAttr.InitializeCustomTimeSource(_timeOffsetEventReleyProvider);
                    curveAttr.OnModify += TextBoxAttribute_OnModify_;
                }
            }

            // カゲ黒カラー補間のアルファは無効にする
            ShadowBlackColorAttr.FindAttributeByIdx(3).IsActiveAttribute = false;
        }

        /// <summary>
        /// 現在の時間オフセットで文字毎のパラメーターを更新します。
        /// </summary>
        internal void DoUpdatePerCharcterTransformAtCurrentTimeOffset()
        {
            var time = (int)this.PerCharTimeOffsetAnmAttr.GetAsFloat();
            _timeOffsetEventReleyProvider.Reley(time, Core.TimeChageEventType.Tick);
        }

        /// <summary>
        ///
        /// </summary>
        void TimeOffsetAnmAttr_OnModify_(LEDataNode sender, int args)
        {
            DoUpdatePerCharcterTransformAtCurrentTimeOffset();
        }

        /// <summary>
        /// アニメーション変更時に、(ペインの)変更イベントを通知します。
        /// </summary>
        void TextBoxAttribute_OnModify_(LEDataNode sender, int args)
        {
            NotifyModifyEvent(this, (int)EventKind.Modify);
        }
    }

    /// <summary>
    /// TextBox ペインの情報
    /// </summary>
    internal class TextBox :
        LEDataNode,             // 内部データノード機構の要素です。
        IDrawable,              // レンダラによって描画されます
        ITextBox,               // 外部モジュールに公開されるインタフェース
        IVertexColor4Settable,
        IPaneExParamater,       // ペイン拡張パラメータです。
        IRevHWMaterialHolder,   // レボリューション固有のマテリアル情報を保持します
        IDisposable
    {
        public const string TextBoxNodeName = "TextBox";

        #region ---------------- 内部クラス宣言 ----------------
        /// <summary>
        /// TextBox を描画する際に使用されるモジュール。
        /// TextBox内部で使用される関数です。
        /// </summary>
        class TextBoxRenderer :
            NWFont.INW4RTextRenderer
        {

           #region ---------------- 内部クラス宣言 ----------------
            class LetterMotion
            {
                public FVec2 Scale;
                public FVec3 Rotate;
                public FVec3 Trans;
                public byte? TopR;
                public byte? TopG;
                public byte? TopB;
                public byte? TopA;
                public byte? BtmR;
                public byte? BtmG;
                public byte? BtmB;
                public byte? BtmA;

                public LetterMotion()
                {
                    this.Scale = new FVec2();
                    this.Rotate = new FVec3();
                    this.Trans = new FVec3();
                }
            }

            /// <summary>
            /// 描画パラメータ
            /// </summary>
            class RenderedTextBoxCache
            {
                public string         FormedString;
                public FVec2          OffsetPos;
                public LetterMotion[] LetterMotions;
                public VerticalLocation LetterMotionsOriginV = VerticalLocation.Center;
                public float LetterMotionsOriginVOffset;

                /// <summary>
                /// パラメータの内容が妥当か判定します。(TODO:)
                /// </summary>
                public bool IsObsoleteFor_( TextBox currentTextBox )
                {
                    return true;
                }
            }
            #endregion ---------------- 内部クラス宣言 ----------------

            NWFont.NW4RTextWriter	_textWriter  = null;
            RevHWMaterial           _textMaterial = null;
            ILEFont                  _currentFont = null;
            IRenderer               _renderer    = null;
            RenderedTextBoxCache    _renderedTextBoxCache = new RenderedTextBoxCache();
            Color[]					 _vtxColorSet = new Color[4];

            bool _shadowEnabled = false;
            bool _autoShadowAlpha = false;
            bool _originToCenter = false; // 各文字の原点をテキストボックスの中心にする
            float _textBoxWidthForOriginToCenter = 0.0f;
            float _textBoxHeightForOriginToCenter = 0.0f;
            FVec2 _shadowScale = new FVec2();

            float _italicOffset = 0.0f;

            bool  _isPackedFont;
            bool  _isPackedFontBorderDisabled;

            bool _isBorderEffectEnabled;

            /// <summary>
            /// フォントを取得します。
            /// </summary>
            public ILEFont CurrentFont
            {
                get{ return _currentFont;}
                set{ _currentFont = value;}
            }

            /// <summary>
            /// タグプロセッサをリセットします。
            /// </summary>
            public void ResetTagProcessor(bool isExtendedTagEnabled)
            {
                _textWriter.ResetTagProcessor(isExtendedTagEnabled);
            }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public TextBoxRenderer()
            {
                _textWriter = new NWFont.NW4RTextWriter( this );
            }

            #region NWFont.INW4RTextRenderer メンバ

            float CalcLineMaxWidth(float maxWidth, float[] lineWidth, int line)
            {
                float lineMaxWidth = maxWidth;
                if (lineWidth != null && line < lineWidth.Length)
                {
                    lineMaxWidth += lineWidth[line];
                }
                if (lineMaxWidth < 0.0f)
                {
                    lineMaxWidth = 0.0f;
                }
                return lineMaxWidth;
            }

            /// <summary>
            /// 指定したピクセル幅以内になるように適切に、
            /// 改行を挿入した文字列を返します。
            /// </summary>
            /// <param name="srcStr"></param>
            /// <param name="maxWidth"></param>
            /// <returns></returns>
            string GetSuitableLengthString_( string srcStr, float maxWidth, float[] lineWidth )
            {
                int    length = 0;
                string resultStr = String.Empty;
                string subStr    = srcStr;
                int line = 0;

                while( length <= subStr.Length )
                {
                    float strW = _textWriter.CalcStringWidth( subStr, length );

                    // 指定された長さよりも長い行が発見された
                    if (strW > CalcLineMaxWidth(maxWidth, lineWidth, line))
                    {
                        // 改行コードを挿入して、もう一度スタートからやり直す。
                        int   okLength = length - 1;
                        if( okLength <= 0 )
                        {
                            if (lineWidth != null && line < lineWidth.Length)
                            {
                                // 行ごとの幅とオフセットが設定されているときに
                                // 1文字でペインからはみ出してしまう場合、
                                // 1文字も含まずに改行を挿入します。
                                okLength = 0;
                            }
                            else
                            {
                                // 1文字でペインからはみ出してしまう場合、
                                // 最低でも一文字づつ改行を挿入していきます。
                                okLength = 1;
                            }
                        }

                        if (okLength == 0)
                        {
                            resultStr += "\n";
                        }
                        else
                        {
                            // 適切な長さ分の文字列を resultStr に追加します。
                            resultStr += subStr.Substring(0, okLength) + "\n";
                            // 後半の未調査部分 を新たな subStr とします。
                            subStr = subStr.Substring(okLength, subStr.Length - okLength);
                        }
                        // subStr の先頭から調査再開。
                        length     = 0;

                        line++;
                    }
                    else
                    {
                        length++;
                    }
                }

                return resultStr + subStr;
            }

            /// <summary>
            /// 文字列パラメータのアンカーポイント設定を解釈し、
            /// フォントライブラリに設定を反映します。
            /// また、描画の際に使用する、位置オフセット量も計算します。
            /// </summary>
            /// <param name="lt"></param>
            /// <param name="offset"></param>
            private FVec2 CalcStringOffsetPos_( string text, TextBox textBox )
            {
                FVec2 offset = FVec2.Empty;

                float strW = _textWriter.CalcStringWidth( text, text.Length );
                float strH = _textWriter.CalcStringHeight( text, text.Length );

#if false
                // 文字列が一行の場合(改行の個数が0)には行間分の幅を減らしておく
                int lineCount = text.Length - text.Replace( '\n'.ToString(), "").Length;
                if( lineCount == 0 )
                {
                    strH -= textBox.LineSpace;
                }
#endif

                IPane     ownerPane = textBox.OwnerPane;
                FVec2     diff = new FVec2( ownerPane.RenderingWidth - strW,
                                            ownerPane.RenderingHeight - strH);

                if (textBox.LineWidthOffsetEnabled)
                {
                    // 行ごとの幅とオフセットが有効になっている場合は左上揃え固定にする
                    offset.X = 0;
                    offset.Y = 0;
                }
                else
                {
                    // 水平方向について
                    switch (textBox.BasePosTypeH)
                    {
                        case HorizontalLocation.Left: offset.X = 0; break;
                        case HorizontalLocation.Right: offset.X = diff.X; break;
                        case HorizontalLocation.Center: offset.X = diff.X / 2; break;
                        default:
                            break;
                    }

                    // 垂直方向について
                    switch (textBox.BasePosTypeV)
                    {
                        case VerticalLocation.Top: offset.Y = 0; break;
                        case VerticalLocation.Bottom: offset.Y = diff.Y; break;
                        case VerticalLocation.Center: offset.Y = diff.Y / 2; break;
                        default: Debug.Assert(false); break;
                    }
                }

                return offset;
            }

            /// <summary>
            ///
            /// </summary>
            static NWFont.NW4RTextWriter.PositionFlag GetNW4RFlag_( HorizontalLocation Hlocation )
            {
                switch( Hlocation )
                {
                    case HorizontalLocation.Left:
                    return NWFont.NW4RTextWriter.PositionFlag.HORIZONTAL_ALIGN_LEFT;
                    case HorizontalLocation.Center:
                    return NWFont.NW4RTextWriter.PositionFlag.HORIZONTAL_ALIGN_CENTER;
                    case HorizontalLocation.Right:
                    return NWFont.NW4RTextWriter.PositionFlag.HORIZONTAL_ALIGN_RIGHT;
                    default: Debug.Assert( false );
                    return NWFont.NW4RTextWriter.PositionFlag.HORIZONTAL_ALIGN_LEFT;
                }
            }

            static NWFont.NW4RTextWriter.PositionFlag GetNW4RFlag_( LineAlignment lineAlignment, HorizontalLocation Hlocation )
            {
                switch( lineAlignment )
                {
                    case LineAlignment.NotSpecified:
                    return GetNW4RFlag_( Hlocation );
                    case LineAlignment.Left:
                    return NWFont.NW4RTextWriter.PositionFlag.HORIZONTAL_ALIGN_LEFT;
                    case LineAlignment.Center:
                    return NWFont.NW4RTextWriter.PositionFlag.HORIZONTAL_ALIGN_CENTER;
                    case LineAlignment.Right:
                    return NWFont.NW4RTextWriter.PositionFlag.HORIZONTAL_ALIGN_RIGHT;
                    default: Debug.Assert( false );
                    return NWFont.NW4RTextWriter.PositionFlag.HORIZONTAL_ALIGN_LEFT;
                }
            }

            /// <summary>
            /// TextWriterを初期化します。
            /// </summary>
            void SetupTextWriter_(
                NWFont.INWFont      nwFont,
                TextBox             textBox )
            {
                _textWriter.SetFont( nwFont );

                // フォントスケールを設定します。
                _textWriter.SetFontSize( textBox.FontSize.X, textBox.FontSize.Y );

                // 文字間隔を設定します。
                _textWriter.SetCharSpace( textBox.CharSpace );
                _textWriter.SetLineSpace( textBox.LineSpace );

                _textWriter.SetDrawFromRightToLeftEnabled(textBox.DrawFromRightToLeft);

                uint alignHFlag;
                if (textBox.LineWidthOffsetEnabled)
                {
                    // 行ごとの幅とオフセットが有効になっている場合は左上揃え固定にする
                    alignHFlag = (uint)NWFont.NW4RTextWriter.PositionFlag.HORIZONTAL_ALIGN_LEFT;
                }
                else
                {
                    alignHFlag = (uint)GetNW4RFlag_(textBox.LineAlignment, textBox.BasePosTypeH);
                }

                _textWriter.SetDrawFlag( alignHFlag | (uint)NWFont.NW4RTextWriter.PositionFlag.VERTICAL_ORIGIN_TOP );

                if (textBox.OwnerPane != null)
                {
                    // TextBox の GetSuitableLengthString_ で自動改行処理を行っているため、
                    // TextWriter での自動改行は行わない。
                    _textWriter.ResetWidthLimit();
                }
            }

            /// <summary>
            ///制御コードを除いた実際の文字列を取得します。
            /// ex. ^FF ^0F など ハットの後の16進2桁
            /// </summary>
            string RemoveContorlCharacter_( string contestsText )
            {
                // ^00 - ^1f を削除する。
                contestsText = Regex.Replace( contestsText, @"\^[01][a-fA-F0-9]", "" );
                // ^^(連続ハット)以外の^を削除する。
                contestsText = Regex.Replace( contestsText, @"\^(.?)", "$1" );

                return contestsText;
            }

            /// <summary>
            /// 文字列描画まえの準備計算を行います。
            /// </summary>
            void PreCalcBeforeDrawing_
            (
                RenderedTextBoxCache    textBoxCache,
                NWFont.INWFont      nwFont,
                TextBox                 textBox
            )
            {
                // TextWriterを初期化します。
                SetupTextWriter_( nwFont, textBox );

                // ペインからはみ出さないように適当に改行を加えた文字列を生成します。
                textBoxCache.FormedString =
                    GetSuitableLengthString_( RemoveContorlCharacter_( textBox.ContentsText ),
                                              textBox.OwnerPane.RenderingWidth,
                                              textBox.LineWidthOffsetEnabled ? textBox.LineWidth : null );

                // 1文字アニメーションを計算します。
                textBoxCache.LetterMotions = CalcLetterMotions_(textBoxCache.FormedString, textBox);
                textBoxCache.LetterMotionsOriginV = textBox.PerCharacterTransformOriginV;
                textBoxCache.LetterMotionsOriginVOffset = textBox.PerCharacterTransformOriginVOffset;

                // アンカー設定を解釈し、表示位置のオフセットなどを計算します。
                textBoxCache.OffsetPos = CalcStringOffsetPos_( textBoxCache.FormedString, textBox );
            }

            /// <summary>
            /// 文字数でカーブを均等評価して、値を LetterMotionとして格納する。
            /// </summary>
            private LetterMotion[] CalcLetterMotions_(string formedString, TextBox textBox)
            {
                if (!textBox.IsPerCharTransformEnabled || formedString.Length <= 0)
                {
                    return null;
                }

                float stepT = 0.0f;
                if (IsPerCharTransformFixSpaceEnabled(textBox))
                {
                    stepT = textBox.PerCharTransformFixSpaceWidth;
                }
                else if (textBox.PerCharacterTransformSplitByCharWidth)
                {
                    if (!textBox.ILEFont.IsFileNotFound())
                    {
                        float totalT = 0.0f;
                        SetupTextWriter_(textBox.ILEFont.NWFont, textBox);
                        int begin = textBox.PerCharacterTransformOriginToCenter ? 0 : 1;
                        for (int i = begin; i < formedString.Length; i++)
                        {
                            float width = _textWriter.CalcStringWidth(formedString[i].ToString(), 1) + _textWriter.GetCharSpace();
                            totalT += width;
                        }
                        stepT = (float)textBox.PerCharacterTransformTimeWidth / totalT;
                    }
                }
                else
                {
                    stepT = formedString.Length == 1 ? 0.0f : (float)textBox.PerCharacterTransformTimeWidth / (formedString.Length - 1);
                }

                var loopType = textBox.PerCharacterTransformLoopType;
                LetterMotion[] letterMotions = new LetterMotion[formedString.Length];
                float time = textBox.PerCharacterTransformTimeOffset;

                if (IsPerCharTransformFixSpaceEnabled(textBox))
                {
                    // 文字揃えにより開始位置を変える
                    switch (textBox.PerCharTransformFixSpaceOrigin)
                    {
                        case HorizontalLocation.Left:
                            // 何もしない
                            break;
                        case HorizontalLocation.Center:
                            time = textBox.PerCharacterTransformTimeOffset + (textBox.PerCharacterTransformTimeWidth - textBox.PerCharTransformFixSpaceWidth * formedString.Length) / 2.0f;
                            break;
                        case HorizontalLocation.Right:
                            time = textBox.PerCharacterTransformTimeOffset + textBox.PerCharacterTransformTimeWidth - textBox.PerCharTransformFixSpaceWidth * formedString.Length;
                            break;
                        default:
                            Debug.Assert(false);
                            break;
                    }
                }

                for (int i = 0; i < formedString.Length; i++)
                {
                    float width = 0.0f;
                    // 文字ごとの時間を計算する。ループ指定されている場合は、範囲を求めて、ループする。
                    if (IsPerCharTransformFixSpaceEnabled(textBox))
                    {
                        time += stepT / 2.0f;
                    }
                    else if (textBox.PerCharacterTransformSplitByCharWidth)
                    {
                        width = _textWriter.CalcStringWidth(formedString[i].ToString(), 1) + _textWriter.GetCharSpace();
                        if (textBox.PerCharacterTransformOriginToCenter)
                        {
                            // 文字の中心を基準にするために、文字幅の半分を 2 回に分けて足す(前半)
                            time += width * stepT / 2.0f;
                        }
                        else if (i > 0)
                        {
                            time += width * stepT;
                        }
                    }
                    else
                    {
                        // time += stepT; とすると計算誤差が累積して互換性が失われるためその都度乗算で求める。
                        time = textBox.PerCharacterTransformTimeOffset + stepT * (float)i;
                    }

                    letterMotions[i] = new LetterMotion();

                    letterMotions[i].Scale.X = TryGetSubAttrCurveValueOrDefalut_(loopType, textBox.PerCharTransformSIAnmAttr, 0, time, 1.0f);
                    letterMotions[i].Scale.Y = TryGetSubAttrCurveValueOrDefalut_(loopType, textBox.PerCharTransformSIAnmAttr, 1, time, 1.0f);

                    letterMotions[i].Trans.X = TryGetSubAttrCurveValueOrDefalut_(loopType, textBox.PerCharTransformTIAnmAttr, 0, time, 0.0f);
                    letterMotions[i].Trans.Y = -TryGetSubAttrCurveValueOrDefalut_(loopType, textBox.PerCharTransformTIAnmAttr, 1, time, 0.0f);
                    letterMotions[i].Trans.Z = TryGetSubAttrCurveValueOrDefalut_(loopType, textBox.PerCharTransformTIAnmAttr, 2, time, 0.0f);

                    letterMotions[i].Rotate.X = -TryGetSubAttrCurveValueOrDefalut_(loopType, textBox.PerCharTransformRIAnmAttr, 0, time, 0.0f);
                    letterMotions[i].Rotate.Y = -TryGetSubAttrCurveValueOrDefalut_(loopType, textBox.PerCharTransformRIAnmAttr, 1, time, 0.0f);
                    letterMotions[i].Rotate.Z = -TryGetSubAttrCurveValueOrDefalut_(loopType, textBox.PerCharTransformRIAnmAttr, 2, time, 0.0f);

                    letterMotions[i].TopR = TryGetSubAttrCurveByteValueOrNull_(loopType, textBox.PerCharTransformTopColorIAnmAttr, 0, time);
                    letterMotions[i].TopG = TryGetSubAttrCurveByteValueOrNull_(loopType, textBox.PerCharTransformTopColorIAnmAttr, 1, time);
                    letterMotions[i].TopB = TryGetSubAttrCurveByteValueOrNull_(loopType, textBox.PerCharTransformTopColorIAnmAttr, 2, time);
                    letterMotions[i].TopA = TryGetSubAttrCurveByteValueOrNull_(loopType, textBox.PerCharTransformTopColorIAnmAttr, 3, time);

                    letterMotions[i].BtmR = TryGetSubAttrCurveByteValueOrNull_(loopType, textBox.PerCharTransformBottomColorIAnmAttr, 0, time);
                    letterMotions[i].BtmG = TryGetSubAttrCurveByteValueOrNull_(loopType, textBox.PerCharTransformBottomColorIAnmAttr, 1, time);
                    letterMotions[i].BtmB = TryGetSubAttrCurveByteValueOrNull_(loopType, textBox.PerCharTransformBottomColorIAnmAttr, 2, time);
                    letterMotions[i].BtmA = TryGetSubAttrCurveByteValueOrNull_(loopType, textBox.PerCharTransformBottomColorIAnmAttr, 3, time);

                    if (IsPerCharTransformFixSpaceEnabled(textBox))
                    {
                        time += stepT / 2.0f;
                    }
                    else if (textBox.PerCharacterTransformOriginToCenter && textBox.PerCharacterTransformSplitByCharWidth)
                    {
                        // 文字の中心を基準にするために、文字幅の半分を 2 回に分けて足す(後半)
                        time += width * stepT / 2.0f;
                    }
                }

                return letterMotions;
            }

            /// <summary>
            /// 文字毎のアニメーションの文字幅固定機能の有効・無効を取得します。
            /// </summary>
            private static bool IsPerCharTransformFixSpaceEnabled(TextBox textBox)
            {
                return textBox.IsPerCharTransformFixSpaceEnabled && textBox.PerCharacterTransformOriginToCenter;
            }

            /// <summary>
            /// カーブでのループ時間を計算します。
            /// </summary>
            private static float CalcCurveLocalLoopTime_(float time, float firstTime, float lastTime)
            {
                if (firstTime != lastTime)
                {
                    if (time > lastTime)
                    {
                        time = firstTime + (time - lastTime) % (lastTime - firstTime);
                    }
                    else if (time < firstTime)
                    {
                        time = lastTime + (time - firstTime) % (lastTime - firstTime);
                    }
                }

                return time;
            }

            /// <summary>
            /// サブアトリビュートカーブの値取得を試みます。なにも無ければデフォルト値が返ります。
            /// </summary>
            private static float TryGetSubAttrCurveValueOrDefalut_(PerCharTransformLoopType loopType, IAnmAttribute attr, int subIndex, float time, float defaultValue)
            {
                IAnmAttribute subAttr = attr.FindSubAttributeByIdx(subIndex);
                Debug.Assert(subAttr != null);

                if (loopType == PerCharTransformLoopType.Loop)
                {
                    float firstTime = float.MaxValue, lastTime = float.MinValue;
                    subAttr.GetFrameRangeRecursive(ref firstTime, ref lastTime);
                    if (firstTime == float.MaxValue)
                    {
                        firstTime = 0; // 見つからなかった場合は 0 とする
                    }
                    if (lastTime == float.MinValue)
                    {
                        lastTime = 0; // 見つからなかった場合は 0 とする
                    }

                    time = CalcCurveLocalLoopTime_(time, firstTime, lastTime);
                }

                return subAttr.HasKey() ? subAttr.ICurrentAnimationCurve.Evaluate(time) : defaultValue;
            }

            /// <summary>
            /// サブアトリビュートカーブの値取得を試みます。なにも無ければゼロが返ります。
            /// </summary>
            private static byte? TryGetSubAttrCurveByteValueOrNull_(PerCharTransformLoopType loopType, IAnmAttribute attr, int subIndex, float time)
            {
                IAnmAttribute subAttr = attr.FindSubAttributeByIdx(subIndex);
                Debug.Assert(subAttr != null);

                if (loopType == PerCharTransformLoopType.Loop)
                {
                    float firstTime = float.MaxValue, lastTime = float.MinValue;
                    subAttr.GetFrameRangeRecursive(ref firstTime, ref lastTime);
                    if (firstTime == float.MaxValue)
                    {
                        firstTime = 0; // 見つからなかった場合は 0 とする
                    }
                    if (lastTime == float.MinValue)
                    {
                        lastTime = 0; // 見つからなかった場合は 0 とする
                    }

                    time = CalcCurveLocalLoopTime_(time, firstTime, lastTime);
                }

                return subAttr.HasKey() ? (byte?)subAttr.ICurrentAnimationCurve.Evaluate(time) : null;
            }

            /// <summary>
            /// 文字を描画します。
            /// </summary>
            /// <param name="g"></param>
            /// <param name="leteerPane"></param>
            public void DrawLetter( IRenderer renderer, TextBox  textBox )
            {
                Debug.Assert(textBox.ILEFont != null);

                // フォントが発見されていない場合は描画処理をしません。
                if (textBox.ILEFont.IsFileNotFound())
                {
                    return;
                }

                _renderer = renderer;
                _currentFont = textBox.ILEFont;
                _textMaterial = textBox._revHWMaterial;

                // フチ
                _isPackedFont = textBox.ILEFont.NWFont.IsPackedFormatFont();
                _isPackedFontBorderDisabled = textBox.InvisibleBorderEnabled;
                _isBorderEffectEnabled = textBox.ILEFont.NWFont.IsBorderEffectEnabled();

                // 必要であれば、事前計算を行います。
                if (_renderedTextBoxCache.IsObsoleteFor_(textBox))
                {
                    PreCalcBeforeDrawing_(_renderedTextBoxCache, textBox.ILEFont.NWFont, textBox);
                }

                // テクスチャ行列をリセットします。
                _renderer.SetTextureMtx(0, FVec2.Empty, FVec2.One, 0.0f);

                // 斜体描画用のオフセット
                float italicOffset = textBox.ItalicFactor * _currentFont.Width * _textWriter.GetScaleH();

                // サンプリング設定(文字を書くのに適当な設定を行っておきます)。
                // RGBA形式のビットマップフォントでは、ガンマ変換を行います。
                bool needToSrgbFetch = textBox.ILEFont.NWFont.IsNeededToSrgbFetch();
                _renderer.SetTextureSamplingState(0, AttrTexWrap.Clamp, AttrTexWrap.Clamp, AttrTexFilterMin.Linear, AttrTexFilterMag.Linear, needToSrgbFetch);

                // 影の描画
                if (textBox.ShadowEnabled)
                {
                    FloatColor shadowWhiteColor = new FloatColor(textBox.ShadowWhiteColor);
                    FloatColor shadowBlackColor = new FloatColor(textBox.ShadowBlackColor);
                    shadowBlackColor.A = 0.0f;

                    // マテリアルカラーの設定
                    // HSV補正処理を施して、白黒補間を設定します。
                    renderer.SetMatrialColorBlend(shadowWhiteColor, shadowBlackColor, false);


                    this._shadowEnabled = textBox.ShadowEnabled;
                    this._autoShadowAlpha = textBox.PerCharacterTransformAutoShadowAlpha;
                    if (textBox.PerCharacterTransformOriginToCenter)
                    {
                        this._originToCenter = true;
                        this._textBoxWidthForOriginToCenter = textBox.OwnerPane.Width;
                        this._textBoxHeightForOriginToCenter = textBox.OwnerPane.Height;
                    }
                    this._shadowScale = textBox.ShadowScale;

                    _textWriter.SetCursor(_renderedTextBoxCache.OffsetPos.X, _renderedTextBoxCache.OffsetPos.Y);
                    this._italicOffset = italicOffset +
                        (textBox.ShadowItalicFactor * _currentFont.Width * _textWriter.GetScaleH() * textBox.ShadowScale.X);

                    // 変換を乗算
                    renderer.PushMtx();
                    {
                        Matrix34 mT = new Matrix34();
                        mT[0, 3] = textBox.ShadowOffset.X;
                        mT[1, 3] = -textBox.ShadowOffset.Y;

                        renderer.CurrentMtx34 = renderer.CurrentMtx34 * mT;

                        _vtxColorSet[0] = textBox.ShadowTopColor.AsColor;
                        _vtxColorSet[1] = _vtxColorSet[0];
                        _vtxColorSet[2] = textBox.ShadowBottomColor.AsColor;
                        _vtxColorSet[3] = _vtxColorSet[2];

                        _textWriter.Print(_renderedTextBoxCache.FormedString, textBox.LineWidthOffsetEnabled ? textBox.LineOffset : null);

                        this._shadowEnabled = false;
                        this._autoShadowAlpha = false;
                        this._originToCenter = false;
                    }
                    renderer.PopMtx();
                }

                IMaterial mat = textBox.IMaterial;
                if (textBox.DoubleDrawnBorderEnabled)
                {
                    DrawLetterImpl_(renderer, textBox, italicOffset, mat.BlackColor.ToRGBAColor(), mat.BlackColor.ToRGBAColor());

                    _isPackedFontBorderDisabled = true;
                    DrawLetterImpl_(renderer, textBox, italicOffset, mat.WhiteColor.ToRGBAColor(), mat.BlackColor.ToRGBAColor());
                }
                else
                {
                    DrawLetterImpl_(renderer, textBox, italicOffset, mat.WhiteColor.ToRGBAColor(), mat.BlackColor.ToRGBAColor());
                }

                // 行ごとの幅とオフセットの描画
                if (textBox.LineWidthOffsetEnabled && textBox.OwnerPane.IsSelected)
                {
                    for (int i = 0; i < ITextBoxHelper.LineOffsetWidthCountMax; i++)
                    {
                        if ((textBox.FontSize.Y + textBox.LineSpace) * (float)i >= textBox.OwnerPane.Height)
                        {
                            break;
                        }

                        float left;
                        float right;

                        ITextBoxHelper.ChangeWidthOffsetToLeftRight(out left, out right, textBox.LineOffset[i], textBox.LineWidth[i]);
                        float top = textBox.FontSize.Y * ((float)i + 0.5f) + textBox.LineSpace * (float)i;
                        const float rectSize = 6.0f;
                        _renderer.FillRectangle(-left - rectSize, top - rectSize / 2.0f, 0.0f, rectSize, rectSize);
                        _renderer.FillRectangle(textBox.OwnerPane.RenderingWidth + right, top - rectSize / 2.0f, 0.0f, rectSize, rectSize);
                    }
                }

                _renderer = null;
            }

            private void DrawLetterImpl_(IRenderer renderer, TextBox textBox, float italicOffset, RGBAColor wc, RGBAColor bc)
            {
                // 通常の文字の描画
                if (textBox.PerCharacterTransformOriginToCenter)
                {
                    this._originToCenter = true;
                    this._textBoxWidthForOriginToCenter = textBox.OwnerPane.Width;
                    this._textBoxHeightForOriginToCenter = textBox.OwnerPane.Height;
                }
                {
                    // マテリアルカラーの設定
                    // HSV補正処理を施して、白黒補間を設定します。
                    {
                        IMaterial mat = textBox.IMaterial;

                        // ラインタイム側は、補正処理をしていないので合わせておく。
                        FloatColor tweakedWhite = mat.WhiteColor;
                        FloatColor tweakedBlack = mat.BlackColor;
                        tweakedBlack.A = 0.0f;

                        renderer.SetMatrialColorBlend(tweakedWhite, tweakedBlack, false);

                    }

                    _textWriter.SetCursor(_renderedTextBoxCache.OffsetPos.X, _renderedTextBoxCache.OffsetPos.Y);
                    this._italicOffset = italicOffset;

                    // 文字頂点色の設定
                    _vtxColorSet[0] = textBox.TopColor.AsColor;
                    _vtxColorSet[1] = _vtxColorSet[0];
                    _vtxColorSet[2] = textBox.BottomColor.AsColor;
                    _vtxColorSet[3] = _vtxColorSet[2];

                    // 描画(TextWriterモジュールに描画を委譲します。)
                    _textWriter.Print(_renderedTextBoxCache.FormedString, textBox.LineWidthOffsetEnabled ? textBox.LineOffset : null);
                }
                this._originToCenter = false;
            }

            // 色を乗算する関数
            Color MultiplyColor_(Color origCol, byte? a, byte? r, byte? g, byte? b)
            {
                int col32 = 0;

                if (a != null)
                {
                    col32 |= (((origCol.A * a.Value / 255) & 0xff) << 24);
                }
                else
                {
                    col32 |= origCol.A << 24;
                }

                if (r != null)
                {
                    col32 |= (((origCol.R * r.Value / 255) & 0xff) << 16);
                }
                else
                {
                    col32 |= origCol.R << 16;
                }

                if (g != null)
                {
                    col32 |= (((origCol.G * g.Value / 255) & 0xff) << 8);
                }
                else
                {
                    col32 |= origCol.G << 8;
                }

                if (b != null)
                {
                    col32 |= (((origCol.B * b.Value / 255) & 0xff) << 0);
                }
                else
                {
                    col32 |= origCol.B << 0;
                }

                return Color.FromArgb(col32);
            }

            // 文字毎の変換の色情報を取得する関数
            private void AcquireLetterMotionColor(
                out byte? topA, out byte? topR, out byte? topG, out byte? topB,
                out byte? bottomA, out byte? bottomR, out byte? bottomG, out byte? bottomB,
                LetterMotion letterMotion)
            {
                if (_shadowEnabled)
                {
                    topA = _autoShadowAlpha ? letterMotion.TopA : null;
                    topR = null;
                    topG = null;
                    topB = null;
                    bottomA = _autoShadowAlpha ? letterMotion.BtmA : null;
                    bottomR = null;
                    bottomG = null;
                    bottomB = null;
                }
                else
                {
                    topA = letterMotion.TopA;
                    topR = letterMotion.TopR;
                    topG = letterMotion.TopG;
                    topB = letterMotion.TopB;
                    bottomA = letterMotion.BtmA;
                    bottomR = letterMotion.BtmR;
                    bottomG = letterMotion.BtmG;
                    bottomB = letterMotion.BtmB;
                }
            }

            /// <summary>
            /// NW4RTextWriter からコールバックされる描画関数
            /// 一文字の描画ごとにコールバックされます。
            /// </summary>
            /// <param name="glyph"></param>
            public void DrawChar( NWFont.NW4RGlyph glyph, float x, float y, int index )
            {
                // サイズには、スケールを乗じる必要があります。
                float fW		= glyph.glyphWidth * _textWriter.GetScaleH();
                float fH		= glyph.height     * _textWriter.GetScaleV();

                // 描画位置をRectangleに設定します
                float        charZ      = _textMaterial.IMaterial.OwnerPane != null ? _textMaterial.IMaterial.OwnerPane.Z : 0.0f;
                FVec3        imgPos     = new FVec3(x, y, charZ);
                FVec2        imgSize    = new FVec2( fW, fH );

                Color[] vtxMultipliedColorSet = _vtxColorSet;

                // 1文字ごとの変換があれば適用します。
                if (_renderedTextBoxCache.LetterMotions != null)
                {
                    LetterMotion letterMotion = _renderedTextBoxCache.LetterMotions[index];
                    Matrix34 mtxLetter = CalcPerCharMtx_(x, y, fW, fH, charZ, letterMotion);

                    _renderer.PushMtx();
                    _renderer.CurrentMtx34 = _renderer.CurrentMtx34 * mtxLetter;

                    byte? topA, topR, topG, topB;
                    byte? bottomA, bottomR, bottomG, bottomB;
                    AcquireLetterMotionColor(
                        out topA, out topR, out topG, out topB,
                        out bottomA, out bottomR, out bottomG, out bottomB,
                        letterMotion);
                    vtxMultipliedColorSet = new Color[4];
                    vtxMultipliedColorSet[0] = MultiplyColor_(_vtxColorSet[0], topA, topR, topG, topB);
                    vtxMultipliedColorSet[1] = vtxMultipliedColorSet[0];
                    vtxMultipliedColorSet[2] = MultiplyColor_(_vtxColorSet[2], bottomA, bottomR, bottomG, bottomB);
                    vtxMultipliedColorSet[3] = vtxMultipliedColorSet[2];
                }

                _renderer.RectangleTopXOffset = this._italicOffset;
                _renderer.FontHeight = _currentFont.Height * _textWriter.GetScaleV();
                if (this._shadowEnabled)
                {
                    imgPos.Y = y + (-glyph.height * this._shadowScale.Y + glyph.height) * _textWriter.GetScaleV();

                    imgSize.X *= this._shadowScale.X;
                    imgSize.Y *= this._shadowScale.Y;
                }

                float        u0 = (float)glyph.cellX / glyph.texWidth;
                float        v0 = (float)glyph.cellY / glyph.texHeight;
                float        u1 = (float)(glyph.cellX + glyph.rawWidth) / glyph.texWidth;
                float        v1 = (float)(glyph.cellY + glyph.rawHeight)/ glyph.texHeight ;

                // フォント画像の一部を描画します
                SheetBitmap	 sheetBmp = _currentFont.GetFontBitmapBySheetIdx( glyph.sheetNo );
                if (sheetBmp == null)
                {
                    return;
                }

                // フォントテクスチャが動的に更新される可能性があるので毎回チェックする。
                RendererTextrureFormat fontTextureFormat = RendererTextrureFormat.ARGB; // TODO : フォーマットにあわせて最適なものを選ぶ
                _renderer.TryUpdateTextureIfOld(sheetBmp.Bitmap, sheetBmp.LastModifyTime, fontTextureFormat);

                TexCoord4 coord = new TexCoord4();

                coord.LT = new FVec2( u0, v0 );
                coord.RT = new FVec2( u1, v0 );
                coord.LB = new FVec2( u0, v1 );
                coord.RB = new FVec2( u1, v1 );

                // カラー設定を反映します。(TODO:)

                // ブレンド設定
                {
                    var pe = _textMaterial.PEData;
                    PEBlend colBlend = pe.UseDefaultBlendSettings ? PEBlend.DefaultBlendColor : pe.Blend;
                    PEBlend alpBlend = pe.UseColorBlendSettingsForAlpha ? PEBlend.DefaultBlendAlpha : pe.BlendAlpha;

                    _renderer.SetBlendMode(
                        colBlend.Type,
                        colBlend.SrcFactor, colBlend.DstFactor, colBlend.BlendOp,
                        alpBlend.SrcFactor, alpBlend.DstFactor, alpBlend.BlendOp);

                    _renderer.SetAlphaCompare(!pe.UseDefaultAlphaTestSettings, pe.ACompare.Comp, pe.ACompare.Ref);
                }

                // フチ設定
                if(_isPackedFont || _isBorderEffectEnabled)
                {
                    _renderer.SetPackedFontMode(_isPackedFontBorderDisabled);
                }

                // sRGB フェッチの設定
                _renderer.SetTextureSamplingSrgbState(0, glyph.isBitmapFont && LECore.Structures.FontFactory.TextureConverterUtility.IsNeededToSrgbFetchFormat(glyph.sheetFormat));

                // マテリアルカラーの設定
                // 文字の背景との正しいブレンドのために、
                // BlackColorに文字色を設定しています。
                // ↑ 現状はしていません。
                _renderer.SetTextureState(sheetBmp.Bitmap, 0, fontTextureFormat);
                _renderer.DrawImage(imgPos, imgSize, new TexCoord4[] { coord }, vtxMultipliedColorSet);

                _renderer.RectangleTopXOffset = 0.0f;
                _renderer.FontHeight = 0.0f;

                // 1文字ごとの変換を無効にします
                if (_renderedTextBoxCache.LetterMotions != null)
                {
                    _renderer.PopMtx();
                }
            }

            /// <summary>
            /// 1文字ごとの、変換行列を計算します。
            /// </summary>
            private Matrix34 CalcPerCharMtx_(float x, float y, float fW, float fH, float charZ, LetterMotion lm)
            {
                float transformOriginY;
                if (_originToCenter)
                {
                    transformOriginY = y;
                }
                else
                {
                    transformOriginY = _renderedTextBoxCache.LetterMotionsOriginV == VerticalLocation.Center ? y + fH * 0.5f : y + fH;
                    transformOriginY += -_renderedTextBoxCache.LetterMotionsOriginVOffset; // フォントは Y 軸がレイアウトと逆なのでオフセットの正負を反転する
                }
                var mS = SharpDX.Matrix.Scaling(lm.Scale.X, lm.Scale.Y, 1.0f);
                var mO = SharpDX.Matrix.Translation(-(x + fW * 0.5f), -transformOriginY, -charZ);
                var mR = SharpDX.Matrix.RotationX(MathUtil.DegToRad(lm.Rotate.X)) * SharpDX.Matrix.RotationY(MathUtil.DegToRad(lm.Rotate.Y)) * SharpDX.Matrix.RotationZ(MathUtil.DegToRad(lm.Rotate.Z));
                var mT = SharpDX.Matrix.Translation(
                    // _originToCenter == true の場合は、各文字の原点をテキストボックスの中心にする
                    _originToCenter ? (int)(-x + _textBoxWidthForOriginToCenter / 2.0f - fW * 0.5f + lm.Trans.X) : lm.Trans.X,
                    _originToCenter ? (int)(_textBoxHeightForOriginToCenter / 2.0f - transformOriginY + lm.Trans.Y) : lm.Trans.Y,
                    lm.Trans.Z);
                Matrix34 mRes = new Matrix34();
                mRes.Set(mO * mS * mR * SharpDX.Matrix.Invert(mO) * mT);

                return mRes;
            }

            /// <summary>
            /// 文字列の幅を取得します。
            /// </summary>
            /// <returns></returns>
            public FVec2 CalcStringSize( TextBox  textBox )
            {
                // フォントが発見されていない場合はサイズ計算をしません。
                if (textBox.ILEFont.IsFileNotFound())
                {
                    return textBox.OwnerPane.Size.FVec2;
                }

                SetupTextWriter_( textBox.ILEFont.NWFont, textBox );

                string contStr = RemoveContorlCharacter_( textBox.ContentsText );

               float w = _textWriter.CalcStringWidth( contStr, contStr.Length );
               float h = _textWriter.CalcStringHeight( contStr, contStr.Length );
               return new FVec2( w, h );
            }



            #endregion NWFont.INW4RTextRenderer メンバ
        }

        #endregion ---------------- 内部クラス宣言 ----------------

        #region	 ------------------- フィールド -------------------

        // ------------ アニメーションしないパラメータ
        string               _contentsText = "Input Text";
        string               _textID = string.Empty;

        ILEFont               _font;
        // フォント名：この名前をもとに描画時にマネージャに問い合わせて、
        // 描画に使用するフォントを取得します。
        string                _fontName;

        FVec2                _fontSize     = new FVec2( 16.0f, 16.0f );
        FVec2                 _fontSizeOriginal = FVec2.Empty;

        uint                 _allocByte    = 0;
        bool                 _specifiedStringLength = false;

        float                _charSpace    = 0.0f;
        float                _lineSpace    = 0.0f;
        HorizontalLocation   _basePosTypeH = HorizontalLocation.Center;
        VerticalLocation     _basePosTypeV = VerticalLocation.Center;
        LineAlignment			_lineAlignment = LineAlignment.NotSpecified;
        bool                 _centerCeilingEnabled = false;
        bool                 _drawFromRightToLeft = false;

        // 自身の描画に使用されるモジュール
        TextBoxRenderer _textBoxRenderer  = new TextBoxRenderer();

        readonly Material         _material              = null;
        readonly RevHWMaterial    _revHWMaterial         = null;
        readonly TextBoxAttribute _textBoxAnmAttr        = null;
        readonly IPane            _ownerPane             = null;

        /// <summary>
        /// 斜体傾き係数
        /// </summary>
        float _italicFactor = 0.0f;

        //----------------------------------------------------------
        // 影関連
        //----------------------------------------------------------

        /// <summary>
        /// 影が有効かどうか
        /// </summary>
        bool _shadowEnabled = false;

        /// <summary>
        /// 影のオフセット
        /// </summary>
        FVec2 _shadowOffset = new FVec2(1.0f, -1.0f);

        /// <summary>
        /// 影のスケール
        /// </summary>
        FVec2 _shadowScale = FVec2.One;

        /// <summary>
        /// 影斜体傾き係数
        /// </summary>
        float _shadowItalicFactor = 0.0f;

        /// <summary>
        /// 影上端色
        /// </summary>
        RGBAColor _shadowTopColor = RGBAColor.Black;

        /// <summary>
        /// 影下端色
        /// </summary>
        RGBAColor _shadowBottomColor = RGBAColor.Black;

        /// <summary>
        /// 文字変化のループ
        /// </summary>
        PerCharTransformLoopType _perCharacterTransformLoopType = PerCharTransformLoopType.Loop;

        /// <summary>
        /// 文字変化の原点（垂直）
        /// </summary>
        VerticalLocation _perCharacterTransformOriginV = VerticalLocation.Center;

        /// <summary>
        /// 文字変化の原点のオフセット（垂直）
        /// </summary>
        float _perCharacterTransformOriginVOffset = 0.0f;

        /// <summary>
        /// 文字変化の文字幅による分割
        /// </summary>
        bool _perCharacterTransformSplitByCharWidth = false;

        /// <summary>
        /// 文字変化の影のアルファを文字のアルファに自動追従させる
        /// </summary>
        bool _perCharacterTransformAutoShadowAlpha = false;

        /// <summary>
        /// 文字変化の各文字の原点をテキストボックスの中心にする
        /// </summary>
        bool _perCharacterTransformOriginToCenter = false;

        /// <summary>
        ///
        /// </summary>
        bool _isPerCharTransformEnabled = false;

        /// <summary>
        /// 文字毎のアニメーションの簡単設定
        /// </summary>
        EasySettingOfPerCharTransform[] _easySettingOfPerCharTransform = null;

        /// <summary>
        /// 文字毎のアニメーションの文字幅固定機能の有効・無効
        /// </summary>
        bool _isPerCharTransformFixSpaceEnabled = false;

        /// <summary>
        /// 文字毎のアニメーションの文字幅固定の文字幅
        /// </summary>
        float _perCharTransformFixSpaceWidth = 0.0f;

        /// <summary>
        /// 文字毎のアニメーションの文字幅固定の文字揃え
        /// </summary>
        HorizontalLocation _perCharTransformFixSpaceOrigin = HorizontalLocation.Center;

        /// <summary>
        /// 文字毎のアニメーションの円形配置の開始角度
        /// </summary>
        float _perCharTransformEasyCircleBeginAngle = 180.0f;

        /// <summary>
        /// 文字毎のアニメーションの円形配置の終了角度
        /// </summary>
        float _perCharTransformEasyCircleEndAngle = 360.0f;

        /// <summary>
        /// 文字毎のアニメーションの円形配置の円の半径X
        /// </summary>
        float _perCharTransformEasyCircleRadiusX = 100.0f;

        /// <summary>
        /// 文字毎のアニメーションの円形配置の円の半径Y
        /// </summary>
        float _perCharTransformEasyCircleRadiusY = 100.0f;

        /// <summary>
        /// 文字毎のアニメーションの円形配置の分割数
        /// </summary>
        int _perCharTransformEasyCircleDivisionNumber = 10;

        /// <summary>
        /// 文字毎のアニメーションの円形配置の文字の向き
        /// </summary>
        float _perCharTransformEasyCircleCharAngle = 90.0f;

        /// <summary>
        /// 縁取りOFF
        /// </summary>
        bool _invisibleBorderEnabled = false;

        /// <summary>
        /// 2度書き縁取り
        /// </summary>
        bool _doubleDrawnBorderEnabled = false;

        /// <summary>
        /// 行ごとの幅
        /// </summary>
        float[] _lineWidth = new float[ITextBoxHelper.LineOffsetWidthCountMax];

        /// <summary>
        /// 行ごとのオフセット
        /// </summary>
        float[] _lineOffset = new float[ITextBoxHelper.LineOffsetWidthCountMax];

        /// <summary>
        /// 行ごとの幅とオフセットの有効無効
        /// </summary>
        bool _lineWidthOffsetEnabled = false;

        /// <summary>
        /// 拡張タグの解釈の有効無効
        /// </summary>
        bool _isExtendedTagEnabled = false;

        #endregion	 ------------------- フィールド -------------------

        #region	 ------------------- プロパティ -------------------

        /// <summary>
        /// 内容文字列
        /// </summary>
        public string ContentsText
        {
            get{ return _contentsText;}
            set
            {
                if( _contentsText != value )
                {
                    _contentsText = value;
                    NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.TextboxContentsChanged);
                }
            }
        }

        /// <summary>
        /// メッセージ識別文字列
        /// </summary>
        public string TextID
        {
            get { return _textID; }
            set
            {
                if (_textID != value)
                {
                    _textID = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// フォントサイズ
        /// </summary>
        public FVec2 FontSize
        {
            get{ return _fontSize;}
            set
            {
                if( _fontSize != value )
                {
                    _fontSize = value; NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// フォントサイズ
        /// </summary>
        public FVec2 FontSizeOriginal
        {
            get { return _fontSizeOriginal; }
            set
            {
                if (_fontSizeOriginal != value)
                {
                    _fontSizeOriginal = value; NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// フォントスケール
        /// </summary>
        public FVec2  FontScale
        {
            get
            {
                if( _font != null )
                {
                    return new FVec2(
                         _fontSize.X / (float)_font.Width,
                         _fontSize.Y / (float)_font.Height );
                }
                else
                {
                    return new FVec2( 1.0f, 1.0f );
                }
            }
        }

        /// <summary>
        /// フォント実体
        /// </summary>
        public ILEFont ILEFont
        {
            get{ return _font;}
            private set
            {
                if( _font != value )
                {
                    _font = value;
                    NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.TextboxFontChanged);
                }
            }
        }

        /// <summary>
        /// フォント名を設定します。
        /// </summary>
        public string FontName
        {
            set
            {
                if( _fontName != value )
                {
                    _fontName = value;
                    SetupFont_();
                }
            }
            get { return _fontName; }
        }

        public Material Material
        {
            get{ return _material; }
        }

        public IMaterial IMaterial
        {
            get{ return _material as IMaterial; }
        }

        public RevHWMaterial RevHWMaterial
        {
            get{ return _revHWMaterial;}
        }

        public float LineSpace
        {
            get{ return _lineSpace;}
            set
            {
                if( _lineSpace != value )
                {
                    _lineSpace = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public float CharSpace
        {
            get{ return _charSpace;}
            set
            {
                if( _charSpace != value )
                {
                    _charSpace = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public HorizontalLocation  BasePosTypeH
        {
            get{ return _basePosTypeH; }
            set
            {
                if( _basePosTypeH != value )
                {
                    _basePosTypeH = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public VerticalLocation  BasePosTypeV
        {
            get{ return _basePosTypeV; }
            set
            {
                if( _basePosTypeV != value )
                {
                    _basePosTypeV = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public LineAlignment LineAlignment
        {
            get { return _lineAlignment; }
            set
            {
                _lineAlignment = value;
                NotifyChangeToScene_();
            }
        }

        public bool CenterCeilingEnabled
        {
            get { return _centerCeilingEnabled; }
            set
            {
                _centerCeilingEnabled = value;
                NotifyChangeToScene_();
            }
        }

        public bool DrawFromRightToLeft
        {
            get { return _drawFromRightToLeft; }
            set
            {
                _drawFromRightToLeft = value;
                NotifyChangeToScene_();
            }
        }

        public uint AllocStringLength
        {
            get{ return _allocByte;}
            set
            {
                if( _allocByte != value )
                {
                    _allocByte = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public bool SpecifiedStringLength
        {
            get{ return _specifiedStringLength;}
            set
            {
                if( _specifiedStringLength != value )
                {
                    _specifiedStringLength = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public TextBoxAttribute TextBoxAnmAttr
        {
            get{ return _textBoxAnmAttr;}
        }

        /// <summary>
        /// UV座標を列挙します。
        /// </summary>
        public IEnumerable<TexCoord4> TexCoords
        {
            get { yield break; }
        }

        /// <summary>
        /// 描画可能か？
        /// </summary>
        bool _IsReadyToDraw
        {
            get { return ILEFont != null && ILEFont.IsReadyToDraw; }
        }

        #endregion	 ------------------- プロパティ -------------------


        #region privete
        /// <summary>
        /// フォントを初期化します。
        /// </summary>
        bool SetupFont_()
        {
            if( _ownerPane != null &&
                _ownerPane.OwnerSubScene != null )
            {
                this.ILEFont = _ownerPane.OwnerSubScene.ILEFontManager.FindFontByName( this._fontName );
                return true;
            }
            else
            {
                return false;
            }
        }
        #endregion
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextBox( Pane ownerPane, string contents, string fontName )
            :base( ownerPane, TextBoxNodeName )
        {
            Debug.Assert( ownerPane != null );

            _ownerPane = ownerPane;
            FontName = fontName;

            _textBoxAnmAttr  = new TextBoxAttribute( this );
            _material        = new Material( this, ownerPane.PaneName );
            _revHWMaterial   = new RevHWMaterial( this, _material );

            _revHWMaterial.TexIndirectMtxSet.SetIsActiveAttributeAll(false);
            _revHWMaterial.RevHWMaterialAttr.IsActiveAttribute = false;

            // テキストで、黒カラー補間のアルファ値は無効
            _material.BlackColorAnmAttr.FindAttributeByIdx(3).IsActiveAttribute = false;
            _textBoxAnmAttr.ShadowBlackColorAttr.FindAttributeByIdx(3).IsActiveAttribute = false;

            this.ContentsText = contents;

            ownerPane.BindPaneExData( this );
        }

        /// <summary>
        /// リソースを開放します。
        /// </summary>
        public void Dispose()
        {
            _textBoxAnmAttr.Dispose();
            _material.Dispose();
            _revHWMaterial.Dispose();
        }

        public FVec2 CalcStringSize()
        {
            return (ILEFont != null) ? _textBoxRenderer.CalcStringSize(this) : FVec2.Empty;
        }

        /// <summary>
        /// 内容文字列ぴったりにペインサイズを調整します。
        /// </summary>
        public void AdjustPaneSize()
        {
            if( ILEFont != null )
            {
                Pane ownerPane = OwnerPane as Pane;

                // 計測のために、ペインサイズを一旦適当に大きな値を設定します。
                float fontWidth = this.ILEFont != null ? (float)this.ILEFont.Width : LEFont.InitialFontSize;
                ownerPane.Width = fontWidth * this.ContentsText.Length * 2;

                // 規定の幅を、文字列の幅とします。
                FVec2 strSize = _textBoxRenderer.CalcStringSize( this );

                // 一文字のサイズが中途半端な小数値になるような、
                // 場合に、処理系毎で数値計算誤差が異なることによって、
                // 意図しない改行がおこるので、少しマージンを設けておく。
                //
                // 一文字のサイズが中途半端な小数値となるのは、
                // FontScale.X が 1.0f では無い場合と仮定しておく。
                // 文字のサイズ自体が、中途半端な小数値であることはありえないはず。
                float margine = this.FontScale.X != 1.0f ? 0.00001f : 0.0f;
                ownerPane.Width = strSize.X + margine;
                ownerPane.Height = strSize.Y;
            }
        }

        /// <summary>
        /// フォント参照を更新します。
        /// </summary>
        public void UpdateFontReference()
        {
            SetupFont_();
        }

        /// <summary>
        /// 変更をシーンに通知します。
        /// </summary>
        void NotifyChangeToScene_()
        {
            NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
        }

        #region IDrawable メンバ

        /// <summary>
        /// 描画を行います。
        /// Nullペインの描画処理の後に呼ばれます。
        /// </summary>
        public void Draw( IRenderer renderer, DrawableOption option )
        {
            if (option.CheckActive(DrawableOptionFlag.IgnorePaneImageDrawing))
            {
                return;
            }

            Debug.Assert( _ownerPane != null );

            if( _IsReadyToDraw )
            {
                // TextBoxRenderer に 描画処理を委譲します。
                _textBoxRenderer.DrawLetter(renderer, this);
            }
            else
            {
                if (this.ILEFont != null && !this.ILEFont.IsFileNotFound())
                {
                    // フォント初期化終了時アクション
                    var guiThreadContext = SynchronizationContext.Current;
                    Action onFinishFontInit = new Action(() =>
                    {
                        // 更新イベントをGUIスレッドで発行する。
                        guiThreadContext.Post((arg) =>
                        {
                            if (this.GetOwnerSubScene() != null)
                            {
                                (this.GetOwnerSubScene() as SubScene).RaisePaneModifyEvent();
                            }
                        }, null);
                    });

                    this.ILEFont.InitFontAsync(onFinishFontInit);

                    if (option.CheckActive(DrawableOptionFlag.SyncTextureInitialization))
                    {
                        // InitFontAsync は他のメソッドで既に呼ばれている可能性があるので、単純にスリープしながら待つ
                        for (int i=0;i<1000;i++)
                        {
                            if (!this.ILEFont.IsWaitingUpdate)
                            {
                                DbgConsole.WriteLine($"Waited {i * 10} ms");
                                break;
                            }

                            Thread.Sleep(10);
                        }

                        if (_IsReadyToDraw)
                        {
                            _textBoxRenderer.DrawLetter(renderer, this);
                        }
                        else
                        {
                            DbgConsole.WriteLine($"Failed to Draw TextBox {this.Name}");
                        }
                    }
                }
            }
        }

        #endregion IDrawable メンバ

        #region ITextBox メンバ

        public RGBAColor     TopColor
        {
            get{ return _textBoxAnmAttr.TopColor;}
            set{ _textBoxAnmAttr.TopColor = value;}
        }

        public RGBAColor     BottomColor
        {
            get{ return _textBoxAnmAttr.BottomColor;}
            set{ _textBoxAnmAttr.BottomColor = value;}
        }

        /// <summary>
        /// 影黒カラー補間
        /// </summary>
        public RGBAColor ShadowBlackColor
        {
            get { return _textBoxAnmAttr.ShadowBlackColorAttr.GetAsRBGA() ; }
            set { _textBoxAnmAttr.ShadowBlackColorAttr.SetValue(value); }
        }

        /// <summary>
        /// 影白カラー補間
        /// </summary>
        public RGBAColor ShadowWhiteColor
        {
            get { return _textBoxAnmAttr.ShadowWhiteColorAttr.GetAsRBGA(); }
            set { _textBoxAnmAttr.ShadowWhiteColorAttr.SetValue(value); }
        }

        /// <summary>
        /// 斜体傾き係数
        /// </summary>
        public float ItalicFactor
        {
            get { return _italicFactor; }
            set
            {
                if (_italicFactor != value)
                {
                    _italicFactor = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 影が有効かどうか
        /// </summary>
        public bool ShadowEnabled
        {
            get { return _shadowEnabled; }
            set
            {
                if (_shadowEnabled != value)
                {
                    _shadowEnabled = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 影のオフセット
        /// </summary>
        public FVec2 ShadowOffset
        {
            get { return _shadowOffset; }
            set
            {
                if (_shadowOffset != value)
                {
                    _shadowOffset = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 影のスケール
        /// </summary>
        public FVec2 ShadowScale
        {
            get { return _shadowScale; }
            set
            {
                if (_shadowScale != value)
                {
                    _shadowScale = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 影斜体傾き係数
        /// </summary>
        public float ShadowItalicFactor
        {
            get { return _shadowItalicFactor; }
            set
            {
                if (_shadowItalicFactor != value)
                {
                    _shadowItalicFactor = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 影上端色
        /// </summary>
        public RGBAColor ShadowTopColor
        {
            get { return _shadowTopColor; }
            set
            {
                if (_shadowTopColor != value)
                {
                    _shadowTopColor = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 影下端色
        /// </summary>
        public RGBAColor ShadowBottomColor
        {
            get { return _shadowBottomColor; }
            set
            {
                if (_shadowBottomColor != value)
                {
                    _shadowBottomColor = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 文字変化のオフセット時間
        /// </summary>
        public float PerCharacterTransformTimeOffset
        {
            get { return this.PerCharTimeOffsetAnmAttr.GetAsFloat(); }
            set
            {
                if (this.PerCharTimeOffsetAnmAttr.GetAsFloat() != value)
                {
                    this.PerCharTimeOffsetAnmAttr.SetAsFloat(value);
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字変化の時間幅
        /// </summary>
        public float PerCharacterTransformTimeWidth
        {
            get { return this.PerCharTimeWidthAnmAttr.GetAsFloat(); }
            set
            {
                if (this.PerCharTimeWidthAnmAttr.GetAsFloat() != value)
                {
                    this.PerCharTimeWidthAnmAttr.SetAsFloat(value);
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字変化のループ
        /// </summary>
        public PerCharTransformLoopType PerCharacterTransformLoopType
        {
            get { return _perCharacterTransformLoopType; }
            internal set
            {
                if (_perCharacterTransformLoopType != value)
                {
                    _perCharacterTransformLoopType = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字変化の原点
        /// </summary>
        public VerticalLocation PerCharacterTransformOriginV
        {
            get { return _perCharacterTransformOriginV; }
            internal set
            {
                if (_perCharacterTransformOriginV != value)
                {
                    _perCharacterTransformOriginV = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字変化の原点のオフセット
        /// </summary>
        public float PerCharacterTransformOriginVOffset
        {
            get { return _perCharacterTransformOriginVOffset; }
            internal set
            {
                if (_perCharacterTransformOriginVOffset != value)
                {
                    _perCharacterTransformOriginVOffset = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字変化の文字幅による分割
        /// </summary>
        public bool PerCharacterTransformSplitByCharWidth
        {
            get { return _perCharacterTransformSplitByCharWidth; }
            internal set
            {
                if (_perCharacterTransformSplitByCharWidth != value)
                {
                    _perCharacterTransformSplitByCharWidth = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字変化の影のアルファを文字のアルファに自動追従させる
        /// </summary>
        public bool PerCharacterTransformAutoShadowAlpha
        {
            get { return _perCharacterTransformAutoShadowAlpha; }
            internal set
            {
                if (_perCharacterTransformAutoShadowAlpha != value)
                {
                    _perCharacterTransformAutoShadowAlpha = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字変化の各文字の原点をテキストボックスの中心にする
        /// </summary>
        public bool PerCharacterTransformOriginToCenter
        {
            get { return _perCharacterTransformOriginToCenter && IsPerCharTransformEnabled; }
            internal set
            {
                if (_perCharacterTransformOriginToCenter != value)
                {
                    _perCharacterTransformOriginToCenter = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字変化の有効無効
        /// </summary>
        public bool IsPerCharTransformEnabled
        {
            get { return _isPerCharTransformEnabled; }
            internal set
            {
                if (_isPerCharTransformEnabled != value)
                {
                    _isPerCharTransformEnabled = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの簡単設定
        /// </summary>
        public EasySettingOfPerCharTransform[] EasySettingOfPerCharTransform
        {
            get { return _easySettingOfPerCharTransform; }
            internal set
            {
                if (_easySettingOfPerCharTransform != value)
                {
                    _easySettingOfPerCharTransform = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの文字幅固定機能の有効・無効
        /// </summary>
        public bool IsPerCharTransformFixSpaceEnabled
        {
            get { return _isPerCharTransformFixSpaceEnabled; }
            internal set
            {
                if (_isPerCharTransformFixSpaceEnabled != value)
                {
                    _isPerCharTransformFixSpaceEnabled = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの文字幅固定の文字幅
        /// </summary>
        public float PerCharTransformFixSpaceWidth
        {
            get { return _perCharTransformFixSpaceWidth; }
            internal set
            {
                if (_perCharTransformFixSpaceWidth != value)
                {
                    _perCharTransformFixSpaceWidth = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの文字幅固定の文字揃え
        /// </summary>
        public HorizontalLocation PerCharTransformFixSpaceOrigin
        {
            get { return _perCharTransformFixSpaceOrigin; }
            internal set
            {
                if (_perCharTransformFixSpaceOrigin != value)
                {
                    _perCharTransformFixSpaceOrigin = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の開始角度
        /// </summary>
        public float PerCharTransformEasyCircleBeginAngle
        {
            get { return _perCharTransformEasyCircleBeginAngle; }
            internal set
            {
                if (_perCharTransformEasyCircleBeginAngle != value)
                {
                    _perCharTransformEasyCircleBeginAngle = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の終了角度
        /// </summary>
        public float PerCharTransformEasyCircleEndAngle
        {
            get { return _perCharTransformEasyCircleEndAngle; }
            internal set
            {
                if (_perCharTransformEasyCircleEndAngle != value)
                {
                    _perCharTransformEasyCircleEndAngle = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の円の半径X
        /// </summary>
        public float PerCharTransformEasyCircleRadiusX
        {
            get { return _perCharTransformEasyCircleRadiusX; }
            internal set
            {
                if (_perCharTransformEasyCircleRadiusX != value)
                {
                    _perCharTransformEasyCircleRadiusX = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の円の半径Y
        /// </summary>
        public float PerCharTransformEasyCircleRadiusY
        {
            get { return _perCharTransformEasyCircleRadiusY; }
            internal set
            {
                if (_perCharTransformEasyCircleRadiusY != value)
                {
                    _perCharTransformEasyCircleRadiusY = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の分割数
        /// </summary>
        public int PerCharTransformEasyCircleDivisionNumber
        {
            get { return _perCharTransformEasyCircleDivisionNumber; }
            internal set
            {
                if (_perCharTransformEasyCircleDivisionNumber != value)
                {
                    _perCharTransformEasyCircleDivisionNumber = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の文字の向き
        /// </summary>
        public float PerCharTransformEasyCircleCharAngle
        {
            get { return _perCharTransformEasyCircleCharAngle; }
            internal set
            {
                if (_perCharTransformEasyCircleCharAngle != value)
                {
                    _perCharTransformEasyCircleCharAngle = value;
                    NotifyModifyEvent(this, (int)EventKind.Modify);
                }
            }
        }

        /// <summary>
        /// 縁取りOFF
        /// </summary>
        public bool InvisibleBorderEnabled
        {
            get { return _invisibleBorderEnabled; }
            set
            {
                if (_invisibleBorderEnabled != value)
                {
                    _invisibleBorderEnabled = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 2度書き縁取り
        /// </summary>
        public bool DoubleDrawnBorderEnabled
        {
            get { return _doubleDrawnBorderEnabled; }
            set
            {
                if (_doubleDrawnBorderEnabled != value)
                {
                    _doubleDrawnBorderEnabled = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 行ごとの幅とオフセットの有効無効
        /// </summary>
        public bool LineWidthOffsetEnabled
        {
            get { return _lineWidthOffsetEnabled && !PerCharacterTransformOriginToCenter; }
            set
            {
                if (_lineWidthOffsetEnabled != value)
                {
                    _lineWidthOffsetEnabled = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 行ごとの幅
        /// </summary>
        public float[] LineWidth
        {
            get { return _lineWidth; }
            set
            {
                if (_lineWidth != value)
                {
                    _lineWidth = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 行ごとの幅の取得
        /// </summary>
        public float GetLineWidth(int index)
        {
            return _lineWidth[index];
        }

        /// <summary>
        /// 行ごとの幅の設定
        /// </summary>
        public void SetLineWidth(int index, float value)
        {
            if (_lineWidth[index] != value)
            {
                _lineWidth[index] = value;
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// 行ごとのオフセット
        /// </summary>
        public float[] LineOffset
        {
            get { return _lineOffset; }
            set
            {
                if (_lineOffset != value)
                {
                    _lineOffset = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 行ごとのオフセット
        /// </summary>
        public float GetLineOffset(int index)
        {
            return _lineOffset[index];
        }

        /// <summary>
        /// 行ごとのオフセットの設定
        /// </summary>
        public void SetLineOffset(int index, float value)
        {
            if (_lineOffset[index] != value)
            {
                _lineOffset[index] = value;
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// 拡張タグの解釈の有効無効
        /// </summary>
        public bool IsExtendedTagEnabled
        {
            get { return _isExtendedTagEnabled; }
            set
            {
                if (_isExtendedTagEnabled != value)
                {
                    _isExtendedTagEnabled = value;
                    _textBoxRenderer.ResetTagProcessor(value);
                    NotifyChangeToScene_();
                }
            }
        }

        // IVertexColor4Holder
        public IAnmAttribute VtxCol_LTIAnmAttr { get { return _textBoxAnmAttr.TopColorAnmAttr; } }
        public IAnmAttribute VtxCol_RTIAnmAttr { get { return _textBoxAnmAttr.TopColorAnmAttr; } }
        public IAnmAttribute VtxCol_LBIAnmAttr { get { return _textBoxAnmAttr.BottomColorAnmAttr; } }
        public IAnmAttribute VtxCol_RBIAnmAttr { get { return _textBoxAnmAttr.BottomColorAnmAttr; } }

        public RGBAColor VtxCol_LT
        {
            get { return _textBoxAnmAttr.TopColor; }
            set { if (_textBoxAnmAttr.TopColor != value) { _textBoxAnmAttr.TopColor = value; } }
        }

        public RGBAColor VtxCol_RT
        {
            get { return _textBoxAnmAttr.TopColor; }
            set { if (_textBoxAnmAttr.TopColor != value) { _textBoxAnmAttr.TopColor = value; } }
        }

        public RGBAColor VtxCol_LB
        {
            get { return _textBoxAnmAttr.BottomColor; }
            set { if (_textBoxAnmAttr.BottomColor != value) { _textBoxAnmAttr.BottomColor = value; } }
        }

        public RGBAColor VtxCol_RB
        {
            get { return _textBoxAnmAttr.BottomColor; }
            set { if (_textBoxAnmAttr.BottomColor != value) { _textBoxAnmAttr.BottomColor = value; } }
        }

        #region IAnmAttribute 取得関連
        public IAnmAttribute TopColorIAnmAttr     { get{ return TopColorAnmAttr;} }
        public IAnmAttribute BottomColorIAnmAttr  { get{ return BottomColorAnmAttr;} }
        public IAnmAttribute ShadowBlackColorIAnmAttr { get { return TextBoxAnmAttr.ShadowBlackColorAttr; } }
        public IAnmAttribute ShadowWhiteColorIAnmAttr { get { return TextBoxAnmAttr.ShadowWhiteColorAttr; } }
        public IAnmAttribute PerCharTransformTimeOffsetIAnmAttr { get { return PerCharTimeOffsetAnmAttr; } }
        public IAnmAttribute PerCharTransformTimeWidthIAnmAttr { get { return PerCharTimeWidthAnmAttr; } }
        public IAnmAttribute PerCharTransformSIAnmAttr { get { return TextBoxAnmAttr.PerCharTransformSAttr; } }
        public IAnmAttribute PerCharTransformTIAnmAttr {get { return TextBoxAnmAttr.PerCharTransformTAttr; }}
        public IAnmAttribute PerCharTransformRIAnmAttr { get { return TextBoxAnmAttr.PerCharTransformRAttr; } }
        public IAnmAttribute PerCharTransformTopColorIAnmAttr { get { return TextBoxAnmAttr.PerCharTransformTopColorAttr; } }
        public IAnmAttribute PerCharTransformBottomColorIAnmAttr { get { return TextBoxAnmAttr.PerCharTransformBottomColorAttr; } }

        public AnmAttribute TopColorAnmAttr
        {
            get{ return TextBoxAnmAttr.TopColorAnmAttr;}
        }

        public AnmAttribute BottomColorAnmAttr
        {
            get{ return TextBoxAnmAttr.BottomColorAnmAttr;}
        }

        public AnmAttribute PerCharTimeOffsetAnmAttr
        {
            get{ return TextBoxAnmAttr.PerCharTimeOffsetAnmAttr;}
        }

        public AnmAttribute PerCharTimeWidthAnmAttr
        {
            get { return TextBoxAnmAttr.PerCharTimeWidthAnmAttr; }
        }

        public AnmAttribute PerCharTransformSAttr
        {
            get { return TextBoxAnmAttr.PerCharTransformSAttr; }
        }

        public AnmAttribute PerCharTransformTAttr
        {
            get { return TextBoxAnmAttr.PerCharTransformTAttr; }
        }

        public AnmAttribute PerCharTransformRAttr
        {
            get { return TextBoxAnmAttr.PerCharTransformRAttr; }
        }

        public AnmAttribute PerCharTransformTopColorAttr
        {
            get { return TextBoxAnmAttr.PerCharTransformTopColorAttr; }
        }

        public AnmAttribute PerCharTransformBottomColorAttr
        {
            get { return TextBoxAnmAttr.PerCharTransformBottomColorAttr; }
        }

        #endregion IAnmAttribute 取得関連

        #endregion ITextBox メンバ

        #region IRevHWMaterialHolder メンバ
        public IRevHWMaterial[] IRevHWMaterial
        {
            get{ return new IRevHWMaterial[] { _revHWMaterial } ;}
        }
        #endregion IRevHWMaterialHolder メンバ

        #region IPaneExParamater
        /// <summary>
        /// 自身を持っているペインの参照です。
        /// </summary>
        public IPane OwnerPane
        {
            get{ return _ownerPane;}
        }

        /// <summary>
        /// マテリアル名を更新します。
        /// </summary>
        public void UpdateMatarialName()
        {
            this.Material.MaterialName = this.OwnerPane.PaneName;
        }

        /// <summary>
        /// シーン登録時の初期化処理をおこないます。
        /// </summary>
        public void OnJoinSceneInitialize()
        {
            // フォントの初期化をします。
            SetupFont_();

            // 文字毎の変換を更新します。
            _textBoxAnmAttr.DoUpdatePerCharcterTransformAtCurrentTimeOffset();
        }

        /// <summary>
        /// 編集対象に設定される直前の初期化処理
        /// </summary>
        public void FirstTimeInitialize()
        {
            // 文字毎の変換を更新します。
            _textBoxAnmAttr.DoUpdatePerCharcterTransformAtCurrentTimeOffset();
        }

        /// <summary>
        /// 警告文字列を取得します。UIの各所で表示します。
        /// </summary>
        public string WarningMsg
        {
            get
            {
                if (this.ILEFont != null)
                {
                    if (!string.IsNullOrEmpty(this.ILEFont.InitializationErrorMessage))
                    {
                        return this.ILEFont.InitializationErrorMessage;
                    }

                    if (!this.ILEFont.NWFont.IsAvailableBorder() && (this.InvisibleBorderEnabled || this.DoubleDrawnBorderEnabled))
                    {
                        return LECoreStringResMgr.Get("LECORE_UNACCEPTABLE_BORDER", this.ILEFont.FontName);
                    }

                    if ((LEFontHelper.IsBitmapFontFile(this.ILEFont.FontPath) || LEFontHelper.IsComplexFont(this.ILEFont)) &&
                        !LEFontHelper.IsAcceptableBntxPlatform(this.ILEFont, LECore.LayoutEditorCore.PlatformDetail.PlatformPreference))
                    {
                        return LECoreStringResMgr.Get("LECORE_UNACCEPTABLE_PLATFORM",
                                    Path.GetFullPath(this.ILEFont.FontPath),
                                    this.ILEFont.BntxPlatform,
                                    LECore.LayoutEditorCore.PlatformDetail.UIPlatformName)
                            + System.Environment.NewLine
                            + LECoreStringResMgr.Get("LECORE_CHECK_PLATFORM");
                    }

                    if (this.FontSizeOriginal != FVec2.Empty &&
                        this.FontSizeOriginal != new FVec2(this.ILEFont.Width, this.ILEFont.Height) &&
                        !LayoutEditorCore.KeepingSizeScaleEnabled)
                    {
                        return LECoreStringResMgr.Get("LECORE_FONT_WARN_SIZECHANGE");
                    }
                }

                return string.Empty;
            }
        }

        /// <summary>
        /// 非選択状態でも境界を書くかどうか
        /// </summary>
        public bool PaneBoundDrawEnabled
        {
            get { return true; }
        }
        #endregion IPaneExParamater
    }
}
