﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;

namespace LECore.Structures
{
    using Core;
    using Nsrif.Attributes;

    #region LETexGenMethod

    /// <summary>
    /// テクスチャ座標生成方法。
    /// </summary>
    public enum LETexGenMethod
    {
        /// <summary>ＵＶマッピング０。</summary>
        UV0,
        /// <summary>ＵＶマッピング１。</summary>
        UV1,
        /// <summary>ＵＶマッピング２。</summary>
        UV2,
        /// <summary>正射投影マッピング。</summary>
        OrthogonalProjection,
        /// <summary>ペイン基準投影マッピング。</summary>
        PaneBasedProjection,
        /// <summary>透視投影マッピング。</summary>
        PerspectiveProjection,
        /// <summary>ペイン基準透視投影マッピング。</summary>
        PaneBasedPerspectiveProjection,
    }
    #endregion

    #region TexGen

    /// <summary>
    /// ヘルパクラス
    /// </summary>
    public static class TexGenHelper
    {
        /// <summary>
        ///
        /// </summary>
        private static SharpDX.Matrix CalcProjectionScaleMtx_(
            IPaneExParamater paneEx, ITexGen texGen, ITextureImage texImg)
        {
            FVec2 projScaleAdj = new FVec2();

            if (texGen.IsFittingLayoutSizeEnabled)
            {
                FVec2 scrnSize = paneEx.GetBackGroundScreenSize();
                projScaleAdj.Set(scrnSize.X / texImg.Size.X, scrnSize.Y / texImg.Size.Y);
            }
            else if (texGen.IsFittingPaneSizeEnabled && paneEx.OwnerPane != null)
            {
                projScaleAdj.Set(paneEx.OwnerPane.Size.X / texImg.Size.X, paneEx.OwnerPane.Size.Y / texImg.Size.Y);
            }
            else
            {
                projScaleAdj.Set(FVec2.One);
            }

            projScaleAdj.X *= texGen.ProjectionScale.X;
            projScaleAdj.Y *= texGen.ProjectionScale.Y;

            return SharpDX.Matrix.Scaling(projScaleAdj.X, projScaleAdj.Y, 1.0f);
        }

        /// <summary>
        /// 投影行列を計算します。
        /// </summary>
        public static Matrix34 CalcTextureProjectionMtx(
            Matrix34 paneCenterWorldMtx, IPaneExParamater paneEx, ITexGen texGen, ITextureImage texImg)
        {
            SharpDX.Matrix mxtProjS = CalcProjectionScaleMtx_(paneEx, texGen, texImg);
            SharpDX.Matrix mxtProjT = SharpDX.Matrix.Translation(texGen.ProjectionOffset.X, texGen.ProjectionOffset.Y, 0.0f);

            SharpDX.Matrix mxtProj = mxtProjS * mxtProjT;

            // ペイン相対投影の補正行列を適用
            if (texGen.Method == LETexGenMethod.PaneBasedProjection)
            {
                SharpDX.Matrix mtxPaneSRTAdj = SharpDX.Matrix.Scaling(1.0f, -1.0f, 1.0f) * paneCenterWorldMtx.ToSharpDXMatrix34();

                // 設定によっては、平行移動成分だけを補正対象とする。
                if (!texGen.IsAdjustProjectionPaneSREnabled)
                {
                    var vecWorld = mtxPaneSRTAdj.TranslationVector;
                    mtxPaneSRTAdj = SharpDX.Matrix.Translation(vecWorld.X, vecWorld.Y, vecWorld.Z);
                }

                mxtProj = mxtProj * mtxPaneSRTAdj;
            }

            mxtProj.Invert();

            Matrix34 projMtx34 = new Matrix34();
            projMtx34.Set(mxtProj);

            return projMtx34;
        }
    }

    /// <summary>
    /// テクスチャ座標生成クラス。
    /// 外部公開インタフェース
    /// </summary>
    public interface ITexGen
    {
        IAnmAttribute IAnmAttribute { get; }
        int SlotIdx { get; }
        LETexGenMethod Method { get; }
        AttrTexCoordMtx MtxID { get; }
        ITexMtx ITexMtx { get; }

        FVec2 ProjectionScale { get; }
        FVec2 ProjectionOffset { get; }
        FVec3 ProjectionRotate { get; }
        float ProjectionFovy { get; }
        bool IsFittingLayoutSizeEnabled { get; }
        bool IsFittingPaneSizeEnabled { get; }
        bool IsAdjustProjectionPaneSREnabled { get; }

        bool IsFittingPaneSizeSettable { get; }


        bool IsUVMethod { get; }
    }


    /// <summary>
    /// テクスチャ座標生成クラス。
    /// </summary>
    internal sealed class TexGen
        :ITexGen,IDisposable
    {
       public const string TexGenNodeName = "TexGen";

        // アトリビュート
        private LETexGenMethod      _method = LETexGenMethod.UV0;

        // テクスチャ行列(アニメーションアトリビュート)
        private readonly TexMtx _texMtx;
        private readonly Material _owner;
        private readonly int _slotIdx;

        private FVec2 _projectionScale;
        private FVec2 _projectionOffset;
        private FVec3 _projectionRotate;
        private float _projectionFovy;
        private bool _isFittingLayoutSizeEnabled = true;
        private bool _IsFittingPaneSizeEnabled = false;
        private bool _isAdjustProjectionPaneSREnabled = true;


        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexGen( IMaterialTexMap ownerTex, IMaterial owner)
        {
            Ensure.Argument.NotNull(ownerTex);
            Ensure.Argument.NotNull(owner);

            _owner = owner as Material;
            _slotIdx = ownerTex.SlotIdx;

            this.ProjectionScale = FVec2.One;
            this.ProjectionOffset = FVec2.Empty;
            this.ProjectionRotate = FVec3.Empty;
            this.ProjectionFovy = 40.0f;

            string mtxName = string.Format( "texMtx_{0}", ownerTex.SlotIdx );
            // 規定値では、自分のテクスチャ番号と同一のテクスチャ行列を使用します。
            _texMtx = (ownerTex as MaterialTexMap).OwnerMaterial.GetTextureMtx( ownerTex.SlotIdx );
        }

        #region IDisposable メンバ

        public void Dispose()
        {
            _texMtx.Dispose();
        }

        #endregion

        private void NotifyModifyEvent_()
        {
            _owner.NotifyModifyEvent(_owner, (int)SceneModifyEventArgs.Kind.PaneModify);
        }

        #endregion

        #region プロパティ

        /// <summary>
        /// 持ち主マテリアル
        /// </summary>
        public IMaterial Owner
        {
            get { return _owner; }
        }

        /// <summary>
        /// スロット番号
        /// </summary>
        public int SlotIdx
        {
            get { return _slotIdx; }
        }

        public IAnmAttribute IAnmAttribute
        {
            get{ return _texMtx;}
        }

        /// <summary>
        /// 座標生成方法。
        /// </summary>
        public LETexGenMethod Method
        {
            get { return _method;  }
            set { _method = value; NotifyModifyEvent_(); }
        }

        /// <summary>
        /// マトリクスＩＤ。
        /// </summary>
        public AttrTexCoordMtx MtxID
        {
            get
            {
                return ( _texMtx != null ) ? _texMtx.AttrTexCoordMtx : AttrTexCoordMtx.Identity;
            }
        }

        /// <summary>
        /// テクスチャ行列。
        /// </summary>
        public ITexMtx ITexMtx
        {
            get { return _texMtx; }
        }

        /// <summary>
        /// ＵＶマッピング設定かどうか。
        /// </summary>
        public bool IsUVMethod
        {
            get { return LETexGenMethod.UV0 <= _method && _method <= LETexGenMethod.UV2; }
        }

        /// <summary>
        /// 投影スケール
        /// </summary>
        public FVec2 ProjectionScale
        {
            get { return _projectionScale; }
            set
            {
                if (_projectionScale != value)
                {
                    _projectionScale = value;
                    NotifyModifyEvent_();
                }
            }
        }

        /// <summary>
        /// 投影オフセット
        /// </summary>
        public FVec2 ProjectionOffset
        {
            get { return _projectionOffset; }
            set
            {
                if (_projectionOffset != value)
                {
                    _projectionOffset = value;
                    NotifyModifyEvent_();
                }
            }
        }

        /// <summary>
        /// 投影回転
        /// </summary>
        public FVec3 ProjectionRotate
        {
            get { return _projectionRotate; }
            set
            {
                if (_projectionRotate != value)
                {
                    _projectionRotate = value;
                    NotifyModifyEvent_();
                }
            }
        }

        /// <summary>
        /// 投影画角
        /// </summary>
        public float ProjectionFovy
        {
            get { return _projectionFovy; }
            set
            {
                if (_projectionFovy != value)
                {
                    _projectionFovy = value;
                    NotifyModifyEvent_();
                }
            }
        }

        /// <summary>
        /// レイアウトサイズに一致するように自動スケールするか
        /// </summary>
        public bool IsFittingLayoutSizeEnabled
        {
            get
            {
                return (this.Method != LETexGenMethod.OrthogonalProjection && this.Method != LETexGenMethod.PerspectiveProjection) ? false : _isFittingLayoutSizeEnabled;
            }

            set
            {
                if (_isFittingLayoutSizeEnabled != value)
                {
                    _isFittingLayoutSizeEnabled = value;
                    NotifyModifyEvent_();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsFittingPaneSizeEnabled
        {
            get
            {
                return !this.IsFittingPaneSizeSettable ? false : _IsFittingPaneSizeEnabled;
            }

            set
            {
                if (_IsFittingPaneSizeEnabled != value)
                {
                    _IsFittingPaneSizeEnabled = value;
                    NotifyModifyEvent_();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsAdjustProjectionPaneSREnabled
        {
            get
            {
                // ペインの SR を考慮して投影機能は透視投影でも平行投影でも同じ結果になり紛らわしいため、透視投影の場合は強制的に OFF に設定しています。
                return this.Method != LETexGenMethod.PaneBasedProjection ? false : _isAdjustProjectionPaneSREnabled;
            }

            set
            {
                if (_isAdjustProjectionPaneSREnabled != value)
                {
                    _isAdjustProjectionPaneSREnabled = value;
                    NotifyModifyEvent_();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsFittingPaneSizeSettable
        {
            get { return this.Method == LETexGenMethod.PaneBasedProjection || this.Method == LETexGenMethod.PaneBasedPerspectiveProjection; }
        }

        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(ITexGen src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _method = src.Method;
            _texMtx.Set(src.ITexMtx);
            _projectionScale.Set(src.ProjectionScale);
            _projectionOffset.Set(src.ProjectionOffset);
            _projectionRotate.Set(src.ProjectionRotate);
            _projectionFovy = src.ProjectionFovy;
            _isFittingLayoutSizeEnabled = src.IsFittingLayoutSizeEnabled;
            _IsFittingPaneSizeEnabled = src.IsFittingPaneSizeEnabled;
            _isAdjustProjectionPaneSREnabled = src.IsAdjustProjectionPaneSREnabled;

            NotifyModifyEvent_();
        }
        #endregion


    }
    #endregion
}
