﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;

namespace LECore.Structures
{
    using Core;
    using Structures.Nsrif.Attributes;

    #region ------------ 外部公開インタフェース

    /// <summary>
    /// テクスチャ
    /// 外部公開インタフェース
    /// </summary>
    public interface ITexIndirectStage
    {
        /// <summary>
        /// ステージ番号
        /// </summary>
        int Index        { get; }
        /// <summary>
        /// テクスチャマップオーダー。
        /// </summary>
        int TexMapOrder { get; }
        /// <summary>
        /// テクスチャ座標オーダー。
        /// </summary>
        int TexCoordOrder { get; }
        /// <summary>
        /// テクスチャ座標スケールS
        /// </summary>
        AttrIndScale  TexCoordScaleS { get; }
        /// <summary>
        /// テクスチャ座標スケールT
        /// </summary>
        AttrIndScale  TexCoordScaleT { get; }
    }

    /// <summary>
    ///
    /// </summary>
    public interface ITexIndirectStageSet
    {
        int NumTexIndirectStages
        {
            get;
        }

        int NumStageUsed
        {
            get;
        }

        ITexIndirectStage this[ int index]
        {
            get;
        }
    }

    #endregion ------------ 外部公開インタフェース

    /// <summary>
    ///
    /// </summary>
    internal class TexIndirectStageSet :
        LEDataNode,
        ITexIndirectStageSet
    {
        const int     _NumTexIndirectStages = 4;
        const string  TexIndirectStageSetNodeName = "IndirectStageSet";
        int           _numStageUsed = 0;

        readonly TexIndirectStage[]  _texIndirectStageSet = new TexIndirectStage[_NumTexIndirectStages];

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TexIndirectStageSet( LEDataNode ownerNode )
            :base( ownerNode, TexIndirectStageSetNodeName )
        {
            for( int i = 0; i < _NumTexIndirectStages; i++ )
            {
                _texIndirectStageSet[i] = new TexIndirectStage( this, i );
                _texIndirectStageSet[i].Index = i;
            }
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        public void Set( ITexIndirectStageSet src )
        {
            for( int i = 0; i < NumTexIndirectStages; i++ )
            {
                _texIndirectStageSet[i].Set( src[i] );
            }

            this.NumStageUsed =  src.NumStageUsed;
        }

        #region ITexIndirectStageSet

        public int NumTexIndirectStages
        {
            get{ return _NumTexIndirectStages;}
        }

        public int NumStageUsed
        {
            get{ return _numStageUsed;}
            set
            {
                Debug.Assert( value >=0 && value <= _NumTexIndirectStages );
                if( _numStageUsed != value )
                {
                    _numStageUsed = value;
                    NotifyModifyEvent( this, EventKind.Modify );
                }
            }
        }

        public ITexIndirectStage this[ int index]
        {
            get
            {
                if( index < _NumTexIndirectStages )
                {
                    return _texIndirectStageSet[index];
                }
                return null;
            }
        }
        #endregion ITexIndirectStageSet
    }

    /// <summary>
    /// TexIndirectMtx の概要の説明です。
    /// </summary>
    internal class TexIndirectStage :
        LEDataNode,
        ITexIndirectStage
    {
        const string  TexIndirectStageNodeName = "IndirectStage_{0}";
        int           _index = 0;
        int           _texMapOrder = 0;
        int           _texCoordOrder = 0;
        AttrIndScale  _texCoordScaleS = AttrIndScale.Its1;
        AttrIndScale  _texCoordScaleT = AttrIndScale.Its1;




        /// <summary>
        /// 変更の通知をします。
        /// </summary>
        void NotifyChange_()
        {
            NotifyModifyEvent( this, EventKind.Modify );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TexIndirectStage( LEDataNode ownerNode, int index )
            : base( ownerNode, string.Format( TexIndirectStageNodeName, index.ToString() ) )
        {

        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        public void Set( ITexIndirectStage src )
        {
            this.Index           = src.Index;
            this.TexCoordOrder   = src.TexCoordOrder;
            this.TexCoordScaleS  = src.TexCoordScaleS;
            this.TexCoordScaleT  = src.TexCoordScaleT;
            this.TexMapOrder     = src.TexMapOrder;
        }

        #region ITexIndirectStage メンバ

        /// <summary>
        /// ステージ番号
        /// </summary>
        public    int Index
        {
            get{ return _index;}
            set
            {
                if( _index != value )
                {
                    _index = value;
                    NotifyModifyEvent( this, EventKind.Modify );
                }
            }
        }
        /// <summary>
        /// テクスチャマップオーダー。
        /// </summary>
        public    int TexMapOrder
        {
            get{ return _texMapOrder;}
            set
            {
                if( _texMapOrder != value )
                {
                    _texMapOrder = value;
                    NotifyModifyEvent( this, EventKind.Modify );
                }
            }
        }
        /// <summary>
        /// テクスチャ座標オーダー。
        /// </summary>
        public    int TexCoordOrder
        {
            get{ return _texCoordOrder;}
            set
            {
                if( _texCoordOrder != value )
                {
                    _texCoordOrder = value;
                    NotifyModifyEvent( this, EventKind.Modify );
                }
            }
        }
        /// <summary>
        /// テクスチャ座標スケールS
        /// </summary>
        public      AttrIndScale  TexCoordScaleS
        {
            get{ return _texCoordScaleS;}
            set
            {
                if( _texCoordScaleS != value )
                {
                    _texCoordScaleS = value;
                    NotifyModifyEvent( this, EventKind.Modify );
                }
            }
        }
        /// <summary>
        /// テクスチャ座標スケールT
        /// </summary>
        public     AttrIndScale  TexCoordScaleT
        {
            get{ return _texCoordScaleT;}
            set
            {
                if( _texCoordScaleT != value )
                {
                    _texCoordScaleT = value;
                    NotifyModifyEvent( this, EventKind.Modify );
                }
            }
        }

        #endregion ITexIndirectStage メンバ
    }
}

