﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;

namespace LECore.Structures
{
    using Core;
    using LECore.Structures.Nsrif.Attributes;

    /// <summary>
    /// インダイレクト行列セットです。
    /// </summary>
    internal class TexIndirectMtxSet
    {
        const int NumIndirectTextureMtx = 1;
        const string TexIndirectMtxSetNodeName = "IndirectMtxSet";

        readonly TexMtx[] _texIndirectMtxSet = new TexMtx[NumIndirectTextureMtx];

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TexIndirectMtxSet(LEDataNode ownerNode)
        {

            for (int i = 0; i < _texIndirectMtxSet.Length; i++)
            {
                _texIndirectMtxSet[i] = new TexMtx(ownerNode, "IndirectMtx");

                if (LayoutEditorCore.PlatformDetail.IndirectTextureEnabled)
                {
                    // インダイレクト行列は Trans無効
                    AnmAttributeHelper.SetActiveRecursive(_texIndirectMtxSet[i].TransAnmAttr, false);
                }
                else
                {
                    AnmAttributeHelper.SetActiveRecursive(_texIndirectMtxSet[i], false);
                }
            }
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        public void Set(TexIndirectMtxSet src)
        {
            for (int i = 0; i < _texIndirectMtxSet.Length; i++)
            {
                _texIndirectMtxSet[i].Set(src[i]);
            }
        }

        public void SetIsActiveAttributeAll(bool flag)
        {
            for (int i = 0; i < _texIndirectMtxSet.Length; i++)
            {
                _texIndirectMtxSet[i].IsActiveAttribute = flag;
            }
        }

        /// <summary>
        /// 行列を設定します。
        /// </summary>
        public void SetIndirectMtx(ITexMtx srcMtx, int idx)
        {
            TexMtx dstMtx = this[idx] as TexMtx;
            if (dstMtx != null)
            {
                dstMtx.Set(srcMtx);
            }
            else
            {
                Debug.Assert(false);
            }
        }

        #region IIndirectTexMtxSet

        public int NumTexIndirectMtx
        {
            get { return NumIndirectTextureMtx; }
        }

        public ITexMtx this[int index]
        {
            get
            {
                if (index < _texIndirectMtxSet.Length)
                {
                    return _texIndirectMtxSet[index];
                }
                return null;
            }
        }
        #endregion IIndirectTexMtxSet
    }

    /// <summary>
    /// マテリアル
    /// 外部モジュール公開インタフェース
    /// </summary>
    public interface IRevHWMaterial
    {
        /// <summary>
        /// アニメーションアトリビュートを取得します。
        /// </summary>
        IAnmAttribute AttributeRef { get;}
        IColorAssignSet IColorAssignSet { get;}

        IPEData IPEData { get;}
        ITevData ITevData { get;}

        string MaterialName { get;}
        IMaterialTexMap[] IMaterialTexMapSet { get;}
        ITexMtx[] IIndirectMaterialTexMtxSet { get; }
        IMaterial IMaterial { get;}
        ICombinerUserShader ICombinerUserShader { get; }

        int UserMaxNumTexCoord { get;}

        /// <summary>
        /// マテリアル詳細編集が有効かどうか
        /// </summary>
        bool LowLevelCombinerSettingsEnabled {get;}

        /// <summary>
        /// コンバイナユーザーシェーダー編集が有効かどうか
        /// </summary>
        bool CombinerUserShaderSettingsEnabled { get; }
    }

    /// <summary>
    /// Revolution ハードウェア固有の表現による、マテリアル情報。
    /// アニメーションアトリビュート
    /// </summary>
    internal class RevHWMaterialAttr : AnmAttribute
    {
        #region -------------- 定数 --------------

        static readonly AnmAttrDescripter TevDataAttrDescs =
            new AnmAttrDescripter( AttributeType.Combined, "TevData", null, null );

        static readonly AnmAttrDescripter SelfAttrDesc =
            new AnmAttrDescripter(AttributeType.Combined, "LowLevelMaterial", null, null);

        #endregion -------------- 定数 --------------

        #region -------------- プロパティ --------------

        public TevData TevData
        {
            get { return FindSubAttributeByIdx(0) as TevData; }
        }
        #endregion -------------- プロパティ --------------

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RevHWMaterialAttr(LEDataNode owner, Material material)
            : base(owner, null, SelfAttrDesc)
        {
            RevHWMaterial ownerMat = owner as RevHWMaterial;
            Debug.Assert(ownerMat != null);

            new TevData(this, TevDataAttrDescs, ownerMat);
        }
    }

    /// <summary>
    /// Revolution ハードウェア固有の表現による、マテリアル情報。
    /// 通常のマテリアルと重複する情報を持つことがあります。
    ///
    /// 本クラスは、通常のマテリアルが変更された場合、
    /// 変更イベントを処理して、内容を同一に保つ責任を持ちます。
    /// 参照関係は、RevHWMaterial => Material の片方向である点に注意が必要です。
    /// ( Material クラス 側 は本クラスが存在することさえ知らない）
    ///
    /// </summary>
    internal class RevHWMaterial :
        LEDataNode,
        IRevHWMaterial,           // 外部モジュール公開インタフェース
        IDisposable
    {
        public const string RevHWMaterialNodeName = "DetailedMaterial";

        /// <summary>
        /// カラーアサイン情報
        /// </summary>
        readonly ColorAssignSet _colorAssignSet = null;
        /// <summary>
        /// PE 関連パラメータ
        /// </summary>
        readonly Material _material = null;
        readonly PEData _PEData = null;
        readonly RevHWMaterialAttr _revHWMatAttr = null;
        readonly TexIndirectMtxSet _indirectMtxSet = null;
        bool _lowLevelCombinerSettingsEnabled = false;
        bool _combinerUserShaderSettingsEnabled = false;
        readonly CombinerUserShader _combinerUserShader = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RevHWMaterial( LEDataNode owner, Material material )
            : base( owner, RevHWMaterialNodeName )
        {
            _material = material;
            _material.OnTextureCountChanged += new Action(() => UpdataTevStageNumber_());
            _colorAssignSet = new ColorAssignSet( this );
            _indirectMtxSet = new TexIndirectMtxSet(this);
            _PEData = new PEData( this );
            _revHWMatAttr = new RevHWMaterialAttr( this, material );
            _combinerUserShader = new CombinerUserShader(this);
        }

        /// TEVステージ数を調整します。
        /// </summary>
        private void UpdataTevStageNumber_()
        {
            // 詳細モードのときは、マニュアル設定されるので調整しません。
            if (!(LayoutEditorCore.PlatformDetail.LowLevelCombinerSettingsEnabled && LowLevelCombinerSettingsEnabled))
            {
                (this.TevData.TevStages as TevStages).NumStages = Math.Max(0, this._material.IMaterialTexMapSet.Length - 1);
            }
        }

        /// <summary>
        /// リソースを開放します。
        /// </summary>
        public void Dispose()
        {
            _revHWMatAttr.Dispose();
        }

        /// <summary>
        /// マテリアルの使用するテクスチャを追加します。
        /// </summary>
        public MaterialTexMap RegisterMatTexture( string texName, int index, AttrTextureResourceType resType )
        {
            return _material.RegisterMatTexture( texName, index, resType );
        }

        /// <summary>
        /// マテリアルの使用するテクスチャを削除します。
        /// </summary>
        public void RemoveMatTexture( int index )
        {
            _material.RemoveMatTexture( index );
        }

        /// <summary>
        /// レジスタカラーアニメーションアトリビュートを集めます。
        /// </summary>
        public void CollectTEVRegisterColorAnimAttrSet( List<IAnmAttribute> attrSet )
        {
        }

        #region ------------- プロパティ -------------

        public ColorAssignSet ColorAssignSet
        {
            get { return _colorAssignSet; }
        }

        public PEData PEData
        {
            get { return _PEData; }
        }

        public TevData TevData
        {
            get { return _revHWMatAttr.TevData; }
        }

        public CombinerUserShader CombinerUserShader
        {
            get { return _combinerUserShader; }
        }

        internal TexIndirectMtxSet TexIndirectMtxSet
        {
            get { return _indirectMtxSet; }
        }

        internal RevHWMaterialAttr RevHWMaterialAttr
        {
            get { return _revHWMatAttr; }
        }

        /// <summary>
        /// マテリアル詳細編集が有効かどうか
        /// </summary>
        public bool LowLevelCombinerSettingsEnabled
        {
            get
            {
                return _lowLevelCombinerSettingsEnabled;
            }

            internal set
            {
                if (_lowLevelCombinerSettingsEnabled != value)
                {
                    _lowLevelCombinerSettingsEnabled = value;
                    NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.MaterialEditMode);
                }
            }
        }

        /// <summary>
        /// コンバイナユーザーシェーダ編集が有効かどうか
        /// </summary>
        public bool CombinerUserShaderSettingsEnabled
        {
            get
            {
                return _combinerUserShaderSettingsEnabled;
            }

            internal set
            {
                if (_combinerUserShaderSettingsEnabled != value)
                {
                    _combinerUserShaderSettingsEnabled = value;
                    NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.MaterialEditMode);
                }
            }
        }

        #endregion ------------- プロパティ -------------


        #region ------------ IRevHWMaterial メンバ ------------
        /// <summary>
        /// アニメーションアトリビュートを取得します。
        /// </summary>
        public IAnmAttribute AttributeRef
        {
            get { return _revHWMatAttr; }
        }

        /// <summary>
        /// IColorAssign
        /// </summary>
        public IColorAssignSet IColorAssignSet
        {
            get { return _colorAssignSet; }
        }

        /// <summary>
        /// IPEData
        /// </summary>
        public IPEData IPEData
        {
            get { return _PEData; }
        }

        public ICombinerUserShader ICombinerUserShader
        {
            get { return _combinerUserShader; }
        }

        /// <summary>
        /// TEVData
        /// </summary>
        public ITevData ITevData
        {
            get { return _revHWMatAttr.TevData as ITevData; }
        }

        /// <summary>
        /// テクスチャマップ配列を取得します。
        /// </summary>
        public IMaterialTexMap[] IMaterialTexMapSet
        {
            get
            {
                return _material.IMaterialTexMapSet;
            }
        }

        /// <summary>
        /// インダイレクト行列です。
        /// </summary>
        public ITexMtx[] IIndirectMaterialTexMtxSet
        {
            get
            {
                List<ITexMtx> result = new List<ITexMtx>();
                for(int i = 0; i < _indirectMtxSet.NumTexIndirectMtx;i++)
                {
                    result.Add(_indirectMtxSet[i]);
                }

                return result.ToArray();
            }
        }

        /// <summary>
        /// マテリアル名
        /// </summary>
        public string MaterialName
        {
            get { return _material.MaterialName; }
            set { _material.MaterialName = value; NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify); }
        }

        /// <summary>
        /// 使用可能なテクスチャ座標の数
        /// </summary>
        public int UserMaxNumTexCoord
        {
            get { return _material.UserMaxNumTexCoord; }
        }

        public IMaterial IMaterial
        {
            get { return _material; }
        }

        #endregion ------------ IRevHWMaterial メンバ ------------
    }
}
