﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Drawing.Drawing2D;
using System.Text.RegularExpressions;

namespace LECore.Structures
{
    using Core;
    using LECore.Util;

    /// <summary>
    /// PaneAnimationHelper
    /// </summary>
    public static class PaneAnimationHelper
    {
        /// <summary>
        /// ペインが持つすべてのアクティブアトリビュートを取得します。
        /// </summary>
        public static IAnmAttribute[] GetAllActiveAnmAttribute(IPane pane)
        {
            List<IAnmAttribute> attributeSet = new List<IAnmAttribute>();
            LEDataNode dataNode = pane as LEDataNode;

            GetAllAnmAttribute_(dataNode, attributeSet);

            return attributeSet.ToArray();
        }

        /// <summary>
        /// アニメーションの対象となるアトリビュートのセットを取得します。
        /// </summary>
        public static IAnmAttribute[] GetAnimationTargetAttributeSet(
            this IPane pane, ParamaterKind fileType)
        {
            List<IAnmAttribute> attrSet = new List<IAnmAttribute>();
            switch (fileType)
            {
                case ParamaterKind.Animation_TexturePattern:
                case ParamaterKind.Animation_TextureSRT:
                case ParamaterKind.Animation_MaterialColors:
                    pane.IMaterial.ToList().ForEach((mat) => attrSet.AddRange(mat.GetAnimationTargetAttributeSet(fileType)));
                    pane.EnumrateDetailMaterial().ToList().ForEach((mat) => attrSet.AddRange(mat.GetAnimationTargetAttributeSet(fileType)));
                    break;
                case ParamaterKind.Animation_IndirectTextureSRT:
                case ParamaterKind.Animation_AlphaTest:
                    PaneHelper.GetRevHWMatFromPane(pane).ToList().ForEach((mat) => attrSet.AddRange(mat.GetAnimationTargetAttributeSet(fileType)));
                    break;
                case ParamaterKind.Animation_VertexColors: attrSet.AddRange(GetAnimAttrSet_VertexColors_(pane)); break;
                case ParamaterKind.Animation_PaneSRT: attrSet.AddRange(GetAnimAttrSet_PaneSRT_(pane)); break;
                case ParamaterKind.Animation_Visivility: attrSet.AddRange(GetAnimAttrSet_Visivility_(pane)); break;
                case ParamaterKind.Animation_FontShadowColor: attrSet.AddRange(GetAnimAttrSet_FontShadow_(pane)); break;
                case ParamaterKind.Animation_PerCharacterTransform: attrSet.AddRange(GetAnimAttrSet_PerCharTransformEvalTimeOffset_(pane)); break;
                case ParamaterKind.Animation_PerCharacterTransformCurve: attrSet.AddRange(GetAnimAttrSet_PerCharTransformCurve_(pane)); break;
                case ParamaterKind.Animation_Window: attrSet.AddRange(GetAnimAttrSet_Window(pane)); break;
                case ParamaterKind.Animation_ExtUserData: attrSet.AddRange(GetAnimAttrSet_ExtUserData_(pane)); break;
                case ParamaterKind.Animation_Pane:
                case ParamaterKind.Animation_Material:
                case ParamaterKind.Animation_All:
                    break;
                case ParamaterKind.Animation_MaskTextureSRT:
                    attrSet.AddRange(GetAnimAttrSet_MaskTextureSRT(pane));
                    break;
                case ParamaterKind.Animation_DropShadow:
                    attrSet.AddRange(GetAnimAttrSet_DropShadow(pane));
                    break;
                case ParamaterKind.Animation_ProceduralShape:
                    attrSet.AddRange(GetAnimAttrSet_ProceduralShape(pane));
                    break;
                default:
                    Debug.Assert(false);
                    break;
            }

            return attrSet.ToArray();
        }

        /// <summary>
        /// ペインSRT
        /// </summary>
        static IAnmAttribute[] GetAnimAttrSet_PaneSRT_(IPane pane)
        {
            ArrayList attrSet = new ArrayList();

            attrSet.Add(pane.IPaneAttribute.TransIAttr);
            attrSet.Add(pane.IPaneAttribute.SizeIAttr);
            attrSet.Add(pane.IPaneAttribute.ScaleIAttr);
            attrSet.Add(pane.IPaneAttribute.RotAngIAttr);

            return attrSet.ToArray(typeof(IAnmAttribute)) as IAnmAttribute[];
        }

        /// <summary>
        /// ビシビリティ
        /// </summary>
        static IAnmAttribute[] GetAnimAttrSet_Visivility_(IPane pane)
        {
            ArrayList attrSet = new ArrayList();

            attrSet.Add(pane.IPaneAttribute.VisibleIAttr);

            return attrSet.ToArray(typeof(IAnmAttribute)) as IAnmAttribute[];
        }

        /// <summary>
        /// 頂点カラー
        /// </summary>
        static IAnmAttribute[] GetAnimAttrSet_VertexColors_(IPane pane)
        {
            ArrayList attrSet = new ArrayList();

            // 透明度アトリビュート
            attrSet.Add(pane.IPaneAttribute.TransparencyIAttr);

            // 種類ごとに...
            switch (pane.PaneKind)
            {
                case PaneKind.Textbox:
                    attrSet.Add(pane.ITextBox.TopColorIAnmAttr);
                    attrSet.Add(pane.ITextBox.BottomColorIAnmAttr);
                    break;
                case PaneKind.Picture:
                case PaneKind.Window:
                    IVertexColor4Holder vtx4 = pane.IPaneExParamater as IVertexColor4Holder;
                    attrSet.Add(vtx4.VtxCol_LBIAnmAttr);
                    attrSet.Add(vtx4.VtxCol_LTIAnmAttr);
                    attrSet.Add(vtx4.VtxCol_RBIAnmAttr);
                    attrSet.Add(vtx4.VtxCol_RTIAnmAttr);
                    break;
                case PaneKind.Null:
                case PaneKind.Bounding:
                case PaneKind.Capture:
                case PaneKind.Alignment:
                case PaneKind.Scissor:
                    // Do Nothing.
                    break;
            }
            return attrSet.ToArray(typeof(IAnmAttribute)) as IAnmAttribute[];
        }

        /// <summary>
        /// フォント影
        /// </summary>
        static IAnmAttribute[] GetAnimAttrSet_FontShadow_(IPane pane)
        {
            var textBox = pane.ITextBox as TextBox;
            if (textBox == null)
            {
                return new IAnmAttribute[0];
            }

            return new IAnmAttribute[] {
                textBox.TextBoxAnmAttr.ShadowBlackColorAttr,
                textBox.TextBoxAnmAttr.ShadowWhiteColorAttr };
        }

        /// <summary>
        /// 文字カーブ評価時間
        /// </summary>
        static IAnmAttribute[] GetAnimAttrSet_PerCharTransformEvalTimeOffset_(IPane pane)
        {
            var textBox = pane.ITextBox as TextBox;
            if (textBox == null)
            {
                return new IAnmAttribute[0];
            }

            return new IAnmAttribute[] {
                textBox.PerCharTransformTimeOffsetIAnmAttr,
                textBox.PerCharTransformTimeWidthIAnmAttr
            };
        }

        /// <summary>
        /// 文字カーブ評価時間
        /// </summary>
        static IAnmAttribute[] GetAnimAttrSet_PerCharTransformCurve_(IPane pane)
        {
            var textBox = pane.ITextBox as TextBox;
            if (textBox == null)
            {
                return new IAnmAttribute[0];
            }

            return new IAnmAttribute[] {
                textBox.PerCharTransformSAttr,
                textBox.PerCharTransformTAttr,
                textBox.PerCharTransformRAttr,
                textBox.PerCharTransformTopColorAttr,
                textBox.PerCharTransformBottomColorAttr
            };
        }

        static IAnmAttribute[] GetAnimAttrSet_Window(IPane pane)
        {
            var wnd = pane.ILEWindow as LEWindow;
            if (wnd == null)
            {
                return new IAnmAttribute[0];
            }

            return new IAnmAttribute[] { wnd.FrameSizeAnmAttr };
        }

        /// <summary>
        /// 拡張ユーザーデータ
        /// </summary>
        static IAnmAttribute[] GetAnimAttrSet_ExtUserData_(IPane pane)
        {
            ArrayList attrSet = new ArrayList();

            foreach (IUserDataElement element in pane.IUserDataHolder.UserDataElementSet)
            {
                if (element.IAnmAttribute != null)
                {
                    attrSet.Add(element.IAnmAttribute);
                }
            }

            return attrSet.ToArray(typeof(IAnmAttribute)) as IAnmAttribute[];
        }

        /// <summary>
        /// マスク
        /// </summary>
        static IAnmAttribute[] GetAnimAttrSet_MaskTextureSRT(IPane pane)
        {
            var mask = pane.IMask as Mask;
            if (mask == null)
            {
                return new IAnmAttribute[0];
            }

            return new IAnmAttribute[] {
                mask.MtxTransAttr,
                mask.MtxScaleAttr,
                mask.MtxRotateAttr,
            };
        }

        /// <summary>
        /// ドロップシャドウ
        /// </summary>
        static IAnmAttribute[] GetAnimAttrSet_DropShadow(IPane pane)
        {
            var dropShadow = pane.IDropShadow as DropShadow;
            if (dropShadow == null)
            {
                return new IAnmAttribute[0];
            }

            return new IAnmAttribute[] {
                dropShadow.StrokeSizeAttr,
                dropShadow.StrokeColorAttr,
                dropShadow.OuterGlowColorAttr,
                dropShadow.OuterGlowSpreadAttr,
                dropShadow.OuterGlowSizeAttr,
                dropShadow.DropShadowColorAttr,
                dropShadow.DropShadowAngleAttr,
                dropShadow.DropShadowDistanceAttr,
                dropShadow.DropShadowSpreadAttr,
                dropShadow.DropShadowSizeAttr,
            };
        }

        /// <summary>
        /// 角丸
        /// </summary>
        static IAnmAttribute[] GetAnimAttrSet_ProceduralShape(IPane pane)
        {
            if (pane.IPicture == null)
            {
                return new IAnmAttribute[0];
            }
            var proceduralShape = pane.IPicture.IProceduralShape as ProceduralShape;
            if (proceduralShape== null)
            {
                return new IAnmAttribute[0];
            }

            return new IAnmAttribute[] {
                proceduralShape.ExpAttr,
                proceduralShape.RadiusAttr,
                proceduralShape.InnerStrokeSizeAttr,
                proceduralShape.InnerStrokeColorAttr,
                proceduralShape.InnerShadowColorAttr,
                proceduralShape.InnerShadowAngleAttr,
                proceduralShape.InnerShadowDistanceAttr,
                proceduralShape.InnerShadowSizeAttr,
                proceduralShape.ColorOverlayColorAttr,
                proceduralShape.GradationOverlayControlPointAttrArray[0],
                proceduralShape.GradationOverlayControlPointAttrArray[1],
                proceduralShape.GradationOverlayControlPointAttrArray[2],
                proceduralShape.GradationOverlayControlPointAttrArray[3],
                proceduralShape.GradationOverlayColorAttrArray[0],
                proceduralShape.GradationOverlayColorAttrArray[1],
                proceduralShape.GradationOverlayColorAttrArray[2],
                proceduralShape.GradationOverlayColorAttrArray[3],
                proceduralShape.IGradationOverlayAngleAttr,
                proceduralShape.OuterShadowColorAttr,
                proceduralShape.OuterShadowAngleAttr,
                proceduralShape.OuterShadowDistanceAttr,
                proceduralShape.OuterShadowSizeAttr,
            };
        }


        /// <summary>
        ///
        /// </summary>
        static void GetAllAnmAttribute_(IAnmAttribute attribute, List<IAnmAttribute> set)
        {
            if (attribute.IsActiveAttribute)
            {
                set.Add(attribute);
            }

            for (int i = 0; i < attribute.NumSubAttribute; i++)
            {
                IAnmAttribute subAttribute = attribute.FindSubAttributeByIdx(i);
                GetAllAnmAttribute_(subAttribute, set);
            }
        }

        /// <summary>
        /// ペインがアニメーションを持っているか調査します。
        /// </summary>
        static public bool CheckPaneHasPaneAnmKey(IPane pane)
        {
            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                if (CheckPaneHasPaneAnmKey(pane, kind))
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// ペインが指定したキーを持っているか調査します。
        /// </summary>
        static public bool CheckPaneHasPaneAnmKey(IPane pane, ParamaterKind fileType)
        {
            return GetAnimationTargetAttributeSet(pane, fileType).Any((attr) => attr.CheckHaveKeyRecursiveInAllTag() || attr.CheckParameterizedAnimRecursive());
        }

        /// <summary>
        /// ペインが指定したキーを持っているか調査します。
        /// </summary>
        static public bool CheckPaneHasPaneAnmKeyCurrentTag(IPane pane, ParamaterKind fileType)
        {
            return GetAnimationTargetAttributeSet(pane, fileType).Any((attr) => attr.CheckHaveKeyRecursive() || attr.CheckParameterizedAnimRecursive());
        }

        /// <summary>
        /// 指定時間にキーを持っているか再帰的に調べます
        /// </summary>
        static public bool ExistKeyRecursive(IAnmAttribute attr, float time)
        {
            bool ret = false;

            if (attr.ICurrentAnimationCurve?.FindKeyExistAt(time) != null)
            {
                return true;
            }

            for (int i = 0; i < attr.NumSubAttribute; i++)
            {
                ret |= ExistKeyRecursive(attr.FindSubAttributeByIdx(i), time);
            }

            return ret;
        }

        /// <summary>
        /// アニメーションの再評価を行ないます。
        /// </summary>
        static public void EvaluateAnimationForce(IPane pane, int time)
        {
            if (pane == null) return;

            foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
            {
                if(curveAttr.IsReferenceCurves)
                {
                    continue;
                }

                curveAttr.UpdateAttributeValue((int)time);
            }
        }
    }
}
