﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

namespace LECore.Structures
{
    /// <summary>
    /// FontHelper の概要の説明です。
    /// </summary>
    public static class LEFontHelper
    {
        // フォントファイル選択ダイアログフィルタ文字列
        public static readonly string OPEN_FONT_FILE_FILTER_STR;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        static LEFontHelper()
        {
            OPEN_FONT_FILE_FILTER_STR = string.Format(
                "Font file. ({0}; {1}; {2})|{0}; {1}; {2}| all files (*.*)|*.*",
                AppConstants.BFFNTFileExtPattern,
                AppConstants.FCPXFileExtPattern,
                AppConstants.FFNTFileExtPattern);
        }

        /// <summary>
        /// フォント登録を試行します。
        /// </summary>
        internal static void TryRegisterFont(this ILEFontManager fontManger, ILEFont font)
        {
            if (font != null && fontManger.FindFontByName(font.FontName) == null)
            {
                (fontManger as FontManager).CreateLEFont(font.FontName, font.FontPath, font.FontSettings);
            }
        }

        /// <summary>
        /// フォント登録を試行します。
        /// <remarks>
        /// 外部モジュールに公開される関数です。
        /// </remarks>
        /// </summary>
        public static void TryCreateFont(this ILEFontManager fontManger, ILEFont font)
        {
            if (font != null && fontManger.FindFontByName(font.FontName) == null)
            {
                (fontManger as FontManager).CreateILEFont(font.FontName, font.FontPath, font.FontSettings);
            }
        }

        /// <summary>
        /// ファイルが見つからなかったかどうか
        /// </summary>
        public static bool IsFileNotFound(this ILEFont font)
        {
            if (font == null) return true;
            return font.NWFont == null;
        }

        /// <summary>
        /// 複合フォントかどうか
        /// </summary>
        public static bool IsComplexFont(this ILEFont font)
        {
            if (font == null)
            {
                return false;
            }

            return IsComplexFontFile(font.FontPath);
        }

        /// <summary>
        /// スケーラブルフォントかどうか
        /// </summary>
        public static bool IsScalableFont(this ILEFont font)
        {
            return font != null && font.FontSettings is IScalableFontSettings;
        }

        /// <summary>
        /// プロジェクト設定に記述されたフォントかどうか
        /// </summary>
        public static bool IsProjectFont(this ILEFont font)
        {
            if (font.FontSettings != null && IsComplexFontFile(font.FontPath) == false)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        private static bool CompareExtension(string filepath, string ext)
        {
            var fileext = Path.GetExtension(filepath);
            return (fileext != null) && fileext.Equals(ext, StringComparison.OrdinalIgnoreCase);
        }

        // TODO: multiScalableFont を含むか分かりずらいので整理する。IsScalableFont とも挙動が違う
        /// <summary>
        ///
        /// </summary>
        public static bool IsScalableFontFile(string fontPath)
        {
            return
                CompareExtension(fontPath, AppConstants.TTFFileExtExt) ||
                CompareExtension(fontPath, AppConstants.OTFFileExtExt) ||
                IsBfttfScalableFontFile(fontPath) ||
                IsBfotfScalableFontFile(fontPath);
        }


        /// <summary>
        ///
        /// </summary>
        public static bool IsBfttfScalableFontFile(string fontPath)
        {
            return CompareExtension(fontPath, AppConstants.BFTTFFileExt);
        }

        /// <summary>
        ///
        /// </summary>
        public static bool IsBfotfScalableFontFile(string fontPath)
        {
            return CompareExtension(fontPath, AppConstants.BFOTFFileExtExt);
        }

        /// <summary>
        ///
        /// </summary>
        public static bool IsComplexFontFile(string fontPath)
        {
            return CompareExtension(fontPath, AppConstants.FCPXFileExt);
        }

        public static bool IsFontConvertSettingsFile(string fontPath)
        {
            return CompareExtension(fontPath, AppConstants.FFNTFileExt);
        }

        public static bool IsBitmapFontFile(string fontPath)
        {
            return CompareExtension(fontPath, AppConstants.BFFNTFileExt);
        }

        public static bool IsAcceptableBntxPlatform(ILEFont font, PlatformPreference preference)
        {
            return IsAcceptableBntxPlatform(((LEFont)font).BntxPlatform, preference);
        }

        public static bool IsAcceptableBntxPlatform(string bntxPlatform, PlatformPreference preference)
        {
            return bntxPlatform == null ||
                preference.AcceptableBntxPlatformNames.Contains(bntxPlatform.TrimEnd());
        }

        // 末端の要素を列挙します
        public static IEnumerable<NW4R.Font.INWFont> GetLeafs(NW4R.Font.INWFont font)
        {
            var pairFont = font as NW4R.Font.PairFont;
            if (pairFont != null)
            {
                return GetLeafs(pairFont.FirstFont).Concat(GetLeafs(pairFont.SecondFont));
            }
            else
            {
                return Enumerable.Repeat(font, 1);
            }
        }
    }
}
