﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Windows.Forms;
using System.Linq;
using LECore.Save_Load;
using LECore.Manipulator;
using LECore.Structures.Core;
using System.Diagnostics;

namespace LECore.Structures
{
    /// <summary>
    /// レイアウトファイルの更新ユーティリティです。
    /// </summary>
    public static class FileUpdateHelper
    {
        /// <summary>
        /// 更新パラメータ
        /// </summary>
        public struct UpdateParam
        {
            /// <summary>
            /// ファイルパスまたはフォルダパス
            /// </summary>
            public string Path;

            /// <summary>
            /// オーナーウインドウ
            /// </summary>
            public IWin32Window Owner;

            /// <summary>
            /// ダイアログのタイトル
            /// </summary>
            public string DialogTitle;

            /// <summary>
            /// 確認ダイアログを表示するか
            /// </summary>
            public bool DoShowConfirmDialog;

            /// <summary>
            /// 確認ダイアログの文字列
            /// </summary>
            public string ConfirmMessage;

            /// <summary>
            /// パーツレイアウトを更新するか
            /// </summary>
            public bool DoUpdatePartsLayout;

            /// <summary>
            /// アニメーションを更新するか
            /// </summary>
            public bool DoUpdateAnimation;

            /// <summary>
            /// サムネイル画像も更新するか
            /// </summary>
            public bool DoSaveThumbnail;

            /// <summary>
            /// 確認メッセージ表示デリゲート
            /// </summary>
            public Func<string, bool> ShowConfirmMessage;

            /// <summary>
            /// 更新処理の開始デリゲート
            /// </summary>
            public Action BeginProcess;

            /// <summary>
            /// 途中経過表示デリゲート
            /// </summary>
            public Action<string> ReportMessage;

            /// <summary>
            /// 更新処理の終了デリゲート
            /// </summary>
            public Action EndProcess;

            /// <summary>
            /// 処理を中断するか確認するデリゲート
            /// </summary>
            public Func<bool> IsCancelled;

            /// <summary>
            /// エラー表示デリゲート
            /// </summary>
            public Action<string> ReportError;
        }

        /// <summary>
        /// ファイルまたはフォルダを一括更新します。
        /// </summary>
        /// <param name="param">更新パラメータ</param>
        public static void UpdateAll(UpdateParam param)
        {
            // nullチェックを省略するためのReportMessageデリゲート
            Action<string> reportMessage = param.ReportMessage != null ? param.ReportMessage : (string msg) => {};

            var subScenesBeforeUpdateAll = LayoutEditorCore.Scene.ISubSceneSet;

            List<string> filePaths = new List<string>();

            if (File.Exists(param.Path))
            {
                // 単体のファイルを追加
                filePaths.Add(param.Path);
            }
            else if (Directory.Exists(param.Path))
            {
                // フォルダ以下のflytファイルを列挙
                try
                {
                    filePaths.AddRange(Directory.GetFiles(param.Path, AppConstants.LayoutFileExtPattern, SearchOption.AllDirectories));
                }
                catch (Exception e)
                {
                    if (param.ReportError != null)
                    {
                        param.ReportError(string.Format("ERROR !!! - {0}", e.Message));
                    }

                    return;
                }
            }

            // 確認ダイアログを表示
            if (param.DoShowConfirmDialog && param.ShowConfirmMessage != null)
            {
                string confirmMsg = string.Format(param.ConfirmMessage, filePaths.Count);
                bool confirmResult = param.ShowConfirmMessage(confirmMsg);

                if (confirmResult == false)
                {
                    return;
                }
            }

            bool partsLoadResult = true;  // 部品レイアウトの読み込み結果

            LEReportMsgHandler msgHandler = (sender, args) =>
            {
                if (args is RlytConverter.ErrorLoadPaneMessageArgs)
                {
                    partsLoadResult = false;
                }
            };

            // 部品レイアウトの読み込みエラーをハンドルする
            if (param.DoUpdatePartsLayout)
            {
                LECore.LayoutEditorCore.MsgReporter.PreReportMsg += msgHandler;
            }

            // 処理を開始
            if (param.BeginProcess != null)
            {
                param.BeginProcess();
            }

            int numProccessedFiles = 0;
            var loadedSubScenes = new List<ISubScene>();
            {
                foreach (string filePath in filePaths)
                {
                    reportMessage(LECoreStringResMgr.Get("SYSTEM_PROCCESSING") + " ..." + filePath + Environment.NewLine);

                    try
                    {
                        partsLoadResult = true;

                        var loadOption = param.DoUpdateAnimation ? LayoutEditorCore.LoadOption.TryToOpenRlan : LayoutEditorCore.LoadOption.None;
                        ISubScene loadedSubScene = LayoutEditorCore.CreateSubSceneWithoutEventNotifications(filePath, loadOption);

                        if (loadedSubScene == null)
                        {
                            reportMessage("    Load EROOR !!! " + Environment.NewLine);
                            break;
                        }

                        Debug.Assert(!subScenesBeforeUpdateAll.Contains(loadedSubScene));
                        loadedSubScenes.Add(loadedSubScene);

                        // 部品レイアウトを同期
                        if (param.DoUpdatePartsLayout)
                        {
                            // 同期前に部品レイアウトを読み込めたか確認
                            if (partsLoadResult == false)
                            {
                                reportMessage(LECoreStringResMgr.Get("SYSTEM_UPDATE_ERROR_LOAD_PARTS", Path.GetFileName(filePath)) + Environment.NewLine);
                                continue;
                            }

                            // 同期処理
                            SubSceneHelper.ReloadAllPartsPanesWithoutEventNotifications(loadedSubScene);
                        }

                        // 一旦別名ファイルに書き出す
                        string tempSavePath = Path.GetFullPath(filePath) + Path.GetFileNameWithoutExtension(filePath) + "__temp__" + Path.GetExtension(filePath);
                        ExportOption exportOption = new ExportOption() {
                            UseBaseValue = true,
                            Frame = GlobalTime.Inst.Time,
                            CreateThumbnail = param.DoSaveThumbnail,
                            OriginalPath = filePath,
                        };

                        bool bResult = LayoutEditorCore.ExportToFileAllWithoutEventNotifications(loadedSubScene, tempSavePath, exportOption);
                        if (!bResult)
                        {
                            reportMessage("    Save EROOR !!! " + Environment.NewLine);
                            break;
                        }

                        // ファイルを比較して変更があれば、正式に変更を反映する。
                        {
                            // レイアウト
                            {
                                string srcText = GetTrimedFileContents_(File.ReadAllText(filePath));
                                string newText = GetTrimedFileContents_(File.ReadAllText(tempSavePath));

                                if (srcText != newText)
                                {
                                    File.Copy(tempSavePath, filePath, true);
                                }

                                File.Delete(tempSavePath);
                            }

                            // アニメーション
                            string tempAnimFilePath = Path.ChangeExtension(tempSavePath, AppConstants.AnimationFileExt);
                            if (File.Exists(tempAnimFilePath))
                            {
                                string srcAnimFilePath = Path.ChangeExtension(filePath, AppConstants.AnimationFileExt);
                                if (File.Exists(srcAnimFilePath))
                                {
                                    string srcText = GetTrimedFileContents_(File.ReadAllText(srcAnimFilePath));
                                    string newText = GetTrimedFileContents_(File.ReadAllText(tempAnimFilePath));

                                    if (srcText != newText)
                                    {
                                        File.Copy(tempAnimFilePath, srcAnimFilePath, true);
                                    }

                                    File.Delete(tempAnimFilePath);
                                }
                                else
                                {
                                    File.Copy(tempAnimFilePath, srcAnimFilePath, true);
                                    File.Delete(tempAnimFilePath);
                                }
                            }
                        }

                        numProccessedFiles++;

                        // メモリが増大しすぎないようにときどきリソースを開放する
                        if (loadedSubScenes.Count == 20)
                        {
                            SceneManipulator sceneMnp = new SceneManipulator();
                            sceneMnp.BindTarget(LayoutEditorCore.Scene);

                            Debug.Assert(loadedSubScenes.Intersect(LayoutEditorCore.Scene.PartsSubScenes.Select(x => x.SubScene)).Count() == 0);

                            // 大丈夫だと思うが、念のために Except を入れておく
                            foreach (ISubScene subScene in loadedSubScenes.Except(
                                LayoutEditorCore.Scene.PartsSubScenes.Select(x => x.SubScene).ToArray()))
                            {
                                sceneMnp.RemoveSubScene(subScene);
                            }

                            loadedSubScenes.Clear();
                        }
                    }
                    catch (Exception exp)
                    {
                        reportMessage(string.Format("   ERROR !!! - {0}", exp.Message) + Environment.NewLine);
                    }

                    // キャンセル判定
                    if (param.IsCancelled != null && param.IsCancelled())
                    {
                        break;
                    }
                }

                // 後始末、読み込んだデータをまとめて消去します。
                {
                    SceneManipulator sceneMnp = new SceneManipulator();
                    sceneMnp.BindTarget(LayoutEditorCore.Scene);

                    var subScenes = LayoutEditorCore.Scene.ISubSceneSet.Except(subScenesBeforeUpdateAll).ToArray();
                    foreach (ISubScene removedScene in subScenes)
                    {
                        sceneMnp.RemoveSubScene(removedScene);
                    }

                    // LayoutEditorCore.Scene.PartsSubScenes に削除されたインスタンスへの参照が残っている可能性があるので初期化が必要
                    sceneMnp.RefreshPartsSubScenes(LayoutEditorCore.Scene.PartsRootPath);
                }
            }

            // ハンドラの登録を解除
            if (param.DoUpdatePartsLayout)
            {
                LECore.LayoutEditorCore.MsgReporter.PreReportMsg -= msgHandler;
            }

            // 結果の報告
            {
                reportMessage(LECoreStringResMgr.Get("SYSTEM_EXECUTE_NUM", string.Format("({0}/{1})", numProccessedFiles, filePaths.Count)) + Environment.NewLine);
                reportMessage(LECoreStringResMgr.Get("SYSTEM_REEXPORT_NOCHANGE_NOTICE") + Environment.NewLine);

                if (param.EndProcess != null)
                {
                    param.EndProcess();
                }
            }
        }

        /// <summary>
        /// 不要なタグを除去した内容文字列を取得します。
        /// </summary>
        private static string GetTrimedFileContents_(string content)
        {
            // create タグを除去する
            {
                int index1 = content.IndexOf("<create user");
                if (index1 != -1)
                {
                    int index2 = content.IndexOf("/>", index1);
                    if (index2 != -1)
                    {
                        content = content.Remove(index1, index2 - index1 + 2);
                    }
                }
            }

            // generator タグを除去する（ツールのバージョンが書いてある）
            {
                int index1 = content.IndexOf("<generator");
                if (index1 != -1)
                {
                    int index2 = content.IndexOf("/>", index1);
                    if (index2 != -1)
                    {
                        content = content.Remove(index1, index2 - index1 + 2);
                    }
                }
            }

            // <partsDerivativeBaseLastModifyDate> タグ
            {
                int index1 = content.IndexOf("<partsDerivativeBaseLastModifyDate>");
                if (index1 != -1)
                {
                    int index2 = content.IndexOf("</partsDerivativeBaseLastModifyDate>", index1);
                    if (index2 != -1)
                    {
                        content = content.Remove(index1, index2 - index1 + "</partsDerivativeBaseLastModifyDate>".Length);
                    }
                }
            }

            // fontFile タグを全て除去する
            // <fontFile Path="..\..\..\..\sample.bfnt" /> のようなパス指定のファイルがCドライブ直下に置かれていたりすると
            // 親フォルダを辿れず <fontFile Path="sample.bfnt"> などに書き換わってしまい、それを無視する
            for (;;)
            {
                bool replaced = false;

                int index1 = content.IndexOf("<fontFile");
                if (index1 != -1)
                {
                    int index2 = content.IndexOf("/>", index1);
                    if (index2 != -1)
                    {
                        content = content.Remove(index1, index2 - index1 + 2);
                        replaced = true;
                    }
                }

                if (replaced == false)
                {
                    break;
                }
            }

            // "-0" を "0" に置換する
            content = content.Replace("\"-0\"", "\"0\"");

            return content;
        }

        /// <summary>
        /// 複数のファイルから SHA1 を計算する。
        /// </summary>
        /// <param name="filePaths">計算対象ファイル</param>
        private static string ComputeSha1(IEnumerable<string> filePaths)
        {
            using (var memoryStream = new System.IO.MemoryStream())
            using (var writer = new BinaryWriter(memoryStream))
            {
                try
                {
                    foreach (var filePath in filePaths)
                    {
                        if (File.Exists(filePath))
                        {
                            using (var fs = new System.IO.FileStream(filePath, System.IO.FileMode.Open, System.IO.FileAccess.Read))
                            {
                                writer.Write("f"); // 将来のファイル以外の情報をハッシュに含める余地を残すためにある
                                writer.Write(fs.Length);
                                writer.Flush();
                                fs.CopyTo(memoryStream);
                            }
                        }
                        else
                        {
                            return string.Empty;
                        }
                    }

                    var sha1 = new System.Security.Cryptography.SHA1Managed();
                    memoryStream.Position = 0;
                    return string.Concat(sha1.ComputeHash(memoryStream).Select(x => x.ToString("x2")));
                }
                catch
                {
                    return string.Empty;
                }
            }
        }

        /// <summary>
        /// flyt と flan から SHA1 を計算する
        /// </summary>
        /// <param name="filePath">計算対象 flyt ファイル</param>
        public static string ComputeSha1FromFlytAndFlan(string flytPath)
        {
            var paths = new List<string>();
            paths.Add(flytPath);
            var flanPath = Path.ChangeExtension(flytPath, AppConstants.AnimationFileExtNoDot);
            if (File.Exists(flanPath))
            {
                paths.Add(flanPath);
            }

            return ComputeSha1(paths);
        }
    }
}
