﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;

namespace LECore.Structures.Core.Command
{
    using GroupMgrParamaters = LEGroupMgrModifyEventArgs.GroupMgrParamaters;
    using LECore.Save_Load;


    /// <summary>
    ///
    /// </summary>
    internal interface ICommandMemento<TSrcInstaceType>
    {
        void CopyTo( TSrcInstaceType dstInstance );
    }

    /// <summary>
    /// メメントオブジェクトを作成する、Undoコマンドを生成する、
    /// ファクトリクラスです。
    ///
    /// TInstaceType に メメントオブジェクトの作成ソースとなる
    /// クラスの型を指定してください。
    /// </summary>
    internal class MementoCommandFactory<TInstaceType>
    {
        /// <summary>
        /// Undo/Rodo処理の実態となる、変更操作を実装する関数。
        /// 本関数内で、パラメータを変更します。
        /// </summary>
        public delegate void InstanceModifyFunction();

        /// <summary>
        /// 実際に、メメントオブジェクトを作成する処理を実装する関数
        /// </summary>
        public delegate ICommandMemento<TInstaceType> MementoCreateFunction( TInstaceType src );

        /// <summary>
        /// 実際に、メメントオブジェクトを作成する処理を実装する関数
        /// </summary>
        readonly MementoCreateFunction _MementoCreateFunction;

      private readonly ISubScene commandCreatorSubScene = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MementoCommandFactory(
            MementoCreateFunction mementoCreateFunction, ISubScene commandCreatorSubScene)
        {
            _MementoCreateFunction = mementoCreateFunction;
      this.commandCreatorSubScene = commandCreatorSubScene;
        }

      ///
      public MementoCommandFactory( MementoCreateFunction mementoCreateFunction)
      {
          _MementoCreateFunction = mementoCreateFunction;
      }

        /// <summary>
        /// 対象を変更し、変更操作をコマンドとして登録します。
        /// </summary>
        public void Modify( TInstaceType instance, InstanceModifyFunction instanceModifyFunction )
        {
            ICommandMemento<TInstaceType> beforeModify = _MementoCreateFunction( instance );

            instanceModifyFunction();

            ICommandMemento<TInstaceType> afterModify = _MementoCreateFunction( instance );

            CommandFactory commandFactory = Scene.CurrentSubScene.CommandFactory;
            commandFactory.MakeMementoModifyCmd<TInstaceType>( instance, beforeModify, afterModify );
        }

        /// <summary>
        /// 対象を変更を試み、変更された場合は変更操作をコマンドとして登録します。
        /// </summary>
        public bool Modify( TInstaceType instance, Func<bool> instanceModifyFunction)
        {
            ICommandMemento<TInstaceType> beforeModify = _MementoCreateFunction(instance);

            bool modified = instanceModifyFunction();

            if (modified)
            {
                ICommandMemento<TInstaceType> afterModify = _MementoCreateFunction(instance);

                CommandFactory commandFactory = Scene.CurrentSubScene.CommandFactory;
                commandFactory.MakeMementoModifyCmd<TInstaceType>(instance, beforeModify, afterModify);
            }

            return modified;
        }
    }
}
