﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Collections;
using System.Collections.Generic;
using System.Reflection;

namespace LECore.Structures.Core.Command
{
    using GroupMgrParamaters = LEGroupMgrModifyEventArgs.GroupMgrParamaters;
    using LECore.Save_Load;
    using LECore.Manipulator;

    /// <summary>
    /// コマンドの種類を表す列挙子です。
    /// </summary>
    public enum CommandType
    {
        ModAttribute,
        ModSubAttribute,
        RestoreSubAttribute,
        ModAnmCurve,
        ModScene,
        ModProperty,
        CopyPane,
        ModMemento,
        Composed,
        ModifyAnmCurve, //@@
        ModAnmMode,
        AddAnmCurve,
        RemoveAnmCurve,
        ModTagName,
        ScaleAnimation,
        Action,
        Invalid
    }

    /// <summary>
    /// すべてのコマンドクラスが実装すべきインタフェースです。
    /// </summary>
    public interface ICommand
    {
        /// <summary>
        /// コマンドの種類を取得します。
        /// </summary>
        CommandType GetCmdType();

        /// <summary>
        /// コマンドを取り消します(実行前の状態に設定する)
        /// </summary>
        void Undo();

        /// <summary>
        /// コマンドをやり直します(実行後の状態に設定する)
        /// </summary>
        void Redo();

        /// <summary>
        /// コマンドの内容を説明する文字列を取得します。
        /// </summary>
        string ToString();
    }


    #region アトリビュート更新コマンド
    /// <summary>
    /// アトリビュートの値を変更するコマンドです。
    /// </summary>
    internal class AttrModifyCmd : ICommand
    {
        // 変更対象アトリビュート
        AnmAttribute _targetAttr = null;

        //
        // 以下のパラメータは、アトリビュートの種類ごとに異なります。
        // ただし、コマンドクラスからは、それらを意識する必要は無いはずです。
        //
        // 変更前のアトリビュート値
        object _beforeVal = null;
        // 変更後のアトリビュート値
        object _afterVal = null;
        // 変更前のアトリビュート底値
        object _beforeBaseVal = null;
        // 変更後のアトリビュート底値
        object _afterBaseVal = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="target">対象アトリビュート</param>
        /// <param name="beforeVal">変更前の値</param>
        /// <param name="afterVal">変更後の値</param>
        public AttrModifyCmd( AnmAttribute target, object beforeVal, object afterVal, object beforeBaseValue, object afterBaseValue )
        {
            _targetAttr = target;
            _beforeVal = beforeVal;
            _afterVal = afterVal;
            _beforeBaseVal = beforeBaseValue;
            _afterBaseVal = afterBaseValue;
        }

        #region ICommand メンバ

        public CommandType GetCmdType()
        {
            return CommandType.ModAttribute;
        }

        public void Undo()
        {
            _targetAttr.SetValue( _beforeVal );
            _targetAttr.SetBaseValue( _beforeBaseVal );
        }

        public void Redo()
        {
            _targetAttr.SetValue( _afterVal );
            _targetAttr.SetBaseValue( _afterBaseVal );
        }

        public override string ToString()
        {
            return string.Format( "type = {0} _beforeVal = {1} afterVal = {2}",
                                  GetCmdType().ToString(),
                                  _beforeVal.ToString(),
                                  _afterVal.ToString() );
        }

        #endregion
    }
    #endregion アトリビュート更新コマンド

    #region ユーザーデータエレメント更新コマンド
    internal class UserDataElementModifyCmd : ICommand
    {
        /// <summary>
        /// コマンドの種類
        /// </summary>
        public enum ModifyCmdType
        {
            ElementAdd,
            ElementReplace,
            ElementRemove,
            ElementSet,
            ElementIndexChange
        }

        /// <summary>
        /// コマンドのパラメータ
        /// </summary>
        public struct ModifyCmdParam
        {
            /// <summary>
            /// 変更対象のユーザーデータホルダを取得または設定します。
            /// </summary>
            public UserDataHolder TargetHolder { get; set; }

            /// <summary>
            /// 変更対象のユーザーデータエレメントを取得または設定します。
            /// </summary>
            public UserDataElement Element { get; set; }

            /// <summary>
            /// 置き換えるユーザーデータエレメントを取得または設定します。
            /// </summary>
            public UserDataElement ReplaceElement { get; set; }

            /// <summary>
            /// ユーザーデータエレメントの変更前の番号を取得または設定します。
            /// </summary>
            public int BeforeElementIndex { get; set; }

            /// <summary>
            /// ユーザーデータエレメントの変更後の番号を取得または設定します。
            /// </summary>
            public int AfterElementIndex { get; set; }

            /// <summary>
            /// ユーザーデータアトリビュートの変更前の番号を取得または設定します。
            /// </summary>
            public int BeforeAttributeIndex { get; set; }

            /// <summary>
            /// ユーザーデータアトリビュートの変更後の番号を取得または設定します。
            /// </summary>
            public int AfterAttributeIndex { get; set; }

            /// <summary>
            /// ユーザーデータの変更前の名前を取得または設定します。
            /// </summary>
            public string BeforeName { get; set; }

            /// <summary>
            /// ユーザーデータの変更後の名前を取得または設定します。
            /// </summary>
            public string AfterName { get; set; }

            /// <summary>
            /// ユーザーデータの変更前の上書き設定を取得または設定します。
            /// </summary>
            public bool BeforeOverwrite { get; set; }

            /// <summary>
            /// ユーザーデータの変更後の上書き設定を取得または設定します。
            /// </summary>
            public bool AfterOverwrite { get; set; }

            /// <summary>
            /// ユーザーデータの変更前の種類を取得または設定します。
            /// </summary>
            public UserDataKind BeforeKind { get; set; }

            /// <summary>
            /// ユーザーデータの変更後の種類を取得または設定します。
            /// </summary>
            public UserDataKind AfterKind { get; set; }

            /// <summary>
            /// ユーザーデータの変更前の値を取得または設定します。
            /// </summary>
            public object BeforeValue { get; set; }

            /// <summary>
            /// ユーザーデータの変更後の値を取得または設定します。
            /// </summary>
            public object AfterValue { get; set; }

            /// <summary>
            /// ユーザーデータの変更前の値を取得または設定します。
            /// </summary>
            public object BeforeBaseValue { get; set; }

            /// <summary>
            /// ユーザーデータの変更後の値を取得または設定します。
            /// </summary>
            public object AfterBaseValue { get; set; }
        }

        /// <summary>
        /// コマンドの種類
        /// </summary>
        ModifyCmdType _modCmdType;

        /// <summary>
        /// コマンドのパラメータ
        /// </summary>
        ModifyCmdParam _modCmdParam;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="modCmdType">コマンドの種類</param>
        /// <param name="modCmdParam">コマンドのパラメータ</param>
        public UserDataElementModifyCmd(ModifyCmdType modCmdType, ModifyCmdParam modCmdParam)
        {
            _modCmdType = modCmdType;
            _modCmdParam = modCmdParam;
        }

        #region ICommand メンバ

        /// <summary>
        /// コマンドの種類を取得します。
        /// </summary>
        /// <returns>コマンドの種類を返します。</returns>
        public CommandType GetCmdType()
        {
            return CommandType.ModSubAttribute;
        }

        /// <summary>
        /// アンドゥ
        /// </summary>
        public void Undo()
        {
            switch (_modCmdType)
            {
                case ModifyCmdType.ElementAdd:
                    _modCmdParam.TargetHolder.RemoveUserDataElement(_modCmdParam.Element);
                    break;

                case ModifyCmdType.ElementReplace:
                    _modCmdParam.TargetHolder.RemoveUserDataElement(_modCmdParam.ReplaceElement);
                    _modCmdParam.TargetHolder.InsertUserDataElement(_modCmdParam.Element, _modCmdParam.BeforeElementIndex, _modCmdParam.BeforeAttributeIndex);
                    break;

                case ModifyCmdType.ElementRemove:
                    _modCmdParam.TargetHolder.InsertUserDataElement(_modCmdParam.Element, _modCmdParam.BeforeElementIndex, _modCmdParam.BeforeAttributeIndex);
                    break;

                case ModifyCmdType.ElementSet:
                    _modCmdParam.TargetHolder.SetUserDataPamaters(_modCmdParam.AfterName, _modCmdParam.BeforeName, _modCmdParam.BeforeOverwrite, _modCmdParam.BeforeKind, _modCmdParam.BeforeValue, _modCmdParam.BeforeBaseValue);
                    break;

                case ModifyCmdType.ElementIndexChange:
                    _modCmdParam.TargetHolder.RemoveUserDataElement(_modCmdParam.Element);
                    _modCmdParam.TargetHolder.InsertUserDataElement(_modCmdParam.Element, _modCmdParam.BeforeElementIndex, _modCmdParam.BeforeAttributeIndex);
                    break;
            }
        }

        /// <summary>
        /// リドゥ
        /// </summary>
        public void Redo()
        {
            switch (_modCmdType)
            {
                case ModifyCmdType.ElementAdd:
                    _modCmdParam.TargetHolder.AddUserDataElement(_modCmdParam.Element);
                    break;

                case ModifyCmdType.ElementReplace:
                    _modCmdParam.TargetHolder.RemoveUserDataElement(_modCmdParam.Element);
                    _modCmdParam.TargetHolder.InsertUserDataElement(_modCmdParam.ReplaceElement, _modCmdParam.BeforeElementIndex, _modCmdParam.BeforeElementIndex);
                    break;

                case ModifyCmdType.ElementRemove:
                    _modCmdParam.TargetHolder.RemoveUserDataElement(_modCmdParam.Element);
                    break;

                case ModifyCmdType.ElementSet:
                    _modCmdParam.TargetHolder.SetUserDataPamaters(_modCmdParam.BeforeName, _modCmdParam.AfterName, _modCmdParam.AfterOverwrite, _modCmdParam.AfterKind, _modCmdParam.AfterValue, _modCmdParam.AfterBaseValue);
                    break;

                case ModifyCmdType.ElementIndexChange:
                    _modCmdParam.TargetHolder.RemoveUserDataElement(_modCmdParam.Element);
                    _modCmdParam.TargetHolder.InsertUserDataElement(_modCmdParam.Element, _modCmdParam.AfterElementIndex, _modCmdParam.AfterAttributeIndex);
                    break;
            }
        }

        /// <summary>
        /// コマンドの内容を説明する文字列を取得します。
        /// </summary>
        /// <returns>コマンドの内容を説明する文字列を返します。</returns>
        public override string ToString()
        {
            return string.Format("type = {0} _modCmdType = {1}", GetCmdType().ToString(), _modCmdType.ToString());
        }
        #endregion
    }
    #endregion

    #region ユーザーデータ復元コマンド
    internal class UserDataRestoreCmd : ICommand
    {
        /// <summary>
        /// コマンドの種類
        /// </summary>
        public enum ModifyCmdType
        {
            OnUndo,
            OnRedo
        }

        /// <summary>
        /// コマンドの種類
        /// </summary>
        private ModifyCmdType _modCmdType;

        /// <summary>
        /// 変更対象のユーザーデータホルダ
        /// </summary>
        private UserDataHolder _targetHolder;

        /// <summary>
        /// エレメントリスト
        /// </summary>
        private UserDataElement[] _elements;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="modCmdType">コマンドの種類</param>
        /// <param name="elements">エレメントリスト</param>
        public UserDataRestoreCmd(ModifyCmdType modCmdType, UserDataHolder targetHolder, UserDataElement[] elements)
        {
            _modCmdType = modCmdType;
            _targetHolder = targetHolder;
            _elements = elements;
        }

        #region ICommand メンバ

        /// <summary>
        /// コマンドの種類を取得します。
        /// </summary>
        /// <returns>コマンドの種類を返します。</returns>
        public CommandType GetCmdType()
        {
            return CommandType.RestoreSubAttribute;
        }

        /// <summary>
        /// アンドゥ
        /// </summary>
        public void Undo()
        {
            if (_modCmdType == ModifyCmdType.OnUndo)
            {
                _targetHolder.RestoreUserDataElement(_elements);
            }
        }

        /// <summary>
        /// リドゥ
        /// </summary>
        public void Redo()
        {
            if (_modCmdType == ModifyCmdType.OnRedo)
            {
                _targetHolder.RestoreUserDataElement(_elements);
            }
        }

        /// <summary>
        /// コマンドの内容を説明する文字列を取得します。
        /// </summary>
        /// <returns>コマンドの内容を説明する文字列を返します。</returns>
        public override string ToString()
        {
            return string.Format("type = {0} _modCmdType = {1}", GetCmdType().ToString(), _modCmdType.ToString());
        }
        #endregion
    }
    #endregion

    #region プロパティ変更コマンド
    /// <summary>
    /// プロパティの値を変更するコマンドです。
    /// </summary>
    internal class PropModifyCmd : ICommand
    {
        // 変更対象
        IValueAccessor  _target = null;

        // 以下のパラメータは、プロパティの種類ごとに異なります。
        // ただし、コマンドクラスからは、それらを意識する必要は無いはずです。
        //
        // 変更前のプロパティ値
        object _beforeVal = null;
        // 変更後のプロパティ値
        object _afterVal = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="target">対象</param>
        /// <param name="beforeVal">変更前の値</param>
        /// <param name="afterVal">変更後の値</param>
        public PropModifyCmd( IValueAccessor target, object beforeVal, object afterVal )
        {
            _target = target;
            _beforeVal = beforeVal;
            _afterVal = afterVal;
        }

        #region ICommand メンバ
        public CommandType GetCmdType()
        {
            return CommandType.ModProperty;
        }

        public void Undo()
        {
            _target.SetValue( _beforeVal );
        }

        public void Redo()
        {
            _target.SetValue( _afterVal );
        }

        public override string ToString()
        {
            return string.Format( "type = {0} _beforeVal = {1} afterVal = {2}",
                                  GetCmdType().ToString(),
                                  _beforeVal.ToString(),
                                  _afterVal.ToString() );
        }
        #endregion
    }
    #endregion プロパティ変更コマンド

    #region プロパティ変更コマンド
    /// <summary>
    /// クラスインスタンスのプロパティを更新するコマンドです。
    /// </summary>
    internal class PropertyModifyCmd : ICommand
    {
        readonly object _ownerInstance = null;
        readonly PropertyInfo _propertyInfo = null;
        readonly object _beforeVal = null;
        readonly object _afterVal = null;


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PropertyModifyCmd(
            object ownerInstance,
            string propertyName,
            object beforeVal,
            object afterVal )
        {
            _propertyInfo = ownerInstance.GetType().GetProperty( propertyName );
            Debug.Assert( _propertyInfo != null );

            _ownerInstance = ownerInstance;
            _beforeVal = beforeVal;
            _afterVal = afterVal;
        }

        #region ICommand メンバ

        public CommandType GetCmdType()
        {
            return CommandType.ModProperty;
        }

        public void Undo()
        {
            _propertyInfo.SetValue( _ownerInstance, _beforeVal, null );
        }

        public void Redo()
        {
            _propertyInfo.SetValue( _ownerInstance, _afterVal, null );
        }

        public override string ToString()
        {
            return string.Format( "type = {0} [{1}.{2}] _beforeVal = {3} afterVal = {4}",
                GetCmdType().ToString(),
                _ownerInstance.GetType().Name,
                _propertyInfo.Name,
                _beforeVal.ToString(),
                _afterVal.ToString() );
        }

        #endregion
    }
    #endregion

    #region アニメーションカーブ変更コマンド
    /// <summary>
    /// アニメーションカーブを変更するコマンドです。
    /// </summary>
    internal class AnmCurveModCmd :
        ICommand
    {
        /// <summary>
        /// コマンドの種類
        /// </summary>
        public enum ModifyCmdType
        {
            KeyAdd,
            KeyRemove,
            KeyModify,
            ChangeInterporationMode,
      ChangePreInfinity,        //@@
      ChangePostInfinity,        //@@
            Invalid
        }

        /// <summary>
        /// ModifyCmdType.KeyModify の時に使用されるパラメータの型です。
        /// </summary>
        public class AnmKeyMement
        {
            readonly AnmKeyFrame _cloneKeyBeforeMod = null;
            readonly AnmKeyFrame _cloneKeyAfterMod = null;


            public AnmKeyFrame CloneKeyBeforeMod
            {
                get { return _cloneKeyBeforeMod; }
            }

            public AnmKeyFrame CloneKeyAfterMod
            {
                get { return _cloneKeyAfterMod; }
            }

            public AnmKeyMement
            (
                AnmKeyFrame cloneKeyBeforeMod,
                AnmKeyFrame cloneKeyAfterMod
            )
            {
                _cloneKeyBeforeMod = cloneKeyBeforeMod;
                _cloneKeyAfterMod = cloneKeyAfterMod;
            }
        }


        // 変更対象アニメーションカーブ
        AnmCurve _keyOwnerCurve = null;
        AnmKeyFrame _targetKeyFrame = null;
        object _paramater = null;
        ModifyCmdType _modCmdType = ModifyCmdType.Invalid;


        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="anmCurve"></param>
        /// <param name="modCmdType"></param>
        /// <param name="key0"></param>
        /// <param name="key1"></param>
        public AnmCurveModCmd
        (
            ModifyCmdType modCmdType,
            AnmKeyFrame targetKeyFrame,
            object paramater
        )
        {
            _keyOwnerCurve = targetKeyFrame.OwnerAnmCurve;
            _modCmdType = modCmdType;
            _targetKeyFrame = targetKeyFrame;
            _paramater = paramater;
        }

        #region ICommand メンバ

        public CommandType GetCmdType()
        {
            return CommandType.ModAnmCurve;
        }

        /// <summary>
        /// キーフレームを変更する
        /// </summary>
        /// <param name="srcData"></param>
        void ModifyTargetKeyFrame_( AnmKeyFrame srcData )
        {
            if( _targetKeyFrame.Time != srcData.Time )
            {
                _targetKeyFrame.SetParamaters( srcData );

                // キーフレームのソート
                _targetKeyFrame.OwnerAnmCurve.SetCurveStateDirty();
            }
            else
            {
                _targetKeyFrame.SetParamaters( srcData );
            }

            // カーブ全体の再評価
            _targetKeyFrame.OwnerAnmCurve.Update( true );
        }

        /// <summary>
        /// アンドゥ
        /// </summary>
        public void Undo()
        {
            if( _modCmdType == ModifyCmdType.KeyModify )
            {
                AnmKeyMement anmKeyMement = _paramater as AnmKeyMement;

                ModifyTargetKeyFrame_( anmKeyMement.CloneKeyBeforeMod );
            }
            else if( _modCmdType == ModifyCmdType.KeyAdd )
            {
                _keyOwnerCurve.RemoveKeyFrame( _targetKeyFrame );
            }
            else if( _modCmdType == ModifyCmdType.KeyRemove )
            {
                _keyOwnerCurve.AddNewKeyFrame( _targetKeyFrame );
            }
        }

        /// <summary>
        /// リドゥ
        /// </summary>
        public void Redo()
        {
            if( _modCmdType == ModifyCmdType.KeyModify )
            {
                AnmKeyMement anmKeyMement = _paramater as AnmKeyMement;

                ModifyTargetKeyFrame_( anmKeyMement.CloneKeyAfterMod );
            }
            else if( _modCmdType == ModifyCmdType.KeyAdd )
            {
                _keyOwnerCurve.AddNewKeyFrame( _targetKeyFrame );
            }
            else if( _modCmdType == ModifyCmdType.KeyRemove )
            {
                _keyOwnerCurve.RemoveKeyFrame( _targetKeyFrame );
            }
        }

        public override string ToString()
        {
            return string.Format( "type = {0} _modCmdType = {1}",
                GetCmdType().ToString(),
                _modCmdType.ToString() );
        }

        #endregion
    }
    #endregion アニメーションカーブ変更コマンド

    ///
    internal class AnmCurveModifyCommand : ICommand
    {
        public delegate void Setter( AnmCurve anmCurve, object value);

        private AnmCurve targetAnmCurve = null;
        private object beforeValue = null;
        private object afterValue = null;

        private Setter setter = null;

        public AnmCurveModifyCommand( AnmCurve anmCurve, object beforeValue, object afterValue, Setter setter)
        {
            Debug.Assert( anmCurve != null);

            this.setter = setter;

            this.targetAnmCurve = anmCurve;
            this.beforeValue = beforeValue;
            this.afterValue  = afterValue;
        }

        ///
        public CommandType GetCmdType()
        {
            return CommandType.ModifyAnmCurve;
        }

        ///
        void ICommand.Undo()
        {
            this.setter( this.targetAnmCurve, this.beforeValue);
        }

        ///
        void ICommand.Redo()
        {
            this.setter( this.targetAnmCurve, this.afterValue);
        }

        ///
        string ICommand.ToString()
        {
            return string.Format( "type = {0}", GetCmdType().ToString());
        }
    }

    /// <summary>
    /// 任意のアクションを実行するコマンド
    /// </summary>
    internal class ActionCommand : ICommand
    {
        Action action;
        public ActionCommand(Action action)
        {
            this.action = action;
        }

        ///
        public CommandType GetCmdType()
        {
            return CommandType.ModifyAnmCurve;
        }

        ///
        void ICommand.Undo()
        {
            action();
        }

        ///
        void ICommand.Redo()
        {
            action();
        }

        ///
        string ICommand.ToString()
        {
            return string.Format("type = {0}", GetCmdType().ToString());
        }
    }

    #region シーン変更コマンド
    /// <summary>
    /// シーンに関する変更処理コマンド
    /// </summary>
    internal class SceneModifyCmd :
        ICommand
    {


        /// <summary>
        /// 種類
        /// </summary>
        SceneModifyEventArgs.Kind _modCmdType;

        SubScene _targetSubScene = null;
        object _paramater = null;


        /// <summary>
        /// _modCmdType が MakeHierarchy,   // 階層生成
        ///                BreakeHierarchy, // 階層消去
        /// のときに、_paramater の実体として使用されるクラスです。
        ///
        /// 階層構造を構築するノードのペアが格納されています。
        /// </summary>
        public class ConnectinPair
        {
            readonly public Pane _afterParent;
            readonly public Pane _beforeParent;

            readonly public int _afterIndexOfChildren;
            readonly public int _beforeIndexOfChildren;

            readonly public Pane _child;



            public ConnectinPair
            (
                Pane parent,
                Pane child
            )
            {
                _afterParent = parent;
                _beforeParent = child.Parent as Pane;

                _beforeIndexOfChildren = _beforeParent.IndexOf( child );
                _afterIndexOfChildren = parent.Children.Length;

                // 操作前後の親は異なるはず。
                // Debug.Assert( _afterParent != _beforeParent );

                _child = child;
            }

            /// <summary>
            ///
            /// </summary>
            public ConnectinPair
            (
                Pane parent,
                Pane child,
                int newIndex
            )
            {
                _afterParent = parent;
                _beforeParent = child.Parent as Pane;

                _beforeIndexOfChildren = _beforeParent.IndexOf( child );
                _afterIndexOfChildren = newIndex;

                if( _afterIndexOfChildren < _beforeIndexOfChildren )
                {
                    _beforeIndexOfChildren++;
                }

                // 操作前後の親は異なるはず。
                // Debug.Assert( _afterParent != _beforeParent );

                _child = child;
            }
        }




        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SceneModifyCmd
        (
            SceneModifyEventArgs.Kind modCmdType,
            SubScene targetSubScene,
            object paramater
        )
        {
            _modCmdType = modCmdType;
            _targetSubScene = targetSubScene;
            _paramater = paramater;
        }

        #region ICommand メンバ

        public CommandType GetCmdType()
        {
            return CommandType.ModScene;
        }

        /// <summary>
        /// アンドゥ
        /// </summary>
        public void Undo()
        {
            switch( _modCmdType )
            {
                case SceneModifyEventArgs.Kind.PaneAdd:
                {
                    _targetSubScene.RemovePain( _paramater as Pane );
                    break;
                }
                case SceneModifyEventArgs.Kind.PaneRemove:
                {
                    _targetSubScene.AddPane(_paramater as Pane);
                    break;
                }
                case SceneModifyEventArgs.Kind.HierarchyMake:
                {
                    ConnectinPair connectinPair = _paramater as ConnectinPair;
                    Debug.Assert( connectinPair != null );

                    if( !connectinPair._child.IsChildrenOf( connectinPair._beforeParent ) )
                    {
                        connectinPair._beforeParent.AddChildPane( connectinPair._child );
                    }
                    connectinPair._beforeParent.ChangeChildOrder( connectinPair._child, connectinPair._beforeIndexOfChildren );

                    break;
                }
                case SceneModifyEventArgs.Kind.HierarchyReordr: // ChangeOrder
                {
                    ConnectinPair connectinPair = _paramater as ConnectinPair;
                    Debug.Assert( connectinPair != null );
                    connectinPair._beforeParent.ChangeChildOrder( connectinPair._child, connectinPair._beforeIndexOfChildren );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupMake:
                {
                    GroupMgrParamaters groupData = (GroupMgrParamaters)_paramater;
                    groupData.GroupMgr.RemoveGroup(
                        groupData.GroupMgr.FindGroupByName( groupData.Group.GrouprName ) );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupRemove:
                {
                    GroupMgrParamaters groupData = (GroupMgrParamaters)_paramater;
                    groupData.GroupMgr.MakeNewGroup(
                        groupData.Group.GrouprName,
                        groupData.Group.Member );

                    ILEGroup newGroup =
                        groupData.GroupMgr.FindGroupByName( groupData.Group.GrouprName );

                    groupData.GroupMgr.ChangeGroupOrder(
                        newGroup,
                        groupData.OldIndex );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupReorder:
                {
                    GroupMgrParamaters groupData = (GroupMgrParamaters)_paramater;
                    groupData.GroupMgr.ChangeGroupOrder(
                        groupData.Group,
                        groupData.OldIndex );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupAddPane:
                {
                    GroupMgrParamaters groupData = (GroupMgrParamaters)_paramater;

                    groupData.FindGroupByName().RemoveMember( groupData.MemberPane );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupRemovePane:
                {
                    GroupMgrParamaters groupData = (GroupMgrParamaters)_paramater;

                    groupData.FindGroupByName().AddMember( groupData.MemberPane );
                    groupData.FindGroupByName().ChangeMemberOrder(
                        groupData.MemberPane,
                        groupData.OldIndex );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupReorderPane:
                {
                    GroupMgrParamaters groupData = (GroupMgrParamaters)_paramater;

                    groupData.FindGroupByName().ChangeMemberOrder(
                        groupData.MemberPane,
                        groupData.OldIndex );

                    break;
                }
                default:
                Debug.Assert( false ); break;
            }
        }

        /// <summary>
        /// リドゥ
        /// </summary>
        public void Redo()
        {
            switch( _modCmdType )
            {
                case SceneModifyEventArgs.Kind.PaneAdd:
                {
                    _targetSubScene.AddPane(_paramater as Pane);
                    break;
                }
                case SceneModifyEventArgs.Kind.PaneRemove:
                {
                    _targetSubScene.RemovePain( _paramater as Pane );
                    break;
                }
                case SceneModifyEventArgs.Kind.HierarchyMake:
                {
                    ConnectinPair connectinPair = _paramater as ConnectinPair;
                    Debug.Assert( connectinPair != null );

                    if( !connectinPair._child.IsChildrenOf( connectinPair._afterParent ) )
                    {
                        connectinPair._afterParent.AddChildPane( connectinPair._child );
                    }
                    connectinPair._afterParent.ChangeChildOrder( connectinPair._child, connectinPair._afterIndexOfChildren );

                    break;
                }
                case SceneModifyEventArgs.Kind.HierarchyReordr: // ChangeOrder
                {
                    ConnectinPair connectinPair = _paramater as ConnectinPair;
                    Debug.Assert( connectinPair != null );
                    connectinPair._beforeParent.ChangeChildOrder( connectinPair._child, connectinPair._afterIndexOfChildren );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupMake:
                {
                    GroupMgrParamaters groupData = (GroupMgrParamaters)_paramater;
                    groupData.GroupMgr.MakeNewGroup(
                        groupData.Group.GrouprName,
                        groupData.Group.Member );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupRemove:
                {
                    GroupMgrParamaters groupData = (GroupMgrParamaters)_paramater;
                    groupData.GroupMgr.RemoveGroup(
                        groupData.GroupMgr.FindGroupByName( groupData.Group.GrouprName ) );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupReorder:
                {
                    GroupMgrParamaters groupData = (GroupMgrParamaters)_paramater;
                    groupData.GroupMgr.ChangeGroupOrder(
                        groupData.Group,
                        groupData.NewIndex );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupAddPane:
                {
                    GroupMgrParamaters groupData = (GroupMgrParamaters)_paramater;

                    groupData.FindGroupByName().AddMember( groupData.MemberPane );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupRemovePane:
                {
                    GroupMgrParamaters groupData = (GroupMgrParamaters)_paramater;

                    groupData.FindGroupByName().RemoveMember( groupData.MemberPane );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupReorderPane:
                {
                    GroupMgrParamaters groupData = (GroupMgrParamaters)_paramater;

                    groupData.FindGroupByName().ChangeMemberOrder(
                        groupData.MemberPane,
                        groupData.NewIndex );

                    break;
                }
                default:
                Debug.Assert( false ); break;
            }
        }

        public override string ToString()
        {
            return string.Format( "type = {0} _modCmdType = {1} ",
                                  GetCmdType().ToString(),
                                  _modCmdType.ToString() );
        }

        #endregion
    }

    #endregion シーン更新コマンド

    #region ペイン変更コマンド(暫定実装：最終的には消去したい)
    /// <summary>
    /// ペインを更新するコマンドです。
    /// </summary>
    internal class PaneCopyCmd :
        ICommand
    {
        readonly IPane _pane = null;
        readonly IPane _beforeCopy = null;
        readonly IPane _afterCopy = null;

        static readonly PaneParamaterCopyOption _copyOption = new PaneParamaterCopyOption();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PaneCopyCmd(
            IPane pane,
            IPane beforeCopy,
            IPane afterCopy )
        {
            _pane = pane;
            _beforeCopy = beforeCopy;
            _afterCopy = afterCopy;
        }

        #region ICommand メンバ

        public CommandType GetCmdType()
        {
            return CommandType.CopyPane;
        }

        public void Undo()
        {
            _copyOption.SetOption( ParamaterKind.All, true );

            // オプションの追加設定
            PaneParamaterCopyOption tmpOption = new PaneParamaterCopyOption(_copyOption);
            tmpOption.CopyBaseValue = true;

            PaneParamaterPaster.PasteParamaters( tmpOption, _pane, _beforeCopy );
        }

        public void Redo()
        {
            _copyOption.SetOption( ParamaterKind.All, true );

            // オプションの追加設定
            PaneParamaterCopyOption tmpOption = new PaneParamaterCopyOption(_copyOption);
            tmpOption.CopyBaseValue = true;

            PaneParamaterPaster.PasteParamaters(tmpOption, _pane, _afterCopy );
        }

        public override string ToString()
        {
            return string.Format( "type = {0} [ pane = {1} ] ",
                GetCmdType().ToString(),
                _pane.PaneName );
        }

        #endregion
    }
    #endregion ペイン変更コマンド(暫定実装：最終的には消去したい)

    #region TextBox変更コマンド

    internal class MementModifyCommand<TInstanceType> : ICommand
    {
        readonly TInstanceType _target;
        readonly ICommandMemento<TInstanceType> _beforeMod = null;
        readonly ICommandMemento<TInstanceType> _afterMod = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MementModifyCommand(
            TInstanceType target,
            ICommandMemento<TInstanceType> beforeMod,
            ICommandMemento<TInstanceType> afterMod )
        {
            _target = target;
            _beforeMod = beforeMod;
            _afterMod = afterMod;
        }


        #region ICommand メンバ

        /// <summary>
        ///
        /// </summary>
        public CommandType GetCmdType()
        {
            return CommandType.ModMemento;
        }

        /// <summary>
        ///
        /// </summary>
        public void Undo()
        {
            _beforeMod.CopyTo( _target );
        }

        /// <summary>
        ///
        /// </summary>
        public void Redo()
        {
            _afterMod.CopyTo( _target );
        }

        /// <summary>
        ///
        /// </summary>
        public override string ToString()
        {
            return string.Format( "type = {0} [ pane = {1} ] ", GetCmdType().ToString(), "memment" );
        }

        #endregion

    }
    #endregion

    #region アニメーション関連
    /// <summary>
    /// アニメーションモードを切り替えるコマンドです。
    /// </summary>
    internal class AnmModeCmd :
    ICommand
    {
        readonly SubSceneManipulator.AnmConvertParam _beforeMod = null;
        readonly SubSceneManipulator.AnmConvertParam _afterMod = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnmModeCmd(
            SubSceneManipulator.AnmConvertParam beforeMod,
            SubSceneManipulator.AnmConvertParam afterMod)
        {
            _beforeMod = beforeMod;
            _afterMod = afterMod;
        }

        #region ICommand メンバ

        public CommandType GetCmdType()
        {
            return CommandType.ModAnmMode;
        }

        public void Undo()
        {
            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.UndoAnmConvert(_beforeMod);
        }

        public void Redo()
        {
            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.RedoAnmConvert(_afterMod);
        }

        public override string ToString()
        {
            return string.Format("type = {0}", GetCmdType().ToString());
        }

        #endregion
    }

    /// <summary>
    /// 区間タグにアニメーションを追加するコマンドです。
    /// </summary>
    internal class AnmAddCmd :
    ICommand
    {
        readonly ISubScene _subScene = null;
        readonly string _tagName;
        readonly SubSceneManipulator.AnmConvertParam _beforeMod = null;
        readonly SubSceneManipulator.AnmConvertParam _afterMod = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnmAddCmd(
            ISubScene subScene,
            string tagName,
            SubSceneManipulator.AnmConvertParam beforeMod,
            SubSceneManipulator.AnmConvertParam afterMod)
        {
            _subScene = subScene;
            _tagName = tagName;
            _beforeMod = beforeMod;
            _afterMod = afterMod;
        }

        #region ICommand メンバ

        public CommandType GetCmdType()
        {
            return CommandType.AddAnmCurve;
        }

        public void Undo()
        {
            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.UndoAddAnmCurve(_subScene, _beforeMod, _tagName);
        }

        public void Redo()
        {
            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.RedoAddAnmCurve(_afterMod, _tagName);
        }

        public override string ToString()
        {
            return string.Format("type = {0}", GetCmdType().ToString());
        }

        #endregion
    }

    /// <summary>
    /// 区間タグからアニメーションを削除するコマンドです。
    /// </summary>
    internal class AnmRemoveCmd :
    ICommand
    {
        readonly ISubScene _subScene = null;
        readonly string _tagName;
        readonly SubSceneManipulator.AnmConvertParam _beforeMod = null;
        readonly SubSceneManipulator.AnmConvertParam _afterMod = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnmRemoveCmd(
            ISubScene subScene,
            string tagName,
            SubSceneManipulator.AnmConvertParam beforeMod,
            SubSceneManipulator.AnmConvertParam afterMod)
        {
            _subScene = subScene;
            _tagName = tagName;
            _beforeMod = beforeMod;
            _afterMod = afterMod;
        }

        #region ICommand メンバ

        public CommandType GetCmdType()
        {
            return CommandType.RemoveAnmCurve;
        }

        public void Undo()
        {
            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.UndoRemoveAnmCurve(_beforeMod, _tagName);
        }

        public void Redo()
        {
            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.RedoRemoveAnmCurve(_subScene, _afterMod, _tagName);
        }

        public override string ToString()
        {
            return string.Format("type = {0}", GetCmdType().ToString());
        }

        #endregion
    }

    /// <summary>
    /// 区間タグ名を変更するコマンドです。
    /// </summary>
    internal class ModifyTagNameCmd :
    ICommand
    {
        IEnumerable<IPane> _target = null;
        readonly string _beforeMod = null;
        readonly string _afterMod = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ModifyTagNameCmd(
            IEnumerable<IPane> target,
            string beforeMod,
            string afterMod)
        {
            _target = target;
            _beforeMod = beforeMod;
            _afterMod = afterMod;
        }

        #region ICommand メンバ

        public CommandType GetCmdType()
        {
            return CommandType.ModTagName;
        }

        public void Undo()
        {
            foreach (IPane pane in _target)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    curveAttr.ModifyAnmTag(_afterMod as string, _beforeMod as string);
                }
            }
        }

        public void Redo()
        {
            foreach (IPane pane in _target)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    curveAttr.ModifyAnmTag(_beforeMod as string, _afterMod as string);
                }
            }
        }

        public override string ToString()
        {
            return string.Format("type = {0}", GetCmdType().ToString());
        }

        #endregion
    }

    /// <summary>
    /// アニメーションのスケーリングを行うコマンドです。
    /// </summary>
    internal class ScaleAnimationCmd :
    ICommand
    {
        readonly SubSceneManipulator.AnmConvertParam _beforeMod = null;
        readonly SubSceneManipulator.AnmConvertParam _afterMod = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ScaleAnimationCmd(
            SubSceneManipulator.AnmConvertParam beforeMod,
            SubSceneManipulator.AnmConvertParam afterMod)
        {
            _beforeMod = beforeMod;
            _afterMod = afterMod;
        }

        #region ICommand メンバ

        public CommandType GetCmdType()
        {
            return CommandType.ScaleAnimation;
        }

        public void Undo()
        {
            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.UndoScaleAnimation(_beforeMod);
        }

        public void Redo()
        {
            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.RedoScaleAnimation(_afterMod);
        }

        public override string ToString()
        {
            return string.Format("type = {0}", GetCmdType().ToString());
        }

        #endregion
    }
    #endregion


    #region 複合コマンド
    /// <summary>
    /// 複数のコマンドからなるコマンド
    /// 一度に複数のコマンドが実行されるコマンドです。
    ///
    /// 大量のコマンドをまとめて実行するため、コマンド実行前に、
    /// シーンに対して大量の変更を行う事を通知します。
    /// (それによって、不要な再描画によるパフォーマンス低下を回避しています。)
    /// </summary>
    internal class ComposedCmd :
        ICommand
    {
        readonly ISubScene _targetSubScene = null;
        readonly ArrayList _cmds = new ArrayList();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ComposedCmd( ICommand[] srcCmds, ISubScene targetSubScene )
        {
            Debug.Assert( targetSubScene != null );
            Debug.Assert( srcCmds.Length != 0 );

            _targetSubScene = targetSubScene;

            foreach( ICommand cmd in srcCmds )
            {
                _cmds.Add( cmd );
            }
        }

        #region ICommand メンバ

        /// <summary>
        ///
        /// </summary>
        public CommandType GetCmdType()
        {
            return CommandType.Composed;
        }

        /// <summary>
        /// アンドゥ
        /// </summary>
        public void Undo()
        {
            _targetSubScene.BeginMassiveModify();
            // 注意：Undoは登録順とは、逆順に実行する必要があります。
            // Redoの実行順と比較してみてください。
            for( int i = _cmds.Count - 1 ; i >= 0 ; i-- )
            {
                ICommand cmd = _cmds[i] as ICommand;
                cmd.Undo();
            }
            _targetSubScene.EndMassiveModify();
        }

        /// <summary>
        /// リドゥ
        /// </summary>
        public void Redo()
        {
            _targetSubScene.BeginMassiveModify();
            foreach( ICommand cmd in _cmds )
            {
                cmd.Redo();
            }
            _targetSubScene.EndMassiveModify();
        }

        /// <summary>
        ///
        /// </summary>
        public override string ToString()
        {
            return string.Format( "type = {0} numCmd = {1} ",
                GetCmdType().ToString(),
                _cmds.Count.ToString() );
        }

        #endregion
    }
    #endregion 複合コマンド



    /// <summary>
    /// コマンドファクトリクラス
    /// 余り
    /// </summary>
    internal class CommandFactory
    {

        // コマンド・キュー
        readonly SubSceneCmdQueue _cmdQueue = null;


        SubSceneCmdQueue _CmdQueue
        {
            get { return _cmdQueue; }
        }


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CommandFactory( SubSceneCmdQueue cmdQueue )
        {
            Debug.Assert( cmdQueue != null );
            _cmdQueue = cmdQueue;
        }

        #region アトリビュート変更
        /// <summary>
        /// アトリビュート変更コマンドを登録します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="beforeVal"></param>
        /// <param name="afterVal"></param>
        public void MakeAttrModCmd( AnmAttribute target, object beforeVal, object afterVal, object beforeBaseValue, object afterBaseValue )
        {
            AttrModifyCmd cmd = new AttrModifyCmd( target, beforeVal, afterVal, beforeBaseValue, afterBaseValue);

            // グローバル・コマンドキューに登録
            _CmdQueue.RegisterCmd( cmd );
        }
        #endregion アトリビュート変更

        #region サブアトリビュート変更
        /// <summary>
        /// ユーザーデータエレメント追加コマンドを登録します。
        /// </summary>
        /// <param name="targetHolder">変更対象のユーザーデータホルダ</param>
        /// <param name="element">追加するユーザーデータエレメント</param>
        public void MakeUserDataElementAddCmd
        (
            UserDataHolder targetHolder,
            UserDataElement element
        )
        {
            var cmdParam = new UserDataElementModifyCmd.ModifyCmdParam();
            cmdParam.TargetHolder = targetHolder;
            cmdParam.Element = element;

            UserDataElementModifyCmd cmd = new UserDataElementModifyCmd(
                UserDataElementModifyCmd.ModifyCmdType.ElementAdd,
                cmdParam
            );

            _CmdQueue.RegisterCmd( cmd );
        }

        /// <summary>
        /// ユーザーデータエレメント置換コマンドを登録します。
        /// </summary>
        /// <param name="targetHolder">変更対象のユーザーデータホルダ</param>
        /// <param name="element">削除するユーザーデータエレメント</param>
        /// <param name="insertElement">追加するユーザーデータエレメント</param>
        /// <param name="elementIndex">エレメントの番号</param>
        /// <param name="attributeIndex">アトリビュートの番号</param>
        public void MakeUserDataElementReplaceCmd
        (
            UserDataHolder targetHolder,
            UserDataElement element,
            UserDataElement replaceElement,
            int elementIndex,
            int attributeIndex
        )
        {
            var cmdParam = new UserDataElementModifyCmd.ModifyCmdParam();
            cmdParam.TargetHolder = targetHolder;
            cmdParam.Element = element;
            cmdParam.ReplaceElement = replaceElement;
            cmdParam.BeforeElementIndex = elementIndex;
            cmdParam.BeforeAttributeIndex = attributeIndex;

            UserDataElementModifyCmd cmd = new UserDataElementModifyCmd(
                UserDataElementModifyCmd.ModifyCmdType.ElementReplace,
                cmdParam
            );

            _CmdQueue.RegisterCmd( cmd );
        }

        /// <summary>
        /// ユーザーデータエレメント削除コマンドを登録します。
        /// </summary>
        /// <param name="targetHolder">変更対象のユーザーデータホルダ</param>
        /// <param name="element">削除するユーザーデータエレメント</param>
        /// <param name="insertElement">挿入するユーザーデータエレメント</param>
        /// <param name="elementIndex">エレメントの番号</param>
        /// <param name="attributeIndex">アトリビュートの番号</param>
        public void MakeUserDataElementRemoveCmd
        (
            UserDataHolder targetHolder,
            UserDataElement element,
            int elementIndex,
            int attributeIndex
        )
        {
            var cmdParam = new UserDataElementModifyCmd.ModifyCmdParam();
            cmdParam.TargetHolder = targetHolder;
            cmdParam.Element = element;
            cmdParam.BeforeElementIndex = elementIndex;
            cmdParam.BeforeAttributeIndex = attributeIndex;

            UserDataElementModifyCmd cmd = new UserDataElementModifyCmd(
                UserDataElementModifyCmd.ModifyCmdType.ElementRemove,
                cmdParam
            );

            _CmdQueue.RegisterCmd( cmd );
        }

        /// <summary>
        /// ユーザーデータエレメントのセットコマンドを登録します。
        /// </summary>
        /// <param name="targetHolder">変更対象のユーザーデータホルダ</param>
        /// <param name="element">変更対象のユーザーデータエレメント</param>
        /// <param name="beforeName">変更前の名前</param>
        /// <param name="afterName">変更後の名前</param>
        /// <param name="afterName">変更前の上書き設定</param>
        /// <param name="afterName">変更後の上書き設定</param>
        /// <param name="beforeValue">変更前の値</param>
        /// <param name="afterValue">変更後の値</param>
        public void MakeUserDataElementSetCmd
        (
            UserDataHolder targetHolder,
            UserDataElement element,
            string beforeName,
            string afterName,
            bool beforeOverwrite,
            bool afterOverwrite,
            UserDataKind beforeKind,
            UserDataKind afterKind,
            object beforeValue,
            object afterValue,
            object beforeBaseValue,
            object afterBaseValue
        )
        {
            var cmdParam = new UserDataElementModifyCmd.ModifyCmdParam();
            cmdParam.TargetHolder = targetHolder;
            cmdParam.Element = element;
            cmdParam.BeforeName = beforeName;
            cmdParam.AfterName = afterName;
            cmdParam.BeforeOverwrite = beforeOverwrite;
            cmdParam.AfterOverwrite = afterOverwrite;
            cmdParam.BeforeKind = beforeKind;
            cmdParam.AfterKind = afterKind;
            cmdParam.BeforeValue = beforeValue;
            cmdParam.AfterValue = afterValue;
            cmdParam.BeforeBaseValue = beforeBaseValue;
            cmdParam.AfterBaseValue = afterBaseValue;

            UserDataElementModifyCmd cmd = new UserDataElementModifyCmd(
                UserDataElementModifyCmd.ModifyCmdType.ElementSet,
                cmdParam
            );

            _CmdQueue.RegisterCmd( cmd );
        }

        /// <summary>
        /// ユーザーデータエレメントの番号変更コマンドを登録します。
        /// </summary>
        /// <param name="targetHolder">変更対象のユーザーデータホルダ</param>
        /// <param name="element">変更対象のユーザーデータエレメント</param>
        /// <param name="beforeElementIndex">変更前のエレメント番号</param>
        /// <param name="afterElementIndex">変更後のエレメント番号</param>
        /// <param name="beforeAttributeIndex">変更前のアトリビュート番号</param>
        /// <param name="afterAttributeIndex">変更後のアトリビュート番号</param>
        public void MakeUserDataElementIndexChangeCmd
        (
            UserDataHolder targetHolder,
            UserDataElement element,
            int beforeElementIndex,
            int afterElementIndex,
            int beforeAttributeIndex,
            int afterAttributeIndex
        )
        {
            var cmdParam = new UserDataElementModifyCmd.ModifyCmdParam();
            cmdParam.TargetHolder = targetHolder;
            cmdParam.Element = element;
            cmdParam.BeforeElementIndex = beforeElementIndex;
            cmdParam.AfterElementIndex = afterElementIndex;
            cmdParam.BeforeAttributeIndex = beforeAttributeIndex;
            cmdParam.AfterAttributeIndex = afterAttributeIndex;

            UserDataElementModifyCmd cmd = new UserDataElementModifyCmd(
                UserDataElementModifyCmd.ModifyCmdType.ElementIndexChange,
                cmdParam
            );

            _CmdQueue.RegisterCmd( cmd );
        }

        /// <summary>
        /// ユーザーデータエレメントの復元ポイントコマンドを登録します。
        /// </summary>
        /// <param name="targetHolder">変更対象のユーザーデータホルダ</param>
        /// <param name="elements">エレメントリスト</param>
        public void MakeUserDataRestoreOnUndoCmd
        (
            UserDataHolder targetHolder,
            UserDataElement[] elements
        )
        {
            UserDataRestoreCmd cmd = new UserDataRestoreCmd(
                UserDataRestoreCmd.ModifyCmdType.OnUndo,
                targetHolder,
                elements
            );

            _CmdQueue.RegisterCmd( cmd );
        }

        /// <summary>
        /// ユーザーデータエレメントの復元ポイントコマンドを登録します。
        /// </summary>
        /// <param name="targetHolder">変更対象のユーザーデータホルダ</param>
        /// <param name="elements">エレメントリスト</param>
        public void MakeUserDataRestoreOnRedoCmd
        (
            UserDataHolder targetHolder,
            UserDataElement[] elements
        )
        {
            UserDataRestoreCmd cmd = new UserDataRestoreCmd(
                UserDataRestoreCmd.ModifyCmdType.OnRedo,
                targetHolder,
                elements
            );

            _CmdQueue.RegisterCmd( cmd );
        }
        #endregion サブアトリビュート変更

      #region プロパティ変更
      /// <summary>
      /// プロパティ変更コマンドを登録します。
      /// </summary>
      /// <param name="target"></param>
      /// <param name="beforeVal"></param>
      /// <param name="afterVal"></param>
      public void MakePropModCmd( IValueAccessor target, object beforeVal, object afterVal )
      {
          PropModifyCmd cmd = new PropModifyCmd( target, beforeVal, afterVal );

          // グローバル・コマンドキューに登録
          _CmdQueue.RegisterCmd( cmd );
      }
      #endregion プロパティ変更

      /// <summary>
      /// カーブのプロパティ値を変更するコマンドを登録します。
      /// </summary>
      /// <param name="target"></param>
      /// <param name="beforeValue"></param>
      /// <param name="afterValue"></param>
      /// <param name="setter"></param>
      public delegate void Setter( AnmCurve anmCurve, object value);
      public void SetValueCommand( AnmCurve targetAnmCurve, object beforeValue, object afterValue, Setter setter)
      {
          setter( targetAnmCurve, afterValue);

          AnmCurveModifyCommand command =
              new AnmCurveModifyCommand( targetAnmCurve, beforeValue, afterValue,
                                         delegate( AnmCurve anmCurve, object value)
                                             {
                                                 setter( anmCurve, value);
                                             });

          //
          _CmdQueue.RegisterCmd( command);
      }

        /// <summary>
        /// アクションをコマンドにします。
        /// </summary>
        public void SetActionCommand(Action action)
        {
            _CmdQueue.RegisterCmd(new ActionCommand(action));
        }

        #region キー追加、削除
        /// <summary>
        /// アニメーションキー追加コマンドを登録します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="beforeVal"></param>
        /// <param name="afterVal"></param>
        public void MakeAnmKeyAddCmd
        (
            AnmKeyFrame targetKey
        )
        {
            AnmCurveModCmd cmd = new AnmCurveModCmd(
               AnmCurveModCmd.ModifyCmdType.KeyAdd,
               targetKey, null );

            _CmdQueue.RegisterCmd( cmd );
        }

        /// <summary>
        /// アニメーションキー削除コマンドを登録します。
        /// </summary>
        /// <param name="targetKey"></param>
        public void MakeAnmKeyRemoveCmd
        (
            AnmKeyFrame targetKey
        )
        {
            AnmCurveModCmd cmd = new AnmCurveModCmd(
                AnmCurveModCmd.ModifyCmdType.KeyRemove,
                targetKey, null );

            _CmdQueue.RegisterCmd( cmd );
        }

        /// <summary>
        /// アニメーションキー更新コマンドを登録します。
        /// </summary>
        /// <param name="targetKey"></param>
        /// <param name="cloneKeyBefore"></param>
        /// <param name="cloneKeyAfter"></param>
        public void MakeAnmKeyModCmd
        (
            AnmKeyFrame targetKey,
            AnmKeyFrame cloneKeyBefore,
            AnmKeyFrame cloneKeyAfter
        )
        {
            AnmCurveModCmd cmd = new AnmCurveModCmd(
                AnmCurveModCmd.ModifyCmdType.KeyModify,
                targetKey,
                new AnmCurveModCmd.AnmKeyMement( cloneKeyBefore, cloneKeyAfter ) );
            // グローバル・コマンドキューに登録
            _CmdQueue.RegisterCmd( cmd );
        }
        #endregion キー追加、削除

        #region ペイン登録、削除
        /// <summary>
        /// ペイン追加コマンドを登録します。
        /// </summary>
        /// <param name="subScene"></param>
        /// <param name="newPane"></param>
        public void MakeSceneAddPaneCmd
        (
            SubScene subScene,
            Pane newPane
        )
        {
            SceneModifyCmd cmd = new SceneModifyCmd(
                SceneModifyEventArgs.Kind.PaneAdd,  // ペイン追加
                subScene,                              // シーン
                newPane );                             // 新規ペイン

            _CmdQueue.RegisterCmd( cmd );
        }


        /// <summary>
        /// ペイン削除コマンドを登録します。
        /// </summary>
        /// <param name="subScene"></param>
        /// <param name="newPane"></param>
        public void MakeSceneRemovePaneCmd
        (
            SubScene subScene,
            Pane newPane
        )
        {
            SceneModifyCmd cmd = new SceneModifyCmd(
                SceneModifyEventArgs.Kind.PaneRemove,   // ペイン削除
                subScene,                                  // シーン
                newPane );                                 // 新規ペイン

            _CmdQueue.RegisterCmd( cmd );
        }
        #endregion ペイン登録、削除

        #region 階層構築、削除
        /// <summary>
        /// 階層設定コマンドを登録します。
        ///
        /// 階層解除はコマンドRootPaneへの
        /// 階層設定コマンドとして登録してください。
        /// </summary>
        /// <param name="subScene"></param>
        /// <param name="newPane"></param>
        public void MakeMakeHierarchyCmd
        (
            SubScene subScene,
            Pane parent,
            Pane child
        )
        {
            SceneModifyCmd.ConnectinPair connectinPair
                = new SceneModifyCmd.ConnectinPair( parent, child );

            SceneModifyCmd cmd = new SceneModifyCmd(
                SceneModifyEventArgs.Kind.HierarchyMake,// 階層構築
                subScene,                       // シーン
                connectinPair );               // 接続ペア

            _CmdQueue.RegisterCmd( cmd );
        }

        /// <summary>
        ///
        /// </summary>
        public void MakeEditHierarchyOrderCmd
         (
             SubScene subScene,
             Pane parent,
             Pane child,
            int newIndex
         )
        {
            SceneModifyCmd.ConnectinPair connectinPair
                = new SceneModifyCmd.ConnectinPair( parent, child, newIndex );

            SceneModifyCmd cmd = new SceneModifyCmd(
                SceneModifyEventArgs.Kind.HierarchyReordr,// 階層構築
                subScene,                       // シーン
                connectinPair );               // 接続ペア

            _CmdQueue.RegisterCmd( cmd );
        }

        #endregion 階層構築、削除

        #region   グループ構築、削除
        /// <summary>
        /// グループ関連コマンド
        /// </summary>
        public void MakeSceneGroupModCmd
        (
            LEGroupMgrModifyEventArgs eventArgs
        )
        {

            SceneModifyCmd cmd = new SceneModifyCmd(
                eventArgs.EventKind,
                null,
                eventArgs.GroupMgrParams );

            // グローバル・コマンドキューに登録
            _CmdQueue.RegisterCmd( cmd );
        }
        #endregion グループ構築、削除

        #region プロパティ変更
        /// <summary>
        /// プロパティ変更コマンドを登録します。
        /// </summary>
        /// <param name="ownerInstance"></param>
        /// <param name="propertyType"></param>
        /// <param name="beforeVal"></param>
        /// <param name="afterVal"></param>
        public void MakePropertyModifyCmd(
            object ownerInstance,
            string propertyName,
            object beforeVal,
            object afterVal )
        {
            //
            // TODO:beforeVal、afterVal等が 同一のオブジェクトを参照していないかチェックする！
            //
            PropertyModifyCmd cmd =
                new PropertyModifyCmd( ownerInstance, propertyName, beforeVal, afterVal );
            _CmdQueue.RegisterCmd( cmd );
        }

        #endregion プロパティ変更

        #region ペインコピー
        public void MakePaneCopyCmd(
            IPane pane,
            IPane preCopy,
            IPane aftCopy )
        {
            _CmdQueue.RegisterCmd( new PaneCopyCmd( pane, preCopy, aftCopy ) );
        }

        #endregion

        #region TextBox更新コマンド
        /// <summary>
        /// メメント更新コマンド
        /// </summary>
        public void MakeMementoModifyCmd<T>(
            T target,
            ICommandMemento<T> beforeMod,
            ICommandMemento<T> afterMod )
        {
            _CmdQueue.RegisterCmd( new MementModifyCommand<T>( target, beforeMod, afterMod ) );
        }


        #endregion

        #region アニメーション関連
        /// <summary>
        /// アニメーション管理モード変更コマンド
        /// </summary>
        public void MakeAnmModeCmd(
            SubSceneManipulator.AnmConvertParam beforeMod,
            SubSceneManipulator.AnmConvertParam afterMod )
        {
            _CmdQueue.RegisterCmd(new AnmModeCmd(beforeMod, afterMod));
        }

        /// <summary>
        /// アニメーション追加コマンド
        /// </summary>
        public void MakeAnmAddCmd(
            ISubScene sunScene,
            string tagName,
            SubSceneManipulator.AnmConvertParam beforeMod,
            SubSceneManipulator.AnmConvertParam afterMod)
        {
            _CmdQueue.RegisterCmd(new AnmAddCmd(sunScene, tagName, beforeMod, afterMod));
        }

        /// <summary>
        /// アニメーション削除コマンド
        /// </summary>
        public void MakeAnmRemoveCmd(
            ISubScene subScene,
            string tagName,
            SubSceneManipulator.AnmConvertParam beforeMod,
            SubSceneManipulator.AnmConvertParam afterMod)
        {
            _CmdQueue.RegisterCmd(new AnmRemoveCmd(subScene, tagName, beforeMod, afterMod));
        }

        /// <summary>
        /// 区間タグ名変更コマンド
        /// </summary>
        public void MakeModifyTagNameCmd(
            IEnumerable<IPane> target,
            string beforeMod,
            string afterMod)
        {
            _CmdQueue.RegisterCmd(new ModifyTagNameCmd(target, beforeMod, afterMod));
        }

        /// <summary>
        /// アニメーション拡大縮小コマンド
        /// </summary>
        public void MakeScaleAnimationCmd(
            SubSceneManipulator.AnmConvertParam beforeMod,
            SubSceneManipulator.AnmConvertParam afterMod )
        {
            _CmdQueue.RegisterCmd(new ScaleAnimationCmd(beforeMod, afterMod));
        }
        #endregion
    }


    #region コマンドバッファ(サブシーンごとにインスタンスを持つ)

    // コマンドの種類
    public enum CmdQueueChangedEventKind
    {
        Undo,           // Undo
        Redo,           // Redo
        NewReginster,   // 新規登録
        ClearAll        // 全消去
    }

    /// <summary>
    /// コマンドキュー 変更イベントデリゲート
    /// </summary>
    public delegate void QueueStateChangedHandler( ISubSceneCmdQueue sender, CmdQueueChangedEventKind kind, ICommand cmd );



    /// <summary>
    /// サブシーン・コマンドキュー(外部公開インタフェース)
    /// </summary>
    public interface ISubSceneCmdQueue
    {
        event QueueStateChangedHandler QueueStateChangedEvent;

        int NumUndoCommands { get;}
        int NumRedoCommands { get;}

    }

    /// <summary>
    /// グローバル・コマンドキュー
    /// </summary>
    public class SubSceneCmdQueue :
        ISubSceneCmdQueue
    {
        // 最大Undo数
        const int MAX_UNDO_NUM = 255;

        #region 内部型



        /// <summary>
        /// コマンドキューの内部状態を表す情報。
        /// シーンの更新状態を判断するのに使用されます。
        /// </summary>
        public struct InternalState
        {
            public readonly int NumTotalCmdRegistered;
            public readonly int NumUndoCmd;
            public readonly int NumRedoCmd;
            public readonly int NumRedoCmdDropped;

            // 異なるコマンドが実行されている。
            public bool _bDifferentCmdExecuted;


            /// <summary>
            /// 実際実行されているコマンド数を取得します。
            ///
            /// Undoスタックが最大制限に達するまでは、
            /// NumUndoCmdと等しい値になるはずです。
            /// </summary>
            public int NumCmdExecuted
            {
                get { return NumTotalCmdRegistered - NumRedoCmd; }
            }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public InternalState( int numTotalCmdRegistered, int numUndoCmd, int numRedoCmd, int numRedoCmdDropped )
            {
                NumTotalCmdRegistered = numTotalCmdRegistered;
                NumUndoCmd = numUndoCmd;
                NumRedoCmd = numRedoCmd;
                NumRedoCmdDropped = numRedoCmdDropped;
                _bDifferentCmdExecuted = false;
            }

            /// <summary>
            /// ToString()
            /// </summary>
            public override string ToString()
            {
                return "Total = " + NumTotalCmdRegistered.ToString() +
                       "Undo  = " + NumUndoCmd.ToString() +
                       "Redo  = " + NumRedoCmd.ToString();
            }

            /// <summary>
            /// サブシーンが更新されているか判定します。
            /// </summary>
            public bool IsOwnerSubSceneChanged( InternalState another )
            {
                // another で コマンドが登録された直後に...
                // (新規コマンド登録時に、Redoコマンドの破棄が起こる)
                if( another.NumRedoCmd == 0 &&
                    another.NumRedoCmdDropped != this.NumRedoCmdDropped )
                {
                    // another で 実行されたコマンド数が、
                    // 保存状態より、少ない場合は
                    if( another.NumCmdExecuted <= this.NumCmdExecuted )
                    {
                        // 別のコマンドが実行(登録)されている。
                        _bDifferentCmdExecuted = true;
                    }
                }

                // シーンに適用されたコマンドの数が異なる。
                if( another.NumCmdExecuted != this.NumCmdExecuted )
                {
                    return true;
                }
                else if( _bDifferentCmdExecuted )
                {
                    return true;
                }


                // シーンに変更はない。
                return false;
            }
        }

        /// <summary>
        /// コマンド集約を行う際に、
        /// 登録コマンドをキャッシュするデータ構造
        /// </summary>
        struct CmdCache
        {
            ArrayList _cmdSet;
            ISubScene _targetSubScene;


            /// <summary>
            /// コマンド配列
            /// </summary>
            public ArrayList CmdSet
            {
                get { return _cmdSet; }
            }


            /// <summary>
            /// コマンドが実行されるシーン
            /// </summary>
            public ISubScene TargetSubScene
            {
                get { return _targetSubScene; }
            }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="targetScene">コマンドが処理されるシーン</param>
            public CmdCache( ISubScene targetSubScene )
            {
                Debug.Assert( targetSubScene != null );
                _targetSubScene = targetSubScene;
                _cmdSet = new ArrayList();
            }
        }
        #endregion 内部型

        #region ------------------------- フィールド -------------------------
        // コマンドキャッシュスタック
        // Begin-EndCmdPacking() が呼ばれている状態で登録されたコマンドをキャシュしておく
        // 要素が CmdCache のスタックです。
        Stack _cmdCacheStack = new Stack(); // CmdCache

        /// <summary>
        /// コマンドの管理方法をリングバッファによるものから、
        /// 2つのスタックを使用するものに変更しました。
        /// </summary>
        Stack _undoCmdStack = new Stack();
        Stack _redoCmdStack = new Stack();
        int _numCmdRegistered = 0;
        int _numCmdRedoDropped = 0;

        #endregion ------------------------- フィールド -------------------------

        #region ------------------------- 公開イベント -------------------------
        // 更新通知イベント
        public event QueueStateChangedHandler QueueStateChangedEvent;
        #endregion ------------------------- 公開イベント -------------------------

        #region ------------------------- プロパティ -------------------------


        /// <summary>
        /// Undo可能か？
        /// </summary>
        public bool UndoEnable
        {
            get { return _undoCmdStack.Count != 0; }
        }

        /// <summary>
        /// Redo可能か？
        /// </summary>
        public bool RedoEnable
        {
            get { return _redoCmdStack.Count != 0; }
        }

        /// <summary>
        /// 内部状態を取得します。
        /// シーンの更新状態の判定などに使用されます。
        /// </summary>
        public InternalState InternalStateData
        {
            get { return new InternalState( _numCmdRegistered, _undoCmdStack.Count, _redoCmdStack.Count, _numCmdRedoDropped ); }
        }

        /// <summary>
        /// Undoコマンドを登録します。
        /// 最大数制限を処理します。
        ///
        /// FIX_ME:実装方法は効率に問題があるので、データ構造を変更する。
        /// </summary>
        void PushUndoCmd_( ICommand cmd )
        {
            _undoCmdStack.Push( cmd );

            // 制限を越えてしまったら...
            if( _undoCmdStack.Count >= MAX_UNDO_NUM )
            {
                // 最も古いコマンド、（先頭の要素）をひとつ消去する。
                object[] temp = _undoCmdStack.ToArray();
                _undoCmdStack = new Stack();
                for( int i = temp.Length - 2 ; i >= 0 ; i-- )
                {
                    _undoCmdStack.Push( temp[i] );
                }
            }
        }

        /// <summary>
        /// コマンドを集約中か？
        /// </summary>
        bool _IsPackingCmd
        {
            get { return _cmdCacheStack.Count != 0; }
        }

        #endregion //  ------------------------- プロパティ -------------------------


        /// <summary>
        /// コマンドキューの状態をリセットします。
        /// </summary>
        public void Reset()
        {
            _undoCmdStack.Clear();
            _redoCmdStack.Clear();

            if( QueueStateChangedEvent != null )
            {
                QueueStateChangedEvent( this, CmdQueueChangedEventKind.ClearAll, null );
            }
        }

        /// <summary>
        /// シーンをUndoする
        /// </summary>
        public void Undo()
        {
            // Undoを実行する
            if( UndoEnable )
            {
                ICommand cmd = _undoCmdStack.Pop() as ICommand;
                _redoCmdStack.Push( cmd );

                cmd.Undo();

                // 変更イベントを通知します
                if( QueueStateChangedEvent != null )
                {
                    QueueStateChangedEvent( this, CmdQueueChangedEventKind.Undo, cmd );
                }
            }
        }

        /// <summary>
        /// シーンをRedoする
        /// </summary>
        public void Redo()
        {
            // Redoを実行する
            if( RedoEnable )
            {
                ICommand cmd = _redoCmdStack.Pop() as ICommand;
                PushUndoCmd_( cmd );

                cmd.Redo();

                // 変更イベントを通知します
                if( QueueStateChangedEvent != null )
                {
                    QueueStateChangedEvent( this, CmdQueueChangedEventKind.Redo, cmd );
                }
            }
        }

        /// <summary>
        /// コマンドを登録する
        /// </summary>
        /// <param name="newCmd"></param>
        internal void RegisterCmd( ICommand newCmd )
        {
            // コマンドを集約中の場合は、直接キューへの登録は行わず、
            // 一旦、内部キャッシュに記録されます。
            if( _IsPackingCmd )
            {
                CmdCache cmdCache = (CmdCache)_cmdCacheStack.Peek();
                cmdCache.CmdSet.Add( newCmd );
            }
            else
            {

                // 現在のRedoコマンドは消去されRedo不可能となる。
                // 捨てたコマンドは、総登録コマンド数から削除する。
                _numCmdRegistered -= _redoCmdStack.Count;
                _numCmdRedoDropped += _redoCmdStack.Count;
                _redoCmdStack.Clear();

                PushUndoCmd_( newCmd );
                _numCmdRegistered++;

                // 変更イベントを通知します
                if( QueueStateChangedEvent != null )
                {
                    QueueStateChangedEvent( this, CmdQueueChangedEventKind.NewReginster, newCmd );
                }
            }
        }

        /// <summary>
        /// コマンドパッキング開始、複数のコマンドを集約して1つのコマンドに集約するように、
        /// 指示します。
        /// 本関数呼び出し以後の、 RegisterCmd()で登録されたコマンドは、
        /// すべて、一旦クラス内にキャッシングされます。
        /// 対応する、EndCmdPacking()が呼ばれた際に、キャッシングされたコマンドを集約コマンドとして、
        /// キューに登録します。
        ///
        /// 本関数は入れ子で呼び出しが可能です。
        /// （本関数が入れ子で呼び出された場合は、集約されたコマンドが再び、キャッシュ内に記憶され、
        /// 一番最後に呼ばれる、EndCmdPacking()内で、キューにコマンドが登録されます。）
        ///
        /// </summary>
        public void BeginCmdPacking( ISubScene subScene )
        {
            CmdCache cmdCache = new CmdCache( subScene );
            _cmdCacheStack.Push( cmdCache );
        }

        /// <summary>
        /// コマンドパッキング終了
        /// </summary>
        public void EndCmdPacking(bool forceDropCommand)
        {
            Debug.Assert( _IsPackingCmd );
            CmdCache cmdCache = (CmdCache)_cmdCacheStack.Pop();

            ArrayList cmdSet = cmdCache.CmdSet;
            // 複合コマンドを作成して登録します。
            if( cmdSet.Count != 0 )
            {
                ICommand[] srcCmds = cmdSet.ToArray( typeof( ICommand ) ) as ICommand[];
                ComposedCmd composedCmd = new ComposedCmd( srcCmds, cmdCache.TargetSubScene );

                if (!forceDropCommand)
                {
                    RegisterCmd(composedCmd);
                }
            }
        }

        #region -------------- 非公開メソッド --------------


        #endregion // -------------- 非公開メソッド --------------


        #region ISubSceneCmdQueue

        public int NumUndoCommands
        {
            get { return _undoCmdStack.Count; }
        }

        public int NumRedoCommands
        {
            get { return _redoCmdStack.Count; }
        }

        #endregion ISubSceneCmdQueue


    }
    #endregion コマンドバッファ(サブシーンごとにインスタンスを持つ)


}
