﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using System.Diagnostics;

namespace LECore.Structures.Core
{
    /// <summary>
    /// アニメーションカーブの補間タイプ
    /// </summary>
    public enum InterporationType
    {
        None,        // 補間なし
        Fixed,       // 固定値（ユーザによって値が変更されている。）
        Zero,        // ゼロ
        Step,        // ステップ補間
        Liner,       // 線形補間
        Clamped,     // クランプ
        Spline,      // スプライン
    }

    /// <summary>
    /// キーフレーム設定範囲外の計算方法
    /// </summary>
    public enum AnmCurveInfinityType
    {
        Constant,  // 定数
        Cycle       // 循環
    }

    /// <summary>
    /// アトリビュートの種類
    /// </summary>
    public enum AttributeType
    {
        Int        = 0,
        Float      = 1,
        Bool       = 2,
        Enum       = 3,
        Byte       = 4,
        IntVec2    = 5,
        FloatVec2  = 6,
        FloatVec2UV= 7,
        FloatVec3  = 8,
        ByteRGBA4  = 9,
        FloatRGBA4 = 10,
        Combined   = 11,
        RefAttribute = 12,
    }

    /// <summary>
    /// アニメーションマークの変化アクション
    /// </summary>
    public enum AnimationStatusKind
    {
        None,
        SameKey,
        DiffKey,
        HasKey,
        DiffValue,
    }

    /// <summary>
    /// AnmAttribute の
    /// システムコア外部に公開されるインタフェース
    /// </summary>
    public interface IAnmAttribute
    {
        bool HasAnimationCurve{get;}

        string Name{get;}

        string OwnerName{get;}

        string DescriptionName{get;}

        int NumSubAttribute{get;}

        bool HasSubAttribute{get;}

        IAnmCurve ICurrentAnimationCurve {get;}

        IAnmCurve[] Curves {get;}

        IAnmAttribute OwnerNode { get;}
        IAnmAttribute FindSubAttributeByIdx( int idx );

        bool IsActiveAttribute { get;}

        bool EnableLocalize { get;}

        object Value { get; }
    }

    /// <summary>
    /// 補助クラス
    /// </summary>
    public static class AnmCurveHelper
    {
        /// <summary>
        /// 最初のキーを取得する。
        /// </summary>
        public static IAnmKeyFrame GetFirstKey(this IAnmCurve curve)
        {
            IAnmKeyFrame firstKey = null;
            foreach (var key in curve.IKeyFrameSet)
            {
                if (firstKey == null)
                {
                    firstKey = key;
                }
                else
                {
                    if (firstKey.Time >= key.Time)
                    {
                        firstKey = key;
                    }
                }
            }

            return firstKey;
        }

        /// <summary>
        /// キーを持っているか？
        /// </summary>
        public static IAnmKeyFrame GetLastKey(this IAnmCurve curve)
        {
            IAnmKeyFrame lastKey = null;
            foreach (var key in curve.IKeyFrameSet)
            {
                if (lastKey == null)
                {
                    lastKey = key;
                }
                else
                {
                    if (lastKey.Time <= key.Time)
                    {
                        lastKey = key;
                    }
                }
            }

            return lastKey;
        }

        /// <summary>
        /// カーブが一致するか
        /// </summary>
        public static bool IsSame(this IAnmCurve c0, IAnmCurve c1)
        {
            if (!IsSameWithoutKeis(c0, c1) || c0.NumKeyFrame != c1.NumKeyFrame)
            {
                return false;
            }

            for (int i = 0; i < c0.NumKeyFrame; i++)
            {
                var anmKey0 = c0.IKeyFrameSet[i];
                var anmKey1 = c1.IKeyFrameSet[i];

                if (!anmKey0.IsSame(anmKey1))
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// カーブが一致するか（キーは比較しない）
        /// </summary>
        public static bool IsSameWithoutKeis(this IAnmCurve c0, IAnmCurve c1)
        {
            return
                c0.AttrName == c1.AttrName &&
                c0.CustomTimeFrame == c1.CustomTimeFrame &&
                c0.HasCustomTimeSource == c1.HasCustomTimeSource &&
                c0.IsReadOnlyLocked == c1.IsReadOnlyLocked &&
                c0.NumSelectedKey == c1.NumSelectedKey &&
                c0.PostInfinityType == c1.PostInfinityType &&
                c0.PreInfinityType == c1.PreInfinityType &&
                c0.TargetAttrType == c1.TargetAttrType &&
                c0.ViewScale == c1.ViewScale;
        }
    }

    /// <summary>
    /// 補助関数郡
    /// </summary>
    public static class AnmAttributeHelper
    {
        /// <summary>
        /// アニメーションを評価して指定時間の値に更新します。
        /// </summary>
        internal static void EvaluateAnimAll(this IAnmAttribute attribute, float time)
        {
            if (attribute.ICurrentAnimationCurve != null)
            {
                (attribute as AnmAttribute).UpdateAttributeValue((int)time);
            }

            for (int i = 0; i < attribute.NumSubAttribute; i++)
            {
                EvaluateAnimAll(attribute.FindSubAttributeByIdx(i), time);
            }
        }

        /// <summary>
        /// アトリビュートの値を現在の値に戻すアクションの取得
        /// </summary>
        public static Action GetResetValueAction(IAnmAttribute attribute)
        {
            try
            {
                if (attribute is AnmAttribute)
                {
                    object value;
                    ((AnmAttribute)attribute).GetValue(out value);
                    return () =>
                    {
                        ((AnmAttribute)attribute).SetValue(value);
                    };
                }
                else
                {
                    throw new InvalidOperationException();
                }
            }
            catch
            {
                System.Diagnostics.Debug.Assert(false);
                throw;
            }
        }

        /// <summary>
        /// キーを持っているか？
        /// </summary>
        public static bool HasKey(this IAnmAttribute attr)
        {
            return attr.HasAnimationCurve && attr.ICurrentAnimationCurve.NumKeyFrame != 0;
        }

        /// <summary>
        /// キーを持っているか？
        /// 分割モードの場合はすべてのタグ区間についてチェックします
        /// </summary>
        public static bool HasAnyKey(this IAnmAttribute attr)
        {
            AnmAttribute attribute = attr as AnmAttribute;
            return (attribute != null) ? attribute.HasAnyKey : false;
        }

        /// <summary>
        /// 指定したタグ区間にキーを持っているか？
        /// </summary>
        public static bool HasKey(this IAnmAttribute attr, string tag)
        {
            AnmAttribute attribute = attr as AnmAttribute;
            AnmCurve curve = attribute.GetAnmCurve(tag);
            if (curve != null && curve.IKeyFrameSet != null)
            {
                if (curve.IKeyFrameSet.Count() > 0)
                {
                    return true;
                }
            }

            foreach (var subAttr in attribute.SubAttrArray)
            {
                if (subAttr.HasKey(tag))
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 指定フレーム範囲でキーを持っているか？
        /// </summary>
        public static bool HasKeyInRange(this IAnmAttribute attr, int startFrame, int endFrame, string tag = null)
        {
            var curve = tag != null ? attr.GetAnmCurveRelationTag(tag) : attr.ICurrentAnimationCurve;
            // パラメタライズドアニメーションの場合の処理
            if (curve != null && curve.IsParameterizedAnim)
            {
                var parameters = curve.ParameterizedAnimParameters;
                foreach(var parameter in parameters)
                {
                    // パラメタライズドアニメーションは区間が完全に指定フレーム範囲に入っていなければ正しく動作しない
                    if ((float)startFrame <= parameter.offset && parameter.offset + parameter.duration <= (float)endFrame)
                    {
                        return true;
                    }
                }
                return false;
            }

            if (curve == null || curve.NumKeyFrame == 0)
            {
                return false;
            }

            // 最初のキーより前かどうか調査
            if (curve.PreInfinityType != AnmCurveInfinityType.Constant)
            {
                if (startFrame <= curve.GetFirstKey().Time)
                {
                    return true;
                }
            }

            // 最後のキーより後かどうか調査
            if (curve.PostInfinityType != AnmCurveInfinityType.Constant)
            {
                if (endFrame >= curve.GetLastKey().Time)
                {
                    return true;
                }
            }

            for (int i = 0; i < curve.NumKeyFrame; i++)
            {
                float time = curve.IKeyFrameSet[i].Time;
                if (startFrame <= time && endFrame >= time)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// キーを持っているか？
        /// </summary>
        public static int GetSelfSubAttrIndex(this IAnmAttribute attr)
        {
            if (attr.OwnerNode == null)
            {
                return -1;
            }

            for (int i = 0; i < attr.OwnerNode.NumSubAttribute; i++)
            {
                if(object.ReferenceEquals( attr.OwnerNode.FindSubAttributeByIdx(i), attr))
                {
                    return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// 再帰的にアクティブかどうかを調査する。すべての親がアクティブならアクティブ。
        /// </summary>
        public static bool CheckActiveRecursive(this IAnmAttribute attr)
        {
            while(attr.OwnerNode != null)
            {
                if (!attr.IsActiveAttribute)
                {
                    return false;
                }

                attr = attr.OwnerNode;
            }

            return attr.IsActiveAttribute;
        }

        /// <summary>
        /// 再帰的にキーを持つか調査する。
        /// 分割モードの場合選択中のタグ区間についてのみ再帰的に調査する。
        /// </summary>
        public static bool CheckHaveKeyRecursive(this IAnmAttribute attr)
        {
            return CheckHaveKeyRecursiveInRange(attr, int.MinValue, int.MaxValue);
        }

        /// <summary>
        /// 再帰的にキーを持つか調査する。
        /// 分割モードの場合選択中のタグ区間についてのみ再帰的に調査する。
        /// </summary>
        public static bool CheckHaveKeyRecursiveInRange(this IAnmAttribute attr, int startFrame, int endFrame)
        {
            // 自身がキーを持つか調べる
            if (attr.HasKeyInRange(startFrame, endFrame))
            {
                return true;
            }

            // サブアトリビュートについて、キーを持つか調べる
            for (int i = 0; i < attr.NumSubAttribute; i++)
            {
                IAnmAttribute subAttr = attr.FindSubAttributeByIdx(i);
                if (subAttr.CheckHaveKeyRecursiveInRange(startFrame, endFrame))
                {
                    return true;
                }
            }

            // サブアトリビュートも含めて、キーを持つアニメーションカーブが発見されなかった。
            return false;
        }

        /// <summary>
        /// 再帰的にパラメタライズドアニメーションであるかを調査する。
        /// </summary>
        public static bool CheckParameterizedAnimRecursive(this IAnmAttribute attr)
        {
            if (attr.Curves == null)
            {
                return false;
            }
            foreach (IAnmCurve curve in attr.Curves)
            {
                if (curve.IsParameterizedAnim)
                {
                    return true;
                }
            }

            // サブアトリビュートについて、パラメタライズドアニメーションかどうか調べる
            for (int i = 0; i < attr.NumSubAttribute; i++)
            {
                IAnmAttribute subAttr = attr.FindSubAttributeByIdx(i);
                if (subAttr.CheckParameterizedAnimRecursive())
                {
                    return true;
                }
            }

            // サブアトリビュートも含めて、パラメタライズドアニメーションが発見されなかった。
            return false;
        }

        /// <summary>
        /// 再帰的にキーを持つか調査する。
        /// 分割モードの場合すべてのタグ区間について再帰的に調査する。
        /// </summary>
        public static bool CheckHaveKeyRecursiveInAllTag(this IAnmAttribute attr)
        {
            bool bRet = false;
            AnmAttribute attribute = attr as AnmAttribute;

            bRet = attribute.HasAnyKey;
            if (!bRet)
            {
                // サブアトリビュートについて、キーを持つか調べる
                for (int i = 0; i < attr.NumSubAttribute; i++)
                {
                    IAnmAttribute subAttr = attr.FindSubAttributeByIdx(i);
                    bRet |= subAttr.CheckHaveKeyRecursiveInAllTag();
                }
            }

            return bRet;
        }

        /// <summary>
        /// 再帰的にキーを持つか調査する。
        /// </summary>
        public static bool CheckHaveKeyRecursiveInTargetTag(this IAnmAttribute attr, int startFrame, int endFrame, string tagName, bool isSeparateMode)
        {
            bool bRet = false;

            // 自身がキーを持つか調べる
            if (attr.HasKeyInRange(startFrame, endFrame, isSeparateMode ? tagName: AnimCurvesParam.__NoSelected__))
            {
                bRet = true;
            }

            if (!bRet)
            {
                // サブアトリビュートについて、キーを持つか調べる
                for (int i = 0; i < attr.NumSubAttribute; i++)
                {
                    IAnmAttribute subAttr = attr.FindSubAttributeByIdx(i);
                    bRet |= subAttr.CheckHaveKeyRecursiveInTargetTag(startFrame, endFrame, tagName, isSeparateMode);
                }
            }

            return bRet;
        }

        /// <summary>
        /// 開始、終了フレームを計算します。最低一つはキーが存在することを想定しています。
        /// </summary>
        public static bool GetFrameRangeRecursive(this IAnmAttribute attr, ref float startFrame, ref float endFrame)
        {
            if(attr.ICurrentAnimationCurve != null)
            {
                IAnmKeyFrame firstKey = attr.ICurrentAnimationCurve.GetFirstKey();
                if(firstKey != null)
                {
                    startFrame = Math.Min(firstKey.Time, startFrame);
                    endFrame = Math.Max(firstKey.Time, endFrame);
                }

                IAnmKeyFrame lastKey = attr.ICurrentAnimationCurve.GetLastKey();
                if(lastKey != null)
                {
                    startFrame = Math.Min(lastKey.Time, startFrame);
                    endFrame = Math.Max(lastKey.Time, endFrame);
                }

                return firstKey != null || lastKey != null;
            }

            // サブアトリビュートについて、キーを持つか調べる
            bool result = false;
            for (int i = 0; i < attr.NumSubAttribute; i++)
            {
                IAnmAttribute subAttr = attr.FindSubAttributeByIdx(i);
                result |= subAttr.GetFrameRangeRecursive(ref startFrame, ref endFrame);
            }

            return result;
        }

        static public void RemoveKeyFrame(IAnmAttribute attr)
        {
            AnmCurve curve = attr.ICurrentAnimationCurve as AnmCurve;
            if (curve != null)
            {
                curve.ClearKeyFrames();
            }
        }

        static public void RemoveKeyFrameRecursive(IAnmAttribute attr, Func<IAnmAttribute, bool> condition)
        {
            if(condition(attr))
            {
                RemoveKeyFrame(attr);
            }

            for (int i = 0; i < attr.NumSubAttribute; i++)
            {
                RemoveKeyFrameRecursive(attr.FindSubAttributeByIdx(i), condition);
            }
        }

        /// <summary>
        /// 階層以下のActiveを再帰的に設定
        /// </summary>
        internal static void SetActiveRecursive(AnmAttribute attribute, bool flag)
        {
            attribute.IsActiveAttribute = flag;
            for (int i = 0; i < attribute.NumSubAttribute; i++)
            {
                SetActiveRecursive(attribute.FindAttributeByIdx(i), flag);
            }
        }

        /// <summary>
        /// 読み取り専用か
        /// </summary>
        public static bool IsReadOnlyLocked(this IAnmAttribute attribute)
        {
            return attribute != null && attribute.ICurrentAnimationCurve != null && attribute.ICurrentAnimationCurve.IsReadOnlyLocked;
        }

        /// <summary>
        /// 再帰的にキーを持つか調査する。
        /// </summary>
        public static IEnumerable<IAnmAttribute> EnumCurveAttribute(this IAnmAttribute attr)
        {
            // 自分自身がリーフか？
            if (attr.ICurrentAnimationCurve != null)
            {
                yield return attr;
            }

            for (int i = 0; i < attr.NumSubAttribute; i++)
            {
                IAnmAttribute subAttr = attr.FindSubAttributeByIdx(i);

                if (subAttr.ICurrentAnimationCurve != null)
                {
                    yield return subAttr;
                }

                if (subAttr.HasSubAttribute)
                {
                    foreach (IAnmAttribute leaf in EnumCurveAttribute(subAttr))
                    {
                        yield return leaf;
                    }
                }
            }
        }

        /// <summary>
        /// 指定したタグ名に紐付くアニメーションカーブを取得します。
        /// </summary>
        public static IAnmCurve GetAnmCurveRelationTag(this IAnmAttribute attr, string tagName)
        {
            AnmAttribute target = attr as AnmAttribute;
            return target.GetAnmCurve(tagName);
        }

        /// <summary>
        /// アトリビュートが同じ数のカーブを持っているか
        /// </summary>
        public static bool IsSameCurveCount(this IAnmAttribute attr0, IAnmAttribute attr1)
        {
            Debug.Assert(attr0 != null);
            Debug.Assert(attr1 != null);

            if (attr0.Curves == null)
            {
                if (attr1.Curves != null)
                {
                    return false;
                }
                else
                {
                    return true;
                }
            }
            else
            {
                if (attr1.Curves == null)
                {
                    return false;
                }
            }

            return attr0.Curves.Length == attr1.Curves.Length;
        }

        /// <summary>
        /// 現在値が保持される情報かどうか判定します
        /// </summary>
        public static AnimationStatusKind GetMarkColor(this IAnmAttribute attr, bool isSeparateMode)
        {
            object currentValue;
            object baseValue;

            // キーを持っていない
            if (!attr.HasKey())
            {
                // 分割モードの場合
                if (isSeparateMode)
                {
                    // baseValueと現在値が異なる
                    currentValue = GetValue_(attr);
                    baseValue = GetBaseValue_(attr);
                    if (!ObjectToFloatValue_(currentValue).Equals(ObjectToFloatValue_(baseValue)))
                    {
                        return AnimationStatusKind.DiffValue;
                    }
                }

                return AnimationStatusKind.None;
            }

            // キーを持っているが現在時間にキーが無い
            IAnmKeyFrame keyFrame = attr.ICurrentAnimationCurve.FindKeyExistAt(GlobalTime.Inst.Time);
            if (keyFrame == null)
            {
                return AnimationStatusKind.HasKey;
            }

            // キーを持っていて現在時間にキーがある
            // キーの値と現在値を比較
            currentValue = GetValue_(attr);
            if (currentValue == null)
            {
                Debug.Assert(false);
                return AnimationStatusKind.None;
            }

            // 同じ
            if (keyFrame.ValueAsFloat.Equals(ObjectToFloatValue_(currentValue)))
            {
                return AnimationStatusKind.SameKey;
            }

            // 違う
            return AnimationStatusKind.DiffKey;
        }

        public static AnimationStatusKind GetMarkColorRecursive(this IAnmAttribute attr, bool isSeparateMode)
        {
            AnimationStatusKind retColor = AnimationStatusKind.None;

            if (attr.NumSubAttribute == 0)
            {
                return GetMarkColor(attr, isSeparateMode);
            }

            List<AnimationStatusKind> colors = new List<AnimationStatusKind>();
            for (int i = 0; i < attr.NumSubAttribute; i++)
            {
                colors.Add(GetMarkColorRecursive(attr.FindSubAttributeByIdx(i), isSeparateMode));
            }

            int numSameKey = colors.Where(type => type == AnimationStatusKind.SameKey).Count();
            int numNone = colors.Where(type => type == AnimationStatusKind.None).Count();
            int numHasKeyk = colors.Where(type => type == AnimationStatusKind.HasKey).Count();
            int numDiffKey = colors.Where(type => type == AnimationStatusKind.DiffKey).Count();
            int numDiffValue = colors.Where(type => type == AnimationStatusKind.DiffValue).Count();

            if (numDiffKey > 0)
            {
                retColor = AnimationStatusKind.DiffKey;
            }
            else if (numDiffValue > 0)
            {
                retColor = AnimationStatusKind.DiffValue;
            }
            else if (numSameKey > 0)
            {
                retColor = AnimationStatusKind.SameKey;
            }
            else if (numHasKeyk > 0)
            {
                retColor = AnimationStatusKind.HasKey;
            }
            else
            {
                retColor = AnimationStatusKind.None;
            }

            return retColor;
        }

        private static object GetValue_(IAnmAttribute attr)
        {
            AnmAttribute node = attr as AnmAttribute;
            object value;

            (attr as AnmAttribute).GetValue(out value);
            if (value is float)
            {
                // FloatColor対応
                value = (float)value * attr.ICurrentAnimationCurve.ViewScale;
            }

            return value;
        }

        private static object GetBaseValue_(IAnmAttribute attr)
        {
            AnmAttribute node = attr as AnmAttribute;
            if (node == null)
            {
                return null;
            }

            object value;
            node.GetBaseValue(out value);

            if (value is float)
            {
                // FloatColor対応
                value = (float)value * attr.ICurrentAnimationCurve.ViewScale;
            }

            return value;
        }

        private static float ObjectToFloatValue_(object srcVal)
        {
            return Convert.ToSingle(srcVal);
        }
    }

    /// <summary>
    /// キーフレームヘルパー
    /// </summary>
    public static class AnmKeyFrameHelpser
    {
        /// <summary>
        /// キーが一致するかどうか
        /// </summary>
        public static bool IsSame(this IAnmKeyFrame self, IAnmKeyFrame other)
        {
            return
                self.InTangent == other.InTangent &&
                self.OutTangent == other.OutTangent &&
                self.InterporationType == other.InterporationType &&
                self.IsSelected == other.IsSelected &&
                self.TangentModifyEnabled == other.TangentModifyEnabled &&
                self.Time == other.Time &&
                self.TimeAsInt == other.TimeAsInt &&
                self.UnifyTangents == other.UnifyTangents &&
                self.ValueAsFloat == other.ValueAsFloat &&
                self.ValueAsFloatNoEffect == other.ValueAsFloatNoEffect;

        }
    }

    /// <summary>
    /// 編集用UIが参照するインタフェース
    ///
    /// Undo/Redoが適用されるメソッドと、適用されないメソッドが存在します。
    /// 適切に使い分けてください。
    /// </summary>
    public interface IAnmKeyFrame
    {
        /// <summary>
        /// キーの値にフロート値でアクセス可能
        /// </summary>
        float ValueAsFloat
        {
            get;
        }

        /// <summary>
        /// キーの値にフロート値でアクセス可能(データを加工せずにそのまま取得)
        /// </summary>
        float ValueAsFloatNoEffect
        {
            get;
        }


        /// <summary>
        /// 時間の設定、取得
        /// </summary>
        float Time
        {
            get;
        }

        /// <summary>
        /// 時間の設定、取得
        /// </summary>
        int TimeAsInt
        {
            get;
        }

        /// <summary>
        /// 左接線の取得、設定
        /// </summary>
        float InTangent
        {
            get;
        }

        /// <summary>
        /// 右接線の取得、設定
        /// </summary>
        float OutTangent
        {
            get;
        }

        /// <summary>
        /// 補間種類
        /// </summary>
        InterporationType InterporationType
        {
            get;
        }

        /// <summary>
        /// 自分を所有しているアニメーションカーブ
        /// </summary>
        IAnmCurve OwnerIAnmCurve
        {
            get;
        }

        /// <summary>
        /// 選択されているか判定します。
        /// </summary>
        bool IsSelected
        {
            get;
        }

        /// <summary>
        /// In Out Tangentを同期編集するか取得します。
        /// </summary>
        bool         UnifyTangents
        {
            get;
        }

        /// <summary>
        /// スロープ変更が可能か？
        /// </summary>
        bool TangentModifyEnabled
        {
            get;
        }
    }

    /// <summary>
    /// アニメーションカーブ外部公開用インタフェース
    ///
    /// </summary>
    public interface IAnmCurve
    {
        // -------------- アクセサ類
        AttributeType           TargetAttrType   {get;} // 不要になった？

        string                  AttrName         {get;} // 不要になった？
        int                     NumKeyFrame      {get;}

        float FrameOfHead { get; }
        float FrameOfTail { get; }
        float ValueOfHead { get; }
        float ValueOfTail { get; }

        bool IsParameterizedAnim { get; }

        LECore.Structures.SerializableObject.Lan.ParameterizedAnimParameter[] ParameterizedAnimParameters { get; }

        IAnmKeyFrame[]  IKeyFrameSet     {get;}

        bool HasCustomTimeSource { get; }
        int CustomTimeFrame { get; }

        AnmCurveInfinityType PreInfinityType { get;}
        AnmCurveInfinityType PostInfinityType { get;}

        // 選択キーセットを取得します。
        IAnmKeyFrame[]  SelectedKeySet   {get;}
        int              NumSelectedKey { get;}
        bool             IsSelectedKey( IAnmKeyFrame key );

        // 読み取り専用かどうか
        bool IsReadOnlyLocked { get; }

        // カーブの見かけ上のスケール
        float ViewScale { get; set; }

        // 指定キーフレームの前後のキーフレームを取得します。
        IAnmKeyFrame    GetPrevKeyFrame( IAnmKeyFrame key );
        IAnmKeyFrame    GetPrevKeyFrame( float time );
        IAnmKeyFrame    GetNextKeyFrame( IAnmKeyFrame key );
        IAnmKeyFrame    GetNextKeyFrame( float time );

        // 指定時間のキーを取得します。
        IAnmKeyFrame FindKeyExistAt( float time );

        bool                    Contains( IAnmKeyFrame keyFrame );

        // -------------- アニメーション結果の評価(アニメーション結果は常に float 型で出力されます。)
        float                   Evaluate( float time );
    }
}
