﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;


namespace LECore.Structures
{
    //-------------------------------------------------
    #region IAnimShareInfomation

    /// <summary>
    ///
    /// </summary>
    public interface IAnimShareInfomation
    {
        /// <summary>
        /// コメント文字列
        /// </summary>
        string Comment { get; }
        /// <summary>
        /// アニメーション共有元ペイン名前
        /// </summary>
        string SrcPaneName { get; }
        /// <summary>
        /// 対象グループ名
        /// </summary>
        string TargetGroupName { get; }

        bool Valid { get; }
    }

    /// <summary>
    ///
    /// </summary>
    internal class AnimShareInfomation : IAnimShareInfomation
    {
        //----------------------------------------------------------------
        #region フィールド

        string _commnet = string.Empty;
        IPane _srcPane = null;
        ILEGroup _targetGroup = null;

        string _invalidPaneName;
        string _invalidGroupName;

        #endregion フィールド

        //----------------------------------------------------------------
        #region IAnimShareInfomation メンバ

        /// <summary>
        /// コメント文字列
        /// </summary>
        public string Comment
        {
            get { return _commnet; }
        }

        /// <summary>
        /// アニメーション共有元ペイン名前
        /// </summary>
        public string SrcPaneName
        {
            get
            {
                return ( _srcPane != null ) ? _srcPane.PaneName : _invalidPaneName;
            }
        }

        /// <summary>
        /// 対象グループ名
        /// </summary>
        public string TargetGroupName
        {
            get
            {
                return ( _targetGroup != null ) ? _targetGroup.GrouprName : _invalidGroupName;
            }
        }

        /// <summary>
        /// 有効か？
        /// </summary>
        public bool Valid
        {
            get { return _srcPane != null && _targetGroup != null; }
        }

        /// <summary>
        /// 情報を設定します。
        /// </summary>
        public void Set( IPane srcPane, ILEGroup targetGroup, string comment )
        {
            // srcPane == null targetGroup == null の場合も、
            // 有効ではないデータとして、正しく処理する必要があります。

            _srcPane = srcPane;
            _targetGroup = targetGroup;
            _commnet = comment;

            _invalidPaneName = string.Empty;
            _invalidGroupName = string.Empty;
        }

        /// <summary>
        /// 無効な情報として設定します。
        /// </summary>
        public void MarkAsInvalid( string paneName, string groupName )
        {
            _srcPane = null;
            _targetGroup = null;

            _invalidPaneName = paneName;
            _invalidGroupName = groupName;
        }

        #endregion
    }

    #endregion IAnimShareInfomation

    //-------------------------------------------------
    /// <summary>
    /// アニメーションフレーム区間セット
    /// 公開インタフェース
    /// </summary>
    public interface IAnimShareInfomationChunk
    {
        /// <summary>
        /// アニメーション共有情報セット
        /// </summary>
        IAnimShareInfomation[] IAnimShareInfomationSet { get; }
    }

    /// <summary>
    ///
    /// </summary>
    public class AnimShareInfomationChunkManupulator
    {
        #region フィールド
        AnimShareInfomationChunk _target = null;
        #endregion フィールド

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnimShareInfomationChunkManupulator() {}

        /// <summary>
        /// 対象を設定します。
        /// </summary>
        public void BindTarget( IAnimShareInfomationChunk target )
        {
            Debug.Assert( target != null );
            _target = target as AnimShareInfomationChunk;
        }

        /// <summary>
        /// 登録
        /// </summary>
        public IAnimShareInfomation Register( IPane srcPane, ILEGroup targetGroup, string comment )
        {
            return _target.Register( srcPane, targetGroup, comment );
        }

        /// <summary>
        /// 変更
        /// </summary>
        public void Modify( IAnimShareInfomation animShareInfo, IPane srcPane, ILEGroup targetGroup, string comment )
        {
            _target.Modify( animShareInfo, srcPane, targetGroup, comment );
        }

        /// <summary>
        /// 削除
        /// </summary>
        public void Remove( IAnimShareInfomation animShareInfomation )
        {
            _target.Remove( animShareInfomation );
        }

        /// <summary>
        /// 有効かどうか判定し、状態を更新します。
        /// 新たに不正な情報が見つかった場合は true を返します。
        /// </summary>
        public bool UpdateValidityState( ISubScene subScene )
        {
            bool bFonundInvalid = false;
            foreach( IAnimShareInfomation asi in _target.IAnimShareInfomationSet )
            {
                IPane pane = SubSceneHelper.FindPaneByName( subScene, asi.SrcPaneName );
                ILEGroup group = subScene.ILEGroupMgr.FindGroupByName( asi.TargetGroupName );

                // 正常な情報が本当に、正常か確認します。
                if( asi.Valid )
                {
                    if( pane == null || group == null )
                    {
                        // 新たに不正な情報に設定する。
                        _target.MarkAsInvalid( asi, asi.SrcPaneName, asi.TargetGroupName );
                        bFonundInvalid |= true;
                    }
                }
                else
                {
                    if( pane != null && group != null )
                    {
                        // 新たに不正な情報に設定する。
                        _target.Modify( asi, pane, group, asi.Comment );
                        bFonundInvalid |= true;
                    }
                }
            }
            return bFonundInvalid;
        }
    }

    internal delegate void OnAnimShareManagerUpdateHandler();

    /// <summary>
    ///
    /// </summary>
    internal class AnimShareInfomationChunk : IAnimShareInfomationChunk
    {
        //----------------------------------------------------------------
        #region フィールド
        List<IAnimShareInfomation> _animShareInfomationSet = new List<IAnimShareInfomation>();
        #endregion フィールド

        // 変更イベント
        internal event OnAnimShareManagerUpdateHandler OnAnimShareManagerChanged;

        //----------------------------------------------------------------
        #region IAnimShareInfomationSet メンバ

        public IAnimShareInfomation[] IAnimShareInfomationSet
        {
            get { return _animShareInfomationSet.ToArray(); }
        }
        #endregion

        //----------------------------------------------------------------
        #region 登録・削除
        /// <summary>
        /// 登録
        /// </summary>
        public IAnimShareInfomation Register( IPane srcPane, ILEGroup targetGroup, string comment )
        {
            AnimShareInfomation asf = new AnimShareInfomation();
            asf.Set( srcPane, targetGroup, comment );

            _animShareInfomationSet.Add( asf );

            NotifyChanged_();
            return asf;
        }

        /// <summary>
        /// 削除
        /// </summary>
        public void Remove( IAnimShareInfomation animShareInfomation )
        {
            Debug.Assert( _animShareInfomationSet.Contains( animShareInfomation ) );
            _animShareInfomationSet.Remove( animShareInfomation );
            NotifyChanged_();
        }

        /// <summary>
        /// 更新します。
        /// </summary>
        public void Modify( IAnimShareInfomation animShareInfomation, IPane srcPane, ILEGroup targetGroup, string comment )
        {
            Debug.Assert( _animShareInfomationSet.Contains( animShareInfomation ) );

            if( !animShareInfomation.Valid ||
                animShareInfomation.SrcPaneName != srcPane.PaneName ||
                animShareInfomation.TargetGroupName != targetGroup.GrouprName ||
                animShareInfomation.Comment != comment )
            {
                ( animShareInfomation as AnimShareInfomation ).Set( srcPane, targetGroup, comment );
                NotifyChanged_();
            }
        }

        /// <summary>
        /// 不正なデータに設定します。
        /// </summary>
        public void MarkAsInvalid( IAnimShareInfomation animShareInfomation, string paneName, string groupName )
        {
            Debug.Assert( _animShareInfomationSet.Contains( animShareInfomation ) );

            ( animShareInfomation as AnimShareInfomation ).MarkAsInvalid( paneName, groupName );
            NotifyChanged_();
        }

        /// <summary>
        /// 変更を通知します。
        /// </summary>
        void NotifyChanged_()
        {
            if( OnAnimShareManagerChanged != null )
            {
                OnAnimShareManagerChanged();
            }
        }

        #endregion


    }

    /// <summary>
    /// アニメーションフレーム区間セット
    /// 公開インタフェース
    /// <remarks>
    /// 派生部品レイアウト関連で必要となった、ReadOnly属性の対応は行っていません。
    /// </remarks>
    /// </summary>
    public interface IAnimShareInfomationManager
    {
        IEnumerable<IAnimShareInfomationChunk> IAnimShareInfomationChunkSet { get; }
    }

    /// <summary>
    /// アニメーション共有情報マネージャ
    /// </summary>
    internal class AnimShareInfomationManager : IAnimShareInfomationManager
    {
        #region フィールド
        /// <summary>
        /// AnimShareInfomationChunk 配列
        /// 将来の拡張のため、複数のAnimShareInfomationChunkが持てるように
        /// していますが現状は、必ず1つのみ登録するようになっています。
        ///
        /// 登録関数も用意されていません。
        /// </summary>
        List<AnimShareInfomationChunk> _animShareInfomationChunkSet = new List<AnimShareInfomationChunk>();
        #endregion フィールド

        #region イベント
        // 変更イベント
        internal event OnAnimShareManagerUpdateHandler OnAnimShareManagerChanged;
        #endregion イベント

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnimShareInfomationManager()
        {
            // 唯一のAnimShareInfomationChunkを登録します。
            AnimShareInfomationChunk chunk = new AnimShareInfomationChunk();
            chunk.OnAnimShareManagerChanged += chunk_OnAnimShareManagerChanged_;
            _animShareInfomationChunkSet.Add( chunk );
        }

        /// <summary>
        /// AnimShareInfomationChunk変更イベント
        /// </summary>
        void chunk_OnAnimShareManagerChanged_()
        {
            if( OnAnimShareManagerChanged != null )
            {
                // サブシーンなどに変更を通知します。
                OnAnimShareManagerChanged();
            }
        }

        #region IAnimShareInfomationManager メンバ
        /// <summary>
        ///
        /// </summary>
        public IEnumerable<IAnimShareInfomationChunk> IAnimShareInfomationChunkSet
        {
            get { return _animShareInfomationChunkSet.ConvertAll((x) => x as IAnimShareInfomationChunk); }
        }
        #endregion


    }
}
