﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Drawing;

namespace LECore.Structures
{
    using System.Collections.Generic;
    using Core;
    using LECoreInterface;
    using System.Linq;

    /// <summary>
    /// 領域ペインです。
    /// </summary>
    internal class Alignment :
        LEDataNode,
        IAlignment,
        IPaneExParamater,
        IDrawable,
        IDisposable
    {
        public const string AlignmentNodeName = "Alignment";

        private HorizontalLocation _horizontal;
        private VerticalLocation _vertical;
        private float _defaultHorizontalMargin;
        private float _defaultVerticalMargin;
        private bool _isExtendEdgeHorizontalEnabled;
        private bool _isExtendEdgeVerticalEnabled;

        private bool _isAlignmentHorizontalEnabled;
        private bool _isAlignmentVerticalEnabled;

        // ペイン本体への参照
        readonly Pane   _ownerPane = null;

        void NotifyChangeToScene_()
        {
            NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Alignment( Pane ownerPane )
            :base(ownerPane, AlignmentNodeName)
        {
            Debug.Assert( ownerPane != null );
            _ownerPane = ownerPane;
            _ownerPane.BindPaneExData( this );

            // 整列ペインの初期値設定
            this.IsAlignmentHorizontalEnabled = true;
        }

        /// <summary>
        /// リソースを開放します。
        /// </summary>
        public void Dispose()
        {
        }

        #region IAlignment
        public HorizontalLocation Horizontal
        {
            get { return _horizontal; }
            set { if (_horizontal != value) { _horizontal = value; NotifyChangeToScene_(); } }
        }

        public float DefaultHorizontalMargin
        {
            get { return _defaultHorizontalMargin; }
            set { if (_defaultHorizontalMargin != value) { _defaultHorizontalMargin = value; NotifyChangeToScene_(); } }
        }

        public bool IsExtendEdgeHorizontalEnabled
        {
            get { return _isExtendEdgeHorizontalEnabled; }
            set { if (_isExtendEdgeHorizontalEnabled != value) { _isExtendEdgeHorizontalEnabled = value; NotifyChangeToScene_(); } }
        }

        public VerticalLocation Vertical
        {
            get { return _vertical; }
            set { if (_vertical != value) { _vertical = value; NotifyChangeToScene_(); } }
        }

        public float DefaultVerticalMargin
        {
            get { return _defaultVerticalMargin; }
            set { if (_defaultVerticalMargin != value) { _defaultVerticalMargin = value; NotifyChangeToScene_(); } }
        }

        public bool IsExtendEdgeVerticalEnabled
        {
            get { return _isExtendEdgeVerticalEnabled; }
            set { if (_isExtendEdgeVerticalEnabled != value) { _isExtendEdgeVerticalEnabled = value; NotifyChangeToScene_(); } }
        }

        public bool IsAlignmentHorizontalEnabled
        {
            get { return _isAlignmentHorizontalEnabled; }
            set { if (_isAlignmentHorizontalEnabled != value) { _isAlignmentHorizontalEnabled = value; NotifyChangeToScene_(); } }
        }

        public bool IsAlignmentVerticalEnabled
        {
            get { return _isAlignmentVerticalEnabled; }
            set { if (_isAlignmentVerticalEnabled != value) { _isAlignmentVerticalEnabled = value; NotifyChangeToScene_(); } }
        }

        #endregion

        #region IPaneExParamater
        /// <summary>
        /// 自身を持っているペインの参照です。
        /// </summary>
        public IPane OwnerPane
        {
            get{ return _ownerPane;}
        }

        /// <summary>
        /// 自身を持っているペインの参照です。
        /// </summary>
        public void          UpdateMatarialName   ()
        {
            // Do Nothing
        }

        /// <summary>
        /// シーン登録時の初期化処理
        /// </summary>
        public void OnJoinSceneInitialize()
        {
            // 何もしません。
        }

        /// <summary>
        /// 編集対象に設定される直前の初期化処理
        /// </summary>
        public void FirstTimeInitialize()
        {
            // 何もしません。
        }

        /// <summary>
        /// UV座標を列挙します。
        /// </summary>
        public IEnumerable<TexCoord4> TexCoords
        {
            get { yield break; }
        }

        /// <summary>
        /// 警告文字列を取得します。UIの各所で表示します。
        /// </summary>
        public string WarningMsg
        {
            get { return string.Empty; }
        }

        /// <summary>
        /// 境界を書くかどうか
        /// </summary>
        public bool PaneBoundDrawEnabled
        {
            get { return true; }
        }
        #endregion IPaneExParamater

        #region IDrawable
        public void Draw( IRenderer renderer, DrawableOption option )
        {
            System.Drawing.Font sysFont = LayoutEditorCore.GuiFont;
            renderer.DbgDrawString("Alignment_Pane", sysFont, 0, 0 );

            Color  oldColor = renderer.Color;

            renderer.Color = option.SystemOpeBlue;
            renderer.FillRectangle(0, 0, 0, OwnerPane.RenderingWidth, OwnerPane.RenderingHeight);
            renderer.Color = oldColor;
        }
        #endregion IDrawable

        #region 整列処理


        private class AlignInfo
        {
            public IPane Pane { get; set; }
            public float Length { get; set; }
            public float Margin { get; set; }
            public float CenterOffset { get; set; }
        }

        internal struct ModifyAction
        {
            internal Action<IPane, FVec3> transModifyAction;
            internal Action<IPane, float> widthModifyAction;
            internal Action<IPane, float> heightModifyAction;
        }

        static private void MesurePartsHorizontalBoundFromRoot_(IPane rootPane, float transX, ref float left, ref float right)
        {
            foreach(IPane childrend in rootPane.Children)
            {
                MesurePartsHorizontalBound_(childrend, 0.0f, ref left, ref right);
            }
        }

        static private void MesurePartsVerticalBoundFromRoot_(IPane rootPane, float transY, ref float top, ref float bottom)
        {
            foreach(IPane childrend in rootPane.Children)
            {
                MesurePartsVerticalBound_(childrend, 0.0f, ref top, ref bottom);
            }
        }

        static private void MesurePartsHorizontalBound_(IPane pane, float transX, ref float left, ref float right)
        {
            // 非表示なら、サイズ計算の対象にしません。透明度ゼロは、計算対象になる点に注意してください。
            if (pane == null || !pane.Visible)
            {
                // これ以上処理が必要なペイン
                return;
            }

            if (!pane.IsAlignmentIgnore)
            {
                float pane_width = pane.Width;

                // Translateの計算
                transX += pane.X;

                // 範囲を拡張する
                float min = float.MaxValue;
                float max = float.MinValue;

                if (pane.PaneKind == PaneKind.Textbox)
                {
                    ITextBox textBox = pane.ITextBox;

                    float width = (textBox as TextBox).CalcStringSize().X;
                    if (textBox.BasePosTypeH == HorizontalLocation.Right)
                    {
                        // テキストが右揃え
                        max = transX + pane_width / 2.0f;
                        min = max - width;
                    }
                    else if (textBox.BasePosTypeH == HorizontalLocation.Left)
                    {
                        // テキストが左揃え
                        min = transX - pane_width / 2.0f;
                        max = min + width;
                    }
                    else
                    {
                        // テキストが中央揃え
                        min = transX - width / 2.0f;
                        max = transX + width / 2.0f;
                    }
                }
                else if (pane.PaneKind == PaneKind.Picture || pane.PaneKind == PaneKind.Window)
                {
                    min = transX - (pane_width / 2.0f * Math.Abs(pane.Scale.X));
                    max = transX + (pane_width / 2.0f * Math.Abs(pane.Scale.X));
                }

                if (min < max)
                {
                    // ペインの基準位置を反映する
                    if (pane.BasePosTypeH == HorizontalLocation.Left)
                    {
                        min += pane_width / 2.0f;
                        max += pane_width / 2.0f;
                    }
                    else if (pane.BasePosTypeH == HorizontalLocation.Right)
                    {
                        min -= pane_width / 2.0f;
                        max -= pane_width / 2.0f;
                    }
                }

                left = Math.Min(left, min);
                right = Math.Max(right, max);
            }

            // 再帰的に呼び出し
            foreach(IPane childPane in pane.Children)
            {
                MesurePartsHorizontalBound_(childPane, transX, ref left, ref right);
            }
        }

        static private void MesurePartsVerticalBound_(IPane pane, float transY, ref float top, ref float bottom)
        {
            // 非表示なら、サイズ計算の対象にしません。透明度ゼロは、計算対象になる点に注意してください。
            if (pane == null || !pane.Visible)
            {
                // これ以上処理が必要なペイン
                return;
            }

            if (!pane.IsAlignmentIgnore)
            {
                float pane_height = pane.Height;

                // Translateの計算
                transY += pane.Y;

                // 範囲を拡張する
                float min = float.MaxValue;
                float max = float.MinValue;

                if (pane.PaneKind == PaneKind.Textbox)
                {
                    ITextBox textBox = pane.ITextBox;

                    float height = (textBox as TextBox).CalcStringSize().Y;
                    if (textBox.BasePosTypeV == VerticalLocation.Top)
                    {
                        // テキストが上揃え
                        max = transY + pane_height / 2.0f;
                        min = max - height;
                    }
                    else if (textBox.BasePosTypeV == VerticalLocation.Bottom)
                    {
                        // テキストが下揃え
                        min = transY - pane_height / 2.0f;
                        max = min + height;
                    }
                    else
                    {
                        // テキストが中央揃え
                        min = transY - height / 2.0f;
                        max = transY + height / 2.0f;
                    }
                }
                else if (pane.PaneKind == PaneKind.Picture || pane.PaneKind == PaneKind.Window)
                {
                    min = transY - (pane_height / 2.0f * Math.Abs(pane.Scale.Y));
                    max = transY + (pane_height / 2.0f * Math.Abs(pane.Scale.Y));
                }

                if (min < max)
                {
                    // ペインの基準位置を反映する
                    if (pane.BasePosTypeV == VerticalLocation.Bottom)
                    {
                        min += pane_height / 2.0f;
                        max += pane_height / 2.0f;
                    }
                    else if (pane.BasePosTypeV == VerticalLocation.Top)
                    {
                        min -= pane_height / 2.0f;
                        max -= pane_height / 2.0f;
                    }
                }

                top = Math.Max(top, max);
                bottom = Math.Min(bottom, min);
            }

            // 再帰的に呼び出し
            foreach(IPane childPane in pane.Children)
            {
                MesurePartsVerticalBound_(childPane, transY, ref top, ref bottom);
            }
        }

        static private bool MakeHorizontalAlignInfo_(IPane childPane, AlignInfo info)
        {
            if (childPane.IsAlignmentIgnore)
            {
                return false;
            }

            // ペインの種類によって処理が変わる
            // 部品の場合は計算
            // null の場合は配下を計算
            // 計算の結果、幅がゼロなら 非表示状態とみなす
            var parentAlignment = (childPane.Parent as IPane).IAlignment;
            Debug.Assert(parentAlignment != null);

            info.Pane = childPane;
            info.Length = childPane.Width;
            info.Margin = 0.0f;
            info.CenterOffset = 0.0f;

            // 注意：透明度ゼロ（childPane.Transparency == 0）の場合は、幅計算の対象となります。
            bool isVisible = childPane.Visible;

            if (childPane.PaneKind == PaneKind.Parts)
            {
                // 部品のサイズは内部を解析する
                float left = float.MaxValue;
                float right = float.MinValue;
                MesurePartsHorizontalBoundFromRoot_(childPane?.IPartsLayout?.PartsSubScene?.RootIPane, 0.0f, ref left, ref right);

                info.Length = right - left;
                info.CenterOffset = (right + left) / 2.0f;
                if (info.Length == 0.0f)
                {
                    isVisible = false;
                }
            }
            else if(childPane.PaneKind == PaneKind.Null && childPane.HasChildren)
            {
                // 子供を持つ、Null ペインは（部品ペインのように）配下のペインのサイズ総和をサイズとする
                float left = float.MaxValue;
                float right = float.MinValue;
                MesurePartsHorizontalBoundFromRoot_(childPane, 0.0f, ref left, ref right);

                info.Length = right - left;
                info.CenterOffset = (right + left) / 2.0f;
                if (info.Length == 0.0f)
                {
                    isVisible = false;
                }
            }
            else
            {
                if (childPane.PaneKind == PaneKind.Textbox)
                {
                    // CenterOffset
                    {
                        float paneWidth = childPane.Width;
                        float calcWidth = (childPane.ITextBox as TextBox).CalcStringSize().X;
                        info.Length = calcWidth;

                        float diffTextW = -(paneWidth / 2.0f) + (calcWidth / 2.0f);
                        if (childPane.ITextBox.BasePosTypeH == HorizontalLocation.Right)
                        {
                            info.CenterOffset = -diffTextW;
                        }
                        else if (childPane.ITextBox.BasePosTypeH == HorizontalLocation.Left)
                        {
                            info.CenterOffset = diffTextW;
                        }
                    }
                }
                else
                {
                    info.Length = childPane.Width * Math.Abs(childPane.Scale.X);
                }

                // CenterOffset
                {
                    // ペインの基準位置を反映する
                    if (childPane.BasePosTypeH == HorizontalLocation.Left)
                    {
                        info.CenterOffset += childPane.Width / 2.0f;
                    }
                    else if (childPane.BasePosTypeH == HorizontalLocation.Right)
                    {
                        info.CenterOffset -= childPane.Width / 2.0f;
                    }
                }
            }

            // info.Margin
            {
                // 表示されている場合は、マージンを設定します。
                if (isVisible)
                {
                    if(info.Pane.IsAlignmentHorizontalMarginEnabled)
                    {
                        info.Margin = info.Pane.AlignmentHorizontalMargin;
                    }
                    else
                    {
                        info.Margin = parentAlignment.DefaultHorizontalMargin;
                    }
                }
                else
                {
                    info.Margin = 0.0f;
                }
            }

            return isVisible;
        }

        static private bool MakeVerticalAlignInfo_(IPane childPane, AlignInfo info)
        {
            if (childPane.IsAlignmentIgnore)
            {
                return false;
            }

            // ペインの種類によって処理が変わる
            // 部品の場合は計算
            // null の場合は配下を計算
            // 計算の結果、幅がゼロなら 非表示状態とみなす
            var parentAlignment = (childPane.Parent as IPane).IAlignment;
            Debug.Assert(parentAlignment != null);

            info.Pane = childPane;
            info.Length = childPane.Height;
            info.Margin = 0.0f;
            info.CenterOffset = 0.0f;

            // 注意：透明度ゼロ（childPane.Transparency == 0）の場合は、幅計算の対象となります。
            bool isVisible = childPane.Visible;

            if (childPane.PaneKind == PaneKind.Parts)
            {
                // 部品のサイズは内部を解析する
                float top = float.MinValue;
                float bottom = float.MaxValue;
                MesurePartsVerticalBoundFromRoot_(childPane?.IPartsLayout?.PartsSubScene?.RootIPane, 0.0f, ref top, ref bottom);

                info.Length = top - bottom;
                info.CenterOffset = (top + bottom) / 2.0f;
                if (info.Length == 0.0f)
                {
                    isVisible = false;
                }
            }
            else if(childPane.PaneKind == PaneKind.Null && childPane.HasChildren)
            {
                // 子供を持つ、Null ペインは（部品ペインのように）配下のペインのサイズ総和をサイズとする
                float top = float.MinValue;
                float bottom = float.MaxValue;
                MesurePartsVerticalBoundFromRoot_(childPane, 0.0f, ref top, ref bottom);

                info.Length = top - bottom;
                info.CenterOffset = (top + bottom) / 2.0f;
                if (info.Length == 0.0f)
                {
                    isVisible = false;
                }
            }
            else
            {
                if (childPane.PaneKind == PaneKind.Textbox)
                {
                    // CenterOffset
                    {
                        float paneHeight = childPane.Height;
                        float calcHeight = (childPane.ITextBox as TextBox).CalcStringSize().Y;
                        info.Length = calcHeight;

                        float diffTextH = -(paneHeight / 2.0f) + (calcHeight / 2.0f);
                        if (childPane.ITextBox.BasePosTypeV == VerticalLocation.Top)
                        {
                            info.CenterOffset = -diffTextH;
                        }
                        else if (childPane.ITextBox.BasePosTypeV == VerticalLocation.Bottom)
                        {
                            info.CenterOffset = diffTextH;
                        }
                    }
                }
                else
                {
                    info.Length = childPane.Height * Math.Abs(childPane.Scale.Y);
                }

                // CenterOffset
                {
                    // ペインの基準位置を反映する
                    if (childPane.BasePosTypeV == VerticalLocation.Top)
                    {
                        info.CenterOffset -= childPane.Height / 2.0f;
                    }
                    else if (childPane.BasePosTypeV == VerticalLocation.Bottom)
                    {
                        info.CenterOffset += childPane.Height / 2.0f;
                    }
                }
            }

            // info.Margin
            {
                // 表示されている場合は、マージンを設定します。
                if (isVisible)
                {
                    if(info.Pane.IsAlignmentVerticalMarginEnabled)
                    {
                        info.Margin = info.Pane.AlignmentVerticalMargin;
                    }
                    else
                    {
                        info.Margin = parentAlignment.DefaultVerticalMargin;
                    }
                }
                else
                {
                    info.Margin = 0.0f;
                }
            }

            return isVisible;
        }

        static private bool IsHorizontalExpandTarget_(IAlignment alignment, IPane child)
        {
            var children = alignment.OwnerPane.Children;
            switch (alignment.Horizontal)
            {
                case HorizontalLocation.Left: return child == children.LastOrDefault() as IPane;
                case HorizontalLocation.Center: return child == children.FirstOrDefault() as IPane || child == children.LastOrDefault() as IPane;
                case HorizontalLocation.Right: return child == children.FirstOrDefault() as IPane;
                default: return false;
            }
        }

        static private bool IsVerticalExpandTarget_(IAlignment alignment, IPane child)
        {
            var children = alignment.OwnerPane.Children;
            switch (alignment.Vertical)
            {
                case VerticalLocation.Top: return child == children.LastOrDefault() as IPane;
                case VerticalLocation.Center: return child == children.FirstOrDefault() as IPane || child == children.LastOrDefault() as IPane;
                case VerticalLocation.Bottom: return child == children.FirstOrDefault() as IPane;
                default: return false;
            }
        }

        static private void DoHorizontalExpand_(AlignInfo info, float width, bool isLeftEdge, ModifyAction modifyAction)
        {
            modifyAction.widthModifyAction(info.Pane, width);

            if (isLeftEdge)
            {
                if (info.Pane.BasePosTypeH == HorizontalLocation.Center)
                {
                    modifyAction.transModifyAction(info.Pane, new FVec3(info.Pane.X - (width / 2.0f), info.Pane.Y, info.Pane.Z));
                }
                else if (info.Pane.BasePosTypeH == HorizontalLocation.Left)
                {
                    modifyAction.transModifyAction(info.Pane, new FVec3(info.Pane.X - width, info.Pane.Y, info.Pane.Z));
                }
            }
            else
            {
                if (info.Pane.BasePosTypeH == HorizontalLocation.Center)
                {
                    modifyAction.transModifyAction(info.Pane, new FVec3(info.Pane.X + (width / 2.0f), info.Pane.Y, info.Pane.Z));
                }
                else if (info.Pane.BasePosTypeH == HorizontalLocation.Right)
                {
                    modifyAction.transModifyAction(info.Pane, new FVec3(info.Pane.X + width, info.Pane.Y, info.Pane.Z));
                }
            }
        }

        static private void DoVerticalExpand_(AlignInfo info, float height, bool isTopEdge, ModifyAction modifyAction)
        {
            modifyAction.heightModifyAction(info.Pane, height);

            if (isTopEdge)
            {
                if (info.Pane.BasePosTypeV == VerticalLocation.Center)
                {
                    modifyAction.transModifyAction(info.Pane, new FVec3(info.Pane.X, info.Pane.Y + (height / 2.0f), info.Pane.Z));
                }
                else if (info.Pane.BasePosTypeV == VerticalLocation.Top)
                {
                    modifyAction.transModifyAction(info.Pane, new FVec3(info.Pane.X, info.Pane.Y + height, info.Pane.Z));
                }
            }
            else
            {
                if (info.Pane.BasePosTypeV == VerticalLocation.Center)
                {
                    modifyAction.transModifyAction(info.Pane, new FVec3(info.Pane.X, info.Pane.Y - (height / 2.0f), info.Pane.Z));
                }
                else if (info.Pane.BasePosTypeV == VerticalLocation.Bottom)
                {
                    modifyAction.transModifyAction(info.Pane, new FVec3(info.Pane.X, info.Pane.Y - height, info.Pane.Z));
                }
            }
        }

        static private void CollectHorizontalAlignInfo_(IAlignment alignment, IList<AlignInfo> alignInfos, out float totalWidth)
        {
            totalWidth = 0.0f;
            foreach (IPane child in alignment.OwnerPane.Children)
            {
                AlignInfo alignInfo = new AlignInfo();
                if (MakeHorizontalAlignInfo_(child, alignInfo))
                {
                    if (alignment.IsExtendEdgeHorizontalEnabled)
                    {
                        // 端を延長するモードの場合、延長する対象のペインのサイズは、ここでは0と見なす。
                        // ただしマージンだけは計算する
                        if (IsHorizontalExpandTarget_(alignment, child))
                        {
                            alignInfo.Length = 0.0f;
                            alignInfo.CenterOffset = 0.0f;
                        }
                    }

                    totalWidth += alignInfo.Length + alignInfo.Margin;
                    alignInfos.Add(alignInfo);
                }
            }
        }

        static private void CollectVerticalAlignInfo_(IAlignment alignment, IList<AlignInfo> alignInfos, out float totalHeight)
        {
            totalHeight = 0.0f;
            foreach (IPane child in alignment.OwnerPane.Children)
            {
                AlignInfo alignInfo = new AlignInfo();
                if (MakeVerticalAlignInfo_(child, alignInfo))
                {
                    if (alignment.IsExtendEdgeVerticalEnabled)
                    {
                        // 端を延長するモードの場合、延長する対象のペインのサイズは、ここでは0と見なす。
                        // ただしマージンだけは計算する
                        if (IsVerticalExpandTarget_(alignment, child))
                        {
                            alignInfo.Length = 0.0f;
                            alignInfo.CenterOffset = 0.0f;
                        }
                    }

                    totalHeight += alignInfo.Length + alignInfo.Margin;
                    alignInfos.Add(alignInfo);
                }
            }
        }

        static private void DoAlignHorizontalChildren_(IAlignment alignment, IEnumerable<AlignInfo> alignInfos, float totalWidth, ModifyAction modifyAction)
        {
            if (alignment.Horizontal == HorizontalLocation.Center ||
                alignment.Horizontal == HorizontalLocation.Left)
            {
                //=== 中央揃えと左揃え
                // 左端のマージンは考慮しないので引く
                totalWidth -= alignInfos.First().Margin;

                // 左から置いていく
                // 揃える位置は、整列ペインの左端から
                float leftX = alignment.Horizontal == HorizontalLocation.Center ? -totalWidth / 2.0f : -alignment.OwnerPane.Width / 2.0f;
                foreach (var info in alignInfos)
                {
                    if (info != alignInfos.First())
                    {
                        leftX += info.Margin;
                    }

                    float newX = leftX + info.Length / 2.0f - info.CenterOffset;

                    modifyAction.transModifyAction(info.Pane, new FVec3(newX, info.Pane.Y, info.Pane.Z));

                    leftX += info.Length;
                }

                // ExtendEdge
                if (alignment.IsExtendEdgeHorizontalEnabled)
                {
                    if (alignment.Horizontal == HorizontalLocation.Left)
                    {
                        // 右端の要素を延長
                        float width = alignment.OwnerPane.Width - totalWidth;
                        DoHorizontalExpand_(alignInfos.Last(), width, false, modifyAction);
                    }
                    else if (alignment.Horizontal == HorizontalLocation.Center)
                    {
                        // 左端及び右端の要素を延長
                        float width = (alignment.OwnerPane.Width - totalWidth) / 2.0f;
                        DoHorizontalExpand_(alignInfos.Last(), width, false, modifyAction);
                        DoHorizontalExpand_(alignInfos.First(), width, true, modifyAction);
                    }
                }
            }
            else
            {
                //=== 右揃え
                // 右端のマージンは考慮しないので引く
                totalWidth -= alignInfos.Last().Margin;

                // 右揃えでも、「最初の子ペインが一番左になる」のは同じ。
                // このため、逆順にイテレートして置いていく。
                float rightX = alignment.OwnerPane.Width / 2.0f;
                foreach (var info in alignInfos.Reverse<AlignInfo>())
                {
                    if (info != alignInfos.Last())
                    {
                        rightX -= info.Margin;
                    }

                    float newX = rightX - info.Length / 2.0f - info.CenterOffset;

                    modifyAction.transModifyAction(info.Pane, new FVec3(newX, info.Pane.Y, info.Pane.Z));

                    rightX -= info.Length;
                }

                if (alignment.IsExtendEdgeHorizontalEnabled)
                {
                    // 左端の要素を延長
                    float width = alignment.OwnerPane.Width - totalWidth;
                    DoHorizontalExpand_(alignInfos.First(), width, true, modifyAction);
                }
            }
        }

        static private void DoAlignVerticalChildren_(IAlignment alignment, IEnumerable<AlignInfo> alignInfos, float totalHeight, ModifyAction modifyAction)
        {
            if (alignment.Vertical == VerticalLocation.Center ||
                alignment.Vertical == VerticalLocation.Top)
            {
                //=== 中央揃えと上揃え
                // 上端のマージンは考慮しないので引く
                totalHeight -= alignInfos.First().Margin;

                // 上から置いていく
                // 揃える位置は、整列ペインの左端から
                float upY = alignment.Vertical == VerticalLocation.Center ? totalHeight / 2.0f : alignment.OwnerPane.Height / 2.0f;
                foreach (var info in alignInfos)
                {
                    if (info != alignInfos.First())
                    {
                        upY -= info.Margin;
                    }

                    float newY = upY - info.Length / 2.0f - info.CenterOffset;

                    modifyAction.transModifyAction(info.Pane, new FVec3(info.Pane.X, newY, info.Pane.Z));

                    upY -= info.Length;
                }

                // ExtendEdge
                if (alignment.IsExtendEdgeVerticalEnabled)
                {
                    if (alignment.Vertical == VerticalLocation.Top)
                    {
                        // 下端の要素を延長
                        float height = alignment.OwnerPane.Height - totalHeight;
                        DoVerticalExpand_(alignInfos.Last(), height, false, modifyAction);
                    }
                    else if (alignment.Vertical == VerticalLocation.Center)
                    {
                        //上端及び下端の要素を延長
                        float height = (alignment.OwnerPane.Height - totalHeight) / 2.0f;
                        DoVerticalExpand_(alignInfos.Last(), height, false, modifyAction);
                        DoVerticalExpand_(alignInfos.First(), height, true, modifyAction);
                    }
                }
            }
            else
            {
                //=== 下揃え
                // 下端のマージンは考慮しないので引く
                totalHeight -= alignInfos.Last().Margin;

                // 下揃えでも、「最初の子ペインが一番上になる」のは同じ。
                // このため、逆順にイテレートして置いていく。
                float bottomY = -alignment.OwnerPane.Height / 2.0f;
                foreach (var info in alignInfos.Reverse<AlignInfo>())
                {
                    if (info != alignInfos.Last())
                    {
                        bottomY += info.Margin;
                    }

                    float newY = bottomY + info.Length / 2.0f - info.CenterOffset;

                    modifyAction.transModifyAction(info.Pane, new FVec3(info.Pane.X,  newY, info.Pane.Z));

                    bottomY += info.Length;
                }

                if (alignment.IsExtendEdgeVerticalEnabled)
                {
                    // 上端の要素を延長
                    float height = alignment.OwnerPane.Height - totalHeight;
                    DoVerticalExpand_(alignInfos.First(), height, true, modifyAction);
                }
            }
        }

        static internal void HorizontalAlign(IAlignment alignment, ModifyAction modifyAction)
        {
            if (!alignment.IsAlignmentHorizontalEnabled)
            {
                return;
            }

            // 子ペインをなめて情報を蓄積する
            float totalWidth;
            List<AlignInfo> alignInfos = new List<AlignInfo>();
            CollectHorizontalAlignInfo_(alignment, alignInfos, out totalWidth);

            // 処理対象がなければ終了
            if (alignInfos.Count <= 0)
            {
                return;
            }

            DoAlignHorizontalChildren_(alignment, alignInfos, totalWidth, modifyAction);
        }

        static internal void VerticalAlign(IAlignment alignment, ModifyAction modifyAction)
        {
            if (!alignment.IsAlignmentVerticalEnabled)
            {
                return;
            }

            // 子ペインをなめて情報を蓄積する
            float totalHeight;
            List<AlignInfo> alignInfos = new List<AlignInfo>();
            CollectVerticalAlignInfo_(alignment, alignInfos, out totalHeight);

            // 処理対象がなければ終了
            if (alignInfos.Count <= 0)
            {
                return;
            }

            DoAlignVerticalChildren_(alignment, alignInfos, totalHeight, modifyAction);
        }


        static internal void DoAlign(IAlignment alignment, ModifyAction modifyAction)
        {
            var subScene = alignment?.OwnerPane?.OwnerSubScene;
            if (subScene == null)
            {
                return;
            }
            subScene.BeginMassiveModify();
            {
                HorizontalAlign(alignment, modifyAction);

                VerticalAlign(alignment, modifyAction);
            }
            subScene.EndMassiveModify();
        }
        #endregion
    }
}
