﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Diagnostics;
using System.Xml.Serialization;
using System.Linq;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.SerializableObject.Lyt;
    using System.Windows.Forms;


    /// <summary>
    /// rlyt(レイアウト情報)を出力するクラスです。
    /// </summary>
    internal class RlytExporter : BaseSingleExporter
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RlytExporter()
        {
        }

        #region IExporter メンバ
        /// <summary>
        /// 出力ファイルの拡張子を取得します(短縮)。
        /// </summary>
        public override string[] GetFileExtensionShort()
        {
            return new string[] { AppConstants.LayoutFileExtNoDot };
        }

        /// <summary>
        /// 出力ファイルの拡張子を取得します(長い)。
        /// </summary>
        public override string[] GetFileExtensionLong()
        {
            return new string[] { AppConstants.LayoutFileDesc };
        }

        /// <summary>
        /// 出力ファイルの拡張子を取得します(長い)。
        /// </summary>
        public override Type GetSelializableObjType()
        {
            return typeof( Document );
        }

        /// <summary>
        /// 出力データをバリデートします
        /// </summary>
        protected override bool Validate_(SubScene targetScene, LEMsgReporter msgReporter)
        {
            bool isValid = true;

            // 部品設定のバリデート
            try
            {
                msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSSETTING"));

                // 不正なコントロール設定のチェック
                foreach (IControlSettings ctrlSettings in targetScene.IControlSettings)
                {
                    if (ctrlSettings.ContainsNotFoundAnimationTag())
                    {
                        ReportError_(LECoreStringResMgr.Get("LECORE_VALIDATION_CTRL_ANM_CANTFIND"));
                        isValid = false;
                    }

                    if (ctrlSettings.ContainsNotFoundPane())
                    {
                        ReportError_(LECoreStringResMgr.Get("LECORE_VALIDATION_CTRL_PANE_CANTFIND"));
                        isValid = false;
                    }
                }

                // 不正な部品設定のチェック
                if (!targetScene.IPartsSettings.IsInitialState && targetScene.IPartsSettings.IsPartsSizeZero())
                {
                    ReportError_(LECoreStringResMgr.Get("LECORE_VALIDATION_CTRL_PANE_PRATSSIZE_ZERO"));
                    isValid = false;
                }
            }
            finally
            {
                msgReporter.EndPacking();
            }

            return isValid;
        }

        /// <summary>
        /// XML シリアライズ可能な、クラスに変換します。
        /// </summary>
        public override object ConvertToSerializable_(SubScene targetScene, string outPath, LEMsgReporter msgReporter)
        {
            // XML シリアライズ可能な、クラスに変換します。
            RlytConverter converter = new RlytConverter(outPath, msgReporter);
            converter.SaveOptions = this.ExportOption;

            // コンバートクラスを使用して、シーンからアニメーション情報を格納する
            // シリアル化可能クラスへと変換します。
            Document doc = converter.AppToFile(targetScene);
            Debug.Assert(doc != null);

            return doc;
        }

        /// <summary>
        /// ファイル保存をしてもよいかユーザに確認します。
        /// </summary>
        protected override bool ComfirmSaving_(SubScene targetScene, object serializableDocument, string outPath, LEMsgReporter msgReporter)
        {
            var flytDoc = serializableDocument as Document;
            if(flytDoc.body.lyt.paneSet.Length == 0 && File.Exists(outPath))
            {
                string content = File.ReadAllText(outPath);
                if (!content.Contains("<paneSet />"))
                {
                    DialogResult? result = msgReporter.ShowDialog(LECoreStringResMgr.Get("LECORE_DLG_TITLE_SAVE"), LECoreStringResMgr.Get("LECORE_CONFIRM_ZERO_PANE_OVERWRITE"), MessageBoxButtons.OKCancel);
                    if(result.HasValue && result.Value == DialogResult.Cancel)
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        #endregion IExporter メンバ
    }
}
