﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.IO;
using System.Linq;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;

    using LECore.Structures.SerializableObject.Lyt;
    using LECore.Structures.Nsrif.Attributes;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lyt;

    /// <summary>
    /// XMLシリアライズ可能なクラスへの変換。
    /// (基本的なデータ構造に関する部分)
    /// </summary>
    public partial class RlytConverter
    {
        #region App => File

        #region テクスチャ、フォントリソース関連
        /// <summary>
        /// テクスチャリストを構築します。
        ///
        /// 現在は、シーンのテクスチャリストを出力しています。
        /// レイアウトから使用されない画像も含まれます。
        /// </summary>
        FileFmt.TextureFile[] Convert_SOTextureFiles_( ISubScene scene )
        {
            List<FileFmt.TextureFile> texImgs = new List<TextureFile>();

            OutLog_( "LECORE_RLYTSAVE_TEX" );
            ITextureImage[] textures = scene.ITextureMgr.ITextureImageSet.ToArray();
            foreach( ITextureImage texImg in textures )
            {
                try
                {
                    if (texImg.SourceType == TextureSourceType.File)
                    {
                        FileFmt.TextureFile texFile = new TextureFile();

#if false
                        texFile.format = texImg.PixelFmtIsFixed ?
                            FileFmt.TexelFormat.NW4R_TGA : Convert_TexImagePixelFmt_( texImg.PixelFmt );
#else
                        texFile.format = Convert_TexImagePixelFmt_(texImg.PixelFmt);
#endif

                        texFile.imagePath = GetRelativePath_(texImg.FilePath);

                        texImgs.Add(texFile);
                    }
                }
                catch( Exception e )
                {
                    _msgReporter.ReportError( LECoreStringResMgr.Get( "LECORE_RLYTSAVE_TEX" ), e.Message );
                }
            }


            return texImgs.ToArray();
        }

        /// <summary>
        /// フォントリストを構築します。
        /// </summary>
        FileFmt.FontFile[] Convert_SOFontFiles_(ISubScene scene)
        {
            return Convert_SOFontFiles_(scene, (font) => true);
        }

        /// <summary>
        /// フォントリストを構築します。
        ///
        /// 現在は、シーンのテクスチャリストを出力しています。
        /// レイアウトから使用されない画像も含まれます。
        /// </summary>
        FileFmt.FontFile[] Convert_SOFontFiles_( ISubScene scene, Func<ILEFont, bool> condition)
        {
            OutLog_( "LECORE_RLYTSAVE_FONT" );

            List<FileFmt.FontFile> fontFileSet = new List<FileFmt.FontFile>();
            ILEFont[] leFonts = scene.ILEFontManager.ILEFontSet.ToArray();
            ILEFont[] unUsedFonts = SubSceneHelper.GetUnuseFontSet(scene);

            foreach( ILEFont leFont in leFonts )
            {
                if (!condition(leFont))
                {
                    continue;
                }

                // プロジェクト設定に記述されたフォントの場合は、実際に使われているフォントだけ参照情報を保存します。
                if (leFont.IsProjectFont() && unUsedFonts.Contains(leFont))
                {
                    continue;
                }

                try
                {
                    FileFmt.FontFile fontFile = new FileFmt.FontFile();

                    fontFile.name = leFont.FontName;
                    if (!string.IsNullOrEmpty(leFont.FontPath))
                    {
                        fontFile.path = GetRelativePath_(leFont.FontPath);
                    }
                    else
                    {
                        fontFile.path = string.Empty;
                    }

                    fontFileSet.Add( fontFile );
                }
                catch( Exception e )
                {
                    _msgReporter.ReportError( LECoreStringResMgr.Get( "LECORE_RLYTSAVE_FONT" ), e.Message );
                }
            }
            return fontFileSet.ToArray();
        }

        /// <summary>
        /// キャプチャテクスチャリストを構築します。
        ///
        /// 現在は、シーンのキャプチャテクスチャリストを出力しています。
        /// レイアウトから使用されない画像も含まれます。
        /// </summary>
        FileFmt.CaptureTexture[] Convert_SOCaptureTextures_(ISubScene scene)
        {
            List<FileFmt.CaptureTexture> texImgs = new List<FileFmt.CaptureTexture>();

            OutLog_("LECORE_RLYTSAVE_CAPTURETEX");
            ITextureImage[] textures = TextureMgrHelper.GetCaptureTextureArray(scene.ITextureMgr);
            foreach (ITextureImage tex in textures)
            {
                try
                {
                    texImgs.Add(Convert_SOCaptureTexture_(tex.ICaptureTexture));
                }
                catch (Exception e)
                {
                    _msgReporter.ReportError(LECoreStringResMgr.Get("LECORE_RLYTSAVE_CAPTURETEX"), e.Message);
                }
            }

            return texImgs.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="tex"></param>
        /// <returns></returns>
        public FileFmt.CaptureTexture Convert_SOCaptureTexture_(ICaptureTexture tex)
        {
            FileFmt.CaptureTexture result = new FileFmt.CaptureTexture();
            ITextureImage texImg = tex as ITextureImage;

            result.paneName = tex.ReferencedPaneName;
            result.framebufferCaptureEnabled = tex.FrameBufferCaptureEnabled;
            result.captureOnlyFirstFrame = tex.CaptureOnlyFirstFrame;
            result.clearColor = Convert_FloatColor_(tex.ClearColor);
            result.format = Convert_TexImagePixelFmt_(texImg.PixelFmt);
            result.isHidingFromList = texImg.IsHidingFromList;
            result.usage = (FileFmt.CaptureTextureUsage)tex.Usage;

            // 固定のフィルター情報を書き出す
            result.filterCount = 1;
            FileFmt.FilterInformation[] filterInfos = new FilterInformation[1];
            filterInfos[0] = new FilterInformation();
            filterInfos[0].type = FilterType.Copy;
            filterInfos[0].scale = tex.TextureScale;
            result.filters = filterInfos;

            return result;
        }

        #endregion テクスチャ、フォントリソース関連

        #endregion

        #region File => App

        #region テクスチャ・フォントリソース関連
        /// <summary>
        /// すべてのテクスチャを読み込み、マネージャに登録します。
        /// </summary>
        void Convert_TextureFile_( TextureMgr textureMgr, TextureFile[] textureFileSet )
        {
            if( textureFileSet == null )
            {
                return;
            }

            // すべてのテクスチャを読み込み、マネージャに登録します。
            OutLog_( "LECORE_RLYTLOAD_TEX" );
            textureMgr.BeginUpdate();
            foreach( TextureFile textureFile in textureFileSet )
            {
                string textureName = Path.GetFileName( textureFile.imagePath );
                try
                {
                    OutLog_( "LECORE_RLYTLOAD_TEX_NAME", textureName );

                    string completePath = GetAbsolutePath_( textureFile.imagePath );
                    AppData.ITextureImage texImg =
                        textureMgr.RegisterTextureImageFromFile( completePath );

                    // テクセルフォーマットを設定。
                    if( !texImg.PixelFmtIsFixed )
                    {
                        texImg.PixelFmt = Convert_TexelFormat_( textureFile.format );
                    }
                }
                catch
                {
                    ReportErrorMsg_( LECoreStringResMgr.Get( "LECORE_SYS_ERROR_LOAD_TEXTURE" ), textureName );
                    string completePath = GetAbsolutePath_( textureFile.imagePath );
                    // 処理は続行されます。
                }
            }
            textureMgr.EndUpdate();
        }

        /// <summary>
        /// すべてのフォントを読み込み、マネージャに登録します。
        /// </summary>
        void Convert_FontFile_( FontManager fontMgr, FontFile[] fontFileSet )
        {
            if( fontFileSet == null )
            {
                return;
            }

            // すべてのフォントを読み込み、マネージャに登録します。
            OutLog_( "LECORE_RLYTLOAD_FONT" );
            foreach( FontFile fontFile in fontFileSet )
            {
                string fontName = !string.IsNullOrEmpty(fontFile.name) ? fontFile.name : Path.GetFileName(fontFile.path);
                OutLog_( "LECORE_RLYTLOAD_FONT_NAME", fontName );

                try
                {
                    string completePath = GetAbsolutePath_( fontFile.path );
                    ILEFont result = fontMgr.CreateILEFont(fontName, completePath, null);
                    if (!string.IsNullOrEmpty(result.InitializationErrorMessage))
                    {
                        string msg = LECoreStringResMgr.Get("LECORE_SYS_ERROR_LOAD_FONT") + Environment.NewLine + "{1}";
                        ReportErrorMsg_(msg, fontName, result.InitializationErrorMessage);
                    }
                    else if ((LEFontHelper.IsBitmapFontFile(result.FontPath) || LEFontHelper.IsComplexFont(result))&&
                        !LEFontHelper.IsAcceptableBntxPlatform(result, LECore.LayoutEditorCore.PlatformDetail.PlatformPreference))
                    {
                        ReportErrorMsg_(LECoreStringResMgr.Get("LECORE_UNACCEPTABLE_PLATFORM"),
                            Path.GetFullPath(result.FontPath),
                            result.BntxPlatform,
                            LECore.LayoutEditorCore.PlatformDetail.UIPlatformName);
                    }
                }
                catch( Exception e )
                {

                    string msg = LECoreStringResMgr.Get( "LECORE_SYS_ERROR_LOAD_FONT" ) + Environment.NewLine + "{1}";
                    ReportErrorMsg_( msg, fontName, e.Message );
                    // 処理は続行されます。
                }
            }
        }

        /// <summary>
        /// キャプチャテクスチャのファイル情報を、dst で指定したインスタンスへ設定します。
        /// </summary>
        /// <param name="dst"></param>
        /// <param name="src"></param>
        public void Convert_CaptureTexture_(AppData.ICaptureTexture dst, FileFmt.CaptureTexture src)
        {
            AppData.CaptureTexture tmp = dst as AppData.CaptureTexture;
            tmp.FrameBufferCaptureEnabled = src.framebufferCaptureEnabled;
            tmp.CaptureOnlyFirstFrame = src.captureOnlyFirstFrame;
            tmp.ClearColor = Convert_Color4f_(src.clearColor);
            tmp.Format = Convert_TexelFormat_(src.format);
            tmp.TextureScale = src.filters[0].scale;
            tmp.Usage = (AppData.CaptureTextureUsage)src.usage;
        }

        /// <summary>
        /// すべてのキャプチャテクスチャを読み込み、マネージャに登録します。
        /// </summary>
        void Convert_CaptureTextures_(ISubScene subScene, TextureMgr textureMgr, FileFmt.CaptureTexture[] captureTextureSet)
        {
            if (captureTextureSet == null)
            {
                return;
            }

            // すべてのキャプチャテクスチャを読み込み、ペインやテクスチャと関連付けてマネージャに登録します。
            OutLog_("LECORE_RLYTLOAD_CAPTURETEX");
            textureMgr.BeginUpdate();
            foreach (FileFmt.CaptureTexture captureTexture in captureTextureSet)
            {
                try
                {
                    IPane pane = subScene.FindPaneByName(captureTexture.paneName);
                    bool isHidingFromList = captureTexture.isHidingFromList;
                    ITextureImage tex = textureMgr.RegisterCaptureTexture(pane, pane.PaneName, isHidingFromList, (AppData.CaptureTextureUsage)captureTexture.usage);

                    Convert_CaptureTexture_(tex.ICaptureTexture, captureTexture);
                }
                catch
                {
                    ReportErrorMsg_(LECoreStringResMgr.Get("LECORE_SYS_ERROR_LOAD_CAPTURETEXTURE"), captureTexture.paneName);
                    // 処理は続行されます。
                }
            }
            textureMgr.EndUpdate();
        }

        #endregion テクスチャ・フォントリソース関連

        #endregion
    }
}
