﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;

    using LECore.Structures.SerializableObject.Lyt;
    using LECore.Structures.Nsrif.Attributes;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lyt;

    /// <summary>
    /// XMLシリアライズ可能なクラスへの変換。
    /// (詳細なMaterial(RVL)に関する部分)
    /// </summary>
    public partial class RlytConverter
    {
        #region File => App

        #region 型変換関連

        #region TEV関連列挙型 変換
        /// <summary>
        /// コンバインモード
        /// </summary>
        private AttrCombineMode
        Convert_AttrCtrCombineMode_( FileFmt.TevMode src )
        {
            switch( src )
            {
            case TevMode.Replace:     return AttrCombineMode.Replace;
            case TevMode.Modulate:    return AttrCombineMode.Modulate;
            case TevMode.Add:         return AttrCombineMode.Add;
            case TevMode.AddSigned:   return AttrCombineMode.AddSigned;
            case TevMode.Interpolate: return AttrCombineMode.Interpolate;
            case TevMode.Subtract:    return AttrCombineMode.Subtract;
            case TevMode.AddMult:     return AttrCombineMode.AddMultDMP;
            case TevMode.MultAdd:     return AttrCombineMode.MultAddDMP;
            case TevMode.Overlay:    return AttrCombineMode.Overlay;
            case TevMode.Lighten: return AttrCombineMode.Lighten;
            case TevMode.Darken: return AttrCombineMode.Darken;
            case TevMode.Indirect: return AttrCombineMode.Indirect;
            case TevMode.BlendIndirect: return AttrCombineMode.BlendIndirect;
            case TevMode.EachIndirect: return AttrCombineMode.EachIndirect;
            default: Debug.Assert( false ); return AttrCombineMode._Unknown;
            }
        }

        /// <summary>
        /// コンスタントレジスタ
        /// </summary>
        private AttrTevKonst
        Convert_AttrTevKonst_( FileFmt.TevKonstSel src )
        {
            switch( src )
            {
            case TevKonstSel.K0: return AttrTevKonst.K0;
            case TevKonstSel.K1: return AttrTevKonst.K1;
            case TevKonstSel.K2: return AttrTevKonst.K2;
            case TevKonstSel.K3: return AttrTevKonst.K3;
            case TevKonstSel.K4: return AttrTevKonst.K4;
            case TevKonstSel.K5: return AttrTevKonst.K5;
            case TevKonstSel.BlackColor: return AttrTevKonst.BlackColor;
            case TevKonstSel.WhiteColor: return AttrTevKonst.WhiteColor;
            default: Debug.Assert( false ); return AttrTevKonst.K0;
            }
        }

        /// <summary>
        /// スケール
        /// </summary>
        private AttrTevScale
        Convert_AttrTevScale_( FileFmt.TevScale src )
        {
            switch( src )
            {
            case TevScale.V1: return AttrTevScale.Scale1;
            case TevScale.V2: return AttrTevScale.Scale2;
            case TevScale.V4: return AttrTevScale.Scale4;
            default: Debug.Assert( false ); return AttrTevScale.Scale1;
            }
        }

        /// <summary>
        /// ソース
        /// </summary>
        private AttrTevSource
        Convert_AttrTevSource_( FileFmt.TevSource src )
        {
            switch( src )
            {
            case FileFmt.TevSource.Texture0: return AttrTevSource.Texture0;
            case FileFmt.TevSource.Texture1: return AttrTevSource.Texture1;
            case FileFmt.TevSource.Texture2: return AttrTevSource.Texture2;
            case FileFmt.TevSource.Texture3: return AttrTevSource.Texture3;
            case FileFmt.TevSource.Constant: return AttrTevSource.Constant;
            case FileFmt.TevSource.Primary:  return AttrTevSource.Primary;
            case FileFmt.TevSource.Previous: return AttrTevSource.Previous;
            case FileFmt.TevSource.Register: return AttrTevSource.Register;
            default: Debug.Assert( false ); return AttrTevSource.Texture0;
            }
        }

        /// <summary>
        /// オペランドRGB
        /// </summary>
        private AttrTevOp
        Convert_AttrTevOpRgb_( FileFmt.TevOpRgb src )
        {
            switch( src )
            {
            case FileFmt.TevOpRgb.Rgb:      return AttrTevOp.Rgb;
            case FileFmt.TevOpRgb.InvRgb:   return AttrTevOp.InvRgb;
            case FileFmt.TevOpRgb.Alpha:    return AttrTevOp.Alpha;
            case FileFmt.TevOpRgb.InvAlpha: return AttrTevOp.InvAlpha;
            case FileFmt.TevOpRgb.Rrr:      return AttrTevOp.Rrr;
            case FileFmt.TevOpRgb.InvRrr:   return AttrTevOp.InvRrr;
            case FileFmt.TevOpRgb.Ggg:      return AttrTevOp.Ggg;
            case FileFmt.TevOpRgb.InvGgg:   return AttrTevOp.InvGgg;
            case FileFmt.TevOpRgb.Bbb:      return AttrTevOp.Bbb;
            case FileFmt.TevOpRgb.InvBbb:   return AttrTevOp.InvBbb;
            default: Debug.Assert( false ); return AttrTevOp.Rgb;
            }
        }

        /// <summary>
        /// オペランドAlpha
        /// </summary>
        private AttrTevOp
        Convert_AttrTevOpAlpha_( FileFmt.TevOpAlp src )
        {
            switch( src )
            {
            case FileFmt.TevOpAlp.Alpha:    return AttrTevOp.Alpha;
            case FileFmt.TevOpAlp.InvAlpha: return AttrTevOp.InvAlpha;
            case FileFmt.TevOpAlp.R:        return AttrTevOp.R;
            case FileFmt.TevOpAlp.InvR:     return AttrTevOp.InvR;
            case FileFmt.TevOpAlp.G:        return AttrTevOp.G;
            case FileFmt.TevOpAlp.InvG:     return AttrTevOp.InvG;
            case FileFmt.TevOpAlp.B:        return AttrTevOp.B;
            case FileFmt.TevOpAlp.InvB:     return AttrTevOp.InvB;
            default: Debug.Assert( false ); return AttrTevOp.Alpha;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private DirectStageTevArg
        Convert_AttrTevArgRgb_( FileFmt.TevArgRgb src )
        {
            DirectStageTevArg   arg = new DirectStageTevArg();

            arg.Source = Convert_AttrTevSource_( src.source);
            arg.Op     = Convert_AttrTevOpRgb_( src.op);

            return arg;
        }

        /// <summary>
        ///
        /// </summary>
        private DirectStageTevArg
        Convert_AttrTevArgAlpha_( FileFmt.TevArgAlp src )
        {
            DirectStageTevArg   arg = new DirectStageTevArg();

            arg.Source = Convert_AttrTevSource_( src.source);
            arg.Op     = Convert_AttrTevOpAlpha_( src.op);

            return arg;
        }
        #endregion TEV関連列挙型 変換

        /// <summary>
        ///
        /// </summary>
        void Convert_ToTevDirectRgb(
                                      AppData.TevDirectStage dstStage,
                                      FileFmt.Material_CTRTevStageRgb srcStage)
        {
            dstStage.CombineMode = Convert_AttrCtrCombineMode_( srcStage.mode);
            dstStage.Konst       = Convert_AttrTevKonst_( srcStage.konst);
            dstStage.Scale       = Convert_AttrTevScale_( srcStage.scale);
            dstStage.CopyReg     = srcStage.copyReg;

            dstStage.IndirectScale = Convert_FVec2_(srcStage.indirectScale);
            dstStage.IndirectRotate = srcStage.indirectRotate;

            for( int i = 0; i < 3; i++ ) {
                dstStage.TevArgs[i] = Convert_AttrTevArgRgb_(srcStage.arg[i]);
            }
        }

        /// <summary>
        ///
        /// </summary>
        void Convert_ToTevDirectAlpha(
                                      AppData.TevDirectStage dstStage,
                                      FileFmt.Material_CTRTevStageAlpha srcStage)
        {
            dstStage.CombineMode = Convert_AttrCtrCombineMode_( srcStage.mode);
            dstStage.Konst       = Convert_AttrTevKonst_(srcStage.konst);
            dstStage.Scale       = Convert_AttrTevScale_(srcStage.scale);
            dstStage.CopyReg     = srcStage.copyReg;

            for( int i = 0; i < 3; i++ ) {
                dstStage.TevArgs[i] = Convert_AttrTevArgAlpha_(srcStage.arg[i]);
            }
        }

        /// <summary>
        /// ステージのデシリアライズ
        /// FileFmt.Material_RevoTevStage => AppData.TevStage
        /// </summary>
        void Convert_ToTevStage( AppData.TevStage dstStage,
                                 FileFmt.Material_CTRTevStage srcStage )
        {
            Convert_ToTevDirectRgb( dstStage.ColorStage, srcStage.rgb );
            Convert_ToTevDirectAlpha( dstStage.AlphaStage, srcStage.alpha );
        }

        /// <summary>
        /// AppData.IPEData => FileFmt.Material_RevoAlphaCompare
        /// </summary>
        void Convert_ToAppRevoAlphaCompare_( AppData.PEACompare dstACmp, FileFmt.Material_CTRAlphaCompare srcACmp )
        {
            dstACmp.Comp = (AttrCompareFunc)ConvertEnumTypesFromRToL_( dstACmp.Comp, srcACmp.comp );
            dstACmp.Ref = srcACmp.@ref;
        }

        /// <summary>
        /// AppData.IPEData => FileFmt.Material_RevoBlendMode
        /// </summary>
        void Convert_ToFileRevoBlendMode_( AppData.PEBlend dstBlend, FileFmt.Material_CTRBlendMode srcBlend )
        {
            if (srcBlend == null)
            {
                dstBlend.Type = AttrBlendType.None;
            }
            else
            {
                dstBlend.Type = (AttrBlendType)ConvertEnumTypesFromRToL_(dstBlend.Type, srcBlend.type);

                if (srcBlend.blendOpSpecified) { dstBlend.BlendOp = (AttrBlendOp)ConvertEnumTypesFromRToL_(dstBlend.BlendOp, srcBlend.blendOp); }
                if (srcBlend.logicOpSpecified) { dstBlend.LogicOp = (AttrLogicOp)ConvertEnumTypesFromRToL_(dstBlend.LogicOp, srcBlend.logicOp); }
                if (srcBlend.srcFactorSpecified) { dstBlend.SrcFactor = (AttrBlendFactor)ConvertEnumTypesFromRToL_(dstBlend.SrcFactor, srcBlend.srcFactor); }
                if (srcBlend.dstFactorSpecified) { dstBlend.DstFactor = (AttrBlendFactor)ConvertEnumTypesFromRToL_(dstBlend.DstFactor, srcBlend.dstFactor); }
            }
        }
        #endregion

        /// <summary>
        /// FileFmt.Material_Revo => AppData.RevHWMaterial
        /// </summary>
        void Convert_Pane_RevMaterial_( AppData.RevHWMaterial dstMat, FileFmt.Material_CTR srcMat )
        {
            dstMat.PEData.UseDefaultBlendSettings = srcMat.useDefaultBlendSettings;
            dstMat.PEData.UseDefaultAlphaTestSettings = srcMat.useDefaultAlphaTestSettings;

            // PE - AlphaCompare
            Convert_ToAppRevoAlphaCompare_( dstMat.PEData.ACompare, srcMat.alphaCompare );
            Convert_ToFileRevoBlendMode_( dstMat.PEData.Blend, srcMat.blendMode );
            Convert_ToFileRevoBlendMode_(dstMat.PEData.BlendAlpha, srcMat.blendModeAlpha);

            //
            dstMat.MaterialName = srcMat.name;

            //ステージ
            // Tev - TevStage
            if( srcMat.tevStage != null )
            {
                // 古いデータからは、サポートできないデータの入力があり得るので制限します。
                int numStages = Math.Min(TevStages.NumMaxStages, (int)srcMat.tevStageNum);
                dstMat.TevData.TevStages.NumStages = numStages;
                // 現在の仕様では、必ず等しくなるはずです。
                Debug.Assert( srcMat.tevStage.Length == srcMat.tevStageNum );

                for (int i = 0; i < numStages; i++)
                {
                    Convert_ToTevStage( dstMat.TevData.TevStages[i], srcMat.tevStage[i] );
                }
            }

            // Tevカラー
            if (srcMat.tevConstColorFloat != null)
            {
                for (int i = 0; i < srcMat.tevConstColorFloat.Length; i++)
                {
                    Color4f color = srcMat.tevConstColorFloat[i];
                    dstMat.TevData.SetTevColor(new FloatColor(color.r, color.g, color.b, color.a), i);
                }
            }
            else if (srcMat.tevConstColor != null)
            {
                for (int i = 0; i < srcMat.tevConstColor.Length; i++)
                {
                    Color4 c = srcMat.tevConstColor[i];
                    dstMat.TevData.SetTevColor(new FloatColor(System.Drawing.Color.FromArgb(c.a, c.r, c.g, c.b)), i);
                }
            }

            dstMat.LowLevelCombinerSettingsEnabled = srcMat.useLowLevelCombinerSettings;

            // CombinerEditor に関する情報
            dstMat.CombinerUserShaderSettingsEnabled = srcMat.useCombinerUserShaderSettings;
            if (srcMat.CombinerUserShader != null)
            {
                if (!string.IsNullOrWhiteSpace(srcMat.CombinerUserShader.fileName))
                {
                    dstMat.CombinerUserShader.FileName = GetAbsolutePath_(srcMat.CombinerUserShader.fileName);
                }
            }

            // Textrue に関する情報
            // 通常のマテリアルのロードで初期化は完了しているはずなので行わない。
        }


        #endregion

        #region App => File

        #region TEV関連列挙型 変換

        /// <summary>
        ///
        /// </summary>
        private FileFmt.TevMode
        Convert_TevMode_( AttrCombineMode src )
        {
            switch( src )
            {
            case AttrCombineMode.Replace:     return TevMode.Replace;
            case AttrCombineMode.Modulate:    return TevMode.Modulate;
            case AttrCombineMode.Add:         return TevMode.Add;
            case AttrCombineMode.AddSigned:   return TevMode.AddSigned;
            case AttrCombineMode.Interpolate: return TevMode.Interpolate;
            case AttrCombineMode.Subtract:    return TevMode.Subtract;
            case AttrCombineMode.AddMultDMP:  return TevMode.AddMult;
            case AttrCombineMode.MultAddDMP:  return TevMode.MultAdd;
            case AttrCombineMode.Overlay: return TevMode.Overlay;
            case AttrCombineMode.Lighten: return TevMode.Lighten;
            case AttrCombineMode.Darken: return TevMode.Darken;
            case AttrCombineMode.Indirect: return TevMode.Indirect;
            case AttrCombineMode.BlendIndirect: return TevMode.BlendIndirect;
            case AttrCombineMode.EachIndirect: return TevMode.EachIndirect;
            default: Debug.Assert( false ); return TevMode.Replace;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private FileFmt.TevSource
        Convert_TevSource_( AttrTevSource src )
        {
            switch( src )
            {
            case AttrTevSource.Texture0: return FileFmt.TevSource.Texture0;
            case AttrTevSource.Texture1: return FileFmt.TevSource.Texture1;
            case AttrTevSource.Texture2: return FileFmt.TevSource.Texture2;
            case AttrTevSource.Texture3: return FileFmt.TevSource.Texture3;
            case AttrTevSource.Constant: return FileFmt.TevSource.Constant;
            case AttrTevSource.Primary:  return FileFmt.TevSource.Primary;
            case AttrTevSource.Previous: return FileFmt.TevSource.Previous;
            case AttrTevSource.Register: return FileFmt.TevSource.Register;
            default: Debug.Assert( false ); return FileFmt.TevSource.Texture0;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private FileFmt.TevOpRgb
        Convert_TevOpRgb_( AttrTevOp src )
        {
            switch( src )
            {
            case AttrTevOp.Rgb:      return FileFmt.TevOpRgb.Rgb;
            case AttrTevOp.InvRgb:   return FileFmt.TevOpRgb.InvRgb;
            case AttrTevOp.Alpha:    return FileFmt.TevOpRgb.Alpha;
            case AttrTevOp.InvAlpha: return FileFmt.TevOpRgb.InvAlpha;
            case AttrTevOp.Rrr:      return FileFmt.TevOpRgb.Rrr;
            case AttrTevOp.InvRrr:   return FileFmt.TevOpRgb.InvRrr;
            case AttrTevOp.Ggg:      return FileFmt.TevOpRgb.Ggg;
            case AttrTevOp.InvGgg:   return FileFmt.TevOpRgb.InvGgg;
            case AttrTevOp.Bbb:      return FileFmt.TevOpRgb.Bbb;
            case AttrTevOp.InvBbb:   return FileFmt.TevOpRgb.InvBbb;
            default: Debug.Assert( false ); return FileFmt.TevOpRgb.Rgb;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private FileFmt.TevOpAlp
        Convert_TevOpAlpha_( AttrTevOp src )
        {
            switch( src )
            {
            case AttrTevOp.Alpha:    return FileFmt.TevOpAlp.Alpha;
            case AttrTevOp.InvAlpha: return FileFmt.TevOpAlp.InvAlpha;
            case AttrTevOp.R:        return FileFmt.TevOpAlp.R;
            case AttrTevOp.InvR:     return FileFmt.TevOpAlp.InvR;
            case AttrTevOp.G:        return FileFmt.TevOpAlp.G;
            case AttrTevOp.InvG:     return FileFmt.TevOpAlp.InvG;
            case AttrTevOp.B:        return FileFmt.TevOpAlp.B;
            case AttrTevOp.InvB:     return FileFmt.TevOpAlp.InvB;
            default: Debug.Assert( false ); return FileFmt.TevOpAlp.Alpha;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private FileFmt.TevArgRgb
        Convert_TevArgRgb_( DirectStageTevArg src )
        {
            FileFmt.TevArgRgb       arg = new FileFmt.TevArgRgb();

            arg.source = Convert_TevSource_( src.Source);
            arg.op     = Convert_TevOpRgb_( src.Op);

            return arg;
        }

        /// <summary>
        ///
        /// </summary>
        private FileFmt.TevArgAlp
        Convert_TevArgAlpha_( DirectStageTevArg src )
        {
            FileFmt.TevArgAlp   arg = new FileFmt.TevArgAlp();

            arg.source = Convert_TevSource_( src.Source);
            arg.op     = Convert_TevOpAlpha_( src.Op);

            return arg;
        }

        /// <summary>
        /// スケール
        /// </summary>
        FileFmt.TevScale
        Convert_TevScale_( AttrTevScale src )
        {
            switch( src )
            {
            case AttrTevScale.Scale1: return TevScale.V1;
            case AttrTevScale.Scale2: return TevScale.V2;
            case AttrTevScale.Scale4: return TevScale.V4;
            default: Debug.Assert( false ); return TevScale.V1;
            }
        }

        /// <summary>
        /// コンスタントレジスタ
        /// </summary>
        FileFmt.TevKonstSel
        Convert_TevKonst_( AttrTevKonst src )
        {
            switch( src )
            {
            case AttrTevKonst.K0: return TevKonstSel.K0;
            case AttrTevKonst.K1: return TevKonstSel.K1;
            case AttrTevKonst.K2: return TevKonstSel.K2;
            case AttrTevKonst.K3: return TevKonstSel.K3;
            case AttrTevKonst.K4: return TevKonstSel.K4;
            case AttrTevKonst.K5: return TevKonstSel.K5;
            case AttrTevKonst.BlackColor: return TevKonstSel.BlackColor;
            case AttrTevKonst.WhiteColor: return TevKonstSel.WhiteColor;
            default: Debug.Assert( false ); return TevKonstSel.K0;
            }
        }

        /// <summary>
        /// 右の型から左の型への変換
        /// 文字列が同一である型の変換に利用します(大文字、小文字を区別しません)。
        /// </summary>
        object ConvertEnumTypesFromRToL_( Enum type0, Enum type1 )
        {
            return Enum.Parse( type0.GetType(), type1.ToString(), true );
        }

        #endregion TEV関連列挙型 変換

        #region 型変換

        /// <summary>
        /// AppData.IPEData => FileFmt.Material_RevoAlphaCompare
        /// </summary>
        FileFmt.Material_CTRAlphaCompare
        Convert_ToFileRevoAlphaCompare_( AppData.IPEData appPE )
        {
            FileFmt.Material_CTRAlphaCompare dstAlphaCompare = new FileFmt.Material_CTRAlphaCompare();

            dstAlphaCompare.comp = (FileFmt.Compare)ConvertEnumTypesFromRToL_( dstAlphaCompare.comp, appPE.ACompare.Comp );
            dstAlphaCompare.@ref = appPE.ACompare.Ref;
            return dstAlphaCompare;
        }

        /// <summary>
        /// AppData.IPEData => FileFmt.Material_RevoBlendMode
        /// </summary>
        FileFmt.Material_CTRBlendMode
        Convert_ToFileRevoBlendMode_(AppData.PEBlend appPEBlend)
        {
            FileFmt.Material_CTRBlendMode dstBlendMode = new FileFmt.Material_CTRBlendMode();

            dstBlendMode.type = (FileFmt.BlendMode)ConvertEnumTypesFromRToL_(dstBlendMode.type, appPEBlend.Type);
            dstBlendMode.blendOp = (FileFmt.BlendOp)ConvertEnumTypesFromRToL_(dstBlendMode.blendOp, appPEBlend.BlendOp);
            dstBlendMode.logicOp = (FileFmt.LogicOp)ConvertEnumTypesFromRToL_(dstBlendMode.logicOp, appPEBlend.LogicOp);
            dstBlendMode.srcFactor = (BlendFactor)ConvertEnumTypesFromRToL_(dstBlendMode.srcFactor, appPEBlend.SrcFactor);
            dstBlendMode.dstFactor = (BlendFactor)ConvertEnumTypesFromRToL_(dstBlendMode.dstFactor, appPEBlend.DstFactor);

            switch (appPEBlend.Type)
            {
            case AttrBlendType.None:
                break;

            case AttrBlendType.Blend:
                dstBlendMode.blendOpSpecified = true;
                dstBlendMode.srcFactorSpecified = true;
                dstBlendMode.dstFactorSpecified = true;
                break;

            case AttrBlendType.Logic:
                dstBlendMode.logicOpSpecified = true;
                break;
            }

            return dstBlendMode;
        }

        /// <summary>
        /// AppData.TevDirectStage => FileFmt.Material_RevoTevStageColor
        /// </summary>
        FileFmt.Material_CTRTevStageRgb
        Convert_Material_RevoTevStageColor_(AppData.TevDirectStage srcStage)
        {
            Material_CTRTevStageRgb dstStage
                = new Material_CTRTevStageRgb();

            dstStage.mode  = Convert_TevMode_( srcStage.CombineMode);
            dstStage.konst = Convert_TevKonst_( srcStage.Konst);
            dstStage.scale = Convert_TevScale_( srcStage.Scale);
            dstStage.copyReg = srcStage.CopyReg;

            dstStage.indirectScale = Convert_Vec2_(srcStage.IndirectScale);
            dstStage.indirectRotate = srcStage.IndirectRotate;

            dstStage.arg = new TevArgRgb[ 3 ];
            for( int i = 0; i < dstStage.arg.Length; i++ )
            {
                dstStage.arg[ i ] = Convert_TevArgRgb_( srcStage.TevArgs[ i ]);
            }

            return dstStage;
        }

        /// <summary>
        /// AppData.TevDirectStage => FileFmt.Material_RevoTevStageAlpha
        /// </summary>
        FileFmt.Material_CTRTevStageAlpha
        Convert_Material_RevoTevStageAlpha_(AppData.TevDirectStage srcStage)
        {
            Material_CTRTevStageAlpha dstStage
                = new Material_CTRTevStageAlpha();

            dstStage.mode  = Convert_TevMode_( srcStage.CombineMode);
            dstStage.konst = Convert_TevKonst_( srcStage.Konst);
            dstStage.scale = Convert_TevScale_( srcStage.Scale);
            dstStage.copyReg = srcStage.CopyReg;

            dstStage.arg = new TevArgAlp[ 3 ];
            for( int i = 0; i < dstStage.arg.Length; i++ )
            {
                dstStage.arg[ i ] = Convert_TevArgAlpha_( srcStage.TevArgs[ i ]);
            }

            return dstStage;
        }

        /// <summary>
        /// ステージのシリアライズ
        /// AppData.TevStage => FileFmt.Material_RevoTevStage
        /// </summary>
        FileFmt.Material_CTRTevStage
        Convert_RevoTevStage_( AppData.ITevStage srcStage )
        {
            Material_CTRTevStage dstStage =
                new Material_CTRTevStage();

            // カラーステージ
            dstStage.rgb = Convert_Material_RevoTevStageColor_( srcStage.ColorStage );
            // アルファステージ
            dstStage.alpha = Convert_Material_RevoTevStageAlpha_( srcStage.AlphaStage );
            return dstStage;
        }

        /// <summary>
        /// AppData.ITexMtx => FileFmt.Material_RevoIndirectMatrix
        /// </summary>
        FileFmt.TexMatrix
        Convert_IndirectMatrix_( AppData.ITexMtx srcIndMtx )
        {
            FileFmt.TexMatrix dstIndMtx =
                new FileFmt.TexMatrix();

            dstIndMtx.rotate = srcIndMtx.Rotate;

            dstIndMtx.scale = Convert_Vec2_( srcIndMtx.Scale );
            dstIndMtx.translate = Convert_Vec2_( srcIndMtx.Trans );

            return dstIndMtx;
        }
        #endregion

        /// <summary>
        /// AppData.IRevHWMaterial => FileFmt.Material_Revo
        /// </summary>
        FileFmt.Material_CTR Convert_ToFileMaterial_Revo_( AppData.IRevHWMaterial appMat )
        {
            FileFmt.Material_CTR dstMat = new Material_CTR();

            //
            dstMat.name = appMat.MaterialName;

            // PE - AlphaCompare
            dstMat.alphaCompare = Convert_ToFileRevoAlphaCompare_( appMat.IPEData );
            dstMat.blendMode = Convert_ToFileRevoBlendMode_( appMat.IPEData.Blend );
            dstMat.blendModeAlpha = Convert_ToFileRevoBlendMode_(appMat.IPEData.BlendAlpha);
            dstMat.useDefaultBlendSettings = appMat.IPEData.UseDefaultBlendSettings;
            dstMat.useDefaultAlphaTestSettings = appMat.IPEData.UseDefaultAlphaTestSettings;

            // 実際にデータに FloatColor が設定されていたら FloatColor 用のタグを出力する。
            // FloatColor が有効な状態でもなるべく古いタグの状態にしておくため。
            bool floatColorTag = !appMat.IMaterial.BlackColor.ByteConvert || !appMat.IMaterial.WhiteColor.ByteConvert;

            if (floatColorTag)
            {
                // 白黒補間
                dstMat.tevColRegFloat = Convert_FloatColor_(appMat.IMaterial.BlackColor);
                dstMat.tevConstRegFloat = new Color4f[TevColorsConstants.ObsoleteNumTevKColorRegs];
                for (int i = 0; i < dstMat.tevConstRegFloat.Length; i++)
                {
                    dstMat.tevConstRegFloat[i] = Convert_FloatColor_(appMat.IMaterial.WhiteColor);
                }
            }
            else
            {
                // 白黒補間
                dstMat.tevColReg = Convert_Color4_(appMat.IMaterial.BlackColor.ToRGBAColor());
                dstMat.tevConstReg = new Color4[TevColorsConstants.ObsoleteNumTevKColorRegs];
                for (int i = 0; i < dstMat.tevConstReg.Length; i++)
                {
                    dstMat.tevConstReg[i] = Convert_Color4_(appMat.IMaterial.WhiteColor.ToRGBAColor());
                }
            }

            // TEV 定数カラーはまだバイナリ側が FloatColor に対応していないため、必ず Byte カラーのタグを出力する。
            if (appMat.LowLevelCombinerSettingsEnabled || appMat.CombinerUserShaderSettingsEnabled)
            {
                // TEV定数カラー
                dstMat.tevConstColor = new Color4[TevColorsConstants.NumTevKColorRegs];
                for (int i = 0; i < dstMat.tevConstColor.Length; i++)
                {
                    dstMat.tevConstColor[i] = Convert_Color4_(appMat.ITevData.GetTevColor(i).ToRGBAColor());
                }
            }

            //ステージ
            // Tev - TevStage
            // 現在は、使用される情報のみ編集可能となっているので、
            // 保存すべきステージの数 = 使用されるステージの数です。
            int numTevStage = appMat.ITevData.NumStages;
            dstMat.tevStage = new FileFmt.Material_CTRTevStage[numTevStage];
            dstMat.tevStageNum = (byte)numTevStage;
            for( int i = 0 ; i < dstMat.tevStage.Length ; i++ )
            {
                dstMat.tevStage[i] = Convert_RevoTevStage_( appMat.ITevData.GetITevStage( i ) );
            }

            // Textrue に関する情報
            AppData.IMaterialTexMap[] texMaps = appMat.IMaterialTexMapSet;

            dstMat.texMap = new FileFmt.TexMap[texMaps.Length];
            dstMat.texMatrix = new FileFmt.TexMatrix[texMaps.Length];
            dstMat.texCoordGen = new FileFmt.TexCoordGen[texMaps.Length];

            for( int i = 0 ; i < texMaps.Length ; i++ )
            {
                if( texMaps[i] != null )
                {
                    dstMat.texMap[i] = Convert_ToFileTexMap_( texMaps[i] );
                    dstMat.texMatrix[i] = Convert_ToFileTexMtx_( texMaps[i].ITexGen.ITexMtx );
                    dstMat.texCoordGen[i] = Convert_ToFileTexCoordGen_( texMaps[i].ITexGen );
                }
            }

            dstMat.useLowLevelCombinerSettings = appMat.LowLevelCombinerSettingsEnabled;

            // CombinerEditor に関する情報
            dstMat.useCombinerUserShaderSettings = appMat.CombinerUserShaderSettingsEnabled;

            if (dstMat.useCombinerUserShaderSettings)
            {
                dstMat.CombinerUserShader = new FileFmt.Material_CTRCombinerUserShader();

                if (!string.IsNullOrEmpty(appMat.ICombinerUserShader.FileName))
                {
                    dstMat.CombinerUserShader.fileName = GetRelativePath_(appMat.ICombinerUserShader.FileName);
                }
            }

            return dstMat;
        }

#endregion
    }
}
