﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Web;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;

    using LECore.Structures.SerializableObject.Lyt;
    using LECore.Structures.Nsrif.Attributes;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lyt;

    using LECore.Structures.Core.Command;
    using Win32;


    /// <summary>
    /// バージョン文字列比較
    /// </summary>
    public class VersionComparere : IComparer<string>
    {
        static public int DoCompare(string x, string y)
        {
            return new VersionComparere().Compare(x, y);
        }

        public int Compare(string x, string y)
        {
            if (x == y)
            {
                return 0;
            }

            var numx = x.Split('.');
            var numy = y.Split('.');

            // 上位桁から違いを探す
            for (int i = 0; i < numx.Length; i++)
            {
                // x の方が桁数が多いので新しいとみなす。
                if (i >= numy.Length)
                {
                    return 1;
                }

                if (numx[i] == numy[i])
                {
                    continue;
                }
                else
                {
                    try
                    {
                        return int.Parse(numx[i]) - int.Parse(numy[i]);
                    }
                    catch
                    {
                        return StringComparer.Ordinal.Compare(numx[i], numy[i]);
                    }
                }
            }

            // xとyは上位桁ではすべて一致しているが、さらに yの方が桁数が多いので新しいとみなす。
            if (numx.Length < numy.Length)
            {
                return -1;
            }

            // 等しい
            return 0;
        }
    }

    /// <summary>
    /// アプリケーション内部のデータ構造クラスを、
    /// XMLシリアライズ可能な、クラスに変換するクラスです。
    /// </summary>
    public partial class RlytConverter
    {
        //----------------------------------------------------------
        // フィールド：内部状態的なもの
        //----------------------------------------------------------

        SerializableConverterUtil _serializableConverterUtil = null;

        string _VersionString
        {
            get { return _serializableConverterUtil.VersionString; }
        }

        /// <summary>
        /// 相対パス表記の基底となるパス
        /// コンストラクタで指定されます。
        /// </summary>
        string _inputFilePath = string.Empty;

        /// <summary>
        /// エラーメッセージリポーター
        /// </summary>
        readonly LEMsgReporter _msgReporter = null;

        /// <summary>
        /// 出力オプション
        /// </summary>
        public ExportOption SaveOptions
        {
            get;
            set;
        } = new ExportOption();

        /// <summary>
        /// float カラー有効フラグ
        /// </summary>
        public static bool FloatColorWrite { get; set; } = false;

        //----------------------------------------------------------

        /// <summary>
        /// 警告表示を行います。
        /// </summary>
        void ReportErrorMsg_( string msg, params object[] paramaters )
        {
            if( _msgReporter != null )
            {
                _msgReporter.ReportError(
                    LECoreStringResMgr.Get( "LECORE_DLG_TITLE_LOADRLYT" ),
                    msg,
                    paramaters );
            }
        }

        /// <summary>
        /// ログ出力します。
        /// </summary>
        void OutLog_( string msgID, params object[] paramaters )
        {
            string title = LECoreStringResMgr.Get( msgID, paramaters );
            if( _msgReporter != null )
            {
                _msgReporter.OutLog( title, string.Empty );
            }
        }

        /// <summary>
        /// 相対パスを取得します。
        /// </summary>
        string GetRelativePath_( string path )
        {
            if( _inputFilePath == string.Empty )
            {
                return path;
            }

            return Utility.GetRelativePath(_inputFilePath, path);
        }

        /// <summary>
        /// 絶対パスを取得する
        /// </summary>
        string GetAbsolutePath_( string path )
        {
            if( _inputFilePath == string.Empty )
            {
                return path;
            }

            Uri root = new Uri( _inputFilePath );

            return new Uri( root, path ).LocalPath;
        }

        //----------------------------------------------------------
        // コンストラクタ
        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RlytConverter( string inputFilePath, LEMsgReporter msgReporter )
        {
            // 指定された場合は、絶対パスであること。
            Debug.Assert( inputFilePath == string.Empty || Path.IsPathRooted( inputFilePath ) );

            _inputFilePath = inputFilePath;
            _msgReporter = msgReporter;
        }

        //----------------------------------------------------------
        // エントリ
        //----------------------------------------------------------

        /// <summary>
        /// シーンクラスを  FileFmt.Document に変換します。
        /// LECore内部クラス => シリアライズ可能オブジェクト への変換です。
        /// </summary>
        /// <returns></returns>
        public FileFmt.Document AppToFile( ISubScene scene )
        {
            _serializableConverterUtil = new SerializableConverterUtil();

            FileFmt.Document result = new FileFmt.Document();

            // ヘッダ情報関連
            result.head = Convert_head_(scene.DocumentHeader as DocumentHeader);

            result.body = new FileFmt.DocumentBody();

            // Rlyt
            FileFmt.LYT rlyt = new FileFmt.LYT();
            rlyt.paneSet = Convert_SOPaneArray(scene);
            rlyt.groupSet = Convert_SOGroupSet_(scene);
            rlyt.paneHierarchy = Convert_SOPaneHierarchy_(scene);
            rlyt.textureFile = Convert_SOTextureFiles_(scene);
            rlyt.fontFile = Convert_SOFontFiles_(scene);
            rlyt.captureTexture = Convert_SOCaptureTextures_(scene);
            rlyt.screenSetting = Convert_SOScreenSetting_(scene);
            Convert_ToFileParts_(scene, rlyt);

            rlyt.control = Convert_ToFileControlSet_(scene);

            rlyt.userData = Convert_UserDataEx_(scene.IUserDataHolder.UserDataElementSet);

            rlyt.metrics = Convert_Metrics_(scene);
            rlyt.stateMachine = Convert_StateMachine_(scene);

            result.body.lyt = rlyt;

            // result.versionについては、
            // 明示的な設定を行わず、スキーマに設定されている、
            // 既定の値を利用します。

            _serializableConverterUtil = null;

            return result;
        }

        /// <summary>
        /// 統計情報を出力します。
        /// </summary>
        public FileFmt.Metrics Convert_Metrics_(ISubScene subScene)
        {
            SubSceneMetrics srcMetrics = new SubSceneMetrics();
            SubSceneHelper.CalcSubsceneMetrics(subScene, srcMetrics);

            FileFmt.Metrics dstMetrics = new Metrics();
            dstMetrics.totalPaneCount = srcMetrics.GetTotalPaneCount();
            dstMetrics.totalPixelCount = srcMetrics.GetTotalPixcelCount();

            return dstMetrics;
        }

        FileFmt.StateVariables[] Convert_StateMachineStateVariables_(IEnumerable<IStateMachineVariable> srcVars)
        {
            return srcVars.Select(s => new StateVariables()
            {
                name = s.Name,
                value = s.Value,
                type = ConvertEnum_<FileFmt.StateMachineVariableType>(s.VariableType)
            }).ToArray();
        }

        IEnumerable<IStateMachineVariable> ConvertBack_StateMachineStateVariables_(FileFmt.StateVariables[]  srcVars)
        {
            if(srcVars == null)
            {
                return new IStateMachineVariable[0];
            }

            return srcVars.Select(s => new StateMachineVariableData()
            {
                Name = s.name,
                Value = s.value,
                VariableType = ConvertEnum_<AppData.StateMachineVariableType>(s.type)
            }).ToArray();
        }

        static TDst ConvertEnum_<TDst>(Enum src)
        {
            return (TDst)Enum.Parse(typeof(TDst), src.ToString());
        }

        static TDst[] ConvertFlagEnumToArray_<TDst>(Enum flags)
        {
            List<TDst> dst = new List<TDst>();

            foreach (Enum flag in Enum.GetValues(flags.GetType()))
            {
                if (flags.HasFlag(flag))
                {
                    dst.Add(ConvertEnum_<TDst>(flag));
                }
            }

            return dst.ToArray();
        }

        static TDstFlag ConvertBackFlagEnumToArray_<TDstFlag, TSrc>(TSrc[] flags)
        {
            int dst = 0;

            foreach (TSrc flag in flags)
            {
                dst |= (int)Enum.Parse(typeof(TDstFlag), flag.ToString());
            }

            return (TDstFlag)Enum.ToObject(typeof(TDstFlag), dst);
        }

        /// <summary>
        ///
        /// </summary>
        FileFmt.StateFeatureParamater Convert_StateMachineFeatureParamater_(IFeatureParamater param)
        {
            var dstParam = new FileFmt.StateFeatureParamater();

            dstParam.targetName = param.Name;

            dstParam.kind = ConvertFlagEnumToArray_<FileFmt.FeatureParamaterKind>(param.Kind);

            dstParam.x = param.X;
            dstParam.y = param.Y;
            dstParam.z = param.Z;
            dstParam.w = param.W;

            return dstParam;
        }

        /// <summary>
        /// ステートをファイル出力します。
        /// </summary>
        public FileFmt.StateMachine Convert_StateMachine_(ISubScene subScene)
        {
            if(!subScene.IStateMachine.IsEnabled)
            {
                return null;
            }

            FileFmt.StateMachine dst = new StateMachine();

            // StateLayer
            {
                var dstLayers = new List<FileFmt.StateLayer>();
                foreach (var layer in subScene.IStateMachine.Layers)
                {
                    var dstLayer = new StateLayer();

                    dstLayer.name = layer.Name;

                    // State
                    {
                        dstLayer.states = layer.States.Select(state =>
                        {
                            var dstState = new FileFmt.State();
                            dstState.name = state.Name;

                            // StateFeatureParamater
                            {
                                var dstParams = new List<FileFmt.StateFeatureParamater>();
                                foreach (var param in state.Paramaters)
                                {
                                    dstParams.Add(Convert_StateMachineFeatureParamater_(param));
                                }
                                dstState.featureParamaters = dstParams.ToArray();
                            }

                            return dstState;
                        }).ToArray();
                    }

                    // StateTransition
                    {
                        var dstTransitions = new List<FileFmt.StateTransition>();
                        foreach(var transition in layer.Transitions)
                        {
                            var dstTransition = new FileFmt.StateTransition();

                            dstTransition.offset = transition.Offset;
                            dstTransition.totalDuration = transition.TotalDuration;
                            dstTransition.isLoop = transition.IsLoop;
                            dstTransition.isCancelable = transition.IsCancelable;
                            dstTransition.startStateName = transition.StartStateName;
                            dstTransition.endStateName = transition.EndStateName;

                            dstTransition.easing = new FileFmt.StateEasing();
                            dstTransition.easing.type = ConvertEnum_<FileFmt.EasingType>(transition.StateEasing.EasingType);
                            dstTransition.easing.paramater = transition.StateEasing.EasingExtraParamater;

                            {
                                IState state = layer.FindStateByName(transition.StartStateName);
                                Debug.Assert(state != null);

                                dstTransition.tracks = state.Paramaters.Select((paramater) =>
                                {
                                    var track = transition.FindIStateTransitionTrackDataByName(paramater.Name, paramater.Kind);

                                    var dstTrack = new FileFmt.StateTransitionTrack();

                                    dstTrack.targetName = paramater.Name;
                                    dstTrack.kind = ConvertFlagEnumToArray_<FileFmt.FeatureParamaterKind>(paramater.Kind);
                                    Debug.Assert(dstTrack.kind.Length <= 2);

                                    dstTrack.offset = track.Offset;
                                    dstTrack.duration = track.Duration;

                                    dstTrack.easing = new FileFmt.StateEasing();
                                    dstTrack.easing.type = ConvertEnum_<FileFmt.EasingType>(track.StateEasing.EasingType);
                                    dstTrack.easing.paramater = track.StateEasing.EasingExtraParamater;

                                    dstTrack.keies = track.KeyParamaters.Select(key =>
                                    {
                                        var dstKey = new FileFmt.StateTransitionTrackKey();

                                        dstKey.offset = key.Time;
                                        dstKey.easing = new FileFmt.StateEasing();
                                        dstKey.easing.type = ConvertEnum_<FileFmt.EasingType>(key.StateEasing.EasingType);
                                        dstKey.easing.paramater = key.StateEasing.EasingExtraParamater;
                                        dstKey.featureParamater = Convert_StateMachineFeatureParamater_(key.FeatureParamaterData);

                                        var srcEvent = key.StateMachineEvent;
                                        dstKey.stateMachineEvent = new FileFmt.StateMachineEvent();

                                        dstKey.stateMachineEvent.delay = srcEvent.delayFrames;
                                        dstKey.stateMachineEvent.kind = ConvertEnum_<FileFmt.StateMachineEventKind>(srcEvent.kind);
                                        dstKey.stateMachineEvent.param1 = srcEvent.param1 != null ? srcEvent.param1.ToString() : string.Empty;
                                        dstKey.stateMachineEvent.param2 = srcEvent.param2 != null ? srcEvent.param2.ToString() : string.Empty;

                                        return dstKey;
                                    }).ToArray();

                                    return dstTrack;
                                }).ToArray();
                            }

                            var trigger = transition.Trigger;
                            Debug.Assert(trigger != null);
                            dstTransition.trigger = new StateTransitionTrigger();
                            dstTransition.trigger.kind = ConvertEnum_<FileFmt.StateTransitionTriggerKind>(trigger.TriggerKind);
                            dstTransition.trigger.stateMachineOperator = ConvertEnum_<FileFmt.StateMachineOperator>(trigger.StateMachineOperator);
                            dstTransition.trigger.stateMachineVariablesSet = Convert_StateMachineStateVariables_(trigger.StateMachineVariables);

                            dstTransitions.Add(dstTransition);
                        }

                        dstLayer.transitions = dstTransitions.ToArray();
                    }

                    dstLayers.Add(dstLayer);
                }

                dst.layers = dstLayers.ToArray();

                dst.variables = Convert_StateMachineStateVariables_(subScene.IStateMachine.Variables);
            }

            return dst;
        }

        /// <summary>
        /// 重複しているペイン名を取得します。
        /// </summary>
        private string[] RepetitionPane(FileFmt.LYT lyt, ISubScene subScene)
        {
            FileFmt.Pane[] paneSet = lyt.paneSet;

            if (paneSet == null)
            {
                return new string[0];
            }

            return paneSet
                .Where(p => SubSceneHelper.FindPaneByName(subScene, p.name) != null)
                .Select(p => p.name)
                .ToArray<string>();
        }

        /// <summary>
        /// FileFmt.Document を シーンクラス に変換します。
        /// </summary>
        /// <returns></returns>
        public void FileToApp( ISubScene subScene, FileFmt.Document soDocument )
        {
            FileFmt.LYT rlyt = soDocument.body.lyt;

            TextureMgr texMgr = subScene.ITextureMgr as TextureMgr;
            FontManager fontMgr = subScene.ILEFontManager as FontManager;

            OutLog_( "LECORE_RLYTLOAD_BEGIN", Path.GetFileName( _inputFilePath ) );

            // この部分で処理していた、重複ペイン名に関する例外送出は、消去されました。

            subScene.BeginMassiveModify();

            Convert_head_(subScene.DocumentHeader as DocumentHeader, soDocument.head);

            // テクスチャをすべて読み込みます。
            Convert_TextureFile_( texMgr, rlyt.textureFile );

            // フォントをすべて読み込みます
            Convert_FontFile_( fontMgr, rlyt.fontFile );

            // ペイン情報をシーンに読み込みます。
            Convert_LayoutData_PaneSet_( rlyt, subScene );

            // 階層構造を構築します。
            Convert_LayoutData_Herarchy_( rlyt, subScene );

            // グループを構築します。
            Convert_LayoutData_GroupSet_( rlyt, subScene );

            // 背景・スクリーン情報を構築します。
            Convert_ScreenSetting_( rlyt.screenSetting, subScene );

            // 部品設定を構築します。
            Convert_Parts_(rlyt, subScene);

            // キャプチャテクスチャの情報を読み込みます。
            // 内部でペインを参照するためペインの情報が構築済みである必要があります。
            Convert_CaptureTextures_( subScene, texMgr, rlyt.captureTexture );

            // コントロール設定を構築します。
            Convert_Control_(rlyt, subScene);

            // 拡張ユーザ情報を構築します。
            Convert_LayoutUserData_(rlyt, subScene);

            // ステートの読み込み
            Convert_LayoutData_StateMachie_(rlyt, subScene);

            //----------------------------------------------------------
            // 過去の中間ファイルバージョンからの移行に伴う特殊処理をします。
            UpdateOldFormat_(subScene, soDocument);

            // 読み込みデータ内容を調査して、必要ならユーザに警告を示します。
            WarnImportedLayout_(subScene);

            subScene.EndMassiveModify();
        }

        /// <summary>
        /// ステート、ファイル読み込み
        /// </summary>
        void Convert_LayoutData_StateMachie_(FileFmt.LYT rlyt, ISubScene subScene)
        {
            if(rlyt.stateMachine == null)
            {
                return;
            }

            var fileStateMachine = rlyt.stateMachine;
            var stateMachine = (subScene.IStateMachine as StateMachineData);
            stateMachine.IsEnabled = true;

            Debug.Assert(fileStateMachine.layers != null);
            foreach (var fileLayer in fileStateMachine.layers)
            {
                var layer = stateMachine.AddLayer(fileLayer.name);

                // featureParamaters
                {
                    var baseState = fileLayer.states.First();
                    foreach (var param in baseState.featureParamaters)
                    {
                        var paramKind = ConvertBackFlagEnumToArray_<AppData.FeatureParamaterKind, FileFmt.FeatureParamaterKind>(param.kind);
                        stateMachine.AddNewFeatureParamaterToLayer(layer.Name, param.targetName, paramKind);
                    }
                }

                // state
                Debug.Assert(fileLayer.states != null);
                foreach (var fileState in fileLayer.states)
                {
                    stateMachine.AddStateToLayer(fileLayer.name, fileState.name);

                    stateMachine.SetState(fileLayer.name, fileState.name, (s) =>
                    {
                        // 今のところ設定するものなし
                        return true;
                    });

                    // state - featureParamaters
                    if(fileState.featureParamaters != null)
                    {
                        foreach (var fileStateParam in fileState.featureParamaters)
                        {
                            var stateParamKind = ConvertBackFlagEnumToArray_<AppData.FeatureParamaterKind, FileFmt.FeatureParamaterKind>(fileStateParam.kind);
                            stateMachine.SetStateFeatureParamater(layer.Name, fileState.name, fileStateParam.targetName, stateParamKind, (stateParam) =>
                            {
                                stateParam.X = fileStateParam.x;
                                stateParam.Y = fileStateParam.y;
                                stateParam.Z = fileStateParam.z;
                                stateParam.W = fileStateParam.w;
                                return true;
                            });
                        }
                    }
                }

                // transitions
                if(fileLayer.transitions != null)
                {
                    foreach (var fileTransition in fileLayer.transitions)
                    {
                        stateMachine.AddTransitionToLayer(fileLayer.name, fileTransition.startStateName, fileTransition.endStateName);
                        stateMachine.SetTransition(fileLayer.name, fileTransition.startStateName, fileTransition.endStateName, (t) =>
                        {
                            t.Offset = fileTransition.offset;
                            t.TotalDuration = fileTransition.totalDuration;

                            t.StartStateName = fileTransition.startStateName;
                            t.EndStateName = fileTransition.endStateName;

                            t.IsLoop = fileTransition.isLoop;
                            t.IsCancelable = fileTransition.isCancelable;
                            (t.Trigger as StateTransitionTriggerData).StateMachineOperator = ConvertEnum_<AppData.StateMachineOperator>(fileTransition.trigger.stateMachineOperator);
                            (t.Trigger as StateTransitionTriggerData).TriggerKind = ConvertEnum_<AppData.TriggerKind>(fileTransition.trigger.kind);
                            (t.Trigger as StateTransitionTriggerData).StateMachineVariableList.AddRange(ConvertBack_StateMachineStateVariables_(fileTransition.trigger.stateMachineVariablesSet));

                            if (fileTransition.tracks != null)
                            {
                                foreach (var track in fileTransition.tracks)
                                {
                                    var paramKind = ConvertBackFlagEnumToArray_<AppData.FeatureParamaterKind, FileFmt.FeatureParamaterKind>(track.kind);
                                    t.SetStateTransitionTrackDataByName(
                                        track.targetName, paramKind,
                                        track.offset, track.duration,
                                        AppData.IStateEasingHelper.Create(ConvertEnum_<AppData.EasingType>(track.easing.type), track.easing.paramater));

                                    if (track.keies != null)
                                    {
                                        foreach (var key in track.keies)
                                        {
                                            if (paramKind != AppData.FeatureParamaterKind.StateMachineEvent)
                                            {
                                                t.SetStateTransitionTrackKey(
                                                    track.targetName, paramKind,
                                                    key.offset,
                                                    AppData.IStateEasingHelper.Create(ConvertEnum_<AppData.EasingType>(key.easing.type), key.easing.paramater),
                                                    key.featureParamater.x, key.featureParamater.y, key.featureParamater.z, key.featureParamater.w);
                                            }
                                            else
                                            {
                                                t.SetStateTransitionTrackEvent(
                                                    track.targetName,
                                                    key.offset,
                                                    ConvertEnum_<AppData.StateMachineEventKind>(key.stateMachineEvent.kind),
                                                    key.stateMachineEvent.param1, key.stateMachineEvent.param2,
                                                    (uint)key.stateMachineEvent.delay);
                                            }
                                        }
                                    }
                                }
                            }

                            return true;
                        });
                    }
                }
            }

            // TODO:
            if(fileStateMachine.variables != null)
            {
                foreach (var variable in fileStateMachine.variables)
                {

                }
            }
        }

        /// <summary>
        /// 過去の中間ファイルバージョンからの移行に伴う特殊処理をします。
        /// </summary>
        private void UpdateOldFormat_(ISubScene subScene, FileFmt.Document soDocument)
        {
            if (soDocument.version == "1.5.0")
            {
                {
                    // 過去のバージョンでは、ブレンド設定が無視されていた。
                    // 新バージョンでブレンドがサポートされたので、表示結果を維持するために
                    // テキストボックスのブレンド設定は、すべてデフォルト設定に変更する。
                    foreach (var textBoxPane in subScene.FindPanesByKind(AppData.PaneKind.Textbox))
                    {
                        foreach (var material in (textBoxPane.ITextBox as IRevHWMaterialHolder).IRevHWMaterial)
                        {
                            var peData = material.IPEData as PEData;
                            peData.UseDefaultAlphaTestSettings = true;
                            peData.UseDefaultBlendSettings = true;
                        }
                    }

                    // 上書きプロパティの修正
                    foreach (IPane partsPane in subScene.FindPanesByKind(AppData.PaneKind.Parts))
                    {
                        foreach(var partsPropaerty in partsPane.IPartsLayout.PartsPropaerties)
                        {
                            if (partsPropaerty.PaneKind == LECore.Structures.PaneKind.Textbox && partsPropaerty.Paramater != null)
                            {
                                foreach (var material in (partsPropaerty.Paramater as IRevHWMaterialHolder).IRevHWMaterial)
                                {
                                    var peData = material.IPEData as PEData;
                                    peData.UseDefaultAlphaTestSettings = true;
                                    peData.UseDefaultBlendSettings = true;
                                }
                            }
                        }
                    }
                }
            }

            // 1.5.13 より古いデータで対処。
            if (VersionComparere.DoCompare(soDocument.version, "1.5.13") <= 0)
            {
                // 以前の誤っていた挙動がそのまま維持されるようにデータを変換する。
                foreach (IPane pane in subScene.IPaneArray)
                {
                    if (pane.PaneKind != AppData.PaneKind.Parts)
                    {
                        foreach (RevHWMaterial mat in PaneHelper.GetRevHWMatFromPane(pane))
                        {
                            UpdateOldFormat1_5_14_BlendFactorFix_(mat);
                        }
                    }
                    else
                    {
                        // 上書きプロパティの修正
                        foreach (var partsPropaerty in pane.IPartsLayout.PartsPropaerties)
                        {
                            IRevHWMaterialHolder matHolder = partsPropaerty.Paramater as IRevHWMaterialHolder;
                            if (matHolder == null)
                            {
                                continue;
                            }

                            foreach (RevHWMaterial mat in matHolder.IRevHWMaterial)
                            {
                                UpdateOldFormat1_5_14_BlendFactorFix_(mat);
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 不具合修正で互換性に問題がでないように
        /// 以前の挙動と同じようにブレンドファクターの値を書き換えます。
        /// </summary>
        void UpdateOldFormat1_5_14_BlendFactorFix_(RevHWMaterial mat)
        {
            // カラーチャンネルフレンド
            if (mat.IPEData.Blend.SrcFactor == AttrBlendFactor.DstClr)
            {
                mat.IPEData.Blend.SrcFactor = AttrBlendFactor.SrcClr;
            }
            else if (mat.IPEData.Blend.SrcFactor == AttrBlendFactor.InvDstClr)
            {
                mat.IPEData.Blend.SrcFactor = AttrBlendFactor.InvSrcClr;
            }

            // アルファチャンネルフレンド
            if (mat.IPEData.BlendAlpha.SrcFactor == AttrBlendFactor.DstClr)
            {
                mat.IPEData.BlendAlpha.SrcFactor = AttrBlendFactor.SrcClr;
            }
            else if (mat.IPEData.BlendAlpha.SrcFactor == AttrBlendFactor.InvDstClr)
            {
                mat.IPEData.BlendAlpha.SrcFactor = AttrBlendFactor.InvSrcClr;
            }
        }

        /// <summary>
        /// 読み込みデータ内容を調査して、必要ならユーザに警告を示します。
        /// </summary>
        private void WarnImportedLayout_(ISubScene subScene)
        {
            List<ITextBox> textBoxSet = new List<ITextBox>();
            foreach (var textBoxPane in subScene.FindPanesByKind(AppData.PaneKind.Textbox))
            {
                var textBox = textBoxPane.ITextBox;
                if (textBox.ILEFont != null && !textBox.ILEFont.IsFileNotFound())
                {
                    if (textBox.FontSizeOriginal == FVec2.Empty)
                    {
                        continue;
                    }

                    if (textBox.ILEFont.Width != textBox.FontSizeOriginal.X ||
                        textBox.ILEFont.Height != textBox.FontSizeOriginal.Y)
                    {
                        if (LayoutEditorCore.KeepingSizeScaleEnabled)
                        {
                            LECore.Structures.TextBox rawTextBox = textBox as LECore.Structures.TextBox;
                            if (rawTextBox != null)
                            {
                                // スケールを維持したままフォントを変更する。
                                // UI で設定できる精度以下の値を切り捨てる。
                                FVec2 scale = new FVec2(
                                    textBox.FontSize.X / textBox.FontSizeOriginal.X,
                                    textBox.FontSize.Y / textBox.FontSizeOriginal.Y
                                );
                                rawTextBox.FontSize = new FVec2(
                                    (float)Math.Round(textBox.ILEFont.Width * scale.X, 2),
                                    (float)Math.Round(textBox.ILEFont.Height * scale.Y, 2)
                                );
                            }
                        }
                        else
                        {
                            textBoxSet.Add(textBox);
                        }
                    }
                }
            }

            if(textBoxSet.Count > 0)
            {
                // *** 警告：フォント(bffnt)の文字サイズ変更が検出されました ***
                _msgReporter.ReportError("", "*****************************************************");
                _msgReporter.ReportError("",LECoreStringResMgr.Get("LECORE_FONT_WARN_SIZECHANGE"));
                _msgReporter.ReportError("", "");
                foreach (var textBox in textBoxSet)
                {
                    _msgReporter.ReportError("", LECoreStringResMgr.Get("LECORE_FONT_WARN_SIZECHANGE_PANE"), textBox.OwnerPane.PaneName, textBox.FontName, textBox.FontSizeOriginal, new FVec2(textBox.ILEFont.Width, textBox.ILEFont.Height));
                }
                _msgReporter.ReportError("", "*****************************************************");
            }
        }

#region クリップボードから利用される・シリアライズオブジェクト変換関数

#region ペイン
        /// <summary>
        /// ペインをシリアライズ可能なペインクラスに変換します。
        /// </summary>
        internal FileFmt.Pane ConvertPaneToSerializable( AppData.IPane srcPane )
        {
            return Convert_SOPane( srcPane );
        }

        /// <summary>
        /// シリアライズ可能なペインクラスをペインに変換します。
        /// </summary>
        internal AppData.IPane ConvertPaneToInternal( FileFmt.Pane srcPane )
        {
            AppData.IPane result = Convert_Pane_( srcPane );
            return result;
        }

        /// <summary>
        /// ペインの複製を作成します。
        /// </summary>
        internal AppData.IPane MakePaneClone( AppData.IPane srcPane )
        {
            return Convert_Pane_( Convert_SOPane( srcPane ) );
        }


#endregion ペイン

#region 親子階層

        /// <summary>
        /// シリアライズ可能なペイン階層情報に変換します。
        /// </summary>
        public PaneHierarchy
            ConvertPaneHierarchyToSerializable(
               AppData.IPane pane )
        {
            PaneHierarchy dstHierarchy = new PaneHierarchy();
            PaneTree dstPaneTree = new PaneTree();

            Convert_PaneTree_SubTree_( pane, dstPaneTree );

            dstHierarchy.paneTree = dstPaneTree;

            return dstHierarchy;
        }

#endregion 親子階層

#region オブジェクトのスナップショット作成

        // TextBox
        internal FileFmt.TextBox Convert_Item_TextBoxSelf( AppData.ITextBox srcTextBox )
        {
            return this.Convert_Item_TextBoxSelf_( srcTextBox );
        }
        // TextBox
        internal AppData.TextBox Convert_Pane_TextBoxSelf( AppData.Pane dstPane, FileFmt.TextBox srcTextBox )
        {
            return this.Convert_Pane_TextBoxSelf_( dstPane, srcTextBox );
        }

#region UserData
        // ユーザデータ
        internal object[] Convert_Item_UserData( AppData.IPane srcPane )
        {
            return Convert_PaneUserData_( srcPane );
        }

        // ユーザデータ
        internal object[] Convert_Item_UserDataEx(AppData.IUserDataElement[] userDataSet)
        {
            return Convert_UserDataEx_(userDataSet);
        }

        // ユーザデータ
        internal AppData.UserDataElement[] Convert_Pane_UserData( object[] userData )
        {
            return GetUserDataElementSetImpl_(userData);
        }
#endregion UserData

        // Picture
        internal FileFmt.Picture Convert_Item_TextBoxSelf( AppData.IPicture srcPicture )
        {
            return this.Convert_Item_PictureSelf_( srcPicture );
        }
        // Picture
        internal AppData.Picture Convert_Pane_TextBoxSelf( AppData.Pane dstPane, FileFmt.Picture srcPicture )
        {
            return this.Convert_Pane_PictureSelf_( dstPane, srcPicture );
        }

        // ILEWindow
        internal FileFmt.Window Convert_Item_WindowSelf( ILEWindow srcWindow )
        {
            return this.Convert_Item_WindowSelf_( srcWindow );
        }
        // ILEWindow
        internal AppData.ILEWindow Convert_Pane_WindowSelf( AppData.Pane dstPane, FileFmt.Window srcWindow )
        {
            return this.Convert_Pane_WindowSelf_( dstPane, srcWindow );
        }

        // IMaterial
        internal FileFmt.Material Convert_ToFileMaterial( AppData.IMaterial appMat )
        {
            return this.Convert_ToFileMaterial_( appMat );
        }
        // IMaterial
        internal AppData.IMaterial Convert_Pane_Material( FileFmt.Material fileMat, bool flag )
        {
            AppData.Material appMat = new LECore.Structures.Material( null, "material_memento" );
            this.Convert_Pane_Material_( appMat, fileMat );

            return appMat;
        }

        // IRevHWMaterial
        internal FileFmt.Material_CTR Convert_ToFileMaterial_Revo( AppData.IRevHWMaterial appMat )
        {
            return Convert_ToFileMaterial_Revo_( appMat );
        }
        // IRevHWMaterial
        internal AppData.RevHWMaterial Convert_Pane_RevMaterial( FileFmt.Material_CTR fileMat )
        {
            AppData.Material		appMat = new LECore.Structures.Material( null, "material_rvl_memento" );
            AppData.RevHWMaterial appMatRVL = new RevHWMaterial( null, appMat );
            Convert_Pane_RevMaterial_( appMatRVL, fileMat );

            return appMatRVL;
        }

        // Capture
        internal FileFmt.Capture Convert_Item_CaptureSelf(AppData.ICapture srcCapture)
        {
            return this.Convert_Item_CaptureSelf_(srcCapture);
        }
        // Capture
        internal AppData.Capture Convert_Pane_CaptureSelf(AppData.Pane dstPane, FileFmt.Capture srcCapture)
        {
            return this.Convert_Pane_CaptureSelf_(dstPane, srcCapture);
        }

        // Alignment
        internal FileFmt.Alignment Convert_Item_AlignmentSelf(AppData.IAlignment src)
        {
            return this.Convert_Item_AlignmentImpl_(src);
        }

        // Alignment
        internal AppData.Alignment Convert_Pane_AlignmentSelf(AppData.Pane dstPane, FileFmt.Alignment src)
        {
            return this.Convert_Pane_AlignmentSelf_(dstPane, src);
        }

        // Scissor
        internal FileFmt.Scissor Convert_Item_ScissorSelf(AppData.IScissor src)
        {
            return this.Convert_Item_ScissorImpl_(src);
        }

        // Scissor
        internal AppData.Scissor Convert_Pane_ScissorSelf(AppData.Pane dstPane, FileFmt.Scissor src)
        {
            return this.Convert_Pane_ScissorSelf_(dstPane, src);
        }

        /// <summary>
        /// IMaterialTexMap に対応する、FileFmt形式の構造体をまとめる構造体。
        /// </summary>
        internal struct FileFmtMaterialTexMap
        {
            public readonly FileFmt.TexMap			_texMap;
            public readonly FileFmt.TexBlendRatio _texBlendRatio;
            public readonly FileFmt.TexCoordGen _texCoordGen;
            public readonly int _textureIndex;

            public FileFmtMaterialTexMap(
                FileFmt.TexMap			texMap,
                FileFmt.TexBlendRatio	texBlendRatio,
                FileFmt.TexCoordGen	texCoordGen,
                int						textureIndex )
            {
                _texMap = texMap;
                _texBlendRatio = texBlendRatio;
                _texCoordGen = texCoordGen;
                _textureIndex = textureIndex;
            }
        }
        // TexMap
        internal FileFmtMaterialTexMap Convert_ToFileTexMap(
            AppData.IMaterialTexMap appTexMap )
        {

            FileFmtMaterialTexMap		dstTexMap = new FileFmtMaterialTexMap(
                Convert_ToFileTexMap_( appTexMap ),
                null,
                Convert_ToFileTexCoordGen_( appTexMap.ITexGen ),
                appTexMap.SlotIdx );

            return dstTexMap;
        }
        // TexMap
        internal AppData.IMaterialTexMap Convert_MaterialTexMap(
            IMaterial ownerMaterial,
            FileFmtMaterialTexMap fileMtl )
        {
            MaterialTexMap appMtl = new AppData.MaterialTexMap(
                fileMtl._texMap.imageName,
                ownerMaterial as AppData.Material,
                fileMtl._textureIndex );

            Convert_Pane_MaterialTexture_(
                appMtl,
                fileMtl._texMap,
                fileMtl._texBlendRatio,
                fileMtl._texCoordGen );

            return appMtl;
        }

        // TexMtx
        internal FileFmt.TexMatrix Convert_ToFileTexMtx( AppData.ITexMtx appMtx )
        {
            return Convert_ToFileTexMtx_( appMtx );
        }
        // TexMtx
        internal void Convert_Pane_MaterialTexMtx( AppData.TexMtx dstMtx, FileFmt.TexMatrix srcMtx )
        {
            Convert_Pane_MaterialTexMtx_( dstMtx, srcMtx );
        }


#endregion

#endregion

#region private

#region FileFmt.Head ヘッダ情報関連
        /// <summary>
        /// FileFmt.HeadCreate へと変換します。
        /// </summary>
        FileFmt.HeadCreate Convert_headCreate_( object srcData )
        {
            FileFmt.HeadCreate result = new FileFmt.HeadCreate();

            result.date = DateTime.Now;

            result.host = Environment.MachineName;
            result.user = Environment.UserName;

            result.source = string.Empty;

            return result;
        }

        /// <summary>
        /// FileFmt.HeadGenerator へと変換します。
        /// </summary>
        FileFmt.HeadGenerator Convert_headGenerator_( object srcData )
        {
            FileFmt.HeadGenerator rusult = new FileFmt.HeadGenerator();

            rusult.name = LayoutEditorCore.AppName;
            rusult.version = LayoutEditorCore.Version;

            return rusult;
        }

        /// <summary>
        /// FileFmt.Head へと変換します。
        /// </summary>
        FileFmt.Head Convert_head_( DocumentHeader srcData )
        {
            FileFmt.Head result = new FileFmt.Head();

            result.comment = srcData.Comment;
            result.title = srcData.Title;

            // srcData にも代入
            result.create = srcData.Create = srcData.Create ?? Convert_headCreate_( null );
            result.generator = srcData.Generator = srcData.Generator ?? Convert_headGenerator_( null );

            return result;
        }

        private void Convert_head_(DocumentHeader documentHeader, Head head)
        {
            documentHeader.Comment = head.comment;
            documentHeader.Title = head.title;
            documentHeader.Create = head.create;
            documentHeader.Generator = head.generator;
        }


#endregion FileFmt.Head ヘッダ情報関連

#endregion
    }

    /// <summary>
    /// パネル名が重複した用の例外クラス
    /// </summary>
    public class RepetitionPaneNameException : ApplicationException
    {
        private string[]        _names = null;

        public RepetitionPaneNameException( string[] names)
        {
            _names = names;
        }

        public string[] Names
        {
            get { return _names; }
        }

        public override string ToString()
        {
            string message = LECoreStringResMgr.Get( "LECORE_RLYTLOAD_ERROR_REPETITION_PANE_NAME" );

            string names = string.Empty;
            foreach( string name in Names )
            {
                names += name + "\r\n";
            }

            return String.Format( message, names);
        }
    }
}
