﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Linq;

using System.Collections;
using System.Diagnostics;

namespace LECore.Save_Load
{
    using Structures;
    using Structures.Core;
    using Structures.SerializableObject.Lan;
    using static RlanHelper;
    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lan;

    class RltoConverter
    {
        #region ------------ Rlto 書き出し

        /// <summary>
        /// Rlfs の エクスポート
        /// </summary>
        static public FileFmt.AnimContent[] SaveAnimation(AppData.IPane srcPane, string tag)
        {
            return ConvertToAnimContents<TextBox, FileFmt.AnimPerCharacterTransformEvalTarget>(
                tag,
                ParamaterKind.Animation_PerCharacterTransformCurve,
                srcPane,
                x => x.ITextBox is TextBox ? new[] { new AnimContentSource<TextBox>(x.PaneName, (TextBox)x.ITextBox) } : Enumerable.Empty<AnimContentSource<TextBox>>(),
                x => new[] { new AnimTargetSource(x.TextBoxAnmAttr.PerCharTimeOffsetAnmAttr, AnimTargetType.PerCharacterTransformEvalTimeOffset),
                             new AnimTargetSource(x.TextBoxAnmAttr.PerCharTimeWidthAnmAttr, AnimTargetType.PerCharacterTransformEvalTimeWidth) }
                ).ToArray();
        }

        //------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

        /// <summary>
        ///
        /// </summary>
        static public FileFmt.AnimContent[] SavePerCharTransCurveAnim(AppData.IPane srcPane, string tag)
        {
            return ConvertToAnimContents<TextBox, AnimPerCharacterTransformCurveTarget>(
                tag,
                ParamaterKind.Animation_PerCharacterTransformCurve,
                srcPane,
                x => x.ITextBox is TextBox ? new[] { new AnimContentSource<TextBox>(x.PaneName, (TextBox)x.ITextBox) } : Enumerable.Empty<AnimContentSource<TextBox>>(),
                x => GetAnimTargetSourcesFromSubAttributes(x.PerCharTransformSAttr, new[] { AnimTargetType.PerCharacterTransformScaleX, AnimTargetType.PerCharacterTransformScaleY })
                    .Concat(GetAnimTargetSourcesFromSubAttributes(x.PerCharTransformTAttr, new [] { AnimTargetType.PerCharacterTransformTranslateX, AnimTargetType.PerCharacterTransformTranslateY, AnimTargetType.PerCharacterTransformTranslateZ }))
                    .Concat(GetAnimTargetSourcesFromSubAttributes(x.PerCharTransformRAttr, new [] { AnimTargetType.PerCharacterTransformRotateX, AnimTargetType.PerCharacterTransformRotateY, AnimTargetType.PerCharacterTransformRotateZ }))
                    .Concat(GetAnimTargetSourcesFromSubAttributes(x.PerCharTransformTopColorAttr, new [] { AnimTargetType.PerCharacterTransformLT_r, AnimTargetType.PerCharacterTransformLT_g, AnimTargetType.PerCharacterTransformLT_b, AnimTargetType.PerCharacterTransformLT_a }))
                    .Concat(GetAnimTargetSourcesFromSubAttributes(x.PerCharTransformBottomColorAttr, new[] { AnimTargetType.PerCharacterTransformLB_r, AnimTargetType.PerCharacterTransformLB_g, AnimTargetType.PerCharacterTransformLB_b, AnimTargetType.PerCharacterTransformLB_a }))
                ).ToArray();
        }
        #endregion -------------- Rlto


        #region ----------- Rlto 読み込み

        /// <summary>
        /// マテリアルカラー（ハード固有表現）
        /// </summary>
        static public void Load(object dstObject, FileFmt.AnimContent animContent, string tag)
        {
            Pane pane = dstObject as Pane;
            if (pane == null)
            {
                return;
            }

            var ownerTextBox = pane.ITextBox as TextBox;
            if (ownerTextBox == null)
            {
                return;
            }

            if (pane.PaneName != animContent.name)
            {
                return;
            }

            foreach (FileFmt.AnimTarget animTarget in animContent.Items)
            {
                AnmAttribute targetAttr = GetLoadTargetAttr_(animTarget.target, ownerTextBox);
                if (targetAttr != null)
                {
                    RlanHelper.SetHermiteKeyFrame_(targetAttr, animTarget, tag);
                }
            }
        }

        /// <summary>
        /// 対象アトリビュートの発見
        /// </summary>
        static AnmAttribute GetLoadTargetAttr_(AnimTargetType targetType, TextBox ownerTextBox)
        {
            if (ownerTextBox == null)
            {
                return null;
            }

            switch (targetType)
            {
                case FileFmt.AnimTargetType.PerCharacterTransformEvalTimeOffset: return ownerTextBox.TextBoxAnmAttr.PerCharTimeOffsetAnmAttr;
                case FileFmt.AnimTargetType.PerCharacterTransformEvalTimeWidth: return ownerTextBox.TextBoxAnmAttr.PerCharTimeWidthAnmAttr;
                default: return null;
            }
        }

        //------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

        /// <summary>
        /// AnimTargetType に相応しい、アニメーションアトリビュートを取得します。
        /// TODO:
        /// アニメーションアトリビュートは名前で検索可能なので、
        /// 名前でアトリビュートを取得するように変更したい。
        /// セーブフォーマットにアトリビュート名を記述するようにしたい。
        /// </summary>
        /// <param name="type"></param>
        /// <returns></returns>
        static AnmAttribute GetPerCharTransformTargetAnmAttribute_(AppData.Pane dstPane, FileFmt.AnimTargetType type)
        {
            if (dstPane.ITextBox == null)
            {
                return null;
            }

            TextBox tbx = (dstPane.ITextBox as TextBox);

            switch (type)
            {
                case FileFmt.AnimTargetType.PerCharacterTransformScaleX: return tbx.PerCharTransformSAttr.FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.PerCharacterTransformScaleY: return tbx.PerCharTransformSAttr.FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.PerCharacterTransformTranslateX: return tbx.PerCharTransformTAttr.FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.PerCharacterTransformTranslateY: return tbx.PerCharTransformTAttr.FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.PerCharacterTransformTranslateZ: return tbx.PerCharTransformTAttr.FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.PerCharacterTransformRotateX: return tbx.PerCharTransformRAttr.FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.PerCharacterTransformRotateY: return tbx.PerCharTransformRAttr.FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.PerCharacterTransformRotateZ: return tbx.PerCharTransformRAttr.FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.PerCharacterTransformLT_r: return tbx.PerCharTransformTopColorAttr.FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.PerCharacterTransformLT_g: return tbx.PerCharTransformTopColorAttr.FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.PerCharacterTransformLT_b: return tbx.PerCharTransformTopColorAttr.FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.PerCharacterTransformLT_a: return tbx.PerCharTransformTopColorAttr.FindAttributeByIdx(3);
                case FileFmt.AnimTargetType.PerCharacterTransformLB_r: return tbx.PerCharTransformBottomColorAttr.FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.PerCharacterTransformLB_g: return tbx.PerCharTransformBottomColorAttr.FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.PerCharacterTransformLB_b: return tbx.PerCharTransformBottomColorAttr.FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.PerCharacterTransformLB_a: return tbx.PerCharTransformBottomColorAttr.FindAttributeByIdx(3);

                default: Debug.Assert(false); return null;
            }
        }

        /// <summary>
        /// シリアライズデータ => アプリケーションデータ
        /// </summary>
        static public void LoadRltcToPane(object dstObject, FileFmt.AnimContent srcAnimContent, string tag)
        {
            Pane dstPane = dstObject as Pane;
            if (dstPane == null)
            {
                return;
            }

            if (dstPane.PaneName != srcAnimContent.name)
            {
                return;
            }

            foreach (FileFmt.AnimTarget srcAnimTraget in srcAnimContent.Items)
            {
                AnmAttribute dstAttr = GetPerCharTransformTargetAnmAttribute_(dstPane, srcAnimTraget.target);
                Debug.Assert(dstAttr != null);
                RlanHelper.SetHermiteKeyFrame_(dstAttr, srcAnimTraget, tag);
            }
        }
        #endregion --------- Rlto
    }
}
