﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Save_Load;
    using LECore.Structures.Core.Command;

    using RlytConverter = LECore.Save_Load.RlytConverter;
    using TextBoxCommandFactory = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.ITextBox>;
    using Structures.Core;

    /// <summary>
    /// TextBoxManipulator の概要の説明です。
    /// </summary>
    public class TextBoxManipulator : BaseManipulator
    {
        /// <summary>
        ///
        /// </summary>
        static readonly RlytConverter _RlytConverter =
        new RlytConverter( string.Empty, LECore.LayoutEditorCore.MsgReporter );

        // 操作対象
        TextBox          _targetTextBox = null;

        /// <summary>
        /// Undoコマンド生成クラス。
        /// </summary>
        TextBoxCommandFactory _commandFactory;

        /// <summary>
        /// 変更コマンドを登録しつつ、パラメータを更新します。
        /// 任意のValueChangeFunctionでパラメータを更新してください。
        /// </summary>
        void ValueChangeWithCommandRegister_( TextBoxCommandFactory.InstanceModifyFunction valueChangeFunction )
        {
            _commandFactory.Modify( _targetTextBox, valueChangeFunction );
        }

        /// <summary>
        /// フォントマネージャ
        /// </summary>
        ILEFontManager _ILEFontManager
        {
            get { return _targetTextBox.OwnerPane.OwnerSubScene.ILEFontManager; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextBoxManipulator()
        {
            _commandFactory = new TextBoxCommandFactory
                (
                 delegate ( ITextBox mementoSrc ){return new TextBoxMemento( mementoSrc );}
                 );
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        /// <param name="targetTextBox"></param>
        public void BindTarget( ITextBox targetTextBox )
        {
            _targetTextBox = targetTextBox as TextBox;
            Debug.Assert( _targetTextBox != null );
        }

        /// <summary>
        /// 操作対象
        /// </summary>
        public ITextBox ITextBox
        {
            get{ return _targetTextBox;}
        }

        /// <summary>
        /// 内容
        /// </summary>
        public string ContentsText
        {
            set{
                if( _targetTextBox.ContentsText != value )
                {
                    ValueChangeWithCommandRegister_( delegate()
                    {
                        _targetTextBox.ContentsText = value;
                    } );
                }
            }
        }

        /// <summary>
        /// テキストID
        /// </summary>
        public string TextID
        {
            set
            {
                if (_targetTextBox.TextID != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.TextID = value;
                    });
                }
            }
        }

        /// <summary>
        /// 上端色
        /// </summary>
        public RGBAColor TopColor
        {
            set{
                if( _targetTextBox.TopColor != value )
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_targetTextBox.TopColorAnmAttr, value);
                }
            }
        }

        /// <summary>
        /// 下端色
        /// </summary>
        public RGBAColor BottomColor
        {
            set
            {
                if( _targetTextBox.BottomColor != value )
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_targetTextBox.BottomColorAnmAttr, value);
                }
            }
        }

        /// <summary>
        /// フォントサイズ
        /// </summary>
        public FVec2  FontSize
        {
            set
            {
                if( _targetTextBox.FontSize != value )
                {
                    ValueChangeWithCommandRegister_( delegate()
                    {
                        _targetTextBox.FontSize = value;
                    } );
                }
            }
        }

        /// <summary>
        /// 文字間隔
        /// </summary>
        public FVec2  FontBlank
        {
            set
            {
                if( _targetTextBox.CharSpace != value.X ||
                    _targetTextBox.LineSpace != value.Y )
                {
                    ValueChangeWithCommandRegister_( delegate()
                    {
                        _targetTextBox.CharSpace = value.X;
                        _targetTextBox.LineSpace = value.Y;
                    } );
                }
            }
        }

        /// <summary>
        /// ランタイム確保メモリ量
        /// </summary>
        public uint AllocByte
        {
            set
            {
                if( _targetTextBox.AllocStringLength != value )
                {
                    ValueChangeWithCommandRegister_( delegate()
                    {
                        _targetTextBox.AllocStringLength = value;
                    } );
                }
            }
        }

        /// <summary>
        /// 文字列の長さ
        /// </summary>
        public bool SpecifiedStringLength
        {
            set
            {
                if( _targetTextBox.SpecifiedStringLength != value )
                {
                    ValueChangeWithCommandRegister_( delegate()
                    {
                        _targetTextBox.SpecifiedStringLength = value;
                    } );
                }
            }
        }

        /// <summary>
        /// 位置：水平
        /// </summary>
        public HorizontalLocation BasePosTypeH
        {
            set
            {
                if( _targetTextBox.BasePosTypeH != value )
                {
                    ValueChangeWithCommandRegister_( delegate()
                    {
                        _targetTextBox.BasePosTypeH = value;
                    } );
                }
            }
        }

        /// <summary>
        /// 位置：垂直
        /// </summary>
        public VerticalLocation BasePosTypeV
        {
            set
            {
                if( _targetTextBox.BasePosTypeV != value )
                {
                    ValueChangeWithCommandRegister_( delegate()
                    {
                        _targetTextBox.BasePosTypeV = value;
                    } );
                }
            }
        }

        /// <summary>
        /// 行そろえ位置
        /// </summary>
        public LineAlignment LineAlignment
        {
            set
            {
                if( _targetTextBox.LineAlignment != value )
                {
                    ValueChangeWithCommandRegister_( delegate()
                    {
                        _targetTextBox.LineAlignment = value;
                    } );
                }
            }
        }

        /// <summary>
        /// 中央揃えのにじみ防止機能
        /// </summary>
        public bool CenterCeilingEnabled
        {
            set
            {
                if (_targetTextBox.CenterCeilingEnabled != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.CenterCeilingEnabled = value;
                    });
                }
            }
        }

        /// <summary>
        /// 右側の文字から描画する機能
        /// </summary>
        public bool DrawFromRightToLeft
        {
            set
            {
                if (_targetTextBox.DrawFromRightToLeft != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.DrawFromRightToLeft = value;
                    });
                }
            }
        }

        /// <summary>
        /// 斜体傾き係数
        /// </summary>
        public float ItalicFactor
        {
            set
            {
                if (_targetTextBox.ItalicFactor != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.ItalicFactor = value;
                    });
                }
            }
        }

        /// <summary>
        /// 影が有効かどうか
        /// </summary>
        public bool ShadowEnabled
        {
            set
            {
                if (_targetTextBox.ShadowEnabled != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.ShadowEnabled = value;
                    });
                }
            }
        }

        /// <summary>
        /// 影のオフセット
        /// </summary>
        public FVec2 ShadowOffset
        {
            set
            {
                if (_targetTextBox.ShadowOffset != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.ShadowOffset = value;
                    });
                }
            }
        }

        /// <summary>
        /// 影のスケール
        /// </summary>
        public FVec2 ShadowScale
        {
            set
            {
                if (_targetTextBox.ShadowScale != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.ShadowScale = value;
                    });
                }
            }
        }

        /// <summary>
        /// 影斜体傾き係数
        /// </summary>
        public float ShadowItalicFactor
        {
            set
            {
                if (_targetTextBox.ShadowItalicFactor != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.ShadowItalicFactor = value;
                    });
                }
            }
        }

        /// <summary>
        /// 影上端色
        /// </summary>
        public RGBAColor ShadowTopColor
        {
            set
            {
                if (_targetTextBox.ShadowTopColor != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.ShadowTopColor = value;
                    });
                }
            }
        }

        /// <summary>
        /// 影下端色
        /// </summary>
        public RGBAColor ShadowBottomColor
        {
            set
            {
                if (_targetTextBox.ShadowBottomColor != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.ShadowBottomColor = value;
                    });
                }
            }
        }

        /// <summary>
        /// 影黒カラー補間
        /// </summary>
        public RGBAColor ShadowBlackColor
        {
            set
            {
                if (_targetTextBox.ShadowBlackColor != value)
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd((AnmAttribute)_targetTextBox.ShadowBlackColorIAnmAttr, value);
                }
            }
        }

        /// <summary>
        /// 影白カラー補間
        /// </summary>
        public RGBAColor ShadowWhiteColor
        {
            set
            {
                if (_targetTextBox.ShadowWhiteColor != value)
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd((AnmAttribute)_targetTextBox.ShadowWhiteColorIAnmAttr, value);
                }
            }
        }

        /// <summary>
        /// 文字変化：オフセット時間
        /// </summary>
        public float PerCharacterTransformTimeOffset
        {
            set
            {
                if (_targetTextBox.PerCharacterTransformTimeOffset != value)
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmdAsFloat((AnmAttribute)_targetTextBox.PerCharTimeOffsetAnmAttr, value);
                }
            }
        }

        /// <summary>
        /// 文字変化：時間幅
        /// </summary>
        public float PerCharacterTransformTimeWidth
        {
            set
            {
                if (_targetTextBox.PerCharacterTransformTimeWidth != value)
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmdAsFloat((AnmAttribute)_targetTextBox.PerCharTimeWidthAnmAttr, value);
                }
            }
        }

        /// <summary>
        /// 文字変化：ループ
        /// </summary>
        public PerCharTransformLoopType PerCharacterTransformLoopType
        {
            set
            {
                if (_targetTextBox.PerCharacterTransformLoopType != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.PerCharacterTransformLoopType = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字変化：原点（垂直）
        /// </summary>
        public VerticalLocation PerCharacterTransformOriginV
        {
            set
            {
                if (_targetTextBox.PerCharacterTransformOriginV != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.PerCharacterTransformOriginV = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字変化：原点のオフセット（垂直）
        /// </summary>
        public float PerCharacterTransformOriginVOffset
        {
            set
            {
                if (_targetTextBox.PerCharacterTransformOriginVOffset != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.PerCharacterTransformOriginVOffset = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字変化：文字幅による分割
        /// </summary>
        public bool PerCharacterTransformSplitByCharWidth
        {
            set
            {
                if (_targetTextBox.PerCharacterTransformSplitByCharWidth != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.PerCharacterTransformSplitByCharWidth = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字変化：影のアルファを文字のアルファに自動追従させる
        /// </summary>
        public bool PerCharacterTransformAutoShadowAlpha
        {
            set
            {
                if (_targetTextBox.PerCharacterTransformAutoShadowAlpha != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.PerCharacterTransformAutoShadowAlpha = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字変化：各文字の原点をテキストボックスの中心にする
        /// </summary>
        public bool PerCharacterTransformOriginToCenter
        {
            set
            {
                if (_targetTextBox.PerCharacterTransformOriginToCenter != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.PerCharacterTransformOriginToCenter = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字変化の有効無効
        /// </summary>
        public bool IsPerCharTransformEnabled
        {
            set
            {
                if (_targetTextBox.IsPerCharTransformEnabled != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.IsPerCharTransformEnabled = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの簡単設定
        /// </summary>
        public EasySettingOfPerCharTransform[] EasySettingOfPerCharTransform
        {
            set
            {
                if (_targetTextBox.EasySettingOfPerCharTransform != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.EasySettingOfPerCharTransform = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの文字幅固定機能の有効・無効
        /// </summary>
        public bool IsPerCharTransformFixSpaceEnabled
        {
            set
            {
                if (_targetTextBox.IsPerCharTransformFixSpaceEnabled != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.IsPerCharTransformFixSpaceEnabled = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの文字幅固定の文字幅
        /// </summary>
        public float PerCharTransformFixSpaceWidth
        {
            set
            {
                if (_targetTextBox.PerCharTransformFixSpaceWidth != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.PerCharTransformFixSpaceWidth = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの文字幅固定の文字揃え
        /// </summary>
        public HorizontalLocation PerCharTransformFixSpaceOrigin
        {
            set
            {
                if (_targetTextBox.PerCharTransformFixSpaceOrigin != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.PerCharTransformFixSpaceOrigin = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の開始角度
        /// </summary>
        public float PerCharTransformEasyCircleBeginAngle
        {
            set
            {
                if (_targetTextBox.PerCharTransformEasyCircleBeginAngle != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.PerCharTransformEasyCircleBeginAngle = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の終了角度
        /// </summary>
        public float PerCharTransformEasyCircleEndAngle
        {
            set
            {
                if (_targetTextBox.PerCharTransformEasyCircleEndAngle != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.PerCharTransformEasyCircleEndAngle = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の円の半径X
        /// </summary>
        public float PerCharTransformEasyCircleRadiusX
        {
            set
            {
                if (_targetTextBox.PerCharTransformEasyCircleRadiusX != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.PerCharTransformEasyCircleRadiusX = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の円の半径Y
        /// </summary>
        public float PerCharTransformEasyCircleRadiusY
        {
            set
            {
                if (_targetTextBox.PerCharTransformEasyCircleRadiusY != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.PerCharTransformEasyCircleRadiusY = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の分割数
        /// </summary>
        public int PerCharTransformEasyCircleDivisionNumber
        {
            set
            {
                if (_targetTextBox.PerCharTransformEasyCircleDivisionNumber != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.PerCharTransformEasyCircleDivisionNumber = value;
                    });
                }
            }
        }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の文字の向き
        /// </summary>
        public float PerCharTransformEasyCircleCharAngle
        {
            set
            {
                if (_targetTextBox.PerCharTransformEasyCircleCharAngle != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.PerCharTransformEasyCircleCharAngle = value;
                    });
                }
            }
        }

        /// <summary>
        /// 縁取りOFF
        /// </summary>
        public bool InvisibleBorderEnabled
        {
            set
            {
                if (_targetTextBox.InvisibleBorderEnabled != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.InvisibleBorderEnabled = value;
                    });
                }
            }
        }

        /// <summary>
        /// 2度書き縁取り
        /// </summary>
        public bool DoubleDrawnBorderEnabled
        {
            set
            {
                if (_targetTextBox.DoubleDrawnBorderEnabled != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.DoubleDrawnBorderEnabled = value;
                    });
                }
            }
        }

        /// <summary>
        /// 行ごとの幅とオフセットの有効無効
        /// </summary>
        public bool LineWidthOffsetEnabled
        {
            set
            {
                if (_targetTextBox.LineWidthOffsetEnabled != value)
                {
                    ValueChangeWithCommandRegister_(delegate()
                    {
                        _targetTextBox.LineWidthOffsetEnabled = value;
                    });
                }
            }
        }

        /// <summary>
        /// 行ごとの幅
        /// </summary>
        public void SetLineWidth(int index, float value)
        {
            if (_targetTextBox.GetLineWidth(index) != value)
            {
                ValueChangeWithCommandRegister_(delegate()
                {
                    _targetTextBox.SetLineWidth(index, value);
                });
            }
        }

        /// <summary>
        /// 行ごとのオフセット
        /// </summary>
        public void SetLineOffset(int index, float value)
        {
            if (_targetTextBox.GetLineOffset(index) != value)
            {
                ValueChangeWithCommandRegister_(delegate()
                {
                    _targetTextBox.SetLineOffset(index, value);
                });
            }
        }

        /// <summary>
        /// 拡張タグの解釈の有効無効
        /// </summary>
        public bool IsExtendedTagEnabled
        {
            set
            {
                if (_targetTextBox.IsExtendedTagEnabled != value)
                {
                    ValueChangeWithCommandRegister_(delegate ()
                    {
                        _targetTextBox.IsExtendedTagEnabled = value;
                    });
                }
            }
        }

#if true
        /// <summary>
        ///
        /// </summary>
        public bool SameNameButDifferentPaths( string filePath, out ILEFont font)
        {
            font = _ILEFontManager.FindFontByName( filePath);


            if( font != null )
            {
                // プロジェクトで設定されたフォントは別物
                if (font.FontSettings != null && !font.IsComplexFont())
                {
                    return true;
                }

                string pathA = Path.GetFullPath( filePath).ToLower();
                string pathB = Path.GetFullPath( font.FontPath).ToLower();

                if( pathA != pathB )
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public bool ReplaceFontFilePath( string filePath)
        {
            LEFontManagerManipulator fontMgrMnp = new LEFontManagerManipulator();
            fontMgrMnp.BindTarget( _ILEFontManager);

            fontMgrMnp.UpdateFontFilePath( _targetTextBox.ILEFont, filePath);
            return true;
        }
#endif

        /// <summary>
        ///
        /// </summary>
        public bool SetLEFontByFileName(string filePath)
        {
            return SetLEFontByFileName(Path.GetFileName(filePath), filePath);
        }

        /// <summary>
        /// ファイル名を指定して、フォントを変更します。
        /// </summary>
        /// <returns>フォント変更が起こったらtrue</returns>
        public bool SetLEFontByFileName( string fontName, string filePath )
        {
            ILEFont newFont = _ILEFontManager.CreateILEFont(fontName, filePath, null);
            if( newFont != null )
            {
                if( _targetTextBox.ILEFont == null )
                {
                    _targetTextBox.FontName = String.Empty;
                }

                if( _targetTextBox.FontName != newFont.FontName )
                {
                    ValueChangeWithCommandRegister_( delegate()
                    {
                        if (LayoutEditorCore.KeepingSizeScaleEnabled)
                        {
                            // スケールを維持したままフォントを変更する。
                            // UI で設定できる精度以下の値を切り捨てる。
                            FVec2 scale = _targetTextBox.FontScale;
                            _targetTextBox.FontSize = new FVec2(
                                (float)Math.Round(newFont.Width * scale.X, 2),
                                (float)Math.Round(newFont.Height * scale.Y, 2)
                            );
                        }
                        _targetTextBox.FontName = newFont.FontName;
                    } );
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// フォント更新イベントを起こす
        /// </summary>
        public void RaiseChangeEvent()
        {
            (_ILEFontManager as FontManager).RaiseChangeEvent();
        }

        /// <summary>
        /// 内容文字列ぴったりにペインサイズを調整します。
        /// </summary>
        public void AdjustPaneSize()
        {
            // ペインに対する操作なので、ペイン編集コマンドを生成します。
            IPane before = _RlytConverter.MakePaneClone( (_targetTextBox as TextBox).OwnerPane );

            _targetTextBox.AdjustPaneSize();

            IPane after = _RlytConverter.MakePaneClone( ( _targetTextBox as TextBox ).OwnerPane );

            _CommandFactory.MakePaneCopyCmd( ( _targetTextBox as TextBox ).OwnerPane, before, after );
        }

    }
}
