﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Drawing;
using System.Diagnostics;
using System.Text.RegularExpressions;
using System.Linq;
using System.Windows.Forms;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.Core.Command;
    using System.IO;
    using LECore.Structures.LECoreInterface;
    using AnimationCurves = LECore.Structures.Core.AnmAttribute.AnimationCurves;

    /// <summary>
    /// SubSceneManipulator の概要の説明です。
    /// </summary>
    public partial class SubSceneManipulator :
        BaseManipulator
    {
        #region アニメーションモード切り替え

        /// <summary>
        /// アニメーションカーブのコンバートに必要なパラメータを集約したクラスです。
        /// </summary>
        internal class AnmConvertParam
        {
            public SubScene Target { get; set; }
            public bool IsAnimEditSeparate { get; set; }
            public List<AttributeSetParam> AttrParams { get; }

            internal class AttributeSetParam
            {
                public AnmAttribute Attribute { get; }
                public AnimationCurves KeyData { get; }
                public AnimationCurves Curves { get; }

                public AttributeSetParam(AnmAttribute owner, AnimationCurves curves, AnimationCurves keydata)
                {
                    Attribute = owner;
                    Curves = curves;
                    KeyData = keydata;
                }
            }

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public AnmConvertParam()
            {
                AttrParams = new List<AttributeSetParam>();
            }

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public AnmConvertParam(SubScene target, bool isAnimEditSeparate)
                : this()
            {
                Target = target;
                IsAnimEditSeparate = isAnimEditSeparate;
            }

            /// <summary>
            /// アトリビュートのパラメータセットを追加します。
            /// </summary>
            public void AddAttrParams(AnmAttribute owner, AnimationCurves curves, AnimationCurves keydata)
            {
                AttrParams.Add(new AttributeSetParam(owner, curves, keydata));
            }

            /// <summary>
            /// アトリビュートのパラメータセットを追加します。
            /// </summary>
            public void AddAttrParams(AnmAttribute owner, AnimationCurves curves)
            {
                AttrParams.Add(new AttributeSetParam(owner, curves, null));
            }
        }

        /// <summary>
        /// アニメーションモードを切り替えます。
        /// </summary>
        public void AnmConvert()
        {
            // アニメーション区間タグの操作対象をセット
            _animFrameSectionSetMnp.BindTarget(_target.IAnimFrameSectionSet);

            if (!_target.IsAnimEditSeparate)
            {
                // 分割管理モードにコンバート
                AnmConvertSingleToMulti();
            }
            else
            {
                // 一括管理モードにコンバート
                AnmConvertMultiToSingle();
            }
        }

        /// <summary>
        /// 単数カーブを複数カーブへコンバートします。
        /// </summary>
        public void AnmConvertSingleToMulti()
        {
            // ワーク
            IEnumerable<IPane> paneArray = _target.IPaneArray;
            IAnimFrameSectionSet sectionSet = _target.IAnimFrameSectionSet;
            AnmConvertParam beforeVal = new AnmConvertParam(_target, _target.IsAnimEditSeparate); // 変更前のパラメータを保存
            AnmConvertParam afterVal = new AnmConvertParam();

            _target.BeginMassiveModify();

            // 区間タグが無い場合はエラーとする
            if (sectionSet == null ||
                sectionSet.IAnimFrameSectionSet == null ||
                sectionSet.IAnimFrameSectionSet.Count() <= 0)
            {
                // キーがある場合は区間を追加
                int width;
                if (HasAnyKey_(paneArray, out width))
                {
                    _animFrameSectionSetMnp.AddNewFrameSection(AnimFrameSectionSetHelper.DefaultFrameSectionName, "", 0, width, _target);
                }
            }

            // 全てのペインを対象とする
            foreach (IPane pane in paneArray)
            {
                // カーブを持つアトリビュートを列挙し、走査する
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    // 変更前のカーブの状態を保存
                    beforeVal.AddAttrParams(curveAttr, GetCurveData_(curveAttr), GetKeydata_(curveAttr));

                    // アニメーションのコンバートを行う
                    AnmConvertSingleToMulti_(curveAttr, sectionSet);

                    // 変更後のカーブの状態を保存
                    afterVal.AddAttrParams(curveAttr, GetCurveData_(curveAttr), GetKeydata_(curveAttr));
                }
            }

            // アニメーションモードを変更する
            _target.ChangeAnimEditMode(true);

            // 変更後のパラメータを保存
            afterVal.Target = _target;
            afterVal.IsAnimEditSeparate = _target.IsAnimEditSeparate;

            // コマンド送信
            _CommandFactory.MakeAnmModeCmd(beforeVal, afterVal);

            foreach (IAnimFrameSection section in sectionSet.IAnimFrameSectionSet)
            {
                // アニメーション区間タグの変更
                _animFrameSectionSetMnp.SetTagBasic(
                    section, section.Name,
                    section.Comment,
                    0,
                    section.EndFrame - section.StartFrame);
            }

            _target.EndMassiveModify();

            return;
        }

        /// <summary>
        /// 単数カーブを複数カーブへコンバートします。
        /// </summary>
        private void AnmConvertSingleToMulti_(
            AnmAttribute attr,
            IAnimFrameSectionSet sectionSet)
        {
            // タグ区間ごとにキーフレームを抽出する
            List<AnimFrameSection> animFrameSectionList = new List<AnimFrameSection>();
            foreach (AnimFrameSection animFrameSection in sectionSet.IAnimFrameSectionSet)
            {
                animFrameSectionList.Add(animFrameSection);
            }

            AnmCurve srcCurve = attr.IDefaultCurve as AnmCurve;
            var value = srcCurve.Evaluate(0);
            attr.SetBaseValueAsFloat(value);
            bool isFirstSection = true;
            int endIndex = (attr.IsReferenceCurves && animFrameSectionList.Count > 0) ? 1 : animFrameSectionList.Count;
            for (int index = 0; index < endIndex; index++)
            {
                // 現在のセクションと次のセクションを取得しておく
                AnimFrameSection currentSection = animFrameSectionList[index];
                AnimFrameSection nextSection = (index + 1 < animFrameSectionList.Count) ?
                    animFrameSectionList[index + 1] : null;

                // アニメーションカーブをクローンする
                AnmCurve newCurve = srcCurve.Clone(AnmCurve.KeyCloneType.None);

                // 最初のキーフレームを取得する
                AnmKeyFrame firstKeyFrame = isFirstSection ?
                    srcCurve.GetFirstKey() as AnmKeyFrame : GetFirstKeyFrame_(attr, currentSection, true);
                if (firstKeyFrame == null)
                {
                    attr.RegistAnmCurve(currentSection.Name, newCurve); // 空のリストを登録
                    continue;
                }
                isFirstSection = false; // 最初のキーが見つかった

                // 最初のキーフレームを登録
                AddKeyFrame_(newCurve, firstKeyFrame, true);

                // 残りのキーフレームを拾う
                AnmKeyFrame currentKey = firstKeyFrame;
                while (true)
                {
                    // 指定キーフレームの次のキーフレームを拾う
                    AnmKeyFrame nextKey = srcCurve.GetNextKeyFrame(currentKey) as AnmKeyFrame;
                    if (nextKey == null)
                    {
                        // キーフレームが見つからなければ終了
                        break;
                    }
                    currentKey = nextKey;

                    // 終了条件
                    if (!attr.IsReferenceCurves && nextSection != null)
                    {
                        // 次のタグ区間の最初のキーより時間が大きい
                        AnmKeyFrame nextFirstKey = GetFirstKeyFrame_(attr, nextSection, false);
                        if (nextFirstKey == null)
                        {
                            // キーフレームを登録
                            AddKeyFrame_(newCurve, nextKey, true);
                            break;
                        }
                        if (nextKey.Time >= nextFirstKey.Time)
                        {
                            // 最後に拾ったキーが現在のタグ区間内であれば、区間外の最寄キーを拾う
                            if (newCurve.GetLastKey().Time >= currentSection.EndFrame)
                            {
                                break;
                            }
                        }
                    }
                    else
                    {
                        // キーが見つからなければ終了
                        if (nextKey == null)
                        {
                            break;
                        }
                    }

                    // キーフレームを登録
                    AddKeyFrame_(newCurve, nextKey, true);
                }

                // カーブの検証
                if (!ValidateCurve(newCurve, currentSection))
                {
                    // 不正なカーブはキーをすべて消す
                    ClearKeyFrame_(newCurve);
                }
                else
                {
                    // StartFrameが0になるようにオフセットする
                    int offset = currentSection.StartFrame;
                    foreach (AnmKeyFrame keyFrame in newCurve.IKeyFrameSet)
                    {
                        keyFrame.Time -= offset;
                    }
                }

                // テンポラリに登録
                attr.RegistAnmCurve(currentSection.Name, newCurve);
            }

            // 参照セット
            if (attr.IsReferenceCurves && animFrameSectionList.Count > 0)
            {
                AnmCurve curve = attr.GetAnmCurve(animFrameSectionList[0].Name);
                if (curve != null)
                {
                    for (int index = 1; index < animFrameSectionList.Count; index++)
                    {
                        attr.RegistAnmCurve(animFrameSectionList[index].Name, curve);
                    }
                }
            }

            // 一括管理用のカーブはキーフレームを消しておく
            ClearKeyFrame_(srcCurve);
        }

        /// <summary>
        /// カーブが有効かチェックします。
        /// </summary>
        private bool ValidateCurve(AnmCurve curve, AnimFrameSection section)
        {
            bool find = false;
            bool findPrev = false;
            bool findNext = false;
            foreach (AnmKeyFrame keyFrame in curve.IKeyFrameSet)
            {
                if (section.StartFrame > keyFrame.Time)
                {
                    findPrev = true;
                }
                else if (section.EndFrame < keyFrame.Time)
                {
                    findNext = true;
                }
                else
                {
                    find = true;
                }
            }

            return find || (findPrev && findNext);
        }

        /// <summary>
        /// カーブ情報を抽出します。
        /// </summary>
        private AnimationCurves GetCurveData_(AnmAttribute attr)
        {
            var curvedata = new AnimationCurves();
            foreach (string tag in SubSceneHelper.GetTags(_target))
            {
                AnmCurve curve = attr.GetAnmCurve(tag);
                if (curve != null)
                {
                    curvedata.RegistAnmCurve(tag, curve);
                }
            }

            return curvedata;
        }

        /// <summary>
        /// キー情報を抽出します。
        /// </summary>
        private AnimationCurves GetKeydata_(AnmAttribute attr, AnmCurve.KeyCloneType clone = AnmCurve.KeyCloneType.Copy)
        {
            var keydata = new AnimationCurves();
            foreach (string tag in SubSceneHelper.GetTags(_target))
            {
                AnmCurve curve = attr.GetAnmCurve(tag);
                if (curve != null)
                {
                    keydata.RegistAnmCurve(tag, curve.Clone(clone));
                }
            }

            return keydata;
        }

        /// <summary>
        /// 対象のタグ区間に属する最初のキーを探します。
        /// </summary>
        private AnmKeyFrame GetFirstKeyFrame_(AnmAttribute attr, AnimFrameSection section, bool searchForward)
        {
            // StartFrameにキーフレームが存在するか？
            AnmCurve srcCurve = attr.IDefaultCurve as AnmCurve;
            AnmKeyFrame keyFrame = srcCurve.FindKeyExistAt(section.StartFrame) as AnmKeyFrame;
            if (keyFrame == null)
            {
                // 存在しない場合直前のキーフレームを取得する
                keyFrame = srcCurve.GetPrevKeyFrame(section.StartFrame) as AnmKeyFrame;

                // 直前のキーが見つからない場合は直後のキーフレームを拾う
                if (keyFrame == null && searchForward)
                {
                    keyFrame = srcCurve.GetNextKeyFrame(section.StartFrame) as AnmKeyFrame;
                }
            }

            if (keyFrame == null)
            {
                // キーフレームの存在しないカーブ
                return null;
            }

            return keyFrame;
        }

        /// <summary>
        /// カーブにキーを追加します。
        /// </summary>
        private void AddKeyFrame_(AnmCurve curve, AnmKeyFrame key, bool clone, int timeOffset = 0)
        {
            if (clone)
            {
                curve.AddNewKeyFrameWithNoUpdate(new AnmKeyFrame(
                    curve,
                    key.Time + timeOffset,
                    key.Value,
                    key.UnifyTangents,
                    key.InTangent,
                    key.OutTangent,
                    key.InterporationType,
                    key.ViewScale,
                    false));
            }
            else
            {
                curve.AddNewKeyFrameWithNoUpdate(key);
            }
        }

        /// <summary>
        /// カーブからキーを全て削除します。
        /// </summary>
        private void ClearKeyFrame_(AnmCurve curve)
        {
            curve.ClearKeyFramesWithNoUpdate();
        }

        /// <summary>
        /// ペイン配列にキーが存在するかチェックします。
        /// </summary>
        private bool HasAnyKey_(IEnumerable<IPane> paneArray, out int width)
        {
            width = 1;

            foreach (Pane pane in paneArray)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    if (curveAttr.HasAnyKey)
                    {
                        width = curveAttr.ICurrentAnimationCurve.GetLastKey().TimeAsInt;
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// 複数カーブを単数カーブへコンバートします。
        /// </summary>
        public void AnmConvertMultiToSingle()
        {
            // ワーク
            IEnumerable<IPane> paneArray = _target.IPaneArray;
            IAnimFrameSectionSet sectionSet = _target.IAnimFrameSectionSet;
            AnmConvertParam beforeVal = new AnmConvertParam(_target, _target.IsAnimEditSeparate);　// 変更前のパラメータを保存
            AnmConvertParam afterVal = new AnmConvertParam();

            // range、marginの算出
            var maxRangeList = SubSceneHelper.CalcSectionMaxRange(paneArray, sectionSet);
            var maxMarginList = SubSceneHelper.CalcSectionMaxMargin(paneArray, sectionSet);

            // 全てのペインを対象とする
            foreach (IPane pane in paneArray)
            {
                // カーブを持つアトリビュートを列挙し、走査する
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    // 変更前のカーブの状態を保存
                    beforeVal.AddAttrParams(curveAttr, GetCurveData_(curveAttr), GetKeydata_(curveAttr));

                    // アニメーションのコンバートを行う
                    AnmConvertMultiToSingle_(curveAttr, sectionSet, maxRangeList, maxMarginList);

                    // 変更後のカーブの状態を保存
                    afterVal.AddAttrParams(curveAttr, GetCurveData_(curveAttr), GetKeydata_(curveAttr));
                }
            }

            // アニメーションモードを変更する
            _target.ChangeAnimEditMode(false);

            // 変更後のパラメータを保存
            afterVal.Target = _target;
            afterVal.IsAnimEditSeparate = _target.IsAnimEditSeparate;

            // コマンド送信
            _target.BeginMassiveModify();

            _CommandFactory.MakeAnmModeCmd(beforeVal, afterVal);

            // 区間タグの開始終了位置変更
            var sectionParam = SubSceneHelper.CalcSectionParam(sectionSet, maxRangeList, maxMarginList);
            ModifyAnimSectionTag_(sectionSet, sectionParam);

            _target.EndMassiveModify();

            return;
        }

        /// <summary>
        /// 複数カーブを単数カーブへコンバートします。
        /// </summary>
        private void AnmConvertMultiToSingle_(
            AnmAttribute attr,
            IAnimFrameSectionSet sectionSet,
            Dictionary<string, int> rangeList,
            Dictionary<string, Tuple<int, int>> marginList)
        {
            // ワーク
            AnmCurve newCurve = attr.IDefaultCurve as AnmCurve;
            ClearKeyFrame_(newCurve);

            // カーブにキーフレームを設定する
            int offset = 0;
            foreach (AnimFrameSection animFramesection in sectionSet.IAnimFrameSectionSet)
            {
                // maxRangeが無い場合はスキップ
                int maxRange;
                if (!rangeList.TryGetValue(animFramesection.Name, out maxRange))
                {
                    continue;
                }

                // marginが無い場合はスキップ
                Tuple<int, int> margin;
                if (!marginList.TryGetValue(animFramesection.Name, out margin))
                {
                    continue;
                }

                // タグ名に対応するアニメーションが無い場合はスキップ
                AnmCurve anmCurve = attr.GetAnmCurve(animFramesection.Name);
                if (anmCurve != null)
                {
                    // キーを設定する
                    foreach (AnmKeyFrame keyFrame in anmCurve.IKeyFrameSet)
                    {
                        AnmKeyFrame newKey = keyFrame;

                        // キーをセット
                        AnmKeyFrame findKeyFrame = newCurve.FindKeyExistAt(newKey.Time + offset + margin.Item1) as AnmKeyFrame;
                        if (findKeyFrame == null)
                        {
                            AddKeyFrame_(newCurve, newKey, true, offset + margin.Item1);
                        }
                    }
                }

                // オフセット更新
                offset += maxRange + _sectionMargin; // セクション幅 + マージン

                // 参照カーブの場合は即break
                if (attr.IsReferenceCurves)
                {
                    break;
                }
            }

            // preInfinity / postInfinity
            AnmCurve firstCurve = null;
            AnmCurve lastCurve = null;
            foreach (AnimFrameSection animFramesection in sectionSet.IAnimFrameSectionSet)
            {
                AnmCurve anmCurve = attr.GetAnmCurve(animFramesection.Name);
                if (anmCurve != null)
                {
                    if (firstCurve == null)
                    {
                        firstCurve = anmCurve;
                    }
                    lastCurve = anmCurve;
                }
            }
            newCurve.PreInfinityType = (firstCurve != null) ? firstCurve.PreInfinityType : AnmCurveInfinityType.Constant;
            newCurve.PostInfinityType = (lastCurve != null) ? lastCurve.PostInfinityType : AnmCurveInfinityType.Constant;

            // 複数カーブをクリア
            attr.RemoveMultiCurves();
        }

        /// <summary>
        /// アニメーション区間タグのパラメータを変更します。
        /// </summary>
        private void ModifyAnimSectionTag_(
            IAnimFrameSectionSet sectionSet,
            Dictionary<string, Tuple<int, int>> sectionParam)
        {
            foreach (IAnimFrameSection section in sectionSet.IAnimFrameSectionSet)
            {
                Tuple<int, int> param;
                if (sectionParam.TryGetValue(section.Name, out param))
                {
                    _animFrameSectionSetMnp.SetTagBasic(section, section.Name, section.Comment, param.Item1, param.Item2);
                }
            }
        }

        /// <summary>
        /// アトリビュートに区間タグアニメーションを追加します。
        /// </summary>
        public void AddAnimSection(string tagName)
        {
            Debug.Assert(_target != null);
            if (_target == null || !_target.IsAnimEditSeparate)
            {
                return;
            }

            AnmConvertParam beforeValue = new AnmConvertParam(_target, _target.IsAnimEditSeparate);
            AnmConvertParam afterValue = new AnmConvertParam(_target, _target.IsAnimEditSeparate);

            foreach (IPane pane in _target.IPaneArray)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    // before value set
                    beforeValue.AddAttrParams(curveAttr, GetCurveData_(curveAttr));

                    // value set
                    curveAttr.AddAnmTag(tagName);

                    // after value set
                    afterValue.AddAttrParams(curveAttr, GetCurveData_(curveAttr));
                }
            }

            _CommandFactory.MakeAnmAddCmd(_target, tagName, beforeValue, afterValue);
        }

        /// <summary>
        /// アトリビュートから区間タグアニメーションを削除します。
        /// </summary>
        public void RemoveAnimSection(string tagName)
        {
            Debug.Assert(_target != null);
            if (_target == null || !_target.IsAnimEditSeparate)
            {
                return;
            }

            AnmConvertParam beforeValue = new AnmConvertParam(_target, _target.IsAnimEditSeparate);
            AnmConvertParam afterValue = new AnmConvertParam(_target, _target.IsAnimEditSeparate);

            foreach (IPane pane in _target.IPaneArray)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    // before value set
                    beforeValue.AddAttrParams(curveAttr, GetCurveData_(curveAttr));

                    // value set
                    curveAttr.RemoveAnmTag(tagName);

                    // after value set
                    afterValue.AddAttrParams(curveAttr, GetCurveData_(curveAttr));
                }
            }

            _CommandFactory.MakeAnmRemoveCmd(_target, tagName, beforeValue, afterValue);
        }

        /// <summary>
        /// アトリビュートに変更された区間タグ名を設定します。
        /// </summary>
        public void ModifyTagName(
            string oldTag,
            string newTag)
        {
            Debug.Assert(_target != null);
            if (_target == null || !_target.IsAnimEditSeparate)
            {
                return;
            }

            foreach (IPane pane in _target.IPaneArray)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    curveAttr.ModifyAnmTag(oldTag, newTag);
                }
            }

            _CommandFactory.MakeModifyTagNameCmd(_target.IPaneArray, oldTag, newTag);
        }

        /// <summary>
        /// 変更された区間幅に応じてカーブにスケールを掛けます。(Float精度)
        /// </summary>
        public void AdjustAnimCurveFloat(string tagName, float scale)
        {
            Debug.Assert(_target != null);
            if (_target == null || !_target.IsAnimEditSeparate)
            {
                return;
            }

            AnmConvertParam beforeValue = new AnmConvertParam(_target, _target.IsAnimEditSeparate);
            AnmConvertParam afterValue = new AnmConvertParam(_target, _target.IsAnimEditSeparate);

            foreach (IPane pane in _target.IPaneArray)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    AnmCurve curve = curveAttr.ICurrentAnimationCurve as AnmCurve;
                    if (curve != null)
                    {
                        // before value set
                        beforeValue.AddAttrParams(curveAttr, GetCurveData_(curveAttr), GetKeydata_(curveAttr, AnmCurve.KeyCloneType.Clone));

                        // value set
                        foreach (AnmKeyFrame key in curve.IKeyFrameSet)
                        {
                            key.Time = key.Time * scale;
                        }

                        // after value set
                        afterValue.AddAttrParams(curveAttr, GetCurveData_(curveAttr), GetKeydata_(curveAttr, AnmCurve.KeyCloneType.Clone));
                    }
                }
            }

            _CommandFactory.MakeScaleAnimationCmd(beforeValue, afterValue);
        }

        /// <summary>
        /// 変更された区間幅に応じてカーブにスケールを掛けます。(Int精度)
        /// </summary>
        public void AdjustAnimCurveInt(string tagName, float scale)
        {
            Debug.Assert(_target != null);
            if (_target == null || !_target.IsAnimEditSeparate)
            {
                return;
            }

            AnmConvertParam beforeValue = new AnmConvertParam(_target, _target.IsAnimEditSeparate);
            AnmConvertParam afterValue = new AnmConvertParam(_target, _target.IsAnimEditSeparate);

            foreach (IPane pane in _target.IPaneArray)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    AnmCurve curve = curveAttr.GetAnmCurve(tagName);
                    if (curve != null)
                    {
                        // before value set
                        beforeValue.AddAttrParams(curveAttr, GetCurveData_(curveAttr), GetKeydata_(curveAttr, AnmCurve.KeyCloneType.Clone));

                        // value set
                        // 変換後の値が重複しているキーを抽出
                        List<AnmKeyFrame> RemoveKeySet = new List<AnmKeyFrame>();
                        AnmKeyFrame beforeKey = null;
                        foreach (AnmKeyFrame key in curve.IKeyFrameSet)
                        {
                            if (beforeKey != null)
                            {
                                int beforeTime = (int)Math.Round(beforeKey.Time * scale);
                                int currentTime = (int)Math.Round(key.Time * scale);

                                if (beforeTime == currentTime)
                                {
                                    RemoveKeySet.Add(key);
                                }
                            }
                            beforeKey = key;
                        }

                        // 重複するキーを削除
                        foreach (AnmKeyFrame key in RemoveKeySet)
                        {
                            curve.RemoveKeyFrame(key);
                        }

                        // スケール調整
                        foreach (AnmKeyFrame key in curve.IKeyFrameSet)
                        {
                            key.Time = (int)Math.Round(key.Time * scale);
                        }

                        // after value set
                        afterValue.AddAttrParams(curveAttr, GetCurveData_(curveAttr), GetKeydata_(curveAttr, AnmCurve.KeyCloneType.Clone));
                    }
                }
            }

            _CommandFactory.MakeScaleAnimationCmd(beforeValue, afterValue);
        }

        /// <summary>
        /// アニメーションのコンバートをアンドゥします。
        /// </summary>
        public void UndoAnmConvert(object param)
        {
            AnmConvertParam anmConvParam = param as AnmConvertParam;

            SubScene target = anmConvParam.Target;
            this.BindTarget(target);

            _target.IsAnimEditSeparate = anmConvParam.IsAnimEditSeparate;

            foreach (AnmConvertParam.AttributeSetParam attrParams in anmConvParam.AttrParams)
            {
                AnmAttribute ownerAttr = attrParams.Attribute;
                var targetCurves = attrParams.Curves;
                var targetKeyData = attrParams.KeyData;

                ownerAttr.ClearAnmCurve();
                foreach (string tag in SubSceneHelper.GetTags(_target))
                {
                    AnmCurve curve = targetCurves.GetAnmCurve(tag);
                    AnmCurve keydata = targetKeyData.GetAnmCurve(tag);

                    if (curve != null && keydata != null)
                    {
                        curve.ClearKeyFrames();
                        foreach (AnmKeyFrame key in keydata.IKeyFrameSet)
                        {
                            curve.AddNewKeyFrameWithNoUpdate(key);
                        }

                        ownerAttr.RegistAnmCurve(tag, curve);
                    }
                }
            }

            _target.NotifyAnimationEditModeChanged();
        }

        /// <summary>
        /// アニメーションのコンバートをリドゥします。
        /// </summary>
        public void RedoAnmConvert(object param)
        {
            // アンドゥと同じ処理
            UndoAnmConvert(param);
        }

        /// <summary>
        /// アニメーションの追加をアンドゥします。
        /// </summary>
        public void UndoAddAnmCurve(ISubScene subScene, object param, string tagName)
        {
            AnmConvertParam anmConvParam = param as AnmConvertParam;

            foreach (AnmConvertParam.AttributeSetParam attrParams in anmConvParam.AttrParams)
            {
                AnmAttribute ownerAttr = attrParams.Attribute;
                ownerAttr.RemoveAnmTag(tagName);
            }
        }

        /// <summary>
        /// アニメーションの追加をリドゥします。
        /// </summary>
        public void RedoAddAnmCurve(object param, string tagName)
        {
            AnmConvertParam anmConvParam = param as AnmConvertParam;

            foreach (AnmConvertParam.AttributeSetParam attrParams in anmConvParam.AttrParams)
            {
                AnmAttribute ownerAttr = attrParams.Attribute;
                AnmCurve curve = attrParams.Curves.GetAnmCurve(tagName);
                if (curve != null)
                {
                    ownerAttr.RegistAnmCurve(tagName, curve);
                }
            }
        }

        /// <summary>
        /// アニメーションの削除をアンドゥします。
        /// </summary>
        public void UndoRemoveAnmCurve(object param, string tagName)
        {
            this.RedoAddAnmCurve(param, tagName);
        }

        /// <summary>
        /// アニメーションの削除をリドゥします。
        /// </summary>
        public void RedoRemoveAnmCurve(ISubScene subScene, object param, string tagName)
        {
            this.UndoAddAnmCurve(subScene, param, tagName);
        }

        /// <summary>
        /// アニメーションの拡大縮小をアンドゥします。
        /// </summary>
        public void UndoScaleAnimation(object param)
        {
            AnmConvertParam anmConvParam = param as AnmConvertParam;

            SubScene target = anmConvParam.Target;
            this.BindTarget(target);

            foreach (AnmConvertParam.AttributeSetParam attrParams in anmConvParam.AttrParams)
            {
                AnmAttribute ownerAttr = attrParams.Attribute;

                foreach (string tag in SubSceneHelper.GetTags(_target))
                {
                    AnmCurve srcCurve = attrParams.KeyData.GetAnmCurve(tag);
                    AnmCurve dstCurve = attrParams.Curves.GetAnmCurve(tag);
                    if (srcCurve != null && dstCurve != null)
                    {
                        ModifyAnmCurve(srcCurve, dstCurve);
                    }
                    ownerAttr.RegistAnmCurve(tag, dstCurve);
                }
            }
        }

        /// <summary>
        /// アニメーションの拡大縮小をリドゥします。
        /// </summary>
        public void RedoScaleAnimation(object param)
        {
            // アンドゥと同じ処理
            UndoScaleAnimation(param);
        }

        /// <summary>
        /// 拡大縮小のアンドゥ/リドゥによるキー操作を行ないます。
        /// </summary>
        private void ModifyAnmCurve(AnmCurve srcCurve, AnmCurve dstCurve)
        {
            bool bRemove = (srcCurve.NumKeyFrame < dstCurve.NumKeyFrame);

            int key_count = srcCurve.NumKeyFrame;
            for (int i = 0; i < key_count; i++)
            {
                if (dstCurve.NumKeyFrame > i)
                {
                    AnmKeyFrame key = dstCurve.IKeyFrameSet[i] as AnmKeyFrame;
                    key.SetParamaters(srcCurve.IKeyFrameSet[i] as AnmKeyFrame);
                }
                else
                {
                    dstCurve.AddNewKeyFrame(srcCurve.IKeyFrameSet[i] as AnmKeyFrame);
                }
            }

            if (bRemove)
            {
                List<AnmKeyFrame> removeKeyList = new List<AnmKeyFrame>();
                for (int i = key_count; i < dstCurve.NumKeyFrame; i++)
                {
                    removeKeyList.Add(dstCurve.IKeyFrameSet[i] as AnmKeyFrame);
                }

                foreach (AnmKeyFrame key in removeKeyList)
                {
                    dstCurve.RemoveKeyFrame(key);
                }
            }
        }

        /// <summary>
        /// フィッティングを行ないます。
        /// </summary>
        public void FittingSection()
        {
            Debug.Assert(_target != null);
            if (_target == null || !_target.IsAnimEditSeparate)
            {
                return;
            }

            // 選択中のタグ区間に含まれるすべてのカーブを走査し、
            // タグ区間の終了位置をアニメーションキー時間が最も大きかったものに合わせます
            int maxTime = 1; // 最小値は1
            foreach (IPane pane in _target.IPaneArray)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    AnmCurve curve = curveAttr.GetAnmCurve(_target.CurrentTagName);
                    if (curve != null)
                    {
                        IAnmKeyFrame keyFrame = curve.GetLastKey();
                        if (keyFrame != null && maxTime < keyFrame.TimeAsInt)
                        {
                            maxTime = keyFrame.TimeAsInt;
                        }
                    }
                }
            }

            IAnimFrameSection section = _target.IAnimFrameSectionSet.TargetIAnimFrameSection;
            if (section != null)
            {
                _animFrameSectionSetMnp.BindTarget(_target.IAnimFrameSectionSet);
                _animFrameSectionSetMnp.SetTagBasic(section, section.Name, section.Comment, section.StartFrame, maxTime);
            }
        }

        /// <summary>
        /// 前区間へのスナップを行ないます。
        /// </summary>
        public void SnapBeforeKey(IAnmCurve[] targetCurveSet)
        {
            Debug.Assert(_target != null);
            if (_target == null || !_target.IsAnimEditSeparate)
            {
                return;
            }

            _target.BeginMassiveModify();

            foreach (AnmCurve curve in targetCurveSet)
            {
                // 現在区間の前後の区間を取得
                IAnimFrameSection beforeSection;
                IAnimFrameSection afterSection;
                _target.GetBeforeAfterAnimSection(out beforeSection, out afterSection);

                // 前区間のカーブを取得
                IAnmCurve beforeCurve = curve.TargetAttribute.GetAnmCurveRelationTag(beforeSection?.Name);

                // 前区間の値へスナップ
                AnmKeyFrame currentKey = curve?.GetFirstKey() as AnmKeyFrame;
                if (beforeCurve != null && currentKey != null)
                {
                    float snapValue = beforeCurve.Evaluate(beforeSection.EndFrame);

                    _animKeyFrameManipulator.BindTarget(currentKey);
                    _animKeyFrameManipulator.Modify(snapValue, currentKey.Time);
                }
            }

            _target.EndMassiveModify();
        }

        /// <summary>
        /// 後区間へのスナップを行ないます。
        /// </summary>
        public void SnapAfterKey(IAnmCurve[] targetCurveSet)
        {
            Debug.Assert(_target != null);
            if (_target == null || !_target.IsAnimEditSeparate)
            {
                return;
            }

            _target.BeginMassiveModify();

            foreach (AnmCurve curve in targetCurveSet)
            {
                // 現在区間の前後の区間を取得
                IAnimFrameSection beforeSection;
                IAnimFrameSection afterSection;
                _target.GetBeforeAfterAnimSection(out beforeSection, out afterSection);

                // 後区間のカーブを取得
                IAnmCurve afterCurve = curve.TargetAttribute.GetAnmCurveRelationTag(afterSection?.Name);

                // 後区間の値へスナップ
                AnmKeyFrame currentKey = curve?.GetLastKey() as AnmKeyFrame;
                if (afterCurve != null && currentKey != null)
                {
                    float snapValue = afterCurve.Evaluate(afterSection.StartFrame);

                    _animKeyFrameManipulator.BindTarget(currentKey);
                    _animKeyFrameManipulator.Modify(snapValue, currentKey.Time);
                }
            }

            _target.EndMassiveModify();
        }

        #endregion
    }
}
