﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Structures.Core.Command;
    using PictureCommandFactory = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.IPicture>;

    /// <summary>
    /// テクスチャ座標操作クラス
    /// </summary>
    public interface ITexCoordmanipulator
    {
        ITexCoordHolder TexCoordOwner { get; }
        void SetTexCoord(TexCoord4 newTexCoord, int coord4Idx);
        int NumTexCoord { set; }
    }

    /// <summary>
    /// TextBoxManipulator の概要の説明です。
    /// </summary>
    public class PictureManipulator : BaseManipulator, ITexCoordmanipulator
    {
        // 操作対象
        Picture						_target = null;
        PictureCommandFactory _PictureCommandFactory;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PictureManipulator()
        {
            _PictureCommandFactory = new PictureCommandFactory(
                delegate( IPicture srcPicture ){return new PictureMemento( srcPicture );} );
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        /// <param name="targetTextBox"></param>
        public void BindTarget( IPicture target )
        {
            _target = target as Picture;
            Debug.Assert( _target != null );
        }

        /// <summary>
        /// 操作対象を取得します。
        /// </summary>
        public IPicture  IPicture
        {
            get{ return _target;}
        }

        public ITexCoordHolder TexCoordOwner
        {
            get { return this.IPicture; }
        }

        /// <summary>
        /// 詳細なマテリアル設定を使用するか
        /// </summary>
        public bool     UseDetailedMaterial
        {
            set
            {
                if( _target.UseDetailedMaterial != value )
                {
                    _PictureCommandFactory.Modify( _target, delegate()
                    {
                        _target.UseDetailedMaterial = value;
                    } );
                }
            }
        }

        public RGBAColor VtxColor_LT
        {
            set
            {
                if( _target.VtxCol_LT != value )
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_target.VtxCol_LTAnmAttr, value);
                }
            }
        }
        public RGBAColor VtxColor_RT
        {
            set
            {
                if( _target.VtxCol_RT != value )
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_target.VtxCol_RTAnmAttr, value);
                }
            }
        }
        public RGBAColor VtxColor_LB
        {
            set
            {
                if( _target.VtxCol_LB != value )
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_target.VtxCol_LBAnmAttr, value);
                }
            }
        }
        public RGBAColor VtxColor_RB
        {
            set
            {
                if( _target.VtxCol_RB != value )
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_target.VtxCol_RBAnmAttr, value);
                }
            }
        }

        /// <summary>
        /// テクスチャ座標数を設定する。
        /// </summary>
        public int NumTexCoord
        {
            set
            {
                if( _target.NumTexCoord != value )
                {
                    _PictureCommandFactory.Modify( _target, delegate(){ _target.NumTexCoord = value; } );
                }
            }
        }

        /// <summary>
        /// テクスチャ座標数を設定する。
        /// </summary>
        public void SetTexCoord( TexCoord4 srcCoord, int texCoordIdx )
        {
            TexCoord4		dstCoord = new TexCoord4();
            dstCoord.Set( srcCoord );
            TexCoord4		currentCoord = _target.GetTexCoord4( texCoordIdx );

            if( !TexCoord4.CheckSame( dstCoord, currentCoord ) )
            {
                _PictureCommandFactory.Modify( _target, delegate() { _target.SetTexCoord4( dstCoord, texCoordIdx ); } );
            }
        }

        /// <summary>
        /// モデル形状タイプ
        /// </summary>
        public LECore.Structures.SerializableObject.Lyt.ShapeType ShapeType
        {
            set
            {
                if (_target.ShapeType != value)
                {
                    _PictureCommandFactory.Modify( _target, delegate() { _target.ShapeType = value; } );
                }
            }
        }

        /// <summary>
        /// 円形状の分割数
        /// </summary>
        public uint CircleSlice
        {
            set
            {
                if (_target.CircleSlice != value)
                {
                    _PictureCommandFactory.Modify( _target, delegate() { _target.CircleSlice = value; } );
                }
            }
        }

        /// <summary>
        /// 球形状の横分割数
        /// </summary>
        public uint SphereSlice
        {
            set
            {
                if (_target.SphereSlice != value)
                {
                    _PictureCommandFactory.Modify(_target, delegate () { _target.SphereSlice = value; });
                }
            }
        }

        /// <summary>
        /// 球形状の縦分割数
        /// </summary>
        public uint SphereStack
        {
            set
            {
                if (_target.SphereStack != value)
                {
                    _PictureCommandFactory.Modify(_target, delegate () { _target.SphereStack = value; });
                }
            }
        }

        /// <summary>
        /// 角丸四角形の角の半径
        /// </summary>
        public uint RoundRadius
        {
            set
            {
                if (_target.RoundRadius != value)
                {
                    _PictureCommandFactory.Modify(_target, delegate () { _target.RoundRadius = value; });
                }
            }
        }

        /// <summary>
        /// 角丸四角形の角の分割数
        /// </summary>
        public uint RoundSlice
        {
            set
            {
                if (_target.RoundSlice != value)
                {
                    _PictureCommandFactory.Modify(_target, delegate () { _target.RoundSlice = value; });
                }
            }
        }
    }
}
