﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using LECore.Structures.Core;
    using LECore.Structures.Core.Command;

    /// <summary>
    /// AnmCurveManipulator の概要の説明です。
    /// </summary>
    public class AnmCurveManipulator : BaseManipulator
    {
        AnmCurve _targetCurve = null;
        AnmKeyFrameManipulator _keyFrameMnp = null;


        #region --------------- private ---------------
        /// <summary>
        /// アニメーションカーブ中の既存のキーフレームを変更します。
        /// </summary>
        /// <param name="kf"></param>
        /// <param name="currentVal"></param>
        /// <param name="currentTime"></param>
        void ModifyExsistingKey_( IAnmKeyFrame kf, object currentVal, float currentTime )
        {
            // 操作対象に設定
            _keyFrameMnp.BindTarget( kf );

            // 既存キーのアトリビュート値を変更するだけとなります。
            float valAsFloat = AnmAttribute.GetValueAsFloat(_targetCurve.TargetAttrType, currentVal);
            valAsFloat = _targetCurve.ClampTargetMinMax(valAsFloat);

            _keyFrameMnp.Modify( valAsFloat, currentTime );

            // エルミート補間形式の場合は、
            // 新規に挿入されたキーおよび、挿入されたキーの前後のキーの接線情報を計算します。
            // MEMO：現在は計算を行っていません。
            // CalcTangentValue_( kf, true );

            // 更新イベントを通知します。
            //		memo:カーブの再評価を行うように変更しました。
            _targetCurve.Update( true );
        }

        #endregion --------------- private ---------------

          #region プロパティ

          /// <summary>
      /// キー領域以前の値設定。
      /// </summary>
      public AnmCurveInfinityType PreInfinity
      {
          set
          {
              if( _targetCurve.PreInfinityType != value )
              {
                  _CommandFactory.SetValueCommand
                      ( _targetCurve, _targetCurve.PreInfinityType, value,
                        delegate( AnmCurve anmCurve, object newValue)
                            {
                                Debug.Assert( newValue is AnmCurveInfinityType);
                                anmCurve.PreInfinityType = (AnmCurveInfinityType)newValue;
                            });
              }
          }
      }

      /// <summary>
      /// キー領域以後の値設定。
      /// </summary>
      public AnmCurveInfinityType PostInfinity
      {
          set {
              if( _targetCurve.PostInfinityType != value )
              {
                  _CommandFactory.SetValueCommand
                      ( _targetCurve, _targetCurve.PostInfinityType, value,
                        delegate( AnmCurve anmCurve, object newValue)
                            {
                                Debug.Assert( newValue is AnmCurveInfinityType);
                                anmCurve.PostInfinityType = (AnmCurveInfinityType)newValue;
                            });
              }
          }
      }

        #endregion プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnmCurveManipulator()
        {
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        public void BindTarget( IAnmCurve target )
        {
            _targetCurve = target as AnmCurve;
            Debug.Assert( _targetCurve != null );

            if( _keyFrameMnp == null )
            {
                _keyFrameMnp = new AnmKeyFrameManipulator();
            }
        }

        /// <summary/>
        /// 現在の時間、現在の値でキーフレームを作成します。
        ///
        ///
        /// TODO : 既存のカーブ上にキーが生成される場合のカーブ値の求め方を修正する必要がある。
        ///        キー挿入前に...
        ///            以下のパラメータを計算しておく。
        ///               object currentEvalVal = Evaluate( currentTime );
        ///               float currentTangent  = EvaluateTangent( currentTime );
        ///         キー挿入後に...
        ///            既存の曲線の接線の傾きを同一に設定する
        ///
        /// </summary>
        /// <returns></returns>
        public void MakeKeyFrameByCurrentState()
        {
            Debug.Assert( _targetCurve != null );

            if (_targetCurve.IsReadOnlyLocked)
            {
                return;
            }

            // 現在の補間値を計算します。
            int currentTime = GlobalTime.Inst.Time;

            // 現在の値を取得します
            object currentVal = _targetCurve.TargetAttributeValue;

            // TargetAttributeValue は表示スケールを考慮しない値が入っているため
            // カーブに表示スケールが設定されている場合は値にスケールをかけて扱う。
            // TargetAttributeValue はここでしか使用されていない。
            if (_targetCurve.ViewScale != 1.0f)
            {
                float valueAsFloat = Convert.ToSingle(currentVal);
                valueAsFloat *= _targetCurve.ViewScale;
                currentVal = valueAsFloat;
            }

            MakeKeyFrame(currentTime, currentVal);
        }

        /// <summary>
        /// 指定した時間で、キーを作ります。
        /// </summary>
        public void MakeKeyFrame(int currentTime, object currentVal)
        {
            // 同じ時間位置にすでにキーが存在しないか確認します。
            IAnmKeyFrame kf = _targetCurve.FindKeyExistAt(currentTime);
            if (kf != null)
            {
                // すでにキーが存在する場合、
                // 既存キーのアトリビュート値を変更するだけとなります。
                ModifyExsistingKey_(kf, currentVal, currentTime);
            }
            else
            {
                // 新規にキーを挿入します。
                kf = new AnmKeyFrame(_targetCurve,
                                      currentTime,
                                      currentVal,
                                      _targetCurve.ViewScale);

                _targetCurve.InsertNewKeyFrameWithTangentAdjust(kf as AnmKeyFrame);

                // Undoコマンドを登録
                _CommandFactory.MakeAnmKeyAddCmd(kf as AnmKeyFrame);
            }
        }

        /// <summary>
        /// 新規にキーを登録します(オフセットを指定します)。
        /// コピー貼り付けなどに利用されます。
        /// offsetValue には表示スケールの調整されていない生の値が入力されます。
        /// </summary>
        public void AddNewKey( IAnmKeyFrame srcEditableAnmKeyFrame, float offsetTime, float offsetValue )
        {
            AnmKeyFrame srcKey = srcEditableAnmKeyFrame as AnmKeyFrame;

            AddNewKeyFrame(
                srcKey.Time + offsetTime,
                (srcKey.ValueAsFloatNoEffect + offsetValue) * _targetCurve.ViewScale,
                srcKey.UnifyTangents,
                srcKey.InTangentNoEffect * _targetCurve.ViewScale,
                srcKey.OutTangentNoEffect * _targetCurve.ViewScale,
                srcKey.InterporationType);
        }

        /// <summary>
        /// パラメータを明示的に指定してキーを生成します。
        /// </summary>
        public void AddNewKeyFrame( float time, float val, bool bUnifiyTangent, float inTangent, float outTangent, InterporationType type)
        {
            if (_targetCurve.IsReadOnlyLocked)
            {
                return;
            }

            IAnmKeyFrame newKey = _targetCurve.FindKeyExistAt( time );
            if( newKey == null )
            {
                newKey = _targetCurve.AddNewKeyFrame(
                    time,
                    val,
                    bUnifiyTangent,
                    inTangent,
                    outTangent,
                    type,
                    _targetCurve.ViewScale );

                // Undoコマンドを登録
                _CommandFactory.MakeAnmKeyAddCmd( newKey as AnmKeyFrame );
            }
            else
            {
                // すでにキーが存在する場合、
                // 既存キーのアトリビュート値を変更するだけとなります。
                ModifyExsistingKey_( newKey, val, time );
            }
        }

        /// <summary>
        /// アニメーションカーブの、キーフレームを削除します。
        /// </summary>
        public void RemoveKeyFrame( IAnmKeyFrame keyFrame, bool bEvaluate )
        {
            if (_targetCurve.IsReadOnlyLocked)
            {
                return;
            }

            Debug.Assert( _targetCurve != null );

            AnmKeyFrame internalKeyFrame = keyFrame as AnmKeyFrame;
            Debug.Assert( internalKeyFrame != null );

            _targetCurve.RemoveKeyFrame( internalKeyFrame, bEvaluate );

            // Undoコマンドを登録
            if (_CommandFactory != null)
            {
                _CommandFactory.MakeAnmKeyRemoveCmd(internalKeyFrame);
            }
        }

        /// <summary>
        /// カーブを無効化し、変更イベントを通知します。
        /// </summary>
        public void Update( bool bEvaluate )
        {
            _targetCurve.Update( bEvaluate );
        }

        #region 選択関連

        /// <summary>
        /// 選択セットに追加します。
        /// </summary>
        public void AddSelectedSet( IAnmKeyFrame key )
        {
            if (_targetCurve.IsReadOnlyLocked)
            {
                return;
            }

            if( _targetCurve.Contains( key ) )
            {
                _targetCurve.AddSelectedSet( key );
            }
        }

        /// <summary>
        /// 選択セットからキーを取り除きます。
        /// </summary>
        public void RemoveSelectedSet( IAnmKeyFrame key )
        {
            _targetCurve.RemoveSelectedSet( key );
        }

        /// <summary>
        /// 選択キーのリセット
        /// </summary>
        public void ResetSelectedSet()
        {
            _targetCurve.ResetSelectedSet();
        }

        #endregion 選択関連
    }
}
