﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using LECore.Structures.Core;
    using LECore.Structures.Core.Command;



    /// <summary>
    /// AnmAttribute を操作するAPIを提供する、クラスです。
    ///
    /// CoreSystem 外部モジュール(Viewレイヤを実装するモジュール)が、
    /// 内部データ構造の値を操作する場合、XXXXManipulatorクラス群を使用します。
    /// </summary>
    public class AnmAttributeManipulator
    {
        AnmAttribute                _target           = null;
        AnmCurveManipulator         _curveManipulator = null;
        AnmAttributeManipulator     _subAttrMnp       = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="target"></param>
        public AnmAttributeManipulator()
        {
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        /// <param name="target"></param>
        public void BindTarget( IAnmAttribute target )
        {
            _target = target as AnmAttribute;
            Debug.Assert( _target != null );

            // アトリビュートがアニメーションカーブを持っていれば、
            // カーブ編集クラスを初期化します。
            if( _target.ICurrentAnimationCurve != null )
            {
                if( _curveManipulator == null )
                {
                    _curveManipulator = new AnmCurveManipulator();
                }
                _curveManipulator.BindTarget( target.ICurrentAnimationCurve);
            }

            // サブアトリビュート用の操作クラスが必要なら初期化します。
            if( _target.HasSubAttribute && _subAttrMnp == null )
            {
                _subAttrMnp = new AnmAttributeManipulator();
            }
        }

        /// <summary>
        /// 現在の時間、現在の値で、アニメーションキーを生成します。
        /// サブアトリビュートを持つ場合は、それらに対して、再帰的に処理を適用します。
        /// </summary>
        public void MakeKeyFrameByCurrentParams()
        {
            Debug.Assert( _target != null );

            // アトリビュートがアニメーションカーブを持っていれば、キーを作成する。
            if( _target.ICurrentAnimationCurve != null )
            {
                _curveManipulator.MakeKeyFrameByCurrentState();
            }

            // サブアトリビュートに関しても同様にキーを作成する。
            for( int i = 0;i < _target.NumSubAttribute; i++ )
            {
                _subAttrMnp.BindTarget( _target.FindSubAttributeByIdx( i ) );
                _subAttrMnp.MakeKeyFrameByCurrentParams();
            }
        }

        /// <summary>
        /// 再帰的にすべてのキーを削除する。
        /// </summary>
        private void RemoveAllKeyRecursively_(IAnmAttribute attribute)
        {
            if (attribute == null)
            {
                return;
            }

            // すべての区間に対して処理する
            // 注意: Curves の実装の都合上実行順は一定とは限らない
            foreach (var curve in attribute.Curves)
            {
                AnmCurveManipulator curveMnp = new AnmCurveManipulator();
                curveMnp.BindTarget(curve);

                foreach (IAnmKeyFrame key in curve.IKeyFrameSet)
                {
                    curveMnp.RemoveKeyFrame(key, false);
                }
            }

            for (int i = 0; i < attribute.NumSubAttribute; i++)
            {
                RemoveAllKeyRecursively_(attribute.FindSubAttributeByIdx(i));
            }
        }

        /// <summary>
        /// 選択キーのリセット
        /// </summary>
        public void ResetSelectedSet()
        {
            foreach (AnmCurve curve in _target.Curves)
            {
                curve.ResetSelectedSet();
            }
        }

        /// <summary>
        /// 再帰的にすべてのキーを削除する。
        /// </summary>
        internal void RemoveAllKeyRecursively()
        {
            RemoveAllKeyRecursively_(_target);
        }
    }
}
