﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.Core.Command;
    using LECore.Structures.LECoreInterface;
    using AnimFrameSectionCommandCreater = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.IAnimFrameSection>;
    using AnimFrameSectionSetCommandCreater = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.IAnimFrameSectionSet>;


    public class AnimFrameSectionSetManipulator : BaseManipulator
    {
        AnimFrameSectionSet		_target = null;
        AnimFrameSectionCommandCreater _AnimFrameSectionCommandCreater;
        AnimFrameSectionSetCommandCreater _AnimFrameSectionSetCommandCreater;
        SubSceneManipulator     _sunbSceneMnp = null;
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="target"></param>
        public AnimFrameSectionSetManipulator()
            : this(new SubSceneManipulator())
        {

        }

        public AnimFrameSectionSetManipulator(SubSceneManipulator SubSceneMnp)
        {
            _AnimFrameSectionCommandCreater = new AnimFrameSectionCommandCreater(
                delegate( IAnimFrameSection src ) { return new AnimFrameSectionMemento( src ); } );
            _AnimFrameSectionSetCommandCreater = new AnimFrameSectionSetCommandCreater(
                delegate( IAnimFrameSectionSet src ) { return new AnimFrameSectionSetMemento( src ); } );

            _sunbSceneMnp = SubSceneMnp;
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        /// <param name="target"></param>
        public void BindTarget( IAnimFrameSectionSet target )
        {
            _target = target as AnimFrameSectionSet;
            Debug.Assert( _target != null );
        }

        //-----------------------------------------
        // Tag登録、削除 ...
        /// <summary>
        /// 新しいフレーム範囲の登録
        /// </summary>
        public IAnimFrameSection AddNewFrameSection(
            string name,
            string comment,
            int startFrame,
            int endFrame,
            ISubScene subScene)
        {
            IAnimFrameSection result = null;
            if( !_target.CheckExistsByName( name ) )
            {
                _AnimFrameSectionSetCommandCreater.Modify( _target, delegate()
                {
                    result = _target.AddNewFrameSection( name, comment, startFrame, endFrame );
                } );

                Debug.Assert(result != null);

                // 設計的には、もっとも根本である
                // データモデル層、SubScene._animFrameSectionSet_OnAnimSectionTagSetUpdate
                // で処理を行うのが適切ですが、
                // Undo対応を適切に行うには、やや大がかりな変更となることと、実用上の弊害が無いため
                // 現状の実装（Manipulator層での対処）を選択しています。Remove も同様です。
                _sunbSceneMnp.BindTarget(subScene);
                _sunbSceneMnp.AddAnimSection(name);
            }
            return result;
        }

        /// <summary>
        /// 削除
        /// </summary>
        public void Remove( IAnimFrameSection animFrameSection, ISubScene subScene)
        {
            if( _target.CheckExistsByName( animFrameSection.Name ) )
            {
                _AnimFrameSectionSetCommandCreater.Modify( _target, delegate()
                {
                    _target.Remove( animFrameSection );
                } );

                // Manipulator層で SubScene のデータ更新をハンドルします。
                // AddNewFrameSection のコメント参照。
                _sunbSceneMnp.BindTarget(subScene);
                _sunbSceneMnp.RemoveAnimSection(animFrameSection.Name);
            }
        }

        //-----------------------------------------
        // SetOutBinay ...
        /// <summary>
        ///
        /// </summary>
        public bool SetOutBinayFileName(
            IAnimFrameSection target,
            string binName )
        {
            bool result = false;
            if( target.BinarySettings.FileName != binName )
            {
                _AnimFrameSectionCommandCreater.Modify( target, delegate()
                {
                    result = _target.SetOutBinayFileName( target, binName );
                } );
            }
            return result;
        }

        /// <summary>
        ///
        /// </summary>
        public bool SetOutBinayLoopType(
            IAnimFrameSection target,
            AnimationLoopType loopType )
        {
            bool result = false;
            if( target.BinarySettings.LoopType != loopType )
            {
                _AnimFrameSectionCommandCreater.Modify( target, delegate()
                {
                    result = _target.SetOutBinayLoopType( target, loopType );
                } );
            }
            return result;
        }

        /// <summary>
        ///
        /// </summary>
        public bool SetOutBinayParamaterKind(
            IAnimFrameSection target,
            ParamaterKind targetAnimKind )
        {
            bool result = false;
            if( target.BinarySettings.TargetAnimKind != targetAnimKind )
            {
                _AnimFrameSectionCommandCreater.Modify( target, delegate()
                {
                    result = _target.SetOutBinayParamaterKind( target, targetAnimKind );
                } );
            }
            return result;
        }

        /// <summary>
        ///
        /// </summary>
        public bool SetOutBinayActive( IAnimFrameSection target, bool outBinaryActive )
        {
            bool result = false;
            if( target.BinarySettings.Active != outBinaryActive )
            {
                _AnimFrameSectionCommandCreater.Modify( target, delegate()
                {
                    result = _target.SetOutBinayActive( target, outBinaryActive );
                } );
            }
            return result;
        }

        //-----------------------------------------
        // SetTag ...
        /// <summary>
        ///
        /// </summary>
        public bool SetTagIndex( IAnimFrameSection target, int newIndex )
        {
            bool result = false;
            if( _target.GetTagIndex( target ) != newIndex )
            {
                _AnimFrameSectionSetCommandCreater.Modify( _target, delegate()
                {
                    result = _target.SetTagIndex( target, newIndex );
                } );
            }
            return result;
        }

        /// <summary>
        ///
        /// </summary>
        public bool SetTagBasic( IAnimFrameSection target, string name, string comment, int start, int end )
        {
            bool result = false;
            if( target.Name != name ||
                target.Comment != comment ||
                target.StartFrame != start ||
                target.EndFrame != end)
            {
                _AnimFrameSectionCommandCreater.Modify( target, delegate()
                {
                    result = _target.SetTagBasic( target, name, comment, start, end );
                } );
            }
            return result;
        }

        /// <summary>
        ///
        /// </summary>
        bool IsGroupSetSame_( ILEGroup[] group1, ILEGroup[] group2 )
        {
            if( group1.Length != group2.Length )
            {
                return false;
            }

            foreach( ILEGroup currentTarget in group1 )
            {
                if( Array.IndexOf( group2, currentTarget ) == -1 )
                {
                    return false;
                }
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public bool SetTagTargetGroup(
            IAnimFrameSection target,
            ILEGroup[] groups,
            bool bindAnimationRecursively )
        {
            bool result = false;
            if( !IsGroupSetSame_( target.TargetGroup, groups ) )
            {
                _AnimFrameSectionCommandCreater.Modify( target, delegate()
                {
                    result = _target.SetTagTargetGroup(
                        target, groups, bindAnimationRecursively );
                } );
            }
            return result;
        }

        /// <summary>
        ///
        /// </summary>
        public void SetTargetAnimFrameSection(IAnimFrameSection value)
        {
            if(_target.TargetIAnimFrameSection != value)
            {
                _target.TargetIAnimFrameSection = value;
                if (GlobalTime.Inst.UseTargetFrameSectionRangeAsPlayRange)
                {
                    GlobalTime.Inst.FitAndNotifyAfterTargetFrameSectionChanged();
                }
            }
        }

        public void SetUserData(IAnimFrameSection newFs, IUserDataHolder userDataHolder)
        {
            (userDataHolder as UserDataHolder).CloneCopyTo(newFs.UserDataHolder as UserDataHolder);
        }
    }
}
