﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 //
 // 注意：.NET版ライブラリで追加された処理について
 //
 // ●バイトエンディアンの違いを吸収するために、読み込むデータは解釈する前に、バイトスワップが実行されます。
 // ●.NET 上でのテクスチャロード処理を簡略化するために、フォントイメージデータはすべて RGBA8形式に変換されます。
 // ●変換されたデータを格納する、メモリの確保、解放が行われます。
 //
#include "ResFont.h"
#include "inlines.h"
#include "nw4r/misc.h"

#include "ByteOrderUtil.h"
#include "ByteSwapUtil.h"


namespace NW4R
{
namespace Font
{
namespace UnManaged
{




//-------------------------------------------------------------------------
// バイトスワップのユーティリティ群

// using namespace NW4R::Font;





/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ResFont::ResFont()

  Description:  コンストラクタ

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
ResFont::ResFont()
{
}

/*---------------------------------------------------------------------------*
  Name:         ResFont::~ResFont()

  Description:  デストラクタ

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
ResFont::~ResFont()
{
}



/* ------------------------------------------------------------------------
        構築
   ------------------------------------------------------------------------ */
/*---------------------------------------------------------------------------*
  Name:         ResFont::SetResource(void*)

  Description:  オブジェクトにフォントリソースを関連付けます。

  Arguments:    brfnt:  メモリ上にロードしたbrfntファイル先頭へのポインタ。

  Returns:      関連付けに成功したら true そうでなければ false。
 *---------------------------------------------------------------------------*/
bool
ResFont::SetResource(void* brfnt, bool isEndianSwapEnabled)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( brfnt );
    NW4R_ALIGN32_ASSERT( brfnt );
    BinaryFileHeader* fileHeader = reinterpret_cast<BinaryFileHeader*>(brfnt);

    mFontSwapUtil.SetByteSwapEnabled(isEndianSwapEnabled);

    // バイトスワップします
    mFontSwapUtil.SwapBinaryFileHeader_( fileHeader );

    if( !IsManaging(NULL) )
    {
        NW4R_WARNING(FALSE, "Font resource already atached.");
        return false;
    }

    //---- オフセット解決済みのフォントか？
    if (  fileHeader->signature == BINFILE_SIG_FONT_RESOLEVED )
    {
        // FontInfomation を探す
        mFontInfo = static_cast<FontInformation *>(FindBlock(fileHeader, BINBLOCK_SIG_FINF));
    }
    else
    {
        // 再構築する
        mFontInfo = Rebuild(fileHeader);
    }

    if( mFontInfo != NULL )
    {
        SetResourceBuffer(brfnt, mFontInfo, static_cast<FontKerningTable *>(FindBlock(fileHeader, BINBLOCK_SIG_KRNG)));
        return true;
    }

    return false;
}


void* ResFont::FindBlock(BinaryFileHeader* fileHeader, SigWord sigword)
{
    BinaryBlockHeader* blockHeader;
    int nBlocks = 0;

    blockHeader = reinterpret_cast<BinaryBlockHeader*>(
        reinterpret_cast<u8*>(fileHeader) + fileHeader->headerSize
    );

    while (nBlocks < fileHeader->dataBlocks)
    {
        NW4R_POINTER_ASSERT( blockHeader );
        if (blockHeader->kind == sigword)
        {
            return reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader);
        }

        blockHeader = reinterpret_cast<BinaryBlockHeader*>(
            reinterpret_cast<u8*>(blockHeader) + blockHeader->size
        );
        nBlocks++;
    }

    return NULL;
}

/*---------------------------------------------------------------------------*
  Name:         ResFont::RemoveResource()

  Description:  オブジェクトからフォントリソースを切り離します。

  Arguments:    なし。

  Returns:      切り離したフォントリソースへのポインタ。
                フォントが関連付けられていなかった場合はNULLを返します。
 *---------------------------------------------------------------------------*/
void*
ResFont::RemoveResource()
{
    NW4R_POINTER_ASSERT( this );
    NW4R_WARNING(
        ! IsManaging(NULL),
        "ResFont::RemoveResource(): Res font is not loaded.\n"
    );
    return RemoveResourceBuffer();
}

/*---------------------------------------------------------------------------*
  Name:         ResFont::Rebuild( BinaryFileHeader* )

  Description:  メモリ上にロードされたbrfntファイルを再構築して
                使用できるようにします。

  Arguments:    fileHeader: メモリ上にロードしたbrfntファイルの
                            ファイルヘッダへのポインタ

  Returns:      再構築に成功した場合はFINFブロック本体へのポインタを返します。
                再構築に失敗した場合はNULLを返します。
 *---------------------------------------------------------------------------*/
// static
FontInformation*
ResFont::Rebuild( BinaryFileHeader* fileHeader )
{
    NW4R_POINTER_ASSERT( fileHeader );
    NW4R_ALIGN32_ASSERT( fileHeader );
    BinaryBlockHeader* blockHeader;
    FontInformation* info = NULL;
    int nBlocks = 0;

    blockHeader = reinterpret_cast<BinaryBlockHeader*>(
        reinterpret_cast<u8*>(fileHeader) + fileHeader->headerSize
    );

    m_pPtrOffsetBase = reinterpret_cast<char*>(fileHeader);

    while( nBlocks < fileHeader->dataBlocks )
    {
        NW4R_POINTER_ASSERT( blockHeader );


        // バイトスワップします
        mFontSwapUtil.SwapBinaryBlockHeader_( blockHeader );


        switch( blockHeader->kind )
        {
        //--------------------------------------------------
        // INFO ブロック
        case BINBLOCK_SIG_FINF:
            // INFOブロックは1つでなければならない
            NW4R_ASSERT( info == NULL );
            {
                info = reinterpret_cast<FontInformation*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                // バイトスワップします
                mFontSwapUtil.SwapFontInformation_( info );


                NW4R_ASSERT( info->fontType == FONT_TYPE_NNGCTEXTURE );
                NW4R_ASSERT( info->alterCharIndex != GLYPH_INDEX_NOT_FOUND );


                // pGlyph は必須
                NW4R_NULL_ASSERT( info->pGlyph );
                NW4R_POINTER_ASSERT( info->pGlyph );

                // pWidth と pMap はなくても良い
                if( info->pWidth != NULL )
                {
                    NW4R_POINTER_ASSERT( info->pWidth );
                }
                if( info->pMap != NULL )
                {
                    NW4R_POINTER_ASSERT( info->pMap );
                }
            }
            break;

        //--------------------------------------------------
        // TGLP ブロック
        case BINBLOCK_SIG_TGLP:
            // TGLP ブロックも1つでなければならないが複数存在しても致命的ではない
            {
                FontTextureGlyph* glyph = reinterpret_cast<FontTextureGlyph*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                // バイトスワップします
                mFontSwapUtil.SwapFontTextureGlyph_( glyph );
                NW4R_NULL_ASSERT( glyph->sheetImage );

                //---- 32x32 の I4 が最小、1024x1024 の RGBA8 が最大
                NW4R_MIN_ASSERT       ( glyph->cellWidth,     1 );
                NW4R_MIN_ASSERT       ( glyph->cellHeight,    1 );
                NW4R_MINMAX_ASSERT    ( glyph->sheetSize,     32 * 32 / 2,    1024 * 1024 * 4 );
                NW4R_MIN_ASSERT       ( glyph->sheetNum,      1 );
                NW4R_MINMAX_ASSERT    ( glyph->sheetFormat & FONT_SHEETFORMAT_TEXFMT_MASK,   GX_TF_I4,       GX_TF_RGBA8 );
                NW4R_MIN_ASSERT       ( glyph->sheetRow,      1 );
                NW4R_MIN_ASSERT       ( glyph->sheetLine,     1 );
                NW4R_MINMAX_ASSERT    ( glyph->sheetWidth,    32,             1024 );
                NW4R_MINMAX_ASSERT    ( glyph->sheetHeight,   32,             1024 );
            }
            break;

        //--------------------------------------------------
        // CWDHブロック
        case BINBLOCK_SIG_CWDH:
            {
                FontWidth* width = reinterpret_cast<FontWidth*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                // バイトスワップします
                mFontSwapUtil.SwapFontWidth_( width );

                NW4R_ASSERT( width->indexBegin <= width->indexEnd );

                if( width->pNext != NULL )
                {
                    NW4R_POINTER_ASSERT( width->pNext );
                }
            }
            break;

        //--------------------------------------------------
        // CMAPブロック
        case BINBLOCK_SIG_CMAP:
            {
                FontCodeMap* map = reinterpret_cast<FontCodeMap*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                // バイトスワップします
                mFontSwapUtil.SwapFontCodeMap_( map );

                NW4R_ASSERT( map->ccodeBegin <= map->ccodeEnd );
                NW4R_ASSERT( (map->mappingMethod == FONT_MAPMETHOD_DIRECT)
                                || (map->mappingMethod == FONT_MAPMETHOD_TABLE)
                                || (map->mappingMethod == FONT_MAPMETHOD_SCAN) );

                if( map->pNext != NULL )
                {
                    NW4R_POINTER_ASSERT( map->pNext );
                }
            }
            break;
        //--------------------------------------------------
        // KRNGブロック
        case BINBLOCK_SIG_KRNG:
            {
                FontKerningTable* pKerningTable = reinterpret_cast<FontKerningTable*>(reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader));
                mFontSwapUtil.SwapFontKerningMap_(pKerningTable);
            }
            break;

        //--------------------------------------------------
        // unknown
        default:
             NW4R_ASSERTMSG(false, "The font has unknown block('%c%c%c%c').",
                (blockHeader->kind >> 24) & 0xFF,
                (blockHeader->kind >> 16) & 0xFF,
                (blockHeader->kind >>  8) & 0xFF,
                (blockHeader->kind >>  0) & 0xFF
                );
            break;
        }

        blockHeader = reinterpret_cast<BinaryBlockHeader*>(
            reinterpret_cast<u8*>(blockHeader) + blockHeader->size
        );
        nBlocks++;
    }

    return info;
}



} /* UnManaged */
} /* namespace ut */
} /* namespace nw4r */
