﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW4R_UT_FONT_H_
#define NW4R_UT_FONT_H_

#include <dolphin/types.h>
#include <dolphin/gx/GXEnum.h>
#include "CharStrmReader.h"
#include "fontResourceFormat.h"

namespace NW4R
{
namespace Font
{
namespace UnManaged
{

//---- 文字コード型
typedef u16 CharCode;

//---- テクスチャフォーマット型
typedef u16     TexFmt;

#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4201)
#endif
union TextureSize
{
    struct
    {
        u16                 height;   //!< 高さ
        u16                 width;    //!< 幅
    };

    u32                 size;   //!< サイズ
};

#if defined(_WIN32)
#pragma warning(pop)
#endif

class TextureObject
{
public:
    TextureObject()
        : m_Name(0), m_pImage(NULL), m_Size(TextureSize()), m_Format(0), m_SheetNum(0), m_ColorBlackWhiteInterpolationEnabled(true)
    {
        Reset();
    }
    ~TextureObject() { Reset(); }

    void Set(
        u32         name,
        const void* pImage,
        TexFmt      format,
        u16         width,
        u16         height,
        u8          sheetNum,
        bool		blackWhiteInterpolationEnabled);

    void SetTextureSettings(const TextureObject* pSrc);

    void Reset()
    {
        Set(0, NULL, 0, 0, 0, 0, true);
    }

    u32 GetName() const
    {
        return m_Name;
    }

    void SetName(u32 texName)
    {
        m_Name = texName;
    }

    const void* GetImage() const
    {
        return m_pImage;
    }

    u8 GetFormat() const
    {
        return m_Format;
    }

    u8 GetSheetNum() const
    {
        return m_SheetNum;
    }

    const TextureSize GetSize() const
    {
        return m_Size;
    }

    bool IsColorBlackWhiteInterpolationEnabled() const
    {
        return m_ColorBlackWhiteInterpolationEnabled;
    }

    void SetColorBlackWhiteInterpolationEnabled(bool flag)
    {
        m_ColorBlackWhiteInterpolationEnabled = flag;
    }

private:
    u32 m_Name; //!< テクスチャ名
    const void* m_pImage; //!< イメージへのポインタ
    TextureSize m_Size; //!< サイズ
    u8 m_Format; //!< フォーマット
    u8 m_SheetNum; //!< シート数
    bool m_ColorBlackWhiteInterpolationEnabled;	//!< 白黒補完が有効か否か

};

//---- 文字グリフ型
struct Glyph
{
    // リソースの CharWidths とはサイズが異なるので別定義します。
    struct GlyphCharWidths
    {
        s16         left;          // 文字の左スペースの幅
        u16         glyphWidth;    // 文字のグリフ幅
        u16         charWidth;     // 文字の幅 = 左スペース幅 + グリフ幅 + 右スペース幅
        u16         rawWidth;      // テクスチャキャッシュ上の文字幅
    };

    const void* pTexture;
    u32         sheetNo;  // 追加：.NET モジュールでテクスチャの検索を行う際に利用します。
    u16         sheetFormat;     // 追加 : .NET モジュールで sRGB の判定に使用します。
    bool        isBitmapFont;    // 追加 : .NET モジュールで sRGB の判定に使用します。
    GlyphCharWidths widths;
    u16         height;
    u16         rawHeight;
    GXTexFmt    texFormat;
    u16         texWidth;
    u16         texHeight;
    u16         cellX;
    u16         cellY;
    TextureObject* pTextureObject;
    s16			baselineDifference;         //!< 基本となるベースラインからずれた文字の場合その差分が入ります
};

class Font
{
public:
    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */
    enum Type
    {
        TYPE_NULL,          //
        TYPE_ROM,           // RomFont
        TYPE_RESOURCE,       // ResFont
        TYPE_SCALABLE,       // ResFont
        TYPE_PAIR
    };


    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    // 不正な文字コード
    static const CharCode INVALID_CHARACTER_CODE = 0xFFFF;


    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- コンストラクタ/デストラクタ
                                Font()
    : mReaderFunc(&CharStrmReader::ReadNextCharCP1252)
    , m_IsEnableKerning(true)
    {
    }

    virtual                     ~Font()
    {
    }

    //---- フォント全体情報アクセサ
    virtual int                 GetWidth() const = 0;
    virtual int                 GetHeight() const = 0;
    virtual int                 GetAscent() const = 0;
    virtual int                 GetDescent() const = 0;
    virtual int                 GetBaselinePos() const = 0;
    virtual int                 GetCellHeight() const = 0;
    virtual int                 GetCellWidth() const = 0;
    virtual int                 GetMaxCharWidth() const = 0;
    virtual Type                GetType() const = 0;
    virtual GXTexFmt            GetTextureFormat() const = 0;
    virtual int                 GetLineFeed() const = 0;
    virtual const CharWidths    GetDefaultCharWidths() const = 0;

    virtual void                SetDefaultCharWidths(
                                    const CharWidths& widths        // 新しいデフォルト幅
                                ) = 0;

    virtual bool                SetAlternateChar( CharCode c ) = 0; // 新しい代替文字
    virtual void                SetLineFeed( int linefeed ) = 0;    // 新しい改行幅

    //---- 文字単体情報アクセサ
    virtual int                 GetCharWidth(
                                    CharCode c ) const = 0;         // 幅を求める文字
    virtual const CharWidths    GetCharWidths(
                                    CharCode c ) const = 0;         // 幅を求める文字
    virtual void                GetGlyph(
                                    Glyph*      glyphPtr,           // グリフ情報を受け取るバッファ
                                    CharCode    c                   // グリフ情報を取得する文字
                                ) const = 0;
    virtual bool                HasGlyph(CharCode c) const = 0;     // グリフ情報があるか
    virtual int            GetKerning(CharCode c0, CharCode c1) const = 0; // カーニングの値を取得します。
    virtual bool           HasKerning() const = 0; // カーニングの値を持っているか？
    virtual FontType       GetFontType() const = 0;
    virtual bool           IsEnalbeKerning() const {return m_IsEnableKerning;}
    virtual void           SetEnalbeKerning(bool flag) {m_IsEnableKerning = flag;}

    // 追加：.NET モジュールでテクスチャの検索を行う際に利用します。
    virtual int           GetSheetNum() const = 0;

    //---- 文字ストリーム処理
    virtual FontEncoding        GetEncoding() const = 0;
    const CharStrmReader        GetCharStrmReader() const
    {
        NW4R_POINTER_ASSERT( this );
        CharStrmReader reader(mReaderFunc);
        return reader;
    }


protected:
    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    void                        InitReaderFunc(
                                    FontEncoding encoding );

private:
    /* ------------------------------------------------------------------------
            変数
       ------------------------------------------------------------------------ */
    CharStrmReader::ReadNextCharFunc    mReaderFunc;
    bool m_IsEnableKerning;
};




        } /* UnManaged */

    } /* namespace ut */
} /* namespace nw4r */
#endif //  NW4R_UT_FONT_H_
