﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW4R_UT_COLOR_H_
#define NW4R_UT_COLOR_H_

//#include <dolphin.h>
#include <dolphin/gx/GXStruct.h>
#include "inlines.h"

namespace NW4R
{
namespace Font
{
namespace UnManaged
{


struct Color: public GXColor
{
public:
    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    static const int ELEMENT_MIN         = 0;
    static const int ELEMENT_MAX         = 255;
    static const int ALPHA_MIN           = ELEMENT_MIN;
    static const int ALPHA_MAX           = ELEMENT_MAX;
    static const int ALPHA_OPACITY       = ALPHA_MAX;
    static const int ALPHA_TRANSPARENT   = ALPHA_MAX;

    static const u32 BLACK      = 0x000000FF;
    static const u32 GRAY       = 0x808080FF;
    static const u32 WHITE      = 0xFFFFFFFF;
    static const u32 RED        = 0xFF0000FF;
    static const u32 GREEN      = 0x00FF00FF;
    static const u32 BLUE       = 0x0000FFFF;
    static const u32 YELLOW     = 0xFFFF00FF;
    static const u32 MAGENTA    = 0xFF00FFFF;
    static const u32 CYAN       = 0x00FFFFFF;


    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- コンストラクタ
                Color()
    {
        Set(ELEMENT_MAX, ELEMENT_MAX, ELEMENT_MAX, ALPHA_MAX);
    }

                Color(u32 color)
    {
        ToU32ref() = color;
    }

    /*implicit*/Color(const GXColor& color)
    {
        ToU32ref() = *reinterpret_cast<const u32*>(&color);
    }

                Color(
                    int red,
                    int green,
                    int blue,
                    int alpha = ALPHA_OPACITY )
    {
        Set(red, green, blue, alpha);
    }

    //---- デストラクタ
                ~Color()
    {
    }

    //---- 型変換演算子
                operator u32&()
    {
        return ToU32ref();
    }

                operator const u32&() const
    {
        return ToU32ref();
    }

    //---- 明示的な型変換
    u32         ToU32() const
    {
        return ToU32ref();
    }

    //---- アクセサ
    void        Set(
                    int red,
                    int green,
                    int blue,
                    int alpha = ALPHA_OPACITY )
    {
        r = static_cast<u8>(red);
        g = static_cast<u8>(green);
        b = static_cast<u8>(blue);
        a = static_cast<u8>(alpha);
    }

    void        Set(Color color)
    {
        operator =(color);
    }

    //---- 四則演算
    const Color operator +(const Color& right) const
    {
        Color color(
            Min(this->r + right.r, ELEMENT_MAX),
            Min(this->g + right.g, ELEMENT_MAX),
            Min(this->b + right.b, ELEMENT_MAX),
            Min(this->a + right.a, ELEMENT_MAX)
        );
        return color;
    }

    const Color operator -(const Color& right) const
    {
        Color color(
            Max(this->r - right.r, ELEMENT_MIN),
            Max(this->g - right.g, ELEMENT_MIN),
            Max(this->b - right.b, ELEMENT_MIN),
            Max(this->a - right.a, ELEMENT_MIN)
        );
        return color;
    }

    const Color operator *(const Color& right) const
    {
        Color color(
            this->r * right.r / ELEMENT_MAX,
            this->g * right.g / ELEMENT_MAX,
            this->b * right.b / ELEMENT_MAX,
            this->a * right.a / ELEMENT_MAX
        );
        return color;
    }

    const Color operator /(const Color& right) const
    {
        Color color(
            (right.r != 0) ? Min(this->r * ELEMENT_MAX / right.r, ELEMENT_MAX): ELEMENT_MAX,
            (right.g != 0) ? Min(this->g * ELEMENT_MAX / right.g, ELEMENT_MAX): ELEMENT_MAX,
            (right.b != 0) ? Min(this->b * ELEMENT_MAX / right.b, ELEMENT_MAX): ELEMENT_MAX,
            (right.a != 0) ? Min(this->a * ELEMENT_MAX / right.a, ELEMENT_MAX): ELEMENT_MAX
        );
        return color;
    }

    //---- インクリメント、デクリメント
    const Color operator ++()
    {
        (r >= ELEMENT_MAX) ? 0: r++;
        (g >= ELEMENT_MAX) ? 0: g++;
        (b >= ELEMENT_MAX) ? 0: b++;
        (a >= ELEMENT_MAX) ? 0: a++;

        return *this;
    }

    const Color operator ++(int)
    {
        Color old = *this;
        (r >= ELEMENT_MAX) ? 0: r++;
        (g >= ELEMENT_MAX) ? 0: g++;
        (b >= ELEMENT_MAX) ? 0: b++;
        (a >= ELEMENT_MAX) ? 0: a++;

        return old;
    }

    const Color operator --()
    {
        (r <= ELEMENT_MIN) ? 0: r--;
        (g <= ELEMENT_MIN) ? 0: g--;
        (b <= ELEMENT_MIN) ? 0: b--;
        (a <= ELEMENT_MIN) ? 0: a--;

        return *this;
    }

    const Color operator --(int)
    {
        Color old = *this;
        (r <= ELEMENT_MIN) ? 0: r--;
        (g <= ELEMENT_MIN) ? 0: g--;
        (b <= ELEMENT_MIN) ? 0: b--;
        (a <= ELEMENT_MIN) ? 0: a--;

        return old;
    }

    //---- bitごとの和と積
    const Color operator |(const Color& right) const
    {
        return (this->ToU32() | right.ToU32());
    }

    const Color operator &(const Color& right) const
    {
        return (this->ToU32() & right.ToU32());
    }

    Color& operator +=(const Color& rhs)
    {
        *this = *this + rhs;
        return *this;
    }
    Color& operator -=(const Color& rhs)
    {
        *this = *this - rhs;
        return *this;
    }
    Color& operator *=(const Color& rhs)
    {
        *this = *this * rhs;
        return *this;
    }
    Color& operator /=(const Color& rhs)
    {
        *this = *this / rhs;
        return *this;
    }

protected:
    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    u32&        ToU32ref()
    {
        return *reinterpret_cast<u32*>(this);
    }
    const u32&  ToU32ref() const
    {
        return *reinterpret_cast<const u32*>(this);
    }

} ATTRIBUTE_ALIGN(4);



} /* UnManaged */
} /* namespace ut */
} /* namespace nw4r */

#endif //  NW4R_UT_COLOR_H_
