﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace NW4R.ColorPicker
{
    /// <summary>
    /// カラーピッカーコントロールクラス。
    /// </summary>
    /// <remarks>
    /// カラー選択を行う為の汎用的なコンポーネントクラスです。
    /// System.Color 型を扱い、ＲＧＢ、ＲＧＢＡ、ＨＳＶ、ＨＳＶＡの各成分でカラー選択が行えます。
    /// フォームデザイナで特定のフォームに貼り付けて使用して下さい。
    /// <para>
    /// フォームデザイナで使用するには、プロジェクトの参照設定の「参照の追加...」と、
    /// ツールボックスの「アイテムの追加と削除...」で dll ファイルを追加して下さい。
    /// </para>
    /// </remarks>
    /// <example>
    /// <code>
    /// using NW4R.ColorPicker;
    ///
    /// public class Form1() : System.Windows.Forms.Form
    /// {
    ///   // ～省略～
    ///   private ColorPickerControl _colorPicker;
    ///
    ///   public Form1()
    ///   {
    ///     InitializeComponent();
    ///
    ///     // カラー成分と表示方式を設定します。
    ///     // フォームの表示後に動的に切り替えても構いません。
    ///     _colorPicker.ColorMode = ColorMode.RGB;
    ///     _colorPicker.ValueMode = ValueMode.Float;
    ///
    ///     // カラー変更イベントハンドラを登録します。
    ///     _colorPicker.ColorChanged += new ColorChangedEventHandler(colorPicker_ColorChanged)
    ///   }
    ///
    ///   private void colorPicker_ColorChanged(object sender, ColorChangedEventArgs e)
    ///   {
    ///     // カラー変更時のイベントを処理します。
    ///     System.Color color = e.Color;
    ///
    ///     // 編集中（スライダをドラッグしている最中）か編集確定かを判断して処理することができます。
    ///     if (e.EditFixed)
    ///     {
    ///       // 編集確定時の処理
    ///     }
    ///     else
    ///     {
    ///       // 編集中の処理
    ///     }
    ///   }
    /// }
    /// </code>
    /// </example>
    [ToolboxBitmap(typeof(UserControl))]
    public class ColorPickerControl : System.Windows.Forms.UserControl
    {
        private IContainer components;

        private ColorHelper.DoubleColor m_color = new ColorHelper.DoubleColor();

        /// <summary>SV選択バー</summary>
        protected ColorPicker.SaturationValueBar	m_gradCtrl;
        /// <summary>H選択バー</summary>
        protected ColorPicker.HueBar	m_hBar;
        /// <summary>パレットパネル</summary>
        protected ColorPicker.PalettePanel m_pltPanel;
        /// <summary>カラー選択パネル</summary>
        protected ColorPicker.ComponentPanel m_componentPanel;
        /// <summary>パレットにカラーを転送するボタン</summary>
        protected System.Windows.Forms.Button m_colSetButton;
//		private System.Windows.Forms.Button m_colPickButton;
        /// <summary>現在カラー表示パネル</summary>
        protected ColorPicker.ColorBox m_colorPanel;

        //	装飾用
        private Rectangle m_GradBoxWakuOut;
        private Rectangle m_GradBoxWakuIn;
        private Rectangle m_hBarWakuOut;
        protected Button  m_btnToggleRgbHsv;
        protected Button _btnGammaLiner;
        protected Button _btnGammaSRGB;
        private ToolTip _ttpMain;
        protected HexColorBox m_txtHexColorBox;
        private Rectangle m_hBarWakuIn;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorPickerControl()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();
//			hookNotify = RegisterWindowMessage("MOUSEHOOK_MSG");

            // 色情報の同期を行う
            m_color.Set(Color.Black);
            m_componentPanel.Color = m_color.AsColor();
            m_txtHexColorBox.Color = m_color.AsColor();

            //	イベント、プロパティ設定
            this.m_gradCtrl.ColorSelected	+= new SaturationValueBar.ColorSelectedHandler( GradCtrl_ColorSelected );
            this.m_hBar.ColorSelected		+= new ColorSlider.ColorSelectedHandler( HueBar_ColorSelected );
            this.m_pltPanel.ColorSelected	+= new PalettePanel.ColorSelectedHandler( PalettePanel_ColorSelected );
            this.m_componentPanel.ColorChanged	+= new ColorChangedEventHandler( ComponentPanel_ColorSelected );
            this.m_txtHexColorBox.ColorChanged  += new ColorChangedEventHandler( HexColorBox_ColorSelected );
            this.m_txtHexColorBox.HexColorError += new EventHandler( HexColorBox_HexColorError );

            //	スポイトボタンを無効化中はボタンをずらす
            this.m_colSetButton.Location	= new Point( this.m_colSetButton.Location.X, this.m_colSetButton.Location.Y + this.m_colSetButton.Size.Height / 2 );

            this.DoubleBuffered = true;
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if( components != null )
                    components.Dispose();
            }
            base.Dispose( disposing );
        }

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード］エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(ColorPickerControl));
            this.m_colSetButton = new System.Windows.Forms.Button();
            this.m_btnToggleRgbHsv = new System.Windows.Forms.Button();
            this._btnGammaLiner = new System.Windows.Forms.Button();
            this._btnGammaSRGB = new System.Windows.Forms.Button();
            this._ttpMain = new System.Windows.Forms.ToolTip(this.components);
            this.m_txtHexColorBox = new NW4R.ColorPicker.HexColorBox();
            this.m_componentPanel = new NW4R.ColorPicker.ComponentPanel();
            this.m_pltPanel = new NW4R.ColorPicker.PalettePanel();
            this.m_colorPanel = new NW4R.ColorPicker.ColorBox();
            this.m_hBar = new NW4R.ColorPicker.HueBar();
            this.m_gradCtrl = new NW4R.ColorPicker.SaturationValueBar();
            this.SuspendLayout();
            //
            // m_colSetButton
            //
            resources.ApplyResources(this.m_colSetButton, "m_colSetButton");
            this.m_colSetButton.Name = "m_colSetButton";
            this.m_colSetButton.MouseDown += new System.Windows.Forms.MouseEventHandler(this.colSetButton_MouseDown);
            //
            // m_btnToggleRgbHsv
            //
            resources.ApplyResources(this.m_btnToggleRgbHsv, "m_btnToggleRgbHsv");
            this.m_btnToggleRgbHsv.Name = "m_btnToggleRgbHsv";
            this.m_btnToggleRgbHsv.UseVisualStyleBackColor = true;
            this.m_btnToggleRgbHsv.Click += new System.EventHandler(this.Event_btnToggleRgbHsv_Click);
            //
            // _btnGammaLiner
            //
            resources.ApplyResources(this._btnGammaLiner, "_btnGammaLiner");
            this._btnGammaLiner.Name = "_btnGammaLiner";
            this._ttpMain.SetToolTip(this._btnGammaLiner, resources.GetString("_btnGammaLiner.ToolTip"));
            this._btnGammaLiner.UseVisualStyleBackColor = true;
            this._btnGammaLiner.Click += new System.EventHandler(this.Event_btnDeGamma_Click);
            //
            // _btnGammaSRGB
            //
            resources.ApplyResources(this._btnGammaSRGB, "_btnGammaSRGB");
            this._btnGammaSRGB.Name = "_btnGammaSRGB";
            this._ttpMain.SetToolTip(this._btnGammaSRGB, resources.GetString("_btnGammaSRGB.ToolTip"));
            this._btnGammaSRGB.UseVisualStyleBackColor = true;
            this._btnGammaSRGB.Click += new System.EventHandler(this.Event_btnGammaSRGB_Click);
            //
            // m_txtHexColorBox
            //
            resources.ApplyResources(this.m_txtHexColorBox, "m_txtHexColorBox");
            this.m_txtHexColorBox.Name = "m_txtHexColorBox";
            //
            // m_componentPanel
            //
            resources.ApplyResources(this.m_componentPanel, "m_componentPanel");
            this.m_componentPanel.Name = "m_componentPanel";
            //
            // m_pltPanel
            //
            this.m_pltPanel.ColorHNumber = 8;
            this.m_pltPanel.ColorVNumber = 2;
            resources.ApplyResources(this.m_pltPanel, "m_pltPanel");
            this.m_pltPanel.Name = "m_pltPanel";
            this.m_pltPanel.TabStop = false;
            //
            // m_colorPanel
            //
            this.m_colorPanel.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
            resources.ApplyResources(this.m_colorPanel, "m_colorPanel");
            this.m_colorPanel.Name = "m_colorPanel";
            //
            // m_hBar
            //
            this.m_hBar.ColorValue = 0;
            resources.ApplyResources(this.m_hBar, "m_hBar");
            this.m_hBar.MaxValue = 255;
            this.m_hBar.Name = "m_hBar";
            this.m_hBar.TabStop = false;
            //
            // m_gradCtrl
            //
            this.m_gradCtrl.Color = System.Drawing.Color.FromArgb(((int)(((byte)(0)))), ((int)(((byte)(0)))), ((int)(((byte)(0)))));
            resources.ApplyResources(this.m_gradCtrl, "m_gradCtrl");
            this.m_gradCtrl.Name = "m_gradCtrl";
            this.m_gradCtrl.TabStop = false;
            //
            // ColorPickerControl
            //
            this.Controls.Add(this.m_txtHexColorBox);
            this.Controls.Add(this._btnGammaSRGB);
            this.Controls.Add(this._btnGammaLiner);
            this.Controls.Add(this.m_btnToggleRgbHsv);
            this.Controls.Add(this.m_componentPanel);
            this.Controls.Add(this.m_pltPanel);
            this.Controls.Add(this.m_colSetButton);
            this.Controls.Add(this.m_colorPanel);
            this.Controls.Add(this.m_hBar);
            this.Controls.Add(this.m_gradCtrl);
            this.Name = "ColorPickerControl";
            resources.ApplyResources(this, "$this");
            this.ResumeLayout(false);

        }
        #endregion

        #region イベント

        /// <summary>
        /// カラーが変更されたときに発生します。
        /// </summary>
        [Category("カスタム")]
        [Description("カラーが変更されたときに発生します。")]
        public event ColorChangedEventHandler ColorChanged;

        /// <summary>
        /// 16進数カラーが変更でエラーが検出されたときに発生します。
        /// </summary>
        [Category("カスタム")]
        [Description("16進数カラーが変更でエラーが検出されたときに発生します。")]
        public event EventHandler HexColorError;

        #endregion

        #region プロパティ

        /// <summary>
        /// 選択カラー。
        /// </summary>
        [Category("カスタム")]
        [Description("選択カラー。")]
        [DefaultValue(typeof(Color), "Black")]
        public Color Color
        {
            get {	return m_color.AsColor();		}
            set {
                this.m_componentPanel.Color	= value;
                this.m_txtHexColorBox.Color = value;
                m_color.Set(value);
            }
        }

        /// <summary>
        /// カラー値の表示方式。
        /// </summary>
        [Category("カスタム")]
        [Description("カラー値の表示方式。")]
        [DefaultValue(ValueMode.Uchar)]
        public ValueMode	ValueMode
        {
            get	{	return this.m_componentPanel.ValueMode;		}
            set	{	this.m_componentPanel.ValueMode	= value;	}
        }

        /// <summary>
        /// カラー成分のモード。
        /// </summary>
        [Category("カスタム")]
        [Description("カラー成分のモード。")]
        [DefaultValue(ColorMode.RGBA)]
        public ColorMode	ColorMode
        {
            get	{	return this.m_componentPanel.ColorMode;		}
            set
            {
                this.m_componentPanel.ColorMode	= value;

                bool gammaConvertEnalbed = this.ColorMode != ColorMode.HSV && this.ColorMode != ColorMode.HSVA;
                this._btnGammaLiner.Enabled = gammaConvertEnalbed;
                this._btnGammaSRGB.Enabled = gammaConvertEnalbed;
            }
        }

        /// <summary>
        /// カラーパレット（１６色）。
        /// </summary>
        [Category("カスタム")]
        [Description("カラーパレット（16色）。")]
        public Color[]	CustomColors
        {
            get	{	return m_pltPanel.GetPalette();		}
        }

        /// <summary>
        /// テキストボックスの表示位置を右側に設定する。
        /// </summary>
        [Category("カスタム")]
        [Description("テキストボックスの表示位置を右側に設定する。")]
        [DefaultValue(false)]
        public bool		TextBoxIsRight
        {
            get	{	return this.m_componentPanel.TextBoxIsRight;		}
            set	{	this.m_componentPanel.TextBoxIsRight	= value;	}
        }

        #endregion

        #region オーバーライド

        /// <summary>
        /// OnLoadのオーバーライド（プロパティ値からコントロールの設定を初期化する）。
        /// </summary>
        /// <param name="e"></param>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad (e);

            //	プロパティ値からコントロールの設定を初期化
            InitializeControl();
        }

        /// <summary>
        /// OnPaintのオーバーライド（ボーダーの装飾）。
        /// </summary>
        /// <param name="e"></param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint (e);

            Graphics g = e.Graphics;
            if( this.DesignMode )
                InitializeLine();

            Pen		pen	= this.Enabled ? Pens.Black : Pens.DarkGray;
            if( this.m_gradCtrl.Visible )
            {
                ControlPaint.DrawBorder3D( g, m_GradBoxWakuOut );
                g.DrawRectangle( pen, m_GradBoxWakuIn );
            }
            if( this.m_hBar.Visible )
            {
                ControlPaint.DrawBorder3D( g, m_hBarWakuOut );
                g.DrawRectangle( pen, m_hBarWakuIn );
            }
            if( this.m_colorPanel.Visible )
            {
                Point[]	pp	= new Point[3];
                pp[0]	= new Point( this.m_colorPanel.Location.X - 2, this.m_colorPanel.Location.Y + this.m_colorPanel.Size.Height + 2 );
                pp[1]	= new Point( pp[0].X, this.m_colorPanel.Location.Y - 2 );
                pp[2]	= new Point( this.m_colorPanel.Location.X + this.m_colorPanel.Size.Width + 2, pp[1].Y );
                using( Pen tmpPen = new Pen( Color.White, 2 ) )
                {
                    g.DrawLines( tmpPen, pp );
                }
                pp[1]	= new Point( pp[2].X, pp[0].Y );
                using( Pen tmpPen = new Pen( Color.Gray, 2 ) )
                {
                    g.DrawLines( tmpPen, pp );
                }
            }
        }
        /// <summary>
        /// WndProcのオーバーライド
        /// </summary>
        /// <param name="m"></param>
        protected override void WndProc(ref Message m)
        {
            if( this.Capture )
            {
                switch(m.Msg)
                {
                    case 0x0201://WM_MOUSEDOWN:
                    {
                        Bitmap bmp = new Bitmap(Screen.PrimaryScreen.Bounds.Width,
                            Screen.PrimaryScreen.Bounds.Height);
                        using( Graphics g = Graphics.FromImage(bmp) )
                        {
                            IntPtr srcHDC   = DllClass.GetDC(IntPtr.Zero);
                            IntPtr dstHDC   = g.GetHdc();

                            DllClass.BitBlt(
                                dstHDC, 0, 0, bmp.Width, bmp.Height,
                                srcHDC, 0, 0,
                                DllClass.SRCCOPY);

                            g.ReleaseHdc(dstHDC);
                            DllClass.ReleaseDC(srcHDC);

                            if( bmp != null )
                            {
                                int		x	= Control.MousePosition.X;
                                int		y	= Control.MousePosition.Y;
                                if( x < bmp.Width && y < bmp.Height )
                                {
                                    Color	col	= bmp.GetPixel( x, y );
                                    this.m_colorPanel.BackColor	= col;
                                    this.m_colorPanel.Invalidate();
                                }
                            }
                        }
                        break;
                    }
                    case 0x0202://WM_LBUTTONUP:
                    case 0x0204://WM_RBUTTONDOWN:
                    case 0x0207://WM_MBUTTONDOWN:
                    case 0x020B://WM_XBUTTONDOWN:
                    case 0x0290://WM_IME_KEYDOWN:
                        this.Capture	= false;
                        this.Cursor		= Cursors.Default;
                        break;
                }
            }

            base.WndProc(ref m);
        }

        #endregion

        /// <summary>
        /// イベント発行せずに、現在のカラーを変更する。
        /// </summary>
        public void SetDoubleColor(ColorHelper.DoubleColor doubleColor)
        {
            Color color = doubleColor.AsColor();
            if (this.Color != color)
            {
                m_color.Set(doubleColor._a, doubleColor._r, doubleColor._g, doubleColor._b);
                this.m_componentPanel.SetColor(color);
                this.m_txtHexColorBox.Color = color;
                m_colorPanel.BackColor = color;
                HSV hsb = HSV.FromRGB(new RGB(color));
                if (this.m_gradCtrl.UpdateColor(hsb, true))
                {
                    int hh = (int)Math.Round((double)hsb.H * 255 / 360);
                    m_hBar.ColorValue = (hh & 0xFF);
                }
                this.Invalidate();
            }
        }

        /// <summary>
        /// イベント発行せずに、現在のカラーを変更する。
        /// </summary>
        public void SetColor(Color color)
        {
            this.SetDoubleColor(new ColorHelper.DoubleColor(color.A, color.R, color.G, color.B));
        }

        #region カラー変更イベント処理

        /// <summary>
        /// グラデーションピッカーでカラー選択されたときに実行される
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void GradCtrl_ColorSelected( object sender, ColorChangedEventArgs e )
        {
            //	アルファ以外が変更されたか
            if (e.Color.R != this.Color.R || e.Color.G != this.Color.G || e.Color.B != this.Color.B)
            {
                Color select = Color.FromArgb(this.Color.A, e.Color);
                if( this.m_componentPanel.Visible )
                    this.m_componentPanel.UpdateColor( select, true );
                ChangeColor( new ColorChangedEventArgs( select, e.EditFixed ));
            }
            else if( e.EditFixed )
                ChangeColor(new ColorChangedEventArgs(this.Color, e.EditFixed));
            ( (Control)sender ).Invalidate();
        }

        /// <summary>
        /// Hueスライダーでカラー選択されたときに実行される
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void HueBar_ColorSelected( object sender, ColorChangedEventArgs e )
        {
            if( this.m_gradCtrl.UpdateColor( new HSV( (int)Math.Round( (double)( this.m_hBar.ColorValue * 360 / 255 ) ), 100, 100 ), false ) )
            {
                Color select = Color.FromArgb(this.Color.A, this.m_gradCtrl.GetSelectPointColor());
                if( this.m_componentPanel.Visible )
                    this.m_componentPanel.UpdateColor( select, true );
                ChangeColor( new ColorChangedEventArgs( select, e.EditFixed ) );
            }
            else if( e.EditFixed )
                ChangeColor(new ColorChangedEventArgs(this.Color, e.EditFixed));
            ( (Control)sender ).Invalidate();
        }

        /// <summary>
        /// パレットでカラー選択されたときに実行される
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void PalettePanel_ColorSelected( object sender, ColorChangedEventArgs e )
        {
            //	更新
            HSV		hsb	= HSV.FromRGB( new RGB( e.Color ) );
            if( this.m_gradCtrl.UpdateColor( hsb, true ) )
            {
                Color color = Color.FromArgb(this.Color.A, e.Color);
                if( this.m_componentPanel.Visible )
                    this.m_componentPanel.UpdateColor( color, true );
                int		hh	= (int)Math.Round( (double)hsb.H*255/360 );
                if( m_hBar.ColorValue != hh )
                {
                    m_hBar.ColorValue	= ( hh & 0xFF );
                    this.m_hBar.Invalidate();
                }
                ChangeColor( new ColorChangedEventArgs( color, e.EditFixed ) );
            }
        }

        /// <summary>
        /// コンポーネントパネルでカラー選択されたときに実行される
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ComponentPanel_ColorSelected( object sender, ColorChangedEventArgs e )
        {
            //	更新
            HSV		hsb	= ( m_componentPanel.ColorMode == ColorMode.HSVA || m_componentPanel.ColorMode == ColorMode.HSV ) ?
                m_componentPanel.HSV : HSV.FromRGB( new RGB( e.Color ) );
            if( this.m_gradCtrl.UpdateColor( hsb, true ) )
            {
                int		hh	= (int)Math.Round( (double)hsb.H*255/360 );
                if( m_hBar.ColorValue != hh )
                {
                    m_hBar.ColorValue	= ( hh & 0xFF );
                    this.m_hBar.Invalidate();
                }
                ChangeColor( e );
            }
            else if (this.Color.A != e.Color.A)
            {
                ChangeAlpha( e );
            }
            else if( e.EditFixed )
                ChangeColor( e );
        }

        /// <summary>
        /// 16進数値入力でカラー選択されたときに実行される
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void HexColorBox_ColorSelected(object sender, ColorChangedEventArgs e)
        {
            //	更新
            HSV hsb = HSV.FromRGB(new RGB(e.Color));
            if (this.m_gradCtrl.UpdateColor(hsb, true))
            {
                Color color = Color.FromArgb(this.Color.A, e.Color);
                if (this.m_componentPanel.Visible)
                    this.m_componentPanel.UpdateColor(color, true);
                int hh = (int)Math.Round((double)hsb.H * 255 / 360);
                if (m_hBar.ColorValue != hh)
                {
                    m_hBar.ColorValue = (hh & 0xFF);
                    this.m_hBar.Invalidate();
                }
                ChangeColor(new ColorChangedEventArgs(color, e.EditFixed));
                m_componentPanel.ColorMode = m_componentPanel.ColorMode; // ColorModeのsetでcomponentPanelが更新される
            }
        }

        /// <summary>
        /// 16進数値入力でエラーが発生したときに実行される
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void HexColorBox_HexColorError(object sender, EventArgs e)
        {
            HexColorError(sender, e);
        }

        /// <summary>
        /// RGBカラーが変更されたときに呼ばれる
        /// </summary>
        /// <param name="e"></param>
        private void ChangeColor( ColorChangedEventArgs e )
        {
            this.Color = e.Color;

            this.m_colorPanel.BackColor	= Color.FromArgb( e.Color.R, e.Color.G, e.Color.B );
            if( ColorChanged != null )
                ColorChanged( this, e );
        }

        /// <summary>
        /// Alphaカラーが変更されたときに呼ばれる
        /// </summary>
        /// <param name="e"></param>
        private void ChangeAlpha( ColorChangedEventArgs e )
        {
            this.Color = e.Color;
            if( ColorChanged != null )
                ColorChanged( this, e );
        }

        #endregion

        /// <summary>
        /// 選択色をパレットにコピーするボタンが押された
        /// </summary>
        private void colSetButton_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            this.m_pltPanel.UpdatePalette(this.m_pltPanel.Selection, this.Color);
        }

        /// <summary>
        /// プロパティ値からコントロールの設定を初期化
        /// </summary>
        private void	InitializeControl()
        {
            InitializeLine();
            HSV hsb = HSV.FromRGB(new RGB(this.Color));
            this.m_gradCtrl.CreateColorBitmap( hsb );
            this.m_hBar.ColorValue	= (int)Math.Round((double)hsb.H*255/360);
            this.m_hBar.CreateColorBitmap();

            this.m_colorPanel.BackColor = Color.FromArgb(this.Color.R, this.Color.G, this.Color.B);
        }

        /// <summary>
        /// ラインの再設定
        /// </summary>
        private void	InitializeLine()
        {
            int		x, y, w, h;
            x	= this.m_gradCtrl.Location.X;
            y	= this.m_gradCtrl.Location.Y;
            w	= this.m_gradCtrl.Size.Width;
            h	= this.m_gradCtrl.Size.Height;
            m_GradBoxWakuOut	= new Rectangle( x-3, y-3, w+7, h+7 );
            m_GradBoxWakuIn		= new Rectangle( x-1, y-1, w+1, h+1 );
            x	= this.m_hBar.Location.X;
            y	= this.m_hBar.Location.Y;
            w	= this.m_hBar.Size.Width;
            h	= this.m_hBar.Size.Height;
            m_hBarWakuOut		= new Rectangle( x-3, y-3, w+7, h+7 );
            m_hBarWakuIn		= new Rectangle( x-1, y-1, w+1, h+1 );
        }
#if false
        #region フック用変数、関数郡

        //	フック用変数
        static int			m_mouseHook	= 0;
        public const int	WH_MOUSE	= 0x07;

        // フックプロシジャの型宣言
        public delegate int HookProc(int nCode, IntPtr wParam, IntPtr lParam);
        //フックプロシジャのアドレスを保有する変数の宣言.
        HookProc MouseHookProcedure;

        //ポイント構造体の宣言
        [StructLayout(LayoutKind.Sequential)]
        public class POINT
        {
            public int x;
            public int y;
        }
        //MouseHookStruct構造体の宣言
        [StructLayout(LayoutKind.Sequential)]
        public class MouseHookStruct
        {
            public POINT pt;
            public int hwnd;
            public int wHitTestCode;
            public int dwExtraInfo;
        }

        [DllImport("USER32.DLL")]
        public static extern uint RegisterWindowMessage(string identifier);

        [DllImport("KERNEL32.DLL")]
        public static extern void OutputDebugString(string msg);

        [DllImport("MouseHook.DLL")]
        public static extern void SetTarget(IntPtr hwnd);
        // SetWindowsHookExプロシジャのインポートおよび宣言.
        // このAPIでフックプロシジャを登録する.
        [DllImport("user32.dll",CharSet=CharSet.Auto,
             CallingConvention=CallingConvention.StdCall)]
        public static extern int SetWindowsHookEx(
            int idHook, HookProc lpfn, IntPtr hInstance, int threadId);

        //UnhookWindowsHookExプロシジャのインポートおよび宣言.
        //このAPIでフックプロシジャを解放する.
        [DllImport("user32.dll",CharSet=CharSet.Auto,
             CallingConvention=CallingConvention.StdCall)]
        public static extern bool UnhookWindowsHookEx(int idHook);

        //CallNextHookExプロシジャのインポートおよび宣言.
        //このAPIで次の（標準の）プロシジャに制御を渡す.
        [DllImport("user32.dll",CharSet=CharSet.Auto,
             CallingConvention=CallingConvention.StdCall)]
        public static extern int CallNextHookEx(int idHook, int nCode, IntPtr wParam, IntPtr lParam);

        static private Form	m_form;
        // フックプロシジャ
        public static int MouseHookProc(int nCode, IntPtr wParam, IntPtr lParam)
        {
            //Marshall the data from callback.
            MouseHookStruct MyMouseHookStruct = (MouseHookStruct)Marshal.PtrToStructure(lParam, typeof(MouseHookStruct));
            if (nCode < 0)
            {
                return CallNextHookEx(m_mouseHook, nCode, wParam, lParam);
//				return 0;
            }
            else
            {
                //マウス座標を取得する
                String strCaption = "x = " +
                    MyMouseHookStruct.pt.x.ToString("d") +
                    "  y = " +
                    MyMouseHookStruct.pt.y.ToString("d");

                //アクティブなフォームを取得する.
                Form tempForm = Form.ActiveForm;

                //キャプションに表示.
                if( m_form != null )
                    m_form.Text = strCaption;

                return CallNextHookEx(m_mouseHook, nCode, wParam, lParam);
//				return 0;
            }
        }
        #endregion
#endif
        private void colPickButton_Click(object sender, System.EventArgs e)
        {
            this.Capture	= true;
            this.Cursor		= Cursors.Cross;
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス前景色を設定する
        /// </summary>
        /// <param name="ch">設定するチャンネル</param>
        /// <param name="color">前景色</param>
        public void SetTextBoxForeColor(ColorChannel ch, Color color)
        {
            m_componentPanel.SetTextBoxForeColor(ch, color);
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス前景色を取得する
        /// </summary>
        /// <param name="ch">取得するチャンネル</param>
        /// <returns>前景色</returns>
        public Color GetTextBoxForeColor(ColorChannel ch)
        {
            return m_componentPanel.GetTextBoxForeColor(ch);
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス背景色を設定する
        /// </summary>
        /// <param name="ch">設定するチャンネル</param>
        /// <param name="color">背景色</param>
        public void SetTextBoxBackColor(ColorChannel ch, Color color)
        {
            m_componentPanel.SetTextBoxBackColor(ch, color);
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス背景色を取得する
        /// </summary>
        /// <param name="ch">取得するチャンネル</param>
        /// <returns>背景色</returns>
        public Color GetTextBoxBackColor(ColorChannel ch)
        {
            return m_componentPanel.GetTextBoxBackColor(ch);
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックスフォントを設定する
        /// </summary>
        /// <param name="ch">設定するチャンネル</param>
        /// <param name="font">フォント</param>
        public void SetTextBoxFont(ColorChannel ch, Font font)
        {
            m_componentPanel.SetTextBoxFont(ch, font);
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックスフォントを取得する
        /// </summary>
        /// <param name="ch">取得するチャンネル</param>
        /// <returns>フォント</returns>
        public Font GetTextBoxFont(ColorChannel ch)
        {
            return m_componentPanel.GetTextBoxFont(ch);
        }

        /// <summary>
        /// カラーモードのトグル
        /// </summary>
        private void Event_btnToggleRgbHsv_Click(object sender, EventArgs e)
        {
            switch (m_componentPanel.ColorMode)
            {
                case ColorMode.RGB: this.ColorMode = ColorMode.HSV; break;
                case ColorMode.RGBA: this.ColorMode = ColorMode.HSVA; break;
                case ColorMode.HSV: this.ColorMode = ColorMode.RGB; break;
                case ColorMode.HSVA: this.ColorMode = ColorMode.RGBA; break;
            }
        }

        /// <summary>
        /// リニア色空間
        /// </summary>
        private void Event_btnDeGamma_Click(object sender, EventArgs e)
        {
            switch (m_componentPanel.ColorMode)
            {
                case ColorMode.RGB:
                case ColorMode.RGBA:
                    this.SetDoubleColor(ColorHelper.ConvertToLinerGamma(this.m_color));
                    if (ColorChanged != null) { ColorChanged(this, new ColorChangedEventArgs(this.m_color.AsColor(), true)); }
                    break;
                case ColorMode.HSV:
                case ColorMode.HSVA:
                    break;
            }
        }

        private void Event_btnGammaSRGB_Click(object sender, EventArgs e)
        {
            switch (m_componentPanel.ColorMode)
            {
                case ColorMode.RGB:
                case ColorMode.RGBA:
                    this.SetDoubleColor(ColorHelper.ConvertToSRGBGamma(this.m_color));
                    if (ColorChanged != null) { ColorChanged(this, new ColorChangedEventArgs(this.m_color.AsColor(), true)); }
                    break;
                case ColorMode.HSV:
                case ColorMode.HSVA:
                    break;
            }
        }
    }

    /// <summary>
    /// カラーヘルパ
    /// </summary>
    public static class ColorHelper
    {
        /// <summary>
        /// ガンマ補正結果を誤差無く記憶しておくために使われる倍精度カラーです。
        /// </summary>
        public class DoubleColor
        {
            public double _r;
            public double _g;
            public double _b;
            public double _a;

            public DoubleColor()
            {
            }

            public DoubleColor(double a, double r, double g, double b)
            {
                this.Set(a, r, g, b);
            }

            public void Set(double a, double r, double g, double b)
            {
                _a = a;
                _r = r;
                _g = g;
                _b = b;
            }

            public void Set(Color col)
            {
                this.Set(col.A, col.R, col.G, col.B);
            }

            public Color AsColor()
            {
                return Color.FromArgb((int)this._a, (int)this._r, (int)this._g, (int)this._b);
            }
        }

        /// <summary>
        /// リニア色空間への変換
        /// </summary>
        public static DoubleColor ConvertToLinerGamma(DoubleColor col)
        {
            double r = (Math.Pow(col._r / 255.0, 2.2) * 255.0);
            double g = (Math.Pow(col._g / 255.0, 2.2) * 255.0);
            double b = (Math.Pow(col._b / 255.0, 2.2) * 255.0);

            // リニア特性を与える、0～255になっていることに注意が必要
            return new DoubleColor((double)col._a, r, g, b);
        }

        /// <summary>
        /// sRGB色空間への変換
        /// </summary>
        public static DoubleColor ConvertToSRGBGamma(DoubleColor col)
        {
            // sRGB特性を与える、0～255になっていることに注意が必要
            return new DoubleColor(col._a,
                (Math.Pow(((double)col._r / 255.0), 1.0 / 2.2) * 255.0),
                (Math.Pow(((double)col._g / 255.0), 1.0 / 2.2) * 255.0),
                (Math.Pow(((double)col._b / 255.0), 1.0 / 2.2) * 255.0));
        }
    }

    /// <summary>
    /// 外部DLLからExternした関数郡
    /// </summary>
    internal class DllClass
    {
        [DllImport("user32.dll")]
        public static extern IntPtr GetDC(IntPtr hWnd) ;
        [DllImport("user32.dll")]
        public static extern IntPtr ReleaseDC(IntPtr hDc) ;
        public const int SRCCOPY = 0xcc0020 ;
        [DllImport("gdi32.dll")]
        public static extern bool BitBlt(
            IntPtr hdcDst,
            int    xDst,
            int    yDst,
            int    width,
            int    height,
            IntPtr hdcSrc,
            int    xSrc,
            int    ySrc,
            int    rasterOp
            ) ;
    }

    /// <summary>
    /// カラー値の表示方式。
    /// </summary>
    public enum ValueMode : int
    {
        ///	<summary>0～各要素のMAX範囲。</summary>
        Uchar,
        ///	<summary>0.0～1.0範囲。</summary>
        Float
    };

    /// <summary>
    ///	カラー成分のモード。
    /// </summary>
    public enum ColorMode : int
    {
        ///	<summary>RGBA設定。</summary>
        RGBA,
        ///	<summary>RGB設定。</summary>
        RGB,
        ///	<summary>HSVA設定。</summary>
        HSVA,
        ///	<summary>HSV設定。</summary>
        HSV,
        ///	<summary>A設定。</summary>
        A
    };

    /// <summary>
    /// カラーチャンネル種別
    /// </summary>
    public enum ColorChannel : int
    {
        ///	<summary>R</summary>
        R,
        ///	<summary>G</summary>
        G,
        ///	<summary>B</summary>
        B,
        ///	<summary>A</summary>
        A
    }

    /// <summary>
    /// カラー変更イベントを処理するメソッドを表します。
    /// </summary>
    public delegate void ColorChangedEventHandler(object sender, ColorChangedEventArgs e);

    /// <summary>
    /// カラー変更イベントデータ。
    /// </summary>
    public class ColorChangedEventArgs : EventArgs
    {
        private readonly Color m_color;
        private readonly bool m_editFixed;

        /// <summary>
        /// 選択カラー。
        /// </summary>
        public Color Color
        {
            get { return m_color; }
        }
        /// <summary>
        /// エディット終了か否か。
        /// </summary>
        public bool EditFixed
        {
            get { return m_editFixed; }
        }
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="color">選択カラー。</param>
        /// <param name="editfixed">エディット終了か否か。</param>
        public ColorChangedEventArgs( Color color, bool editfixed )
        {
            m_color = color;
            m_editFixed	= editfixed;
        }
    }
}
