﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using BitMiracle.LibTiff.Classic;

namespace TiffBmpCvtr
{
    /// <summary>
    ///
    /// </summary>
    class Program
    {
        /// <summary>
        /// エントリ
        /// </summary>
        static void Main(string[] args)
        {
            string inputFilePath = args[0];
            if (string.Compare(Path.GetExtension(inputFilePath), ".bmp", true) == 0)
            {
                ConvertToTiff_(inputFilePath);
            }
            else
            {
                ConvertToBmp_(inputFilePath);
            }
        }

        //----------------------------------------------------------

        private static void ConvertToBmp_(string inputFilePath)
        {
            Bitmap bmp = Bitmap.FromFile(inputFilePath) as Bitmap;
            string bmpFilePath = Path.ChangeExtension(inputFilePath, ".bmp");
            if (File.Exists(bmpFilePath))
            {
                File.Delete(bmpFilePath);
            }

            bmp.Save(bmpFilePath, ImageFormat.Bmp);
        }

        //----------------------------------------------------------

        private static void ConvertToTiff_(string inputFilePath)
        {
            Bitmap bmp = new Bitmap(inputFilePath);
            Rectangle bmBounds = new Rectangle(0, 0, bmp.Width, bmp.Height);
            BitmapData bmpd = bmp.LockBits(bmBounds, ImageLockMode.ReadOnly, bmp.PixelFormat);

            Bitmap bmp2 = new Bitmap(bmpd.Width, bmpd.Height, bmpd.Stride, PixelFormat.Format32bppArgb, bmpd.Scan0);

            byte[] raster = GetImageRasterBytes_(bmp2, PixelFormat.Format32bppArgb);
            bmp.UnlockBits(bmpd);
            bmp.Dispose();

            string tifFilePath = Path.ChangeExtension(inputFilePath, ".tif");
            if (File.Exists(tifFilePath))
            {
                File.Delete(tifFilePath);
            }

            using (Tiff tif = Tiff.Open(tifFilePath, "w"))
            {
                tif.SetField(TiffTag.IMAGEWIDTH, bmp2.Width);
                tif.SetField(TiffTag.IMAGELENGTH, bmp2.Height);
                tif.SetField(TiffTag.COMPRESSION, Compression.NONE);
                tif.SetField(TiffTag.PHOTOMETRIC, Photometric.RGB);

                tif.SetField(TiffTag.ROWSPERSTRIP, bmp2.Height);

                tif.SetField(TiffTag.XRESOLUTION, bmp2.HorizontalResolution);
                tif.SetField(TiffTag.YRESOLUTION, bmp2.VerticalResolution);

                tif.SetField(TiffTag.BITSPERSAMPLE, 8);
                tif.SetField(TiffTag.SAMPLESPERPIXEL, 4);

                tif.SetField(TiffTag.PLANARCONFIG, PlanarConfig.CONTIG);

                int stride = raster.Length / bmp2.Height;
                ConvertSamples_(raster, bmp2.Width, bmp2.Height);

                int h = bmp2.Height;
                bmp2.Dispose();
                for (int i = 0, offset = 0; i < h; i++)
                {
                    tif.WriteScanline(raster, offset, i, 0);
                    offset += stride;
                }
            }
        }

        //----------------------------------------------------------

        private static byte[] GetImageRasterBytes_(Bitmap bmp, PixelFormat format)
        {
            Rectangle rect = new Rectangle(0, 0, bmp.Width, bmp.Height);
            byte[] bits = null;

            BitmapData bmpdata = bmp.LockBits(rect, ImageLockMode.ReadWrite, format);
            bits = new byte[bmpdata.Stride * bmpdata.Height];
            System.Runtime.InteropServices.Marshal.Copy(bmpdata.Scan0, bits, 0, bits.Length);
            bmp.UnlockBits(bmpdata);

            return bits;
        }

        //----------------------------------------------------------

        private static void ConvertSamples_(byte[] data, int width, int height)
        {
            int stride = data.Length / height;
            const int samplesPerPixel = 4;

            for (int y = 0; y < height; y++)
            {
                int offset = stride * y;
                int strideEnd = offset + width * samplesPerPixel;

                for (int i = offset; i < strideEnd; i += samplesPerPixel)
                {
                    byte temp = data[i + 2];
                    data[i + 2] = data[i];
                    data[i] = temp;
                }
            }
        }
    }
}
