﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Reflection;


namespace NintendoWare.Font
{
    public class ConvertImageFontCache : IDisposable
    {
        public Mutex hashFileMutex;
        private MemoryStream hashSourceStream;
        private BinaryWriter hashSourceBinaryWriter;
        private string cacheKeyHashValue;

        public ConvertImageFontCache()
        {
            hashSourceStream = new MemoryStream();
            hashSourceBinaryWriter = new BinaryWriter(hashSourceStream);

            var assembly = Assembly.GetExecutingAssembly();

            // FontConverterCommon の AssemblyVersion にはビルド日時の情報が含まれる
            AddCacheKey(assembly.GetName().Version.Build);
            AddCacheKey(assembly.GetName().Version.Revision);

            AddCacheKey(GlCm.GetFontConverterVersion());
        }

        public void Dispose()
        {
            if (hashFileMutex != null)
            {
                // hashFileMutex.WaitOne() したので開放する
                hashFileMutex.ReleaseMutex();
                hashFileMutex.Dispose();
                hashFileMutex = null;
            }

            if (hashSourceStream != null)
            {
                hashSourceStream.Dispose();
                hashSourceStream = null;
            }

            if (hashSourceBinaryWriter != null)
            {
                hashSourceBinaryWriter.Dispose();
                hashSourceBinaryWriter = null;
            }
        }

        // キーに追加
        public void AddCacheKey(string value)
        {
            hashSourceBinaryWriter.Write("s");
            hashSourceBinaryWriter.Write(value.Length);
            hashSourceBinaryWriter.Write(value);
        }

        // キーに追加
        public void AddCacheKey(int num)
        {
            hashSourceBinaryWriter.Write("i");
            hashSourceBinaryWriter.Write(num);
        }

        // キーに追加
        public void AddCacheKey(byte[] bytes)
        {
            hashSourceBinaryWriter.Write("b");
            hashSourceBinaryWriter.Write(bytes.Length);
            hashSourceBinaryWriter.Write(bytes);
        }

        // キーに追加
        public void AddCacheKeyFromFile(string path)
        {
            try
            {
                if (File.Exists(path))
                {
                    var bytes = File.ReadAllBytes(path);
                    AddCacheKey(bytes);
                }
            }
            catch
            {
                hashSourceBinaryWriter.Write("e");
            }
        }

        // ハッシュの取得
        public string GetCacheKeyHashValue()
        {
            if (cacheKeyHashValue == null)
            {
                hashSourceBinaryWriter.Flush();
                hashSourceStream.Position = 0;
                var sha1 = new System.Security.Cryptography.SHA1Managed();
                var hash = sha1.ComputeHash(hashSourceStream);
                cacheKeyHashValue = ToBase32(hash);
            }

            return cacheKeyHashValue;
        }

        // bytes を Base32 に変換
        public string ToBase32(byte[] bytes)
        {
            char[] base32Table = Enumerable.Range('A', 'Z' - 'A' + 1).Select(x => (char)x)
                .Concat(Enumerable.Range('2', '7' - '2' + 1).Select(x => (char)x))
                .ToArray();
            string str = "";
            int b = 0;
            int offset = 0;
            for (int i = 0; i < bytes.Length; i++)
            {
                b |= (bytes[i] << offset);
                offset += 8;
                while (offset >= 5)
                {
                    str += base32Table[b & 0x1F];
                    b >>= 5;
                    offset -= 5;
                }
            }
            if (offset > 0)
            {
                str += base32Table[b & 0x1F];
            }

            return str;

        }

        // キャッシュファイルパスを取得
        public string GetCacheFilePath()
        {
            var hash = GetCacheKeyHashValue();
            if (hash != null)
            {
                var directory = ConverterEnvironment.PlatformCacheDirectory;

                if (!Directory.Exists(directory))
                {
                    Directory.CreateDirectory(directory);
                }

                return Path.Combine(directory, hash + ".c" + LibFormat.ExtensionFont);
            }

            return null;
        }

        // ミューテックスを取得する
        public void WaitCacheFileMutex()
        {
            string hashValue = GetCacheKeyHashValue();
            if (hashValue != null)
            {
                hashFileMutex = new System.Threading.Mutex(false, "NW_FontConverter_" + hashValue);
                hashFileMutex.WaitOne();
            }
        }

        // キャッシュファイルをコピー
        public bool CopyToOutput(string outputPath)
        {
            var source = GetCacheFilePath();
            if (File.Exists(source))
            {
                var now = DateTime.Now;

                // 更新日時に関係する問題が起きるのを避けるため各種時間を更新しておく
                File.SetCreationTime(source, now);
                File.SetLastWriteTime(source, now);
                File.SetLastAccessTime(source, now);

                File.Copy(source, outputPath, true);
                return true;
            }

            return false;
        }

        // キャッシュの整理
        public void ClearCache()
        {
            using (var clearCacheMutex = new System.Threading.Mutex(false, "NW_FontConverter_ClearCache"))
            {
                if (clearCacheMutex.WaitOne(0))
                {
                    try
                    {
                        var files = Directory.GetFiles(ConverterEnvironment.PlatformCacheDirectory, "*.c" + LibFormat.ExtensionFont);
                        if (files.Length > ConverterEnvironment.PlatformMaxCacheItemCount)
                        {
                            var info = files.Select(x => new FileInfo(x)).ToArray().OrderBy(x => x.CreationTime).Take(files.Length - ConverterEnvironment.PlatformMaxCacheItemCount);
                            foreach (var item in info)
                            {
                                var hashValue = Path.GetFileNameWithoutExtension(item.FullName).ToUpper();
                                using (var fileMutex = new System.Threading.Mutex(false, "NW_FontConverter_" + hashValue))
                                {
                                    if (fileMutex.WaitOne(0))
                                    {
                                        try
                                        {
                                            File.Delete(item.FullName);
                                        }
                                        finally
                                        {
                                            fileMutex.ReleaseMutex();
                                        }
                                    }
                                }
                            }
                        }
                    }
                    finally
                    {
                        clearCacheMutex.ReleaseMutex();
                    }
                }
            }
        }
    }
}
