﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.Globalization;
    using System.IO;
    using System.Threading;
    using System.Windows.Media;
    using NintendoWare.Font.Win32;
    using HDC = System.IntPtr;
    using LPARAM = System.IntPtr;
    using System.Reflection;

    public enum FontType
    {
        Raster,
        Vector
    }

    public struct IntColor
    {
        private uint value;

        private IntColor(uint val)
        {
            this.value = val;
        }

        public byte R
        {
            get { return (byte)(this.value >> 16); }
        }

        public byte G
        {
            get { return (byte)(this.value >> 8); }
        }

        public byte B
        {
            get { return (byte)this.value; }
        }

        public byte A
        {
            get { return (byte)(this.value >> 24); }
        }

        public static implicit operator uint(IntColor val)
        {
            return val.value;
        }

        public static implicit operator IntColor(uint val)
        {
            return new IntColor(val);
        }
    }

    public class IntArray : List<int>
    {
    }

    public class IntIntMap : Dictionary<int, int>
    {
        public void Add(IntIntMap rhs)
        {
            foreach (var i in rhs)
            {
                int value;
                var found = this.TryGetValue(i.Key, out value);
                if (found)
                {
                    this[i.Key] = value + i.Value;
                }
                else
                {
                    base.Add(i.Key, i.Value);
                }
            }
        }
    }

    public class FontInfo
    {
        public FontInfo(string faceName, FontType type)
        {
            this.FaceName = faceName;
            this.Type = type;
        }

        public string FaceName { get; private set; }

        public FontType Type { get; private set; }
    }

    public class FontMap : Dictionary<string, FontInfo>
    {
    }

    public partial class GlCm
    {
        public const string SystemFontFaceName = "nintendo_NTLG-DB_002";

        static bool _IsSystemFontLoaded
        {
            get { return _SystemFontHandle != IntPtr.Zero; }
        }

        static IntPtr _SystemFontHandle = IntPtr.Zero;

        public static IntColor BMP_RGB(byte r, byte g, byte b)
        {
            return ((uint)r << 16) | ((uint)g << 8) | ((uint)b << 0);
        }

        public static IntColor BMP_RGBA(byte r, byte g, byte b, byte a)
        {
            return ((uint)a << 24) | BMP_RGB(r, g, b);
        }

        public static uint ExtendBits(uint e, int srcLen, int dstLen)
        {
            int srcMax = (1 << srcLen) - 1;
            int dstMax = (1 << dstLen) - 1;
            return (uint)(e * dstMax / srcMax);
        }

        public static uint ExtractBits(uint e, int right, int len)
        {
            return (uint)((e >> right) & ((1 << len) - 1));
        }

        public static uint InverseNumber(uint num, int len)
        {
            int max = (1 << len) - 1;
            return (uint)(max - num);
        }

        public static byte RgbToGrayScale(IntColor c)
        {
            byte r = (byte)ExtractBits(c, 16, 8);
            byte g = (byte)ExtractBits(c, 8, 8);
            byte b = (byte)ExtractBits(c, 0, 8);

            return (byte)((r + g + b) / 3);
        }

        public static IntColor GrayScaleToRgb(uint level)
        {
            return GlCm.BMP_RGB((byte)level, (byte)level, (byte)level);
        }

        public static IntColor ColorToUint(Color c)
        {
            var r = c.R;
            var g = c.G;
            var b = c.B;
            var a = c.A;

            return BMP_RGBA(r, g, b, a);
        }

        public static void MakeListFromMapKey<KeyT, ValueT>(
            List<KeyT> list,
            Dictionary<KeyT, ValueT> map)
        {
            list.Clear();
            list.Capacity = map.Count;
            foreach (var i in map)
            {
                list.Add(i.Key);
            }

            list.Sort();
        }

        public static void MakeListFromMapValue<KeyT, ValueT>(
            List<ValueT> list,
            Dictionary<KeyT, ValueT> map)
        {
            list.Clear();
            list.Capacity = map.Count;
            foreach (var i in map)
            {
                list.Add(i.Value);
            }
        }

        public static FontMap GetInstalledFontNames()
        {
            var fontMap = new FontMap();

            // EnumFontFamiliesEx()に渡す匿名デリゲート
            FONTENUMPROC enumProc = delegate(
                ENUMLOGFONTEX elfe,
                NEWTEXTMETRICEX ntme,
                uint fontType,
                LPARAM param)
            {
                var faceName = elfe.elfLogFont.lfFaceName;
                if (!ConverterEnvironment.IsCtr || faceName[0] != '@') // 縦書きフォント以外
                {
                    if (!fontMap.ContainsKey(faceName))
                    {
                        var innerFontType = (fontType & ~GdiDef.RASTER_FONTTYPE) == 0 ? FontType.Raster : FontType.Vector;
                        fontMap[faceName] = new FontInfo(faceName, innerFontType);
                    }
                }

                return 1;
            };

            HDC hDC = Gdi.CreateDC("DISPLAY", string.Empty, string.Empty, IntPtr.Zero);
            LOGFONT lf = new LOGFONT
            {
                lfCharSet = GdiDef.DEFAULT_CHARSET,
                lfFaceName = string.Empty,
                lfPitchAndFamily = 0
            };

            Gdi.EnumFontFamiliesEx(
                hDC,
                ref lf,
                enumProc,
                IntPtr.Zero,
                0);

            Gdi.DeleteDC(hDC);

            // 読み込みが完了していれば、システムフォント向けのエントリを登録します。
            if (_IsSystemFontLoaded)
            {
                fontMap[SystemFontFaceName] = new FontInfo(SystemFontFaceName, FontType.Vector);
            }

            return fontMap;
        }

        // 入力: (0x|#)?[0-9a-fA-F]{6}
        // 出力: 0x00BBGGRR
        public static IntColor ToColor(string str)
        {
            int pos = 0; // LPCTSTR pos = str;

            if (str.StartsWith("0x", StringComparison.InvariantCulture))
            {
                pos += 2;
            }
            else if (str.StartsWith("#", StringComparison.InvariantCulture))
            {
                pos += 1;
            }

            int len = str.Length - pos;

            if (len != 6)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_ILLEGAL_COLOR_FORMAT, str);
            }

            uint rrggbb;
            bool result = uint.TryParse(str.Substring(pos), NumberStyles.AllowHexSpecifier, null, out rrggbb);

            // RRGGBB -> IntColor
            var color = BMP_RGB(
                (byte)ExtractBits(rrggbb, 16, 8),
                (byte)ExtractBits(rrggbb, 8, 8),
                (byte)ExtractBits(rrggbb, 0, 8));

            if (!result)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_ILLEGAL_COLOR_FORMAT, str);
            }

            return color;
        }

        public static int ROUND_UP(int val, int align)
        {
            return (val + align - 1) & ~(align - 1);
        }

        public static uint ROUND_UP(uint val, int align)
        {
            return (uint)((val + align - 1) & ~(align - 1));
        }

        public static int DIV_UP(int val, int divider)
        {
            return (val + divider - 1) / divider;
        }

        public static uint DIV_UP(uint val, uint divider)
        {
            return (val + divider - 1) / divider;
        }

        public static int ParseHexNumber(string nptr)
        {
            int result;
            if (int.TryParse(nptr, NumberStyles.AllowHexSpecifier, null, out result))
            {
                return result;
            }

            return 0;
        }

        public static void InitLanguage()
        {
            string satelliteAssemblyLocation =
                Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "en");

            if (!Directory.Exists(satelliteAssemblyLocation))
            {
                Thread.CurrentThread.CurrentCulture = new CultureInfo("ja-JP");
                Thread.CurrentThread.CurrentUICulture = new CultureInfo("ja-JP");
            }
            else
            {
                Thread.CurrentThread.CurrentCulture = new CultureInfo("en-US");
                Thread.CurrentThread.CurrentUICulture = new CultureInfo("en-US");
            }
        }

        /// <summary>
        /// システムフォントを初期化します。
        /// </summary>
        public static bool InitSystemFontFromBfttf(string filePath)
        {
            // 読み込み済みなら解放する。
            if (_IsSystemFontLoaded)
            {
                Gdi.RemoveFontMemResourceEx(_SystemFontHandle);
                _SystemFontHandle = IntPtr.Zero;
            }

            var fontBuffer = File.ReadAllBytes(filePath);
            var descrambleResultBuffer = Descramble_(fontBuffer);
            if (descrambleResultBuffer.Length <= 0)
            {
                return false;
            }

            uint fontCount;
            IntPtr fontHandle = Gdi.AddFontMemResourceEx(descrambleResultBuffer, descrambleResultBuffer.Length, IntPtr.Zero, out fontCount);
            if (fontCount <= 0)
            {
                return false;
            }

            _SystemFontHandle = fontHandle;

            return true;
        }

        /// <summary>
        /// スクランブルを解除します。
        /// </summary>
        public static byte[] Descramble_(byte[] data)
        {
            const int HeaderByteSize = 8;
            const UInt32 SCRAMBLE_SIGNETURE = 0x7f9a0218;

            if (data.Length <= HeaderByteSize)
            {
                // サイズが小さすぎるので不正。
                return new byte[0];
            }

            UInt32 SCRAMBLE_KEY = GetUint_(data, 0) ^ SCRAMBLE_SIGNETURE;

            // ttfのサイズを取得
            UInt32 ttfSize = GetUint_(data, 4) ^ SCRAMBLE_KEY;
            if (data.Length < ttfSize)
            {
                // ttf のサイズがデータサイズより大きい場合は不正
                return new byte[0];
            }

            // スクランブル解除
            byte[] dstData = new byte[data.Length - HeaderByteSize];
            for (int i = HeaderByteSize; i < data.Length; i += 4)
            {
                UInt32 temp = GetUint_(data, i);
                temp = temp ^ SCRAMBLE_KEY;

                SetToUint_(temp, dstData, i - HeaderByteSize);
            }

            return dstData;
        }

        /// <summary>
        /// Uint32取得
        /// </summary>
        private static UInt32 GetUint_(byte[] data, int pos)
        {
            return (UInt32)(
                data[pos + 3] |
                data[pos + 2] << 8 |
                data[pos + 1] << 16 |
                data[pos + 0] << 24);
        }

        /// <summary>
        /// Uint32設定
        /// </summary>
        private static void SetToUint_(UInt32 val, byte[] data, int pos)
        {
            data[pos + 3] = (byte)(val & 0xFF);
            data[pos + 2] = (byte)((val >> 8) & 0xFF);
            data[pos + 1] = (byte)((val >> 16) & 0xFF);
            data[pos + 0] = (byte)((val >> 24) & 0xFF);
        }
    }

    [Serializable]
    public class StringList : List<string>
    {
        public StringList()
        {
        }

        public StringList(IEnumerable<string> collection)
            : base(collection)
        {
        }
    }
}
