﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using NintendoWare.Font.Win32;
    using Alpha = System.Byte;
    using HDC = System.IntPtr;

    public class VectorLetterer : Letterer
    {
        private const int SrcGlyphGlayLevels4 = 17;
        private const int SrcGlyphGlayLevels8 = 65;

        private bool isSoftAA;

        public VectorLetterer()
        {
            this.isSoftAA = false;
        }

        public void SetVectorOption(bool isSoftAA)
        {
            this.isSoftAA = isSoftAA;
        }

        //---------------------------------------------------------
        // LetterChar
        //---------------------------------------------------------
        public override bool LetterChar(Glyph glyph, uint c)
        {
            GLYPHMETRICS gm = new GLYPHMETRICS();
            uint reqSize;
            var isDrawGlyph = IsGlyphExists(c);

            if (!isDrawGlyph)
            {
                return false;
            }

            byte[] glyphBmp = new byte[0];

            // get glyph bmp
            {
                uint ret;
                var format = (Bpp <= 4) ? GdiDef.GGO_GRAY4_BITMAP : GdiDef.GGO_GRAY8_BITMAP;

                reqSize = GetGlyphOutlineU(DeviceContext, c, format, gm, 0, null, IsUnicode);

                if (reqSize == 0)
                {
                    // 空白文字等の場合 reqSize == 0 になる。
                    // reqSize == 0 の場合でも
                    // gm.gmBlackBoxX/Y は 0 にならないうえ
                    // バッファを指定するとバッファに詳細不明な値が
                    // 格納されてしまうため reqSize の値を見て分岐する。
                    ret = GetGlyphOutlineU(
                        DeviceContext,
                        c,
                        GdiDef.GGO_METRICS,
                        gm,
                        0,
                        null,
                        IsUnicode);

                    // gm.gmBlackBoxX/Y を 0 に強制する。
                    gm.gmBlackBoxX = 0;
                    gm.gmBlackBoxY = 0;
                }
                else
                {
                    // glyphBmp: topdown
                    Array.Resize(ref glyphBmp, (int)reqSize);
                    ret = GetGlyphOutlineU(DeviceContext, c, format, gm, (int)reqSize, glyphBmp, IsUnicode);
                }

                if (ret == GdiDef.GDI_ERROR)
                {
                    var abc = new ABC[1];
                    var isSuccess = GetCharABCWidthsU(DeviceContext, c, c, abc);

                    if (!isSuccess)
                    {
                        var intAry = new int[1] { (int)abc[0].abcB };
                        var isSuccess2 = GetCharWidth32U(DeviceContext, c, c, intAry);

                        if (!isSuccess2)
                        {
                            Rpt._RPT1("fail to GetGlyphOutline({0:X4})\n", c);
                            return false;
                        }
                    }

                    gm.gmBlackBoxX = 0;
                    gm.gmBlackBoxY = 0;
                    gm.gmCellIncX = (short)(abc[0].abcA + abc[0].abcB + abc[0].abcC);
                    gm.gmCellIncY = 0;
                    gm.gmptGlyphOrigin.x = gm.gmCellIncX;
                    gm.gmptGlyphOrigin.y = 0;
                    for (var i = 0; i < glyphBmp.Length; ++i)
                    {
                        glyphBmp[i] = 0;
                    }
                }
            }

            // calcurate parameter
            {
                int dataWidth = GlCm.ROUND_UP((int)gm.gmBlackBoxX, 4);
                int sizeHeight = (int)((dataWidth > 0) ? reqSize / dataWidth : 0);
                int height = Math.Max((int)gm.gmBlackBoxY, sizeHeight);
                GlyphInfo info = new GlyphInfo();
                IndexImage image = new IndexImage();

                info.BmpWidth = dataWidth;
                info.BmpHeight = height;
                info.GlyphLeft = gm.gmptGlyphOrigin.x;
                info.CharWidth = gm.gmCellIncX;
                info.GlyphAscent = gm.gmptGlyphOrigin.y;
                info.Black = (byte)((1 << Bpp) - 1);
                //// info.bg             = 0xFF;
                info.White = 0;

                if (this.isSoftAA)
                {
                    this.AntiAlias(glyphBmp, info);
                }

                this.MakeImage(image, info.BmpWidth, info.BmpHeight, glyphBmp);

                // set to CGlyph
                SetGlyph(glyph, image, info);
            }

            return true;
        }

        public override void Create(ref LOGFONT logfont, string fontPath, string inputFontFaceName)
        {
            // create DC
            {
                DeviceContext = Gdi.CreateDC("DISPLAY", null, null, IntPtr.Zero);
                if (DeviceContext == IntPtr.Zero)
                {
                    throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_CREATEDC);
                }
            }

            if (this.isSoftAA)
            {
                LOGFONT tempLogFont = new LOGFONT();
                tempLogFont = logfont;
                tempLogFont.lfHeight *= 2;
                SetupFont(ref tempLogFont, fontPath, inputFontFaceName);
            }
            else
            {
                SetupFont(ref logfont, fontPath, inputFontFaceName);
            }

            CalcurateParameter();
        }

        protected override void Dispose(bool disposing)
        {
            if (DeviceContext != IntPtr.Zero)
            {
                if (OldFont != IntPtr.Zero)
                {
                    CleanupFont();

                    // delete dc
                    {
                        Gdi.DeleteDC(DeviceContext);
                    }
                }
            }
        }

        private static uint GetGlyphOutlineU(
            HDC hDC,
            uint c,
            int format,
            GLYPHMETRICS gm,
            int buf,
            byte[] p,
            bool isUnicode)
        {
            var m = new MAT2(new FIXED(0, 1), new FIXED(0, 0), new FIXED(0, 0), new FIXED(0, 1));

            return isUnicode ?
                Gdi.GetGlyphOutlineW(hDC, c, format, gm, buf, p, m) :
                Gdi.GetGlyphOutlineA(hDC, c, format, gm, buf, p, m);
        }

        private void MakeImage(IndexImage idx, int width, int height, byte[] img)
        {
            // TODO: 整理
            if (Bpp <= 4)
            {
                int pos = 0;
                var levels = 1 << Bpp;
                int levelMax = levels - 1;
                int levelOffset = (levelMax > 0) ? SrcGlyphGlayLevels4 / (levelMax * 2) : 0;

                idx.Create(width, height, 8);
                idx.SetGrayScaleTable(Bpp);

                for (int y = 0; y < height; y++)
                {
                    for (int x = 0; x < width; x++)
                    {
                        // カラー成分はBPPに応じた最大値だがアルファ成分は常に最大値は255
                        byte val = img[pos];
                        uint c = (uint)((val + levelOffset) * levelMax / (SrcGlyphGlayLevels4 - 1));
                        Alpha a = (Alpha)(c * 255 / levelMax);
                        Debug.Assert(c <= levelMax);
                        Debug.Assert(a <= 255);
                        Debug.Assert(!(c > 0 && a == 0));

                        idx.SetColorAlpha(x, y, c, a);

                        pos++;
                    }
                }
            }
            else
            {
                int pos = 0;
                var levels = 1 << Bpp;
                int levelMax = levels - 1;
                int levelOffset = (levelMax > 0) ? SrcGlyphGlayLevels4 / (levelMax * 2) : 0;

                idx.Create(width, height, 8);
                idx.SetGrayScaleTable(Bpp);

                for (int y = 0; y < height; y++)
                {
                    for (int x = 0; x < width; x++)
                    {
                        // カラー成分はBPPに応じた最大値だがアルファ成分は常に最大値は255
                        byte val = img[pos];
                        uint c = (uint)((val + levelOffset) * levelMax / (SrcGlyphGlayLevels8 - 1));
                        Alpha a = (Alpha)(c * 255 / levelMax);
                        Debug.Assert(c <= levelMax);
                        Debug.Assert(a <= 255);
                        Debug.Assert(!(c > 0 && a == 0));

                        idx.SetColorAlpha(x, y, c, a);

                        pos++;
                    }
                }
            }
        }

        private void AntiAlias(
            byte[] bmp,
            GlyphInfo info)
        {
            var isOddAscent = (info.GlyphAscent % 2) != 0;
            var isOddDescent = ((info.BmpHeight - info.GlyphAscent) % 2) != 0;
            int sw = info.BmpWidth;
            ////  const int sh = pInfo->bmpHeight;

            // base height  asce desc    asce    height
            // even even -> even even -> a/2   a/2  +d/2   = h/2
            // even odd  -> even odd  -> a/2   a/2  +d/2+1 = h/2+1
            // odd  even -> odd  odd  -> a/2+1 a/2+1+d/2+1 = h/2+1
            // odd  odd  -> odd  even -> a/2+1 a/2+1+d/2   = h/2+1
            if (isOddAscent && isOddDescent)
            {
                info.BmpHeight += 1;
            }

            info.BmpWidth = (info.BmpWidth + 1) / 2;
            info.BmpHeight = (info.BmpHeight + 1) / 2;
            info.GlyphLeft = (info.GlyphLeft + 0) / 2;
            info.CharWidth = (info.CharWidth + 1) / 2;
            info.GlyphAscent = (info.GlyphAscent + 1) / 2;

            if ((info.BmpWidth != 0) && (info.BmpHeight != 0))
            {
                var isOddWidth = (sw % 2) != 0;
                int dw = info.BmpWidth;
                int dh = info.BmpHeight;
                int ps = -(isOddAscent ? sw : 0);
                int pd = 0;

                for (int y = 0; y < dh; ++y)
                {
                    for (int x = 0; x < dw; ++x)
                    {
                        int sum = 0;

                        if ((y != 0) || (!isOddAscent))
                        {
                            sum += bmp[ps + 0];
                            if (((x + 1) < dw) || (!isOddWidth))
                            {
                                sum += bmp[ps + 1];
                            }
                        }

                        if (((y + 1) < dh) || (!isOddDescent))
                        {
                            sum += bmp[ps + sw + 0];
                            if (((x + 1) < dw) || (!isOddWidth))
                            {
                                sum += bmp[ps + sw + 1];
                            }
                        }

                        bmp[pd++] = (byte)((sum + 2) / 4);
                        ps += 2;
                    }

                    ps += (sw * 2) - (dw * 2);
                }
            }
        }
    }
}
