﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    /// <summary>
    ///
    /// </summary>
    public interface IPixelPicker
    {
        int BlockW { get; }

        int BlockH { get; }

        int BlockPixelNum { get; }

        PixelPos GetPixel(int pos);
    }

    /// <summary>
    ///
    /// </summary>
    public struct PixelPos
    {
        private int x;
        private int y;

        public PixelPos(int x, int y)
        {
            this.x = x;
            this.y = y;
        }

        public int X
        {
            get { return this.x; }
            set { this.x = value; }
        }

        public int Y
        {
            get { return this.y; }
            set { this.y = value; }
        }
    }

    /// <summary>
    /// ブロックにおいてピクセルをシーケンスに取得するクラスです。
    /// </summary>
    public class PixelPickerCtr : IPixelPicker
    {
        private readonly RgbImage sheet;
        private readonly int blkXCnt;

        public PixelPickerCtr(RgbImage sheet, int bpp)
        {
            this.sheet = sheet;
            this.blkXCnt = sheet.Width / this.BlockW;
        }

        public int BlockW
        {
            get { return 8; }
        }

        public int BlockH
        {
            get { return 8; }
        }

        public int BlockPixelNum
        {
            get { return this.BlockW * this.BlockH; }
        }

        public PixelPos GetPixel(int pos)
        {
            int sx = 0;
            sx += (pos >> 0) & 0x01;
            sx += (pos >> 1) & 0x02;
            sx += (pos >> 2) & 0x04;
            sx += ((pos / this.BlockPixelNum) % this.blkXCnt) * this.BlockW;

            int sy = 0;
            sy += (pos >> 1) & 0x01;
            sy += (pos >> 2) & 0x02;
            sy += (pos >> 3) & 0x04;
            sy += (pos / this.BlockPixelNum / this.blkXCnt) * this.BlockH;

            return new PixelPos(sx, sy);
        }
    }

    /// <summary>
    /// ブロックにおいてピクセルをシーケンスに取得するクラスです。
    /// </summary>
    public class PixelPickerGL : IPixelPicker
    {
        public PixelPickerGL(RgbImage sheet, int bpp)
        {
            this.BlockW = sheet.Width;
            this.BlockH = sheet.Height;
            this.BlockPixelNum = this.BlockW * this.BlockH;
        }

        public int BlockW { get; private set; }

        public int BlockH { get; private set; }

        public int BlockPixelNum { get; private set; }

        public PixelPos GetPixel(int pos)
        {
            int sx = pos % BlockW;
            int sy = pos / BlockW;
            return new PixelPos(sx, sy);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class PixelPickerCreater
    {
        public static IPixelPicker Create(RgbImage sheet, int bpp)
        {
            return (IPixelPicker)(new PixelPickerCtr(sheet, bpp));

            // Cafe 用 bffnt 読み込み機能は、非可逆圧縮フォーマットが主体のため廃止になっています。
            //return (IPixelPicker)(new PixelPickerGL(sheet, bpp));
        }
    }
}
