﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using System.Globalization;
    using System.Runtime.InteropServices;
    using System.Threading;
    using System.Windows.Controls;

    /// <summary>
    /// トレース出力またはデバッグ出力をTextBoxに転送します。
    /// </summary>
    public class TextBoxTraceListener : TraceListener
    {
        private const int MaxWarning = 100;

        private readonly Delegate appendText;
        private TextBox textBox;
        private int warningNum;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="textBox">TextBoxTraceListenerからの出力を受信するTextBox。</param>
        public TextBoxTraceListener(TextBox textBox)
        {
            if (textBox == null)
            {
                throw new ArgumentNullException("textBox");
            }

            this.textBox = textBox;
            this.appendText = new Action<string>(this.AppendText);
        }

        /// <summary>
        /// 指定したメッセージを書き込みます。
        /// </summary>
        /// <param name="message">書き込むメッセージ</param>
        public override void Write(string message)
        {
            if (!this.EnsureWriter())
            {
                return;
            }

            this.WriteImpl(message);
        }

        /// <summary>
        /// 指定したメッセージと行終端記号を書き込みます。
        /// </summary>
        /// <param name="message">書き込むメッセージ</param>
        public override void WriteLine(string message)
        {
            if (!this.EnsureWriter())
            {
                return;
            }

            this.WriteImpl(message + "\r\n");
            NeedIndent = true;
        }

        [ComVisible(false)]
        public override void TraceEvent(TraceEventCache eventCache, string source, TraceEventType eventType, int id, string message)
        {
            object[] args = null;
            object data1 = null;
            object[] data = null;

            if (eventType != TraceEventType.Error)
            {
                this.warningNum++;
            }

            if (Filter != null && !Filter.ShouldTrace(eventCache, source, eventType, id, message, args, data1, data))
            {
                return;
            }

            if (eventType == TraceEventType.Error)
            {
                this.WriteErrorHeader();
                this.WriteLine(message);

                this.textBox.ScrollToEnd();
            }
            else
            {
                if (this.warningNum < MaxWarning)
                {
                    this.WriteWarningHeader();
                    this.WriteLine(message);
                }
                else if (this.warningNum == MaxWarning)
                {
                    this.WriteLine(Strings.IDS_WARN_OVERMAX_WARNING);
                }
            }
        }

        [ComVisible(false)]
        public override void TraceEvent(TraceEventCache eventCache, string source, TraceEventType eventType, int id, string format, params object[] args)
        {
            object data1 = null;
            object[] data = null;

            if (eventType != TraceEventType.Error)
            {
                this.warningNum++;
            }

            if (Filter != null && !Filter.ShouldTrace(eventCache, source, eventType, id, format, args, data1, data))
            {
                return;
            }

            if (eventType == TraceEventType.Error)
            {
                this.WriteErrorHeader();
                if (args != null)
                {
                    this.WriteLine(string.Format(CultureInfo.InvariantCulture, format, args));
                }
                else
                {
                    this.WriteLine(format);
                }

                this.textBox.ScrollToEnd();
            }
            else
            {
                if (this.warningNum < MaxWarning)
                {
                    this.WriteWarningHeader();
                    if (args != null)
                    {
                        this.WriteLine(string.Format(CultureInfo.InvariantCulture, format, args));
                    }
                    else
                    {
                        this.WriteLine(format);
                    }
                }
                else if (this.warningNum == MaxWarning)
                {
                    this.WriteLine(Strings.IDS_WARN_OVERMAX_WARNING);
                }
            }
        }

        internal bool EnsureWriter()
        {
            return true;
        }

        private void WriteErrorHeader()
        {
            this.WriteLine(Strings.IDS_ERROR_HR);
        }

        private void WriteWarningHeader()
        {
            this.Write(Strings.IDS_WARNING + ": ");
        }

        private void WriteImpl(string message)
        {
            if (NeedIndent)
            {
                WriteIndent();
            }

            if (this.textBox.Dispatcher.CheckAccess())
            {
                this.AppendText(message);
            }
            else
            {
                this.textBox.Dispatcher.BeginInvoke(this.appendText, message);
            }
        }

        private void AppendText(string message)
        {
            this.textBox.AppendText(message);
        }
    }
}
