﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx.h>
#include "detail/util_DebugDrawUtil.h"


void _DrawMeter( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, const char* pString, int value, int maxValue, float x, float y, float w, float h, int align, nn::util::Color4u8 groundColor, bool percentDraw ) NN_NOEXCEPT
{
    char  temp[ 256 ];
    const nn::util::Color4u8 frameColor( 128, 128, 128, 255 );
    const nn::util::Color4u8 greenColor( 128, 255, 128, 255 );
    const nn::util::Color4u8 yellowColor( 255, 255, 128, 255 );
    const nn::util::Color4u8 redColor( 255, 128, 128, 255 );

    pPrimitiveRenderer->SetColor( groundColor );
    pPrimitiveRenderer->Draw2DRect( pCommandBuffer, x, y, w, h );

    float rate = ( static_cast< float >( value ) / maxValue );
    if ( rate < 0.7 )
    {
        pPrimitiveRenderer->SetColor( greenColor );
    }
    else if ( 0.7 <= rate && rate < 0.9 )
    {
        pPrimitiveRenderer->SetColor( yellowColor );
    }
    else
    {
        pPrimitiveRenderer->SetColor( redColor );
    }

    float meterWidth = w * rate;
    if ( meterWidth > w ) meterWidth = w;
    pPrimitiveRenderer->Draw2DRect( pCommandBuffer, x, y + h / 2, meterWidth, h - h / 2 );

    pPrimitiveRenderer->SetColor( frameColor );
    pPrimitiveRenderer->Draw2DFrame( pCommandBuffer, x, y, w, h );

    pTextWriter->SetCursorY( y - 2 );
    pTextWriter->SetFixedWidthEnabled( false );
    pTextWriter->SetCursorX( x + 4 );
    pTextWriter->Print( pString );
    pTextWriter->SetFixedWidthEnabled( true );

    if ( percentDraw )
    {
        float percent = rate * 100.f;
        sprintf( temp, "%3.2f %%", percent );
    }
    else
    {
        sprintf( temp, "%d", static_cast< int >( value ) );
    }

    // align = 0 left
    // align = 1 center
    // align = 2 right
    if ( align == 0 )
    {
        pTextWriter->SetCursorX( x + 4 );
    }
    if ( align == 1 )
    {
        float sw = strlen( temp ) * pTextWriter->GetFixedWidth();
        pTextWriter->SetCursorX( x + ( w / 2 - sw / 2 ) + 4 );
    }
    if ( align == 2 )
    {
        float fw = static_cast< float >( static_cast< int >( pTextWriter->GetFixedWidth() + 1.0f ) );
        float sw = strlen( temp ) * fw;
        pTextWriter->SetCursorX( x + w - sw - 4 );
    }
    pTextWriter->Print( temp );
    pTextWriter->SetFixedWidthEnabled( false );
}



void _DrawMeter( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, const char* pString, MeterItem* meterItemArray, int meterItemCount, int maxValue, float x, float y, float w, float h, int align, nn::util::Color4u8 groundColor, bool percentDraw ) NN_NOEXCEPT
{
    char  temp[ 256 ];
    const nn::util::Color4u8 frameColor( 128, 128, 128, 255 );
    const nn::util::Color4u8 greenColor( 128, 255, 128, 255 );
    const nn::util::Color4u8 yellowColor( 255, 255, 128, 255 );
    const nn::util::Color4u8 redColor( 255, 128, 128, 255 );

    pPrimitiveRenderer->SetColor( groundColor );
    pPrimitiveRenderer->Draw2DRect( pCommandBuffer, x, y, w, h );

    float rateSum  = 0.0f;
    int   valueSum = 0;
    float beforeMeterWidth = 0.0f;
    for( int i =0; i < meterItemCount; i++ )
    {
        float rate = ( static_cast< float >( meterItemArray[ i ].value ) / maxValue );
        pPrimitiveRenderer->SetColor( meterItemArray[ i ].color );
        float meterWidth = w * rate;
        if ( meterWidth > w ) meterWidth = w;
        pPrimitiveRenderer->Draw2DRect( pCommandBuffer, x + beforeMeterWidth, y + h / 2, meterWidth, h - h / 2 );
        rateSum += rate;
        valueSum += meterItemArray[ i ].value;
        beforeMeterWidth = meterWidth;
    }

    pPrimitiveRenderer->SetColor( frameColor );
    pPrimitiveRenderer->Draw2DFrame( pCommandBuffer, x, y, w, h );

    pTextWriter->SetCursorY( y - 2 );
    pTextWriter->SetFixedWidthEnabled( false );
    pTextWriter->SetCursorX( x + 4 );
    pTextWriter->Print( pString );
    pTextWriter->SetFixedWidthEnabled( true );

    if ( percentDraw )
    {
        float percent = rateSum * 100.f;
        sprintf( temp, "%3.2f %%", percent );
    }
    else
    {
        sprintf( temp, "%d", static_cast< int >( valueSum ) );
    }

    // align = 0 left
    // align = 1 center
    // align = 2 right
    if ( align == 0 )
    {
        pTextWriter->SetCursorX( x + 4 );
    }
    if ( align == 1 )
    {
        float sw = strlen( temp ) * pTextWriter->GetFixedWidth();
        pTextWriter->SetCursorX( x + ( w / 2 - sw / 2 ) + 4 );
    }
    if ( align == 2 )
    {
        float fw = static_cast< float >( static_cast< int >( pTextWriter->GetFixedWidth() + 1.0f ) );
        float sw = strlen( temp ) * fw;
        pTextWriter->SetCursorX( x + w - sw - 4 );
    }
    pTextWriter->Print( temp );
    pTextWriter->SetFixedWidthEnabled( false );
}



void _DrawTextBox( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, const char* pString, float x, float y, float w, float h, int align, nn::util::Color4u8 groundColor ) NN_NOEXCEPT
{
    //const nn::util::Color4u8 groundColor( 32, 32, 32, 200 );
    const nn::util::Color4u8 frameColor( 128, 128, 128, 255 );

    pPrimitiveRenderer->SetColor( groundColor );
    pPrimitiveRenderer->Draw2DRect( pCommandBuffer, x, y, w, h );
    pPrimitiveRenderer->SetColor( frameColor );
    pPrimitiveRenderer->Draw2DFrame( pCommandBuffer, x, y, w, h );

    pTextWriter->SetCursorY( y - 2 );

    // align = 0 left
    // align = 1 center
    // align = 2 right
    if ( align == 0 )
    {
        pTextWriter->SetCursorX( x + 4 );
    }
    if ( align == 1 )
    {
        float sw = strlen( pString ) * pTextWriter->GetFixedWidth();
        pTextWriter->SetCursorX( x + ( w / 2 - sw / 2 ) + 4 );
    }
    if ( align == 2 )
    {
        pTextWriter->SetCursorX( x + w - ( strlen( pString ) * pTextWriter->GetFixedWidth() ) - 4 );
    }

    pTextWriter->Print( pString );
    pTextWriter->SetFixedWidthEnabled( false );
}

void _DrawTextBox( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, int parameter, float x, float y, float w, float h, int align, nn::util::Color4u8 groundColor ) NN_NOEXCEPT
{
    char  temp[ 256 ];
    sprintf( temp, "%d", parameter );
    pTextWriter->SetFixedWidthEnabled( true );
    return _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, temp, x, y, w, h, align, groundColor );
}

void _DrawTextBox( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, const char* pString, size_t parameter, float x, float y, float w, float h, int align, nn::util::Color4u8 groundColor ) NN_NOEXCEPT
{
    return _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, pString, static_cast<int>( parameter ), x, y, w, h, align, groundColor );
}

void _DrawTextBox( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, const char* pString, float parameter, float x, float y, float w, float h, int align, nn::util::Color4u8 groundColor ) NN_NOEXCEPT
{
    const nn::util::Color4u8 frameColor( 128, 128, 128, 255 );
    char  temp[ 256 ];

    pPrimitiveRenderer->SetColor( groundColor );
    pPrimitiveRenderer->Draw2DRect( pCommandBuffer, x, y, w, h );
    pPrimitiveRenderer->SetColor( frameColor );
    pPrimitiveRenderer->Draw2DFrame( pCommandBuffer, x, y, w, h );

    pTextWriter->SetCursorY( y - 2 );

    pTextWriter->SetCursorX( x + 4 );
    pTextWriter->SetCursorY( y - 2 );
    pTextWriter->SetFixedWidthEnabled( false );
    pTextWriter->SetCursorX( x + 4 );
    pTextWriter->Print( pString );
    pTextWriter->SetFixedWidthEnabled( true );

    sprintf( temp, "%8.2f", static_cast< float >( parameter ) );

    // align = 0 left
    // align = 1 center
    // align = 2 right
    if ( align == 0 )
    {
        pTextWriter->SetCursorX( x + 4 );
    }
    if ( align == 1 )
    {
        float sw = strlen( temp ) * pTextWriter->GetFixedWidth();
        pTextWriter->SetCursorX( x + ( w / 2 - sw / 2 ) + 4 );
    }
    if ( align == 2 )
    {
        float fw = static_cast< float >( static_cast< int >( pTextWriter->GetFixedWidth() + 1.0f ) );
        float sw = strlen( temp ) * fw;
        pTextWriter->SetCursorX( x + w - sw - 4 );
    }
    pTextWriter->Print( temp );
    pTextWriter->SetFixedWidthEnabled( false );
}

void _DrawTextBox( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, const char* pString, int parameter, float x, float y, float w, float h, int align, nn::util::Color4u8 groundColor ) NN_NOEXCEPT
{
    const nn::util::Color4u8 frameColor( 128, 128, 128, 255 );
    char  temp[ 256 ];

    pPrimitiveRenderer->SetColor( groundColor );
    pPrimitiveRenderer->Draw2DRect( pCommandBuffer, x, y, w, h );
    pPrimitiveRenderer->SetColor( frameColor );
    pPrimitiveRenderer->Draw2DFrame( pCommandBuffer, x, y, w, h );

    pTextWriter->SetCursorY( y - 2 );

    pTextWriter->SetCursorX( x + 4 );
    pTextWriter->SetCursorY( y - 2 );
    pTextWriter->SetFixedWidthEnabled( false );
    pTextWriter->SetCursorX( x + 4 );
    pTextWriter->Print( pString );
    pTextWriter->SetFixedWidthEnabled( true );

    sprintf( temp, "%d", static_cast< int >( parameter ) );

    // align = 0 left
    // align = 1 center
    // align = 2 right
    if ( align == 0 )
    {
        pTextWriter->SetCursorX( x + 4 );
    }
    if ( align == 1 )
    {
        float sw = strlen( temp ) * pTextWriter->GetFixedWidth();
        pTextWriter->SetCursorX( x + ( w / 2 - sw / 2 ) + 4 );
    }
    if ( align == 2 )
    {
        float fw = static_cast< float >( static_cast< int >( pTextWriter->GetFixedWidth() + 1.0f ) );
        float sw = strlen( temp ) * fw;
        pTextWriter->SetCursorX( x + w - sw - 4 );
    }
    pTextWriter->Print( temp );
    pTextWriter->SetFixedWidthEnabled( false );
}

void _DrawTextBox( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, float parameter, const char* pString, float x, float y, float w, float h, nn::util::Color4u8 groundColor ) NN_NOEXCEPT
{
    const nn::util::Color4u8 frameColor( 128, 128, 128, 255 );
    char  temp[ 256 ];

    pPrimitiveRenderer->SetColor( groundColor );
    pPrimitiveRenderer->Draw2DRect( pCommandBuffer, x, y, w, h );
    pPrimitiveRenderer->SetColor( frameColor );
    pPrimitiveRenderer->Draw2DFrame( pCommandBuffer, x, y, w, h );

    pTextWriter->SetCursorY( y - 2 );

    pTextWriter->SetFixedWidthEnabled( false );
    float sw = strlen( pString ) * pTextWriter->GetFixedWidth();
    float sp = x + w - sw - 4;
    pTextWriter->SetCursorX( sp );
    pTextWriter->Print( pString );
    pTextWriter->SetFixedWidthEnabled( true );

    sprintf( temp, "%3.3f", parameter );
    float pw = strlen( temp ) * pTextWriter->GetFixedWidth();
    float pp = x + w - sw - pw - 4;
    pTextWriter->SetCursorX( pp );
    pTextWriter->Print( temp );

    pTextWriter->SetFixedWidthEnabled( false );
}

void _DrawTextBox( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, const char* pString, float parameter, float parameterMax, float x, float y, float w, float h, int align, nn::util::Color4u8 groundColor ) NN_NOEXCEPT
{
    const nn::util::Color4u8 frameColor( 128, 128, 128, 255 );
    char  temp[ 256 ];

    pPrimitiveRenderer->SetColor( groundColor );
    pPrimitiveRenderer->Draw2DRect( pCommandBuffer, x, y, w, h );
    pPrimitiveRenderer->SetColor( frameColor );
    pPrimitiveRenderer->Draw2DFrame( pCommandBuffer, x, y, w, h );

    pTextWriter->SetCursorY( y - 2 );

    pTextWriter->SetFixedWidthEnabled( false );
    pTextWriter->SetCursorX( x + 4 );
    pTextWriter->Print( pString );
    pTextWriter->SetFixedWidthEnabled( true );


    // イレギュラー:プログレスバー
    float rate = parameter / parameterMax;
    pPrimitiveRenderer->Draw2DRect( pCommandBuffer, x + ( w - 4 ) * rate, y, 4, h );


    sprintf( temp, "%4.2f/%4.0f", parameter, parameterMax );

    if ( align == 0 )
    {
        pTextWriter->SetCursorX( x + 4 );
    }
    if ( align == 1 )
    {
        float sw = strlen( temp ) * pTextWriter->GetFixedWidth();
        pTextWriter->SetCursorX( x + ( w / 2 - sw / 2 ) + 4 );
    }
    if ( align == 2 )
    {
        float fw = static_cast< float >( static_cast< int >( pTextWriter->GetFixedWidth() + 1.0f ) );
        float sw = strlen( temp ) * fw;
        pTextWriter->SetCursorX( x + w - sw - 4 );
    }
    pTextWriter->Print( temp );
    pTextWriter->SetFixedWidthEnabled( false );
}
