﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "CaptureFrame.h"

#ifndef _MAX_PATH
#define _MAX_PATH 260  //NOLINT(preprocessor/const)
#endif

#if defined( NN_BUILD_CONFIG_OS_WIN )
//------------------------------------------------------------------------------
//      スクリーンキャプチャ処理
//------------------------------------------------------------------------------
void FrameCapture::SaveCurrentBufferAsPngFile( const char* filePath ) NN_NOEXCEPT
{
    typedef struct
    {
        unsigned char r;
        unsigned char g;
        unsigned char b;
        unsigned char a;
    }Rgb;

    // キャプチャ用のバッファが設定されていない時は何もしない。
    if ( !m_CaptureBuffer ) return;

    // キャプチャ画像を取得
    GLubyte* pBuffer = static_cast<GLubyte*>(m_CaptureBuffer->Map());
    m_CaptureBuffer->InvalidateMappedRange(0, 4 * m_Width * m_Height);

    //  RGB情報の書き込み
    int      real_width = m_Width*4 + m_Width%4;
    uint8_t* bmp_data   = new uint8_t[  m_Width * m_Height * 4 ];
    Rgb*     imagePtr   = reinterpret_cast<Rgb*>( pBuffer );

    for( int i = 0; i < (int)m_Height; i++)
    {
        for( int j = 0; j < (int)m_Width; j++)
        {
            {
                bmp_data[( i * ( m_Width * 3 ) ) + ( j * 3 )]      = imagePtr[i * m_Width + j].r;
                bmp_data[( i * ( m_Width * 3 ) ) + ( j * 3 ) + 1]  = imagePtr[i * m_Width + j].g;
                bmp_data[( i * ( m_Width * 3 ) ) + ( j * 3 ) + 2]  = imagePtr[i * m_Width + j].b;
                bmp_data[( i * ( m_Width * 3 ) ) + ( j * 3 ) + 3]  = imagePtr[i * m_Width + j].a;
            }
        }

        for( int j = m_Width * 4; j < real_width; j++ )
        {
            bmp_data[( i * m_Width ) + j] = 0;
        }
    }

    // キャプチャ画像を保存
    detail::VfxViewerUtilPngIO::WritePng( filePath, (uint8_t*)bmp_data, *m_pHeap, m_Width, m_Height, 8, detail::VfxViewerUtilPngIO::PNG_COLOR_TYPE_RGB, 0, 0, 0 );

    m_CaptureBuffer->Unmap();
    delete bmp_data;
}

void FrameCapture::PushCaptureCommand( nn::gfx::CommandBuffer* pCommandBuffer,
                                       nn::gfx::Buffer* dstBuffer, nn::gfx::Texture* srcTexture,
                                       int width, int height ) NN_NOEXCEPT
{
    m_Width = width;
    m_Height = height;
    m_CaptureBuffer = dstBuffer;

    nn::gfx::BufferTextureCopyRegion region;
    region.SetDefault();
    region.SetBufferImageWidth( m_Width );
    region.SetBufferImageHeight( m_Height );
    region.EditTextureCopyRegion().SetWidth( m_Width );
    region.EditTextureCopyRegion().SetHeight( m_Height );
    pCommandBuffer->CopyImageToBuffer( dstBuffer, srcTexture, region );
    m_CaptureCommandPushed = true;
}

void FrameCapture::CaptureFrameWithDirectoryPath( const char* directoryPath, int interval ) NN_NOEXCEPT
{
    if ( m_CaptureCommandPushed == false )
    {
        m_CaptureIndex = 0;
        return;
    }

    // ScreenCap フォルダが無ければ生成する
    CreateScreenCapDirectory( directoryPath );

    // 連番付きキャプチャ初回時はこれまでのサムネイルを削除
    if ( m_CaptureIndex == 0 )
    {
        ClearScreenCapDirectory( directoryPath );
    }

    if ( ( m_CaptureIndex > 0 && ( m_CaptureIndex - 1 ) % interval == 0 ) )
    {
        char filePath[ _MAX_PATH ];
        sprintf_s( filePath, "%s\\ScreenCap\\ScreenCap_%05d.png", directoryPath, ( m_CaptureIndex - 1 ) );
        CaptureFrameWithFilePath( filePath );
    }

    m_CaptureIndex++;
}

void FrameCapture::CaptureFrameWithFilePath( const char* filePath ) NN_NOEXCEPT
{
    if ( m_CaptureCommandPushed == false )
    {
        return;
    }

    SaveCurrentBufferAsPngFile( filePath );
    m_CaptureCommandPushed = false;
}

void FrameCapture::CreateScreenCapDirectory( const char* directoryPath ) NN_NOEXCEPT
{
    WIN32_FIND_DATA fd = { 0 };
    SetCurrentDirectoryA( directoryPath );
    HANDLE hFindDir = FindFirstFile( L"ScreenCap", &fd );
    if ( hFindDir == INVALID_HANDLE_VALUE )
    {
        CreateDirectory( L"ScreenCap", NULL );
    }
    else
    {
        FindClose( hFindDir );
    }

    SetCurrentDirectoryA( directoryPath );
}

void FrameCapture::ClearScreenCapDirectory( const char* directoryPath ) NN_NOEXCEPT
{
    WIN32_FIND_DATA fd = { 0 };

    SetCurrentDirectoryA( directoryPath );
    SetCurrentDirectory( L"ScreenCap" );

    wchar_t searchBuf[ _MAX_PATH ];
    HANDLE hFind = INVALID_HANDLE_VALUE;

    wsprintf( searchBuf, L"%s", L"*.png" );
    hFind = ( FindFirstFile( searchBuf, &fd ) );
    if ( hFind != INVALID_HANDLE_VALUE )
    {
        do {
            if ( !( fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ) )
            {
                if ( wcsstr( fd.cFileName, L".png" ) != NULL )
                {
                    DeleteFile( fd.cFileName );
                }
            }
        } while ( FindNextFile( hFind, &fd ) );
    }
    FindClose( hFind );

    SetCurrentDirectoryA( directoryPath );
}


#endif
