﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/nn_Macro.h>
#include <nn/util/util_PlacementArray.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>

class GpuStatisticsInfo
{
public:
    /**
    * @brief コンストラクタです。
    */
    GpuStatisticsInfo() NN_NOEXCEPT
    {
        this->SetBufferCount(2);
    }

    /**
    * @brief 各パラメータを既定値に設定するためのヘルパー関数です。
    */
    void SetDefault() NN_NOEXCEPT
    {
        this->SetBufferCount(2);
    };

    /**
    * @brief GpuStatistics のクエリ書き込みの GPU バッファの数を指定します。
    * @param[in] bufferCount GPU バッファの数です。
    * @details GpuStatistics が参照したり出力する結果は (bufferCount - 1) フレーム前のクエリ結果になります。
    */
    void SetBufferCount( int bufferCount ) NN_NOEXCEPT
    {
        this->m_BufferCount = bufferCount;
    }

    /**
    * @brief GpuStatistics のクエリ書き込みの GPU バッファの数を取得します。
    * @return GPU バッファの数です。
    */
    int GetBufferCount() NN_NOEXCEPT
    {
        return this->m_BufferCount;
    }

private:
    int m_BufferCount;
};

class GpuStatistics
{
public:
    //! @brief        コンストラクタです。
    GpuStatistics() NN_NOEXCEPT
        :
        m_TimestampCounter( 0 ),
        m_CurrentBufferIndex( 0 ),
        m_ReferenceBufferIndex( 1 )
    {}

    //! @}
    //! @name 初期化/終了
    //! @{

    //! @brief        クエリ結果の格納メモリとカウンタが書き込まれる GPU バッファを初期化します。
    //! @param[in]    pDevice デバイスへのポインタです。
    //! @param[in]    info 初期化用の情報です。
    //! @param[in]    pMemory 初期用のバッファへのポインタ
    //! @param[in]    memorySize バイトでのバッファサイズ
    //! @param[in]    pMemoryPool GPU カウンタが書き込まれるメモリプールへのポインタ
    //! @param[in]    memoryPoolOffset GPU カウンタが書き込まれるメモリプールへのオフセット
    //! @param[in]    memoryPoolSize GPU カウンタが書き込まれるメモリプールのサイズ
    void Initialize(nn::gfx::Device* pDevice, GpuStatisticsInfo info, void* pMemory, size_t memorySize, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize) NN_NOEXCEPT;

    //! @brief        バッファの終了処理を行います。
    //! @param[in]    pDevice デバイスへのポインタです。
    void Finalize( nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //! @brief        Initialize に必要なバッファのアラインメントを返します。
    //! @return       バッファのアラインメントを返します。
    static size_t GetBufferAlignment() NN_NOEXCEPT;

    //! @brief        Initialize に必要なバッファのサイズを返します。
    //! @return       バッファのサイズを返します。
    static size_t CalculateBufferSize() NN_NOEXCEPT;

    //! @brief        Initialize でメモリプール上に確保するクエリバッファのアラインメントを返します。
    //! @param[in]    pDevice デバイスへのポインタです。
    //! @param[in]    info 初期化用の情報です。
    //! @return       クエリバッファのアラインメントを返します。
    static size_t GetQueryBufferAlignment( nn::gfx::Device* pDevice, GpuStatisticsInfo info) NN_NOEXCEPT;

    //! @brief        Initialize でメモリプール上に確保するクエリバッファのサイズを返します。
    //! @param[in]    pDevice デバイスへのポインタです。
    //! @param[in]    info 初期化用の情報です。
    //! @return       クエリバッファのサイズを返します。
    static size_t CalculateQueryBufferSize( nn::gfx::Device* pDevice, GpuStatisticsInfo info) NN_NOEXCEPT;

    //! @brief        クエリを開始します。
    //! @param[in]    pCommandBuffer クエリを開始する要求コマンドを追加するコマンドバッファです。
    void BeginQuery( nn::gfx::CommandBuffer* pCommandBuffer ) NN_NOEXCEPT;

    //! @brief        クエリを終了します。
    //! @param[in]    pCommandBuffer カウンタを書き込む要求コマンドを追加するコマンドバッファです。
    void EndQuery( nn::gfx::CommandBuffer* pCommandBuffer ) NN_NOEXCEPT;


    //! @brief (info.GetBufferCount() - 1) フレーム前の計測結果を取得します。
    //! @return クエリ結果を返します。
    int64_t GetLastResult(nn::gfx::QueryTarget target) const NN_NOEXCEPT
    {
        if(target == nn::gfx::QueryTarget_Timestamp)
        {
            return m_TimestampCounter;
        }
        else
        {
            return m_CounterBuffer[target].GetValue();
        }
    }

    //! @brief    カウンタの値を読み出します。さらに次フレームの計測結果の格納先バッファを切り替えます。
    //! @details  この関数が呼ばれない場合、正常な測結果を得ることができません。@n
    //! @n        シングルバッファの場合は必ず GPU が動作していないタイミングで呼ぶようにしてください。
    void Next() NN_NOEXCEPT;


    //! @brief    Gpu計測結果をデバッグ表示します。。
    void DrawDebugInfo( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, float x, float y ) NN_NOEXCEPT;

private:
    nn::util::PlacementArray<nn::gfx::Buffer>               m_GpuBuffer;                //!< クエリ結果が書き込まれる GPU バッファ配列へのポインタです。
    nn::util::PlacementArray<nn::gfx::QueryBuffer>          m_CounterBuffer;            //!< GPU バッファからカウンタの値を受け取るバッファの配列です。
    int64_t                                                 m_TimestampCounter;         //!< タイムスタンプのカウンタです。
    GpuStatisticsInfo                                       m_Info;                     //!< 初期化用の情報です。
    int                                                     m_CurrentBufferIndex;
    int                                                     m_ReferenceBufferIndex;
};
