﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/vfx/vfx_System.h>
#include <nn/vfx/viewer/vfx_FileSystem.h>
#include <nn/vfx/viewer/vfx_VwrString.h>

//---------------------------------------------------
//  エフェクトバイナリプレイヤー
//---------------------------------------------------
class EffectBinaryPlayer
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] system        VFXシステム
    //! @param[in] heap          ヒープ
    //! @param[in] resourceID    エフェクトリソースID
    //! @param[in] groupID       エフェクトグループID
    //------------------------------------------------------------------------------
    EffectBinaryPlayer( nn::vfx::System* system, nn::vfx::Heap* heap, int resourceID0, int resourceID1, int groupID ) NN_NOEXCEPT
    {
        m_Side              = 0;
        m_pSystem           = system;
        m_pHeap             = heap;
        m_ResourceId[0]     = resourceID0;
        m_ResourceId[1]     = resourceID1;
        m_GroupId           = groupID;
        m_EmitterSetId      = 0;
        m_pEffectBinary[0]  = nullptr;
        m_pEffectBinary[1]  = nullptr;
        m_EffectBinaryPath  = nn::vfx::viewer::detail::ViewerString();
    }

    //------------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //------------------------------------------------------------------------------
    ~EffectBinaryPlayer() NN_NOEXCEPT
    {
        UnloadEffectBinary( 0 );
        UnloadEffectBinary( 1 );
    }

    //------------------------------------------------------------------------------
    //! @brief  エフェクトバイナリをセットアップします。
    //!
    //! @param[in] effectBinaryPath  ファイルパス
    //! @param[in] pRegisterTextureSlotCallback   ディスクリプタスロット確保関数です。
    //! @param[in] forceReload       ファイルパスが同じでもリロードするか
    //------------------------------------------------------------------------------
    bool SetupEffectBinary( const char* effectBinaryPath, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback, bool forceReload = false ) NN_NOEXCEPT
    {
        size_t   readBufferSize    = 0;

        // パスを比較して異なる場合はセットアップを行う
        if ( forceReload == false && strcmp( m_EffectBinaryPath.CStr(), effectBinaryPath ) == 0 )
        {
            return false;
        }

        // 既存のバイナリを破棄
        m_Side = 1 - m_Side;
        UnloadEffectBinary( m_Side );

        // 再生するエミッタセットIDをリセット
        m_EmitterSetId = 0;

        // バイナリをロードしてシステムにエントリ
        bool ret = nn::vfx::viewer::detail::FileSystem::Load( &m_pEffectBinary[m_Side], &readBufferSize, effectBinaryPath, m_pHeap );
        if ( ret )
        {
            m_pSystem->EntryResource( m_pHeap, m_pEffectBinary[m_Side], m_ResourceId[m_Side], true );
            m_pSystem->GetResource( m_ResourceId[m_Side] )->OutputResourceInfo();
            m_pSystem->GetResource( m_ResourceId[m_Side] )->RegisterTextureViewToDescriptorPool( pRegisterTextureSlotCallback, NULL );
            m_pSystem->CreateEmitterSetId( &m_Handle, m_EmitterSetId, m_ResourceId[m_Side], m_GroupId );
            m_EffectBinaryPath.Copy( effectBinaryPath );
            return  true;
        }
        else
        {
            m_EffectBinaryPath.Copy( effectBinaryPath );
            return false;
        }
    }

    //------------------------------------------------------------------------------
    //! @brief  エフェクトバイナリを破棄します。
    //------------------------------------------------------------------------------
    bool UnloadEffectBinary( int side ) NN_NOEXCEPT
    {
        // リソースに紐づいたエミッタセットを破棄
        m_pSystem->KillEmitterSetGroup( m_GroupId );

        //m_Side = 1 - m_Side;

        // バイナリを破棄
        if ( m_pEffectBinary[side] )
        {
            // 裏のリソースを破棄
            m_pSystem->ClearResource( m_pHeap, m_ResourceId[side] );

            m_pHeap->Free( m_pEffectBinary[side] );
            m_pEffectBinary[side] = nullptr;
            return  true;
        }

        return false;
    }

    //------------------------------------------------------------------------------
    //! @brief  指定IDのエミッタセットを再生します。
    //!
    //! @param[in] index         エミッタセットのID
    //------------------------------------------------------------------------------
    void Play( int index ) NN_NOEXCEPT
    {
        if ( !m_pEffectBinary[m_Side] ) return;

        m_EmitterSetId = index;

        // エミッタセット数を取得する
        int emitterSetCount = m_pSystem->GetResource( m_ResourceId[m_Side] )->GetEmitterSetCount();

        if ( m_EmitterSetId < emitterSetCount )
        {
            m_pSystem->KillEmitterSetGroup( m_GroupId );
            m_pSystem->CreateEmitterSetId( &m_Handle, m_EmitterSetId, m_ResourceId[m_Side], m_GroupId );
        }
    }

    //------------------------------------------------------------------------------
    //! @brief  次のエミッタセットを再生します。
    //------------------------------------------------------------------------------
    void Send() NN_NOEXCEPT
    {
        if ( !m_pEffectBinary[m_Side] ) return;

        // エミッタセット数を取得する
        int emitterSetCount = m_pSystem->GetResource( m_ResourceId[m_Side] )->GetEmitterSetCount();

        if ( m_EmitterSetId + 1 < emitterSetCount )
        {
            m_EmitterSetId++;
            m_pSystem->KillEmitterSetGroup( m_GroupId );
            m_pSystem->CreateEmitterSetId( &m_Handle, m_EmitterSetId, m_ResourceId[m_Side], m_GroupId );
            nn::vfx::detail::OutputLog( "EmitterSet Name : %s\n", m_Handle.GetEmitterSet()->GetName() );
        }
    }

    //------------------------------------------------------------------------------
    //! @brief  前のエミッタセットを再生します。
    //------------------------------------------------------------------------------
    void Prev() NN_NOEXCEPT
    {
        if ( !m_pEffectBinary[m_Side] ) return;

        if ( m_EmitterSetId > 0 )
        {
            m_EmitterSetId--;
            m_pSystem->KillEmitterSetGroup( m_GroupId );
            m_pSystem->CreateEmitterSetId( &m_Handle, m_EmitterSetId, m_ResourceId[m_Side], m_GroupId );
        }
    }

private:
    int                                     m_Side;
    nn::vfx::System*                        m_pSystem;
    nn::vfx::Heap*                          m_pHeap;
    int                                     m_ResourceId[2];
    int                                     m_GroupId;
    int                                     m_EmitterSetId;
    nn::vfx::Handle                         m_Handle;
    void*                                   m_pEffectBinary[2];
    nn::vfx::viewer::detail::ViewerString   m_EffectBinaryPath;
};
