﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d.h>
#include <nn/vfx/vfx_Heap.h>
#include <nns/gfx/gfx_GpuBuffer.h>

namespace nw      {
namespace eftdemo {


#if defined( NN_BUILD_CONFIG_OS_WIN )
//------------------------------------------------
//! @brief テクスチャクラスです。
//------------------------------------------------
class Texture
{
    NN_DISALLOW_COPY( Texture );
public:
    Texture() NN_NOEXCEPT;

    ~Texture() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  初期化を行います。
    //---------------------------------------------------------------------------
    bool Initialize( nn::gfx::Device* pDevice, const char* pFtxbFilePath ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  ディスクリプタスロットの更新を行います。
    //---------------------------------------------------------------------------
    void UpdateDescriptorSlot( vfxdemo::TextureDescriptorIndexAllocator* pTextureDescriptorIndexAllocator ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  破棄を行います。
    //---------------------------------------------------------------------------
    void Finalize( nn::gfx::Device* pDevice ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    nn::gfx::DescriptorSlot GetDescriptorSlot() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( m_pResTextureFile[ m_TextureFace ] );
        return m_DescriptorSlot[ m_TextureFace ];
    }

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_pResTextureFile[ m_TextureFace ] ? true : false;
    }

private:
    nn::gfx::Device*            m_pDevice;
    int                         m_TextureFace;
    nn::gfx::ResTextureFile*    m_pResTextureFile[ 2 ];
    nn::gfx::DescriptorSlot     m_DescriptorSlot[ 2 ];
    uint8_t*                    m_TextureBinaryPtr[ 2 ];
};


//------------------------------------------------
//! @brief  Shader
//------------------------------------------------
class Shader
{
    NN_DISALLOW_COPY( Shader );
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    Shader() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    ~Shader() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  初期化を行います。
    //---------------------------------------------------------------------------
    bool Initialize( nn::gfx::Device* pDevice, nn::gfx::ShaderStage shaderStage, const char* shaderCode ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  破棄を行います。
    //---------------------------------------------------------------------------
    void Finalize( nn::gfx::Device* pDevice ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  シェーダを取得します。
    //---------------------------------------------------------------------------
    nn::gfx::Shader* GetGfxShader() NN_NOEXCEPT
    {
        return &m_Shader[ m_ShaderFace ];
    }

private:
    nn::gfx::Device*            m_pDevice;
    int                         m_ShaderFace;
    nn::gfx::Shader             m_Shader[ 2 ];
    bool                        m_IsInitialized[ 2 ];
};

//------------------------------------------------------------------------------
//  コンバイナプレビューモデルクラス
//------------------------------------------------------------------------------
class CombinerPreviewModel
{
private:
    struct ViewBlock
    {
        nn::util::FloatColumnMajor4x4  projectionMatrix;
        nn::util::FloatColumnMajor4x4  viewMatrix;
        nn::util::Float4    cameraPosition;
        nn::util::Float4    cameraVector;

        void Set(
            const nn::util::Matrix4x3fType& view,
            const nn::util::Matrix4x4fType& projection,
            const nn::util::Vector3fType&   cameraPos ) NN_NOEXCEPT
        {
            // Projection Matrix
            nn::util::MatrixStore( &projectionMatrix, projection );

            // View Matrix(3x4から4x3に変換)
            nn::util::Matrix4x4fType tempView4x4f;
            nn::util::Matrix4x4fType tempViewT4x4f;
            nn::vfx::detail::Matrix4x3fTo4x4f( &tempView4x4f, view );

            // View Matrix
            nn::util::MatrixStore( &viewMatrix, tempView4x4f );

            // Look Direction
            nn::util::MatrixTranspose( &tempViewT4x4f, tempView4x4f );
            cameraVector.x = tempViewT4x4f._m[ 0 ][ 2 ];
            cameraVector.y = tempViewT4x4f._m[ 1 ][ 2 ];
            cameraVector.z = tempViewT4x4f._m[ 2 ][ 2 ];
            cameraVector.w = 0;

            // Eye Position
            nn::util::VectorStore( cameraPosition.v, cameraPos );
            cameraPosition.w = 0;
        }
    };


    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    CombinerPreviewModel() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    ~CombinerPreviewModel() NN_NOEXCEPT;

public:
    //---------------------------------------------------------------------------
    //! @brief   Ige システム初期化
    //---------------------------------------------------------------------------
    void InitializeIgeService() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief   Ige システム破棄
    //---------------------------------------------------------------------------
    void FinalizeIgeService() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  初期化処理を行います。
    //---------------------------------------------------------------------------
    bool Initialize( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  メモリプールから領域を確保します
    //---------------------------------------------------------------------------
    ptrdiff_t AllocateFromMemoryPool( size_t size, size_t alignment /*= DEFAULT_ALIGNMENT*/ ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  モデルファイルをロードします。
    //---------------------------------------------------------------------------
    bool LoadG3dModelBinary( const char* pFmdbFilePath = nullptr ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  テクスチャファイルをロードします。
    //---------------------------------------------------------------------------
    bool LoadTextureBinary( const char* pFtxbFilePath = nullptr ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  ブロックの更新処理を行います。
    //---------------------------------------------------------------------------
    void CalcBlock( const nn::util::Matrix4x3fType* viewMtx ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  描画処理を行います。
    //---------------------------------------------------------------------------
    void Draw( nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::GpuBuffer* pGpuBuffer,
        const nn::util::Matrix4x4fType* projectionMatrix, const nn::util::Matrix4x3fType* viewMatrix, const nn::util::Vector3fType* camPos,
        nn::gfx::DescriptorSlot colorBufferDescSlot, nn::gfx::DescriptorSlot depthBufferDescSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  ディスクリプタスロットの更新を行います。
    //---------------------------------------------------------------------------
    void UpdateDescriptorSlot( vfxdemo::TextureDescriptorIndexAllocator* pTextureDescriptorIndexAllocator,
        vfxdemo::SamplerDescriptorIndexAllocator* pSamplerDescriptorIndexAllocator ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  ディスクリプタスロットの更新を行います。
    //---------------------------------------------------------------------------
    bool IsReload() const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    // フラグメントシェーダの更新
    //---------------------------------------------------------------------------
    void UpdateFragmentShader() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    // シェーダリロードファイルパスを設定
    //---------------------------------------------------------------------------
    void SetReloadShaderFilePath( const char* fshFilePath ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    // テクスチャリロードファイルパスを設定
    //---------------------------------------------------------------------------
    void ReloadTextureFilePath( int index, const char* ftxbFilePath ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    // テクスチャリロードファイルパスを設定
    //---------------------------------------------------------------------------
    void ReloadTextureFilePath(const char* name, const char* ftxbFilePath) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    // テクスチャリロードファイルパスを設定
    //---------------------------------------------------------------------------
    void UpdateShaderParameter(const char* name, float value) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    // 更新データを受信し適用します。
    //---------------------------------------------------------------------------
    void ReceiveUpdateParameter(const char* receiveData) NN_NOEXCEPT;
public:
    //---------------------------------------------------------------------------
    //! @brief  インスタンス取得。
    //---------------------------------------------------------------------------
    static CombinerPreviewModel* GetInstance() NN_NOEXCEPT
    {
        return g_pCombinerPreviewModel;
    }

    //---------------------------------------------------------------------------
    //! @brief  コンバイナプレビューモデルの生成。
    //---------------------------------------------------------------------------
    static void InitializeCombinerPreviewModel( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  コンバイナプレビューモデルの破棄。
    //---------------------------------------------------------------------------
    static void FinalizeCombinerPreviewModel( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  シェーダリロード
    //---------------------------------------------------------------------------
    static void ShaderReload( void* ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  テクスチャリロード
    //---------------------------------------------------------------------------
    static void TextureReload( void* ) NN_NOEXCEPT;

private:
    nn::gfx::Device*            m_pDevice;
    nn::vfx::Heap*              m_pHeap;
    bool                        m_Reload;

    void*                       m_pBinaryLoadBuffer;
    nn::lmem::HeapHandle        m_HeapHandle;

    void*                       m_pMemoryPoolPtr;
    nn::gfx::MemoryPool         m_MemoryPool;
    ptrdiff_t                   m_MemoryPoolOffset;

    nn::g3d::ResFile*           m_pG3dModelResFile;
    nn::gfx::VertexState*       m_pVertexStateArray;
    void*                       m_pVertexStateBuffer;

    nn::g3d::ModelObj*          m_pModelObj;

    int                         m_CurrentFace;

    Shader                      m_VertexShader;
    Shader                      m_FragmentShader;

    Texture                     m_Texture[ 4 ];

    nn::gfx::Sampler            m_Sampler;
    nn::gfx::DescriptorSlot     m_SamplerDescriptorSlot;

    float                       m_ShaderParameter[ 4 ];
    nn::util::Float4            m_VectorParameter0;
    nn::util::Float4            m_VectorParameter1;
    nn::util::Float4            m_VectorParameter2;

    char                        m_TextureFilePath[ 4 ][ _MAX_PATH ];
    char                        m_ReloadFshFilePath[ _MAX_PATH ];

    static CombinerPreviewModel* g_pCombinerPreviewModel;
};
#else
class CombinerPreviewModel
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    CombinerPreviewModel() NN_NOEXCEPT {}

    static CombinerPreviewModel* GetInstance() NN_NOEXCEPT
    {
        return g_pCombinerPreviewModel;
    }

    static void InitializeCombinerPreviewModel( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT
    {
        NN_UNUSED( pDevice );
        NN_UNUSED( pHeap );
        return;
    }

    //---------------------------------------------------------------------------
    // シェーダリロードファイルパスを設定
    //---------------------------------------------------------------------------
    void SetReloadShaderFilePath( const char* fshFilePath ) NN_NOEXCEPT
    {
        NN_UNUSED( fshFilePath );
        return;
    }

    void CalculateCombinerPreviewModel( const nn::util::Matrix4x3fType* viewMatrix ) NN_NOEXCEPT
    {
        NN_UNUSED( viewMatrix );
        return;
    }

    void DrawCombinerPreviewModel( nn::gfx::CommandBuffer*          pCommandBuffer,
        nns::gfx::GpuBuffer*             pGpuBuffer,
        const nn::util::Matrix4x4fType*  projectionMatrix,
        const nn::util::Matrix4x3fType*  viewMatrix,
        const nn::util::Vector3fType*    camPos,
        nn::gfx::DescriptorSlot          colorBufferDescSlot,
        nn::gfx::DescriptorSlot          depthBufferDescSlot ) NN_NOEXCEPT
    {
        NN_UNUSED( pCommandBuffer );
        NN_UNUSED( pGpuBuffer );
        NN_UNUSED( projectionMatrix );
        NN_UNUSED( viewMatrix );
        NN_UNUSED( camPos );
        NN_UNUSED( colorBufferDescSlot );
        NN_UNUSED( depthBufferDescSlot );
        return;
    }

    bool IsReload() const NN_NOEXCEPT
    {
        return false;
    }

    void UpdateDescriptorSlot( vfxdemo::TextureDescriptorIndexAllocator* pTextureDescriptorIndexAllocator,
        vfxdemo::SamplerDescriptorIndexAllocator* pSamplerDescriptorIndexAllocator ) NN_NOEXCEPT
    {
        NN_UNUSED( pTextureDescriptorIndexAllocator );
        NN_UNUSED( pSamplerDescriptorIndexAllocator );
    }

private:
    static CombinerPreviewModel* g_pCombinerPreviewModel;
};
#endif


} // namespace eftdemo
} // namespace nw
