﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.HookCommand;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.ViewModels;
using EffectMaker.UILogic.Manager;
using EffectMaker.UIControls.EffectBrowser.Controls.AddressBar;
using EffectMaker.UIControls.EffectBrowser.Controls.Dialog;
using EffectMaker.UIControls.EffectBrowser.Controls.FileListView;
using EffectMaker.UIControls.EffectBrowser.Controls.FileListView.Base;
using EffectMaker.UIControls.EffectBrowser.Controls.FileTreeView;
using EffectMaker.UIControls.EffectBrowser.Data;
using EffectMaker.UIControls.EffectBrowser.Data.Favorites;
using EffectMaker.UIControls.EffectBrowser.Exceptions;
using EffectMaker.UIControls.EffectBrowser.Utilities;

using FileInfo = EffectMaker.UIControls.EffectBrowser.Data.FileInfo;

namespace EffectMaker.UIControls.EffectBrowser
{
    /// <summary>
    /// エフェクトブラウザのメインパネルです。
    /// </summary>
    public partial class EffectBrowserPanel : UserControl, IMessageFilter
    {
        #region Fields

        /// <summary>
        /// The favorites manager.
        /// </summary>
        private readonly FavoritesManager favoritesManager = new FavoritesManager();

        /// <summary>
        /// The texture preview throttle.
        /// </summary>
        private readonly ThrottleExecuter texturePreviewThrottle = new ThrottleExecuter();

        /// <summary>
        /// The update state throttle.
        /// </summary>
        private readonly ThrottleExecuter updateStateThrottle = new ThrottleExecuter();

        /// <summary>
        /// The color items.
        /// </summary>
        private ToolStripMenuItem[] colorItems;

        /// <summary>
        /// The preview file name.
        /// </summary>
        private string previewFileName;

        /// <summary>
        /// The tsm files edit label colors.
        /// </summary>
        private ToolStripMenuItem[] tsmFilesEditLabelColors;

        /// <summary>
        /// 検索ワード入力欄の初期幅
        /// </summary>
        private readonly int ttbSearchDefalutWidth;

        /// <summary>
        /// ファイルリストビューの表示モード
        /// </summary>
        private ViewMode fileListViewMode = ViewMode.Details;

        /// <summary>
        /// コンピュータタブを一度開いたか
        /// </summary>
        private bool onceExpandedComputer = false;

        /// <summary>
        /// お気に入りタブを一度開いたか
        /// </summary>
        private bool onceExpandedFavorites = false;

        /// <summary>
        /// プレビューの開閉切り替えイベントを無視するかどうか
        /// </summary>
        private bool ignoreShutterEvent = false;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public EffectBrowserPanel()
        {
            this.InitializeComponent();

            this.adbSearch.PromptText = Properties.Resources.SearchBar_PromptText;
            this.adbFavorites.PromptText = Properties.Resources.AddressBar_PromptText;
            this.adbComputer.PromptText = Properties.Resources.AddressBar_PromptText;
            this.ttbSearch.PromptText = Properties.Resources.SearchBox_PromptText;

            this.splSearch.CbxBaseSearchPathFiltering.CheckedChanged +=
                new System.EventHandler(this.CbxBaseSearchPathFiltering_CheckedChanged);

            this.ttbSearchDefalutWidth = this.ttbSearch.Width;
            this.searchFilter.Resize += this.ResizeSearchTextBox;

            this.imageList1.ImageSize = new Size(Constants.LargeIconSize / 2, Constants.LargeIconSize);

            this.splitterShutter.TargetVisibleChanged += (s, e) =>
            {
                if (this.ignoreShutterEvent)
                {
                    return;
                }

                this.ShowMiniPreview = this.splitterShutter.TargetVisible;
            };

            if (!OptionStore.ProjectConfig.IsEftCombinerEditorEnabled)
            {
                this.tsdChangeView.DropDownItems.Remove(this.combinersToolStripMenuItem);
            }
        }

        #endregion

        #region Public Events

        /// <summary>
        /// The eset editing.
        /// </summary>
        public event EventHandler<EsetEditingEventArgs> EsetEditing;

        /// <summary>
        /// The eset exporting.
        /// </summary>
        public event EventHandler<EsetExportingEventArgs> EsetExporting;

        /// <summary>
        /// The prev editing.
        /// </summary>
        public event EventHandler<PrevEditingEventArgs> PrevEditing;

        /// <summary>
        /// The workspace opening.
        /// </summary>
        public event EventHandler<WorkspaceOpeningEventArgs> WorkspaceOpening;

        /// <summary>
        /// The texture selected.
        /// </summary>
        public event EventHandler<TextureSelectedEventArgs> TextureSelected;

        #endregion

        #region Enums

        /// <summary>
        /// ビューモードです。
        /// </summary>
        public enum ViewMode
        {
            /// <summary>
            /// 詳細ビュー(従来どおり)
            /// </summary>
            Details,

            /// <summary>
            /// 拡大アイコンモード(新設)
            /// </summary>
            LargeIcon,
        }

        /// <summary>
        /// ページの種類です。
        /// </summary>
        private enum PageKind
        {
            /// <summary>
            /// コンピュータ
            /// </summary>
            Computer,

            /// <summary>
            /// お気に入り
            /// </summary>
            Favorites,

            /// <summary>
            /// 検索
            /// </summary>
            Search
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets the adress bars.
        /// </summary>
        private IEnumerable<EBAdressBar> AdressBars
        {
            get
            {
                yield return this.adbComputer;
                yield return this.adbFavorites;
                yield return this.adbSearch;
            }
        }

        /// <summary>
        /// Gets the directory.
        /// </summary>
        /// <exception cref="NotSupportedException">
        /// NotSupportedException
        /// </exception>
        private string Directory
        {
            get
            {
                switch (this.SelectedPageKind)
                {
                    case PageKind.Computer:
                        return this.ftvComputer.Directory;

                    case PageKind.Favorites:
                        return this.ftvFavorites.Directory;

                    case PageKind.Search:
                        return this.adbSearch.Directory;
                }

                throw new NotSupportedException();
            }
        }

        /// <summary>
        /// Gets the file list views.
        /// </summary>
        private IEnumerable<EBFileListView> FileListViews
        {
            get
            {
                yield return this.flvComputer;
                yield return this.flvFavorites;
                yield return this.flvSearch;
            }
        }

        /// <summary>
        /// Gets the file tree views.
        /// </summary>
        private IEnumerable<EBFileTreeView> FileTreeViews
        {
            get
            {
                yield return this.ftvComputer;
                yield return this.ftvFavorites;
            }
        }

        /// <summary>
        /// Gets the selected address bar.
        /// </summary>
        /// <exception cref="NotSupportedException">
        /// NotSupportedException
        /// </exception>
        private EBAdressBar SelectedAddressBar
        {
            get
            {
                switch (this.SelectedPageKind)
                {
                    case PageKind.Computer:
                        return this.adbComputer;

                    case PageKind.Favorites:
                        return this.adbFavorites;

                    case PageKind.Search:
                        return this.adbSearch;
                }

                throw new NotSupportedException();
            }
        }

        /// <summary>
        /// Gets the selected file.
        /// </summary>
        /// <exception cref="NotSupportedException">
        /// NotSupportedException
        /// </exception>
        private FileInfo SelectedFile
        {
            get
            {
                switch (this.SelectedPageKind)
                {
                    case PageKind.Computer:
                        return this.flvComputer.SelectedFile;

                    case PageKind.Favorites:
                        return this.flvFavorites.SelectedFile;

                    case PageKind.Search:
                        return this.flvSearch.SelectedFile;
                }

                throw new NotSupportedException();
            }
        }

        /// <summary>
        /// Gets the selected file list view.
        /// </summary>
        /// <exception cref="NotSupportedException">
        /// NotSupportedException
        /// </exception>
        private EBFileListView SelectedFileListView
        {
            get
            {
                switch (this.SelectedPageKind)
                {
                    case PageKind.Computer:
                        return this.flvComputer;

                    case PageKind.Favorites:
                        return this.flvFavorites;

                    case PageKind.Search:
                        return this.flvSearch;
                }

                throw new NotSupportedException();
            }
        }

        /// <summary>
        /// Gets the selected page kind.
        /// </summary>
        private PageKind SelectedPageKind
        {
            get
            {
                return (PageKind)this.tbcTrees.SelectedTab.Tag;
            }
        }

        /// <summary>
        /// 今開いているフォルダの種類を取得します。
        /// </summary>
        private FileKindType OpenedFolderKind
        {
            get
            {
                if (this.SelectedPageKind == PageKind.Search)
                {
                    if (this.splSearch.SearchMode != SearchFileInfo.SearchModeKind.EmitterSetUsedFile)
                    {
                        // 検索ビューなら選択しているファイル種別を返す
                        return this.splSearch.SelectedFileKind;
                    }

                    // 使用エミッタセットの検索中ならその他扱い
                    return FileKindType.Folder;
                }

                // コンピュータ、あるいはお気に入りビューのディレクトリパスを取得
                string directoryPath = string.Empty;
                if (this.SelectedPageKind == PageKind.Computer)
                {
                    directoryPath = this.flvComputer.Directory;
                }
                else if (this.SelectedPageKind == PageKind.Favorites)
                {
                    directoryPath = this.flvFavorites.Directory;
                }

                // パスの末尾によってタイプを分岐
                if (directoryPath.EndsWith(IOConstants.TextureFolderName, true, null))
                {
                    return FileKindType.TextureFile;
                }

                if (directoryPath.EndsWith(IOConstants.PrimitiveFolderName, true, null))
                {
                    return FileKindType.PrimitiveFile;
                }

                if (OptionStore.ProjectConfig.IsEftCombinerEditorEnabled &&
                    directoryPath.EndsWith(IOConstants.CombinerShaderFolderName, true, null))
                {
                    return FileKindType.CombinerFile;
                }

                return FileKindType.Folder;
            }
        }

        /// <summary>
        /// ミニプレビューの表示/非表示を取得または設定します。
        /// </summary>
        public bool ShowMiniPreview
        {
            get { return OptionStore.RootOptions.EffectBrowser.ShowMiniPreview; }
            set { OptionStore.RootOptions.EffectBrowser.ShowMiniPreview = value; }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// Applicationへのメッセージをハンドルします。
        /// </summary>
        /// <param name="m">メッセージ</param>
        /// <returns>メッセージをフィルタでカットするときtrueを返します。</returns>
        [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1126:PrefixCallsCorrectly", Justification = "Reviewed. Suppression is OK here.")]
        public bool PreFilterMessage(ref Message m)
        {
            const int WmKeydown = 0x100;
            const int VkEscape = 0x1B;
            const int VkF = 0x46;

            if (m.Msg == WmKeydown)
            {
                if ((int)m.WParam == VkEscape)
                {
                    if (this.ttbSearch.Focused)
                    {
                        this.ttbSearch.Text = string.Empty;
                        return true;
                    }
                }

                if ((int)m.WParam == VkF)
                {
                    if (ModifierKeys == Keys.Control)
                    {
                        this.ttbSearch.Focus();
                        return true;
                    }
                }
            }

            return false;
        }

        #endregion

        #region Methods

        /// <summary>
        /// The on load.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnLoad(EventArgs e)
        {
            Logger.Log(LogLevels.Debug, "Beginning EffectBrowserPanel.Onload...");

            using (new ProfileTimer("EffectBrowserPanel.Onload"))
            {
                base.OnLoad(e);
                this.InitializeControls();

                this.tdpTexture.SizingGrip = false;
                this.tssFilename.Alignment = ToolStripItemAlignment.Right;
                this.tssState.Alignment = ToolStripItemAlignment.Right;

                this.FileTreeViews.ForEach(ftv =>
                {
                    ftv.Initialize();
                    ////ftv.GotFocus += (ss, ee) => this.ttbSearch.Text = string.Empty;
                });

                this.AdressBars.ForEach(adb => adb.GetDirectoryHistory = () => this.directoryPathHistory);
                this.FileListViews.ForEach(flv =>
                {
                    flv.LoadedEditDataCountChanged += (ss, ee) => this.UpdateStatusBarState();
                    flv.LoadedAssetDataCountChanged += (ss, ee) => this.UpdateStatusBarState();
                    flv.FileOpen += (ss, ee) => this.OnFileOpen();
                    flv.SelectedIndexChanged += (ss, ee) => RunOnIdleProcess.Execute(() =>
                    {
                        this.UpdateStatusBar();
                        this.UpdateFilePreview();
                    });

                    flv.NormalizeColumnHeader();
                });

                this.favoritesManager.Initialize(this.ftvFavorites);

                using (new LockWindowUpdateBlock(this))
                using (new LayoutSuspendBlock(this))
                using (new ControlEventSuppressBlock())
                {
                    this.UpdateAddressBar();
                    this.UpdateStatusBar();
                    this.UpdateView();

                    // コンフィグ読み込み
                    this.LoadConfig();
                    this.FileListViews.ForEach(x => x.UpdateColumnHeader());
                    this.UpdateFilterMenu();

                    // 前回終了時のタブを選択する
                    this.tbcTrees.SelectedIndexChanged += this.ExpandOnce;

                    this.tbcTrees.SelectedIndex = this.setting.FileTreeView.LastTab;
                }

                this.ExpandOnce(null, EventArgs.Empty);
                this.SyncViewMode(null);

                OptionStore.OptionChanged += this.OnGammaCorrectionChanged;

                Application.AddMessageFilter(this);

                // Disposeのフック処理を登録
                new ControlDisposingHook(this.components, this.OnDisposing);
            }
        }

        /// <summary>
        /// Dispose処理を行います。
        /// </summary>
        private void OnDisposing()
        {
            Application.RemoveMessageFilter(this);
            OptionStore.OptionChanged -= this.OnGammaCorrectionChanged;
            this.texturePreviewThrottle.Dispose();
            this.updateStateThrottle.Dispose();

            this.AdressBars.ForEach(adb => adb.Destory());
            this.FileListViews.ForEach(flv => flv.Destory());
        }

        /// <summary>
        /// 初回にタブを開いた際にツリービューの状態を復元するようにします。
        /// </summary>
        /// <param name="sender">未使用</param>
        /// <param name="args">未使用</param>
        private void ExpandOnce(object sender, EventArgs args)
        {
            if (this.tbcTrees.SelectedIndex == 0 && !onceExpandedComputer)
            {
                // コンピュータタブの前回終了時の状態を復元する.
                if (this.setting.FileTreeView.ComputerDirectory != null)
                {
                    this.ftvComputer.Directory = this.setting.FileTreeView.ComputerDirectory;
                    this.ftvComputer.ExpandCurrentDirectoryNode();
                }

                onceExpandedComputer = true;
            }
            else if (this.tbcTrees.SelectedIndex == 1 && !onceExpandedFavorites)
            {
                // お気に入りタブの前回終了時の状態を復元する.
                if (this.setting.FileTreeView.FavoritesDirectory != null)
                {
                    this.ftvFavorites.Directory = this.setting.FileTreeView.FavoritesDirectory;
                    this.ftvFavorites.ExpandCurrentDirectoryNode();
                }

                onceExpandedFavorites = true;
            }

            if (onceExpandedComputer && onceExpandedFavorites)
            {
                this.tbcTrees.SelectedIndexChanged -= this.ExpandOnce;
            }
        }

        /// <summary>
        /// The add favorite.
        /// </summary>
        /// <param name="directory">The directory.</param>
        private void AddFavorite(string directory)
        {
            var defaultName = Path.GetFileName(directory);

            if (string.IsNullOrEmpty(defaultName))
            {
                defaultName = directory;
            }

            using (
                var dialog = new TextInputDialog
                {
                    Title = Properties.Resources.Favorites_InputNameTitle,
                    Message = Properties.Resources.Favorites_InputNameMessage,
                    Text = defaultName
                })
            {
                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    this.favoritesManager.AddBookmark(null, dialog.Text, directory);

                    this.tbcTrees.SelectedTab = this.tbpFavorites;

                    this.UpdateView();
                }
            }
        }

        /// <summary>
        /// The copy file list view column info.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        private void CopyFileListViewColumnInfo(object sender)
        {
            if (sender == this.flvComputer)
            {
                this.flvFavorites.CopyColumnInfo(this.flvComputer);
                this.flvSearch.CopyColumnInfo(this.flvComputer);
            }
            else if (sender == this.flvFavorites)
            {
                this.flvComputer.CopyColumnInfo(this.flvFavorites);
                this.flvSearch.CopyColumnInfo(this.flvFavorites);
            }
            else if (sender == this.flvSearch)
            {
                this.flvComputer.CopyColumnInfo(this.flvSearch);
                this.flvFavorites.CopyColumnInfo(this.flvSearch);
            }

            this.FileListViews.ForEach(x => x.NormalizeColumnHeader());
        }

        /// <summary>
        /// The delete favorite.
        /// </summary>
        private void DeleteFavorite()
        {
            this.favoritesManager.Delete(this.ftvFavorites.SelectedNode);

            this.UpdateView();
        }

        /// <summary>
        /// The file filter checked changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void FileFilterCheckedChanged(object sender, EventArgs e)
        {
            System.Diagnostics.Debug.Assert(sender is ToolStripMenuItem, "sender is ToolStripMenuItem");

            var item = sender as ToolStripMenuItem;
            var kind = (EBFileListView.ColumnKinds)item.Tag;

            this.FileListViews.ForEach(x => x.SetColumnVisible(kind, item.Checked));
        }

        /// <summary>
        /// The file list view_ after column reordered.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void FileListView_AfterColumnReordered(object sender, EventArgs e)
        {
            this.CopyFileListViewColumnInfo(sender);
        }

        /// <summary>
        /// The file list view_ column width changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void FileListView_ColumnWidthChanged(object sender, ColumnWidthChangedEventArgs e)
        {
            this.CopyFileListViewColumnInfo(sender);
        }

        /// <summary>
        /// The file list view_ update state changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void FileListView_UpdateStateChanged(object sender, EventArgs e)
        {
            if (this.IsDisposed)
            {
                return;
            }

            this.updateStateThrottle.Execute(
                10,
                () =>
                    {
                        this.UpdateView();
                        this.UpdateStatusBar();

                        var style = (this.SelectedFileListView.UpdateState
                                     == EBFileListView.UpdateStateKind.Wait)
                                        ? ProgressBarStyle.Blocks
                                        : ProgressBarStyle.Marquee;

                        switch (this.SelectedPageKind)
                        {
                            case PageKind.Computer:
                                this.tpbComputer.Style = style;
                                break;

                            case PageKind.Favorites:
                                this.tpbFavorites.Style = style;
                                break;

                            case PageKind.Search:
                                this.tpbSearch.Style = style;
                                break;
                        }

                        if (string.IsNullOrEmpty(this.SelectedFileListView.SearchWords)
                            || (this.SelectedFileListView.UpdateState != EBFileListView.UpdateStateKind.Wait))
                        {
                            this.lblSearchFilter.Text = string.Empty;
                            this.lblSearchFilter.ForeColor = Color.Black;
                        }
                        else
                        {
                            this.lblSearchFilter.Text = (this.SelectedFileListView.SearchedAllFilesCount
                                                         == 0)
                                                            ? Properties.Resources.Search_NoHit
                                                            : string.Format(
                                                                Properties.Resources.Search_HitCount,
                                                                this.SelectedFileListView.SearchedAllFilesCount);
                            this.lblSearchFilter.ForeColor =
                                (this.SelectedFileListView.SearchedAllFilesCount > 0)
                                    ? Color.Green
                                    : Color.OrangeRed;
                        }
                    });
        }

        /// <summary>
        /// The initialize controls.
        /// </summary>
        private void InitializeControls()
        {
            this.DoubleBuffered = true;

            this.tbpComputer.Tag = PageKind.Computer;
            this.tbpFavorites.Tag = PageKind.Favorites;
            this.tbpSearch.Tag = PageKind.Search;

            this.ftvComputer.DirectoryChanged +=
                (s, e) => this.OnDirectoryChanged((EBFileTreeView)s, this.flvComputer, this.adbComputer);
            this.ftvFavorites.DirectoryChanged +=
                (s, e) => this.OnDirectoryChanged((EBFileTreeView)s, this.flvFavorites, this.adbFavorites);

            this.flvComputer.DirectoryChanged += (s, e) =>
            {
                this.ftvComputer.Directory = this.flvComputer.Directory;

                RunOnIdleProcess.Execute(() => this.ftvComputer.ExpandCurrentDirectoryNode(), 100);
            };

            this.flvFavorites.DirectoryChanged += (s, e) =>
            {
                this.ftvFavorites.Directory = this.flvFavorites.Directory;

                RunOnIdleProcess.Execute(() => this.ftvFavorites.ExpandCurrentDirectoryNode(), 100);
            };

            this.splSearch.KindChanged += (s, e) => this.SyncViewMode(null);
            this.splSearch.SelectedSearchModeChanged += (s, e) => this.SyncViewMode(null);
            this.flvComputer.VisibleChanged += (s, e) => this.SyncViewMode(null);
            this.flvFavorites.VisibleChanged += (s, e) => this.SyncViewMode(null);
            this.flvSearch.VisibleChanged += (s, e) => this.SyncViewMode(null);

            this.flvComputer.GotFocus += (s, e) => this.UpdateView();
            this.flvFavorites.GotFocus += (s, e) => this.UpdateView();
            this.flvSearch.GotFocus += (s, e) => this.UpdateView();

            this.ftvFavorites.AfterSelect += (s, e) => this.UpdateView();
            {
                this.tsmColor0.Image = Constants.LabelColorIcon0;
                this.tsmColor1.Image = Constants.LabelColorIcon1;
                this.tsmColor2.Image = Constants.LabelColorIcon2;
                this.tsmColor3.Image = Constants.LabelColorIcon3;
                this.tsmColor4.Image = Constants.LabelColorIcon4;
                this.tsmColor5.Image = Constants.LabelColorIcon5;
                this.tsmColor6.Image = Constants.LabelColorIcon6;
                this.tsmColor7.Image = Constants.LabelColorIcon7;

                this.tsmColor0.Click += this.TsmFilesEditColor_Click;
                this.tsmColor1.Click += this.TsmFilesEditColor_Click;
                this.tsmColor2.Click += this.TsmFilesEditColor_Click;
                this.tsmColor3.Click += this.TsmFilesEditColor_Click;
                this.tsmColor4.Click += this.TsmFilesEditColor_Click;
                this.tsmColor5.Click += this.TsmFilesEditColor_Click;
                this.tsmColor6.Click += this.TsmFilesEditColor_Click;
                this.tsmColor7.Click += this.TsmFilesEditColor_Click;

                this.tsmColor0.Tag = LabelColorType.Color0;
                this.tsmColor1.Tag = LabelColorType.Color1;
                this.tsmColor2.Tag = LabelColorType.Color2;
                this.tsmColor3.Tag = LabelColorType.Color3;
                this.tsmColor4.Tag = LabelColorType.Color4;
                this.tsmColor5.Tag = LabelColorType.Color5;
                this.tsmColor6.Tag = LabelColorType.Color6;
                this.tsmColor7.Tag = LabelColorType.Color7;

                this.colorItems = new[]
                {
                    this.tsmColor0, this.tsmColor1, this.tsmColor2, this.tsmColor3, this.tsmColor4,
                    this.tsmColor5, this.tsmColor6, this.tsmColor7
                };
                for (int i = 0; i < 8; ++i)
                {
                    this.colorItems[i].Text = string.Format(Properties.Resources.Color_Name, i);
                }
            }

            {
                this.tsmFilesEditLabelColor0.Image = Constants.LabelColorIcon0;
                this.tsmFilesEditLabelColor1.Image = Constants.LabelColorIcon1;
                this.tsmFilesEditLabelColor2.Image = Constants.LabelColorIcon2;
                this.tsmFilesEditLabelColor3.Image = Constants.LabelColorIcon3;
                this.tsmFilesEditLabelColor4.Image = Constants.LabelColorIcon4;
                this.tsmFilesEditLabelColor5.Image = Constants.LabelColorIcon5;
                this.tsmFilesEditLabelColor6.Image = Constants.LabelColorIcon6;
                this.tsmFilesEditLabelColor7.Image = Constants.LabelColorIcon7;

                this.tsmFilesEditLabelColor0.Click += this.TsmFilesEditColor_Click;
                this.tsmFilesEditLabelColor1.Click += this.TsmFilesEditColor_Click;
                this.tsmFilesEditLabelColor2.Click += this.TsmFilesEditColor_Click;
                this.tsmFilesEditLabelColor3.Click += this.TsmFilesEditColor_Click;
                this.tsmFilesEditLabelColor4.Click += this.TsmFilesEditColor_Click;
                this.tsmFilesEditLabelColor5.Click += this.TsmFilesEditColor_Click;
                this.tsmFilesEditLabelColor6.Click += this.TsmFilesEditColor_Click;
                this.tsmFilesEditLabelColor7.Click += this.TsmFilesEditColor_Click;

                this.tsmFilesEditLabelColor0.Tag = LabelColorType.Color0;
                this.tsmFilesEditLabelColor1.Tag = LabelColorType.Color1;
                this.tsmFilesEditLabelColor2.Tag = LabelColorType.Color2;
                this.tsmFilesEditLabelColor3.Tag = LabelColorType.Color3;
                this.tsmFilesEditLabelColor4.Tag = LabelColorType.Color4;
                this.tsmFilesEditLabelColor5.Tag = LabelColorType.Color5;
                this.tsmFilesEditLabelColor6.Tag = LabelColorType.Color6;
                this.tsmFilesEditLabelColor7.Tag = LabelColorType.Color7;

                this.tsmFilesEditLabelColors = new[]
                {
                    this.tsmFilesEditLabelColor0, this.tsmFilesEditLabelColor1,
                    this.tsmFilesEditLabelColor2, this.tsmFilesEditLabelColor3,
                    this.tsmFilesEditLabelColor4, this.tsmFilesEditLabelColor5,
                    this.tsmFilesEditLabelColor6, this.tsmFilesEditLabelColor7
                };
                for (int i = 0; i < 8; ++i)
                {
                    this.tsmFilesEditLabelColors[i].Text = string.Format(Properties.Resources.Color_Name, i);
                }
            }

            {
                this.tsmFileFilterName.Tag = EBFileListView.ColumnKinds.Name;
                this.tsmFileFilterUpdateTimestamp.Tag = EBFileListView.ColumnKinds.UpdateTimestamp;
                this.tsmFileFilterCreateTimestamp.Tag = EBFileListView.ColumnKinds.CreateTimestamp;
                this.tsmFileFilterFileKind.Tag = EBFileListView.ColumnKinds.FileKind;
                this.tsmFileFilterByteSize.Tag = EBFileListView.ColumnKinds.ByteSize;
                this.tsmFileFilterLabelColor.Tag = EBFileListView.ColumnKinds.LabelColor;
                this.tsmFileFilterComment.Tag = EBFileListView.ColumnKinds.Comment;
            }

            this.tsmAddSampleFav.ToolTipText = Properties.Resources.Favorites_AddSampleFolderToolTip;
        }

        /// <summary>
        /// The make new fav folder name.
        /// </summary>
        /// <returns>
        /// The <see cref="string"/>.
        /// </returns>
        private string MakeNewFavFolderName()
        {
            var folderNames =
                this.ftvFavorites.Nodes.OfType<EBFileTreeNode>()
                    .Where(x => x.NodeKind == EBFileTreeNode.NodeKindType.FavFolder)
                    .ToLookup(x => x.Text);

            var defaultName = Properties.Resources.Favorites_DefaultFavFolder;

            for (var i = 0; ; ++i)
            {
                var name = (i == 0) ? defaultName : string.Format("{0} {1}", defaultName, i);

                if (folderNames.Contains(name) == false)
                {
                    return name;
                }
            }
        }

        /// <summary>
        /// The on directory changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="flv">
        /// The flv.
        /// </param>
        /// <param name="adb">
        /// The adb.
        /// </param>
        private void OnDirectoryChanged(EBFileTreeView sender, EBDirectoryFileListView flv, EBHistoryAddressBar adb)
        {
            // アドレスバーとファイルビューのディレクトリをバックアップしておく。
            var prevFlvDirectory = flv.Directory;
            var prevAdbDirectory = adb.Directory;

            // ネットワークパスを追加した場合に発生する例外をここで潰す。
            try
            {
                using (new ControlEventSuppressBlock())
                {
                    flv.Directory = sender.Directory;
                }

                adb.Directory = sender.Directory;
            }
            catch
            {
                // アドレスバーやファイルビューのディレクトリを戻す
                flv.Directory = prevFlvDirectory;
                adb.Directory = prevAdbDirectory;

                // メッセージボックスを表示しておく。
                MessageBox.Show(Properties.Resources.FileNotFound);
            }

            this.UpdateView();
            this.UpdateFilePreview();
            this.SyncViewMode(null);
        }

        /// <summary>
        /// The on file open.
        /// </summary>
        private void OnFileOpen()
        {
            if (this.SelectedFile == null)
            {
                return;
            }

            switch (this.SelectedFile.FileKind)
            {
                case FileKindType.TextureFile:
                    if (this.TextureSelected != null)
                    {
#if false // ダブルクリックでテクスチャを開く機能があるがオミット中
                        var args =
                            new TextureSelectedEventArgs
                            {
                                FilePath = this.SelectedFile.FileFullPath,
                                Channel = ColorChannelFlags.Rgb
                            };

                        this.TextureSelected(this, args);
#endif
                    }

                    break;
                case FileKindType.EsetFile:
                    if (this.EsetEditing != null)
                    {
                        var args = new EsetEditingEventArgs { FilePaths = new[] { this.SelectedFile.FileFullPath } };

                        this.EsetEditing(this, args);
                        this.UpdateView();
                    }

                    break;

                case FileKindType.PreviewFile:
                    if (this.PrevEditing != null && !this.SelectedFile.IsOldData)
                    {
                        var args = new PrevEditingEventArgs { FilePaths = new[] { this.SelectedFile.FileFullPath } };

                        this.PrevEditing(this, args);
                        this.UpdateView();
                    }

                    break;

                case FileKindType.WorkspaceFile:
                    if (this.WorkspaceOpening != null)
                    {
                        var args = new WorkspaceOpeningEventArgs { FilePaths = new[] { this.SelectedFile.FileFullPath } };

                        this.WorkspaceOpening(this, args);
                        this.UpdateView();
                    }

                    break;
            }
        }

        /// <summary>
        /// The on gamma correction changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void OnGammaCorrectionChanged(object sender, EventArgs e)
        {
            this.FileListViews.ForEach(x => x.OnGammaCorrectionChanged());
        }

        /// <summary>
        /// The refresh search.
        /// </summary>
        /// <param name="isUpdateList">
        /// The is update list.
        /// </param>
        private void RefreshSearch(bool isUpdateList)
        {
            if (isUpdateList)
            {
                this.flvSearch.RequestUpdate(
                    this.splSearch.SearchFileInfo,
                    this.splSearch.CbxBaseSearchPathFiltering.Checked);
            }

            this.splSearch.RefreshView();

            this.splSearch.SelectedFilesCount = this.flvSearch.SelectedIndices.Count;
            this.splSearch.FilesCount = this.flvSearch.ItemsCount;

            if (this.splSearch.SearchMode == SearchFileInfo.SearchModeKind.EmitterSetUsedFile)
            {
                this.splSearch.SearchButtonEnabled = Utilities.PathUtility.DirectoryExists(this.Directory)
                                                     && Utilities.PathUtility.FileExists(
                                                         this.splSearch.ReferenceFilePath);
            }
            else
            {
                this.splSearch.SearchButtonEnabled = Utilities.PathUtility.DirectoryExists(this.Directory);
            }

            this.splSearch.ResetButtonEnabled = Utilities.PathUtility.DirectoryExists(this.Directory);
        }

        /// <summary>
        /// The rename favorite.
        /// </summary>
        private void RenameFavorite()
        {
            this.favoritesManager.Rename(this.ftvFavorites.SelectedNode);

            this.UpdateView();
        }

        /// <summary>
        /// The show in explorer.
        /// </summary>
        private void ShowInExplorer()
        {
            System.Diagnostics.Debug.Assert(this.SelectedFileListView.SelectedFile != null, "this.SelectedFileListView.SelectedFile != null");

            Process.Start(
                "EXPLORER.EXE",
                string.Format("/select,{0}", this.SelectedFileListView.SelectedFile.FileFullPath));
        }

        /// <summary>
        /// The update file preview.
        /// </summary>
        private void UpdateFilePreview()
        {
            this.UpdateView();

            if (this.SelectedFile == null)
            {
                this.previewFileName = null;
                this.SetPreviewVisible(false);
                return;
            }

            switch (this.SelectedFile.FileKind)
            {
                case FileKindType.EsetFile:
                case FileKindType.TextureFile:
                case FileKindType.PrimitiveFile:
                    this.previewFileName = this.SelectedFile.FileFullPath;

                    if (this.previewFileName != null)
                    {
                        var delay = BusinessLogic.Manager.TextureManager.Instance.IsCached(this.previewFileName)
                                        ? 8
                                        : 500;

                        this.texturePreviewThrottle.Execute(
                            delay,
                            () =>
                                {
                                    using (new LockWindowUpdateBlock(this.tdpTexture))
                                    {
                                        this.tdpTexture.ColorChannel = ColorChannelFlags.Rgb;
                                        this.tdpTexture.TargetFilePath = this.previewFileName;
                                        this.tdpTexture.FitCenter();
                                    }

                                    this.SetPreviewVisible(this.tdpTexture.IsImageEnabled);
                                });
                    }

                    break;

                default:
                    this.previewFileName = null;
                    this.SetPreviewVisible(false);

                    break;
            }
        }

        /// <summary>
        /// プレビューの表示/非表示を切り替えます。
        /// </summary>
        /// <param name="visible">プレビューの表示/非表示</param>
        private void SetPreviewVisible(bool visible)
        {
            this.ignoreShutterEvent = true;

            if (visible)
            {
                if (this.ShowMiniPreview)
                {
                    this.splitterShutter.TargetVisible = true;
                }

                this.splitterShutter.Enabled = true;
            }
            else
            {
                this.splitterShutter.TargetVisible = false;
                this.splitterShutter.Enabled = false;
            }

            this.ignoreShutterEvent = false;
        }

        /// <summary>
        /// The adb computer_ directory changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void AdbComputer_DirectoryChanged(object sender, EventArgs e)
        {
            this.flvComputer.Directory = this.adbComputer.Directory;

            this.PushDirectoryPathHistory(this.adbComputer.Directory);
        }

        /// <summary>
        /// The adb favorites_ directory changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void AdbFavorites_DirectoryChanged(object sender, EventArgs e)
        {
            this.flvFavorites.Directory = this.adbFavorites.Directory;

            this.PushDirectoryPathHistory(this.adbFavorites.Directory);
        }

        /// <summary>
        /// The adb search_ directory changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void AdbSearch_DirectoryChanged(object sender, EventArgs e)
        {
            this.splSearch.Directory = this.adbSearch.Directory;
            this.flvSearch.Directory = this.adbSearch.Directory;
            this.splSearch.CbxBaseSearchPathFiltering.Enabled =
                Utilities.PathUtility.DirectoryExists(this.adbSearch.Directory);

            this.PushDirectoryPathHistory(this.adbSearch.Directory);
        }

        /// <summary>
        /// The cbx base search path filtering_ checked changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void CbxBaseSearchPathFiltering_CheckedChanged(object sender, EventArgs e)
        {
            this.RefreshSearch(true);
        }

        /// <summary>
        /// The cms files_ opened.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void CmsFiles_Opened(object sender, EventArgs e)
        {
            this.UpdateFilesContextMenu();
        }

        /// <summary>
        /// The flv search_ items count changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void FlvSearch_ItemsCountChanged(object sender, EventArgs e)
        {
            this.splSearch.FilesCount = this.flvSearch.ItemsCount;
        }

        /// <summary>
        /// The flv search_ selected index changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void FlvSearch_SelectedIndexChanged(object sender, EventArgs e)
        {
            RunOnIdleProcess.Execute(() => this.splSearch.SelectedFilesCount = this.flvSearch.SelectedIndices.Count);
        }

        /// <summary>
        /// The spl search_ kind changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void SplSearch_KindChanged(object sender, EventArgs e)
        {
            RunOnIdleProcess.Execute(
                () =>
                    {
                        this.flvSearch.ClearItems();
                        this.RefreshSearch(false);
                    });
        }

        /// <summary>
        /// The spl search_ move all files button click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void SplSearch_MoveAllFilesButtonClick(object sender, EventArgs e)
        {
            Utilities.PathUtility.SetupUnusedDirectory();

            var fileList = this.flvSearch.OpenedAllFiles.Select(x => x.FileFullPath).ToList();

            if (!WorkspaceRootViewModel.CallHookCommand(HookCommand.PreDelete, fileList))
            {
                return;
            }

            var dir =
                this.splSearch.SelectedFileKind == FileKindType.TextureFile ? Constants.TexturesUnusedDirectory :
                this.splSearch.SelectedFileKind == FileKindType.PrimitiveFile ? Constants.PrimitivesUnusedDirectory :
                this.splSearch.SelectedFileKind == FileKindType.CombinerFile ? Constants.CombinersUnusedDirectory :
                string.Empty;

            foreach (var file in this.flvSearch.OpenedAllFiles)
            {
                var dst = Path.Combine(dir, file.FileName);

                // 同一パスへの移動は無視
                if (file.FileFullPath == dst)
                {
                    continue;
                }

                if (File.Exists(dst))
                {
                    File.Delete(dst);
                }

                File.Move(file.FileFullPath, dst);
            }

            if (!WorkspaceRootViewModel.CallHookCommand(HookCommand.PostDelete, fileList))
            {
                return;
            }

            this.RefreshSearch(true);
        }

        /// <summary>
        /// The spl search_ move selected files button click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void SplSearch_MoveSelectedFilesButtonClick(object sender, EventArgs e)
        {
            Utilities.PathUtility.SetupUnusedDirectory();

            var fileList = this.flvSearch.SelectedFiles.Select(x => x.FileFullPath).ToList();

            if (!WorkspaceRootViewModel.CallHookCommand(HookCommand.PreDelete, fileList))
            {
                return;
            }

            var dir = (this.splSearch.SelectedFileKind == FileKindType.TextureFile)
                          ? Constants.TexturesUnusedDirectory
                          : Constants.PrimitivesUnusedDirectory;

            foreach (var file in this.flvSearch.SelectedFiles)
            {
                var dst = Path.Combine(dir, file.FileName);

                // 同一パスへの移動は無視
                if (file.FileFullPath == dst)
                {
                    continue;
                }

                if (File.Exists(dst))
                {
                    File.Delete(dst);
                }

                File.Move(file.FileFullPath, dst);
            }

            if (!WorkspaceRootViewModel.CallHookCommand(HookCommand.PostDelete, fileList))
            {
                return;
            }

            this.RefreshSearch(true);
        }

        /// <summary>
        /// The spl search_ reference file path changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void SplSearch_ReferenceFilePathChanged(object sender, EventArgs e)
        {
            this.RefreshSearch(false);
        }

        /// <summary>
        /// The spl search_ reset button click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void SplSearch_ResetButtonClick(object sender, EventArgs e)
        {
            RunOnIdleProcess.Execute(
                () =>
                    {
                        this.flvSearch.ClearItems();
                        this.RefreshSearch(false);
                        this.UpdateView();
                        this.UpdateStatusBar();
                        this.UpdateFilePreview();
                    });
        }

        /// <summary>
        /// The spl search_ search button click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void SplSearch_SearchButtonClick(object sender, EventArgs e)
        {
            this.flvSearch.RequestUpdate(
                this.splSearch.SearchFileInfo,
                this.splSearch.CbxBaseSearchPathFiltering.Checked);
        }

        /// <summary>
        /// The spl search_ selected search mode changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void SplSearch_SelectedSearchModeChanged(object sender, EventArgs e)
        {
            RunOnIdleProcess.Execute(
                () =>
                    {
                        this.flvSearch.ClearItems();
                        this.RefreshSearch(false);
                    });
        }

        /// <summary>
        /// The tbc trees_ selected index changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TbcTrees_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.UpdateAddressBar();
            this.UpdateView();
            this.UpdateStatusBar();
            this.UpdateFilePreview();
        }

        /// <summary>
        /// The tsb set label color eset_ drop down opening.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsbSetLabelColorEset_DropDownOpening(object sender, EventArgs e)
        {
            foreach (var colorItem in this.colorItems)
            {
                colorItem.Checked = false;
            }

            var selectedFile =
                this.SelectedFileListView.SelectedFiles.FirstOrDefault(x => x.FileKind == FileKindType.EsetFile);
            if (selectedFile != null)
            {
                this.colorItems[(int)selectedFile.LabelColor].Checked = true;
            }
        }

        /// <summary>
        /// サンプルフォルダをお気に入りに追加します。
        /// </summary>
        /// <param name="sender">使用しません</param>
        /// <param name="e">使用しません</param>
        private void TsmAddSampleFav_Click(object sender, EventArgs e)
        {
            if (!this.favoritesManager.AddSampleFolder())
            {
                MessageBox.Show(
                    Properties.Resources.Favorites_SampleFolderNotFound,
                    Properties.Resources.Caption_Information,
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);
            }
            else
            {
                var current = (PageKind)this.tbcTrees.SelectedTab.Tag;
                if (current == PageKind.Computer)
                {
                    MessageBox.Show(
                        Properties.Resources.Favorites_AddSampleFolder,
                        Properties.Resources.Caption_Information,
                        MessageBoxButtons.OK,
                        MessageBoxIcon.Information);
                }
            }
        }

        /// <summary>
        /// The tsm add favorite_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmAddFavorite_Click(object sender, EventArgs e)
        {
            this.AddFavorite(this.Directory);
        }

        /// <summary>
        /// The tsm add folder_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmAddFolder_Click(object sender, EventArgs e)
        {
            this.favoritesManager.AddFavFolder(null, this.MakeNewFavFolderName());

            this.UpdateView();
        }

        /// <summary>
        /// The tsm computer add_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmComputerAdd_Click(object sender, EventArgs e)
        {
            this.AddFavorite(this.Directory);
        }

        /// <summary>
        /// The tsm computer show in explorer_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmComputerShowInExplorer_Click(object sender, EventArgs e)
        {
            if (System.IO.Directory.Exists(this.Directory))
            {
                Process.Start("EXPLORER.EXE", this.Directory);
            }
        }

        /// <summary>
        /// The tsm delete favorite_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmDeleteFavorite_Click(object sender, EventArgs e)
        {
            this.DeleteFavorite();
        }

        /// <summary>
        /// The tsm favorites add_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmFavoritesAdd_Click(object sender, EventArgs e)
        {
            this.AddFavorite(this.Directory);
        }

        /// <summary>
        /// The tsm favorites delete_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmFavoritesDelete_Click(object sender, EventArgs e)
        {
            this.DeleteFavorite();
        }

        /// <summary>
        /// The tsm favorites rename_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmFavoritesRename_Click(object sender, EventArgs e)
        {
            this.RenameFavorite();
        }

        /// <summary>
        /// The tsm file filter reset_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmFileFilterReset_Click(object sender, EventArgs e)
        {
            this.FileListViews.ForEach(x => x.ResetColumnVisible());
            this.UpdateFilterMenu();
        }

        /// <summary>
        /// The tsm files add fav_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmFilesAddFav_Click(object sender, EventArgs e)
        {
            var selectedFile = this.SelectedFileListView.SelectedFile;
            this.AddFavorite(selectedFile != null ? selectedFile.FileFullPath : this.Directory);
        }

        /// <summary>
        /// The tsm files edit color_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmFilesEditColor_Click(object sender, EventArgs e)
        {
            System.Diagnostics.Debug.Assert(sender is ToolStripMenuItem, "sender is ToolStripMenuItem");

            var selectedEsetFiles = this.SelectedFileListView.SelectedFiles.Where(x => x.IsEditable).ToArray();
            if (selectedEsetFiles.Any() == false)
            {
                return;
            }

            var labelColor = (LabelColorType)(sender as ToolStripMenuItem).Tag;

            using (new WaitCursorBlock(this))
            {
                var errorFiles = new ConcurrentDictionary<string, OldFileInfo>();

                // Parallel.ForEach(selectedEsetFiles, selectedFile =>
                selectedEsetFiles.ForEach(
                    selectedFile =>
                        {
                            var oldLabelColor = selectedFile.LabelColor;

                            try
                            {
                                if (selectedFile.LabelColor != labelColor)
                                {
                                    selectedFile.LabelColor = labelColor;
                                    selectedFile.SaveToFile(true);
                                }
                            }
                            catch (InvalidEsetException)
                            {
                                errorFiles[selectedFile.FileFullPath] = new OldFileInfo
                                {
                                    Target = selectedFile,
                                    OldLabelColor = oldLabelColor
                                };
                            }
                        });

                if (errorFiles.Any())
                {
                    using (
                        var multipleLineDialog = new MultipleLineDialog
                        {
                            Message =
                                Properties.Resources
                                .Exception_InvalidEset,
                            Title =
                                Properties.Resources
                                .Dialog_CantWriteFile,
                            Lines = errorFiles.Keys
                        })
                    {
                        multipleLineDialog.ShowDialog();
                    }

                    errorFiles.Values.ForEach(x => x.Target.LabelColor = x.OldLabelColor);
                }

                this.SelectedFileListView.Invalidate();
            }
        }

        /// <summary>
        /// The tsm files edit comment_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmFilesEditComment_Click(object sender, EventArgs e)
        {
            var selectedEsetFiles = this.SelectedFileListView.SelectedFiles.Where(x => x.IsEditable).ToArray();
            if (selectedEsetFiles.Any() == false)
            {
                return;
            }

            using (
                var dialog = new TextInputDialog
                {
                    Title = Properties.Resources.FileEdit_EditCommentTitle,
                    Message = Properties.Resources.FileEdit_EditCommentPrompt,
                    Text = selectedEsetFiles.First().Comment
                })
            {
                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    var comment = dialog.Text;

                    var errorFiles = new ConcurrentDictionary<string, OldFileInfo>();

                    using (new WaitCursorBlock(this))
                    {
                        // Parallel.ForEach(selectedEsetFiles, selectedFile =>
                        selectedEsetFiles.ForEach(
                            selectedFile =>
                                {
                                    var oldComment = selectedFile.Comment;

                                    try
                                    {
                                        if (selectedFile.Comment != comment)
                                        {
                                            selectedFile.Comment = comment;
                                            selectedFile.SaveToFile(true);
                                        }
                                    }
                                    catch (InvalidEsetException)
                                    {
                                        errorFiles[selectedFile.FileFullPath] = new OldFileInfo
                                        {
                                            Target = selectedFile,
                                            OldComment = oldComment
                                        };
                                    }
                                });
                    }

                    if (errorFiles.Any())
                    {
                        using (
                            var multipleLineDialog = new MultipleLineDialog
                            {
                                Message = Properties.Resources.Exception_InvalidEset,
                                Title = Properties.Resources.Dialog_CantWriteFile,
                                Lines = errorFiles.Keys
                            })
                        {
                            multipleLineDialog.ShowDialog();
                        }

                        errorFiles.Values.ForEach(x => x.Target.Comment = x.OldComment);
                    }

                    this.SelectedFileListView.Invalidate();
                }
            }
        }

        /// <summary>
        /// The tsm files edit label color_ drop down opening.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmFilesEditLabelColor_DropDownOpening(object sender, EventArgs e)
        {
            foreach (var colorItem in this.tsmFilesEditLabelColors)
            {
                colorItem.Checked = false;
            }

            var selectedFile =
                this.SelectedFileListView.SelectedFiles.FirstOrDefault(x => x.FileKind == FileKindType.EsetFile);
            if (selectedFile != null)
            {
                this.tsmFilesEditLabelColors[(int)selectedFile.LabelColor].Checked = true;
            }
        }

        /// <summary>
        /// The tsm files edit_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmFilesEdit_Click(object sender, EventArgs e)
        {
            if (this.SelectedFileListView.SelectedFiles.Any(x => x.FileKind == FileKindType.EsetFile))
            {
                if (this.EsetEditing != null)
                {
                    this.EsetEditing(
                        this,
                        new EsetEditingEventArgs
                        {
                            FilePaths =
                                this.SelectedFileListView.SelectedFiles.Select(x => x.FileFullPath)
                        });
                    this.UpdateView();
                }
            }
            else if (this.SelectedFileListView.SelectedFiles.Any(x => x.FileKind == FileKindType.PreviewFile && !x.IsOldData))
            {
                if (this.PrevEditing != null)
                {
                    this.PrevEditing(
                        this,
                        new PrevEditingEventArgs
                        {
                            FilePaths =
                                this.SelectedFileListView.SelectedFiles.Select(x => x.FileFullPath)
                        });
                    this.UpdateView();
                }
            }
            else if (this.SelectedFileListView.SelectedFiles.Any(x => x.FileKind == FileKindType.WorkspaceFile))
            {
                if (this.WorkspaceOpening != null)
                {
                    this.WorkspaceOpening(
                        this,
                        new WorkspaceOpeningEventArgs
                        {
                            FilePaths =
                                this.SelectedFileListView.SelectedFiles.Select(x => x.FileFullPath)
                        });
                    this.UpdateView();
                }
            }
        }

        /// <summary>
        /// The tsm files export_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmFilesExport_Click(object sender, EventArgs e)
        {
            if (this.SelectedFileListView.SelectedFiles.Any(x => x.FileKind == FileKindType.EsetFile))
            {
                if (this.EsetExporting != null)
                {
                    this.EsetExporting(
                        this,
                        new EsetExportingEventArgs
                        {
                            FilePaths =
                                this.SelectedFileListView.SelectedFiles.Select(
                                    x => x.FileFullPath)
                        });
                }
            }
        }

        /// <summary>
        /// The tsm files show in explorer_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmFilesShowInExplorer_Click(object sender, EventArgs e)
        {
            var selectedFile = this.SelectedFileListView.SelectedFile;

            Process.Start(
                "EXPLORER.EXE",
                selectedFile != null ? string.Format("/select,{0}", selectedFile.FileFullPath) : this.Directory);
        }

        /// <summary>
        /// The tsm show in explorer_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TsmShowInExplorer_Click(object sender, EventArgs e)
        {
            this.ShowInExplorer();
        }

        /// <summary>
        /// The ttb search_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TtbSearch_Click(object sender, EventArgs e)
        {
            this.ttbSearch.Focus();
        }

        /// <summary>
        /// ツールストリップビューのヘルプボタンを押したときの処理を行います。
        /// エフェクトブラウザのヘルプを開きます。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void TsbHelp_Click(object sender, EventArgs e)
        {
            HelpManager.ShowHelp("92116287");
        }

        /// <summary>
        /// The ttb search_ got focus.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TtbSearch_GotFocus(object sender, EventArgs e)
        {
            RunOnIdleProcess.Execute(() => this.ttbSearch.SelectAll());
        }

        /// <summary>
        /// The ttb search_ key down.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TtbSearch_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyData == Keys.Return)
            {
                this.SelectedFileListView.Focus();
                this.SelectedFileListView.SelectFirstFile();
            }
        }

        /// <summary>
        /// The ttb search_ text changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TtbSearch_TextChanged(object sender, EventArgs e)
        {
            this.FileListViews.ForEach(x => x.SetSearchWord(this.ttbSearch.Text));
        }

        /// <summary>
        /// ファイルリストビューの切り替えボタンイベントを処理します。
        /// </summary>
        /// <param name="sender">ボタン</param>
        /// <param name="e">イベント引数</param>
        private void TsdChangeView_ButtonClick(object sender, EventArgs e)
        {
            ToolStripMenuItem targetItem = null;
            switch (this.OpenedFolderKind)
            {
                case FileKindType.TextureFile:
                    targetItem = this.texturesToolStripMenuItem;
                    break;
                case FileKindType.PrimitiveFile:
                    targetItem = this.primitivesToolStripMenuItem;
                    break;
                case FileKindType.CombinerFile:
                    targetItem = this.combinersToolStripMenuItem;
                    break;
                default:
                    targetItem = this.otherToolStripMenuItem;
                    break;
            }

            // 開いているフォルダの種類に対応するアイテムをトグルする
            targetItem.Checked = !targetItem.Checked;
        }

        /// <summary>
        /// ファイルリストビューのフォルダ種別ごとのON/OFF切り替えイベントを処理します。
        /// </summary>
        /// <param name="sender">ボタン</param>
        /// <param name="e">イベント引数</param>
        private void TsdChangeView_ItemClick(object sender, EventArgs e)
        {
            this.SyncViewMode(sender);
        }

        /// <summary>
        /// The resize search text box.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void ResizeSearchTextBox(object sender, EventArgs e)
        {
            int width = this.CalcToolStripLeftWidth(this.searchFilter, this.ttbSearch);
            var icon = Properties.Resources.Icon_FileRefineSearch;
            Size size = this.ttbSearch.Size;
            if (width < 1)
            {
                this.ttbSearch.BackgroundIcon = icon;
                this.ttbSearch.Overflow = ToolStripItemOverflow.Always;
                size.Width = this.ttbSearchDefalutWidth;
            }
            else
            {
                this.ttbSearch.BackgroundIcon = width > icon.Width + 8 ? icon : null;
                this.ttbSearch.Overflow = ToolStripItemOverflow.Never;
                size.Width = Math.Min(this.ttbSearchDefalutWidth, width);
            }

            this.ttbSearch.Size = size;
            this.ttbSearch.PromptText = this.ttbSearch.PromptText;
        }

        /// <summary>
        /// ツールストリップに表示されているコントロールの余白を計算します。
        /// </summary>
        /// <param name="toolStrip">ツールストリップ</param>
        /// <param name="ctrl">計算から除外するコントロール</param>
        /// <returns>余白の幅(除外コントロールが複数ある場合は個数で割られた値)を返します。</returns>
        private int CalcToolStripLeftWidth(ToolStrip toolStrip, object ctrl)
        {
            int width = toolStrip.DisplayRectangle.Width;

            if (toolStrip.OverflowButton.Visible)
            {
                width = width - toolStrip.OverflowButton.Width -
                    toolStrip.OverflowButton.Margin.Horizontal;
            }

            int springBoxCount = 0;

            foreach (ToolStripItem item in toolStrip.Items)
            {
                if (item.IsOnOverflow || !item.Visible)
                {
                    continue;
                }

                if (item.GetType() == ctrl.GetType())
                {
                    springBoxCount++;
                    width -= item.Margin.Horizontal;
                }
                else
                {
                    width = width - item.Width - item.Margin.Horizontal;
                }
            }

            if (springBoxCount > 1)
            {
                width /= springBoxCount;
            }

            return width;
        }

        /// <summary>
        /// 今開いているフォルダに応じてファイルリストのビューモードを同期します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        private void SyncViewMode(object sender)
        {
            this.tsdChangeView.ToolTipText =
                this.SelectedPageKind == PageKind.Search
                    ? Properties.Resources.FileIconSizeChangeButton_HintSearch
                    : Properties.Resources.FileIconSizeChangeButton_HintNormal;

            var folderKind = this.OpenedFolderKind;

            if (folderKind == FileKindType.TextureFile &&
                (sender == null || sender == this.texturesToolStripMenuItem))
            {
                this.fileListViewMode = this.texturesToolStripMenuItem.Checked ?
                    ViewMode.LargeIcon : ViewMode.Details;
            }
            else if (folderKind == FileKindType.PrimitiveFile &&
                (sender == null || sender == this.primitivesToolStripMenuItem))
            {
                this.fileListViewMode = this.primitivesToolStripMenuItem.Checked ?
                    ViewMode.LargeIcon : ViewMode.Details;
            }
            else if (folderKind == FileKindType.CombinerFile &&
                (sender == null || sender == this.combinersToolStripMenuItem))
            {
                this.fileListViewMode = this.combinersToolStripMenuItem.Checked ?
                    ViewMode.LargeIcon : ViewMode.Details;
            }
            else if (folderKind == FileKindType.Folder &&
                (sender == null || sender == this.otherToolStripMenuItem))
            {
                this.fileListViewMode = this.otherToolStripMenuItem.Checked ?
                    ViewMode.LargeIcon : ViewMode.Details;
            }

            // ビューモードを切り替える
            switch (this.fileListViewMode)
            {
                case ViewMode.Details:
                    this.tsdChangeView.Image = Properties.Resources.Icon_EB_ChangeView_Off;
                    this.tsmFileFilter.Visible = true;
                    this.flvComputer.View = this.flvFavorites.View = this.flvSearch.View = View.Details;
                    break;

                case ViewMode.LargeIcon:
                    this.tsdChangeView.Image = Properties.Resources.Icon_EB_ChangeView_On;
                    this.tsmFileFilter.Visible = false;
                    this.flvComputer.View = this.flvFavorites.View = this.flvSearch.View = View.LargeIcon;
                    break;
            }
        }

        #endregion

        /// <summary>
        /// The old file info.
        /// </summary>
        private class OldFileInfo
        {
            #region Public Properties

            /// <summary>
            /// Gets or sets the old comment.
            /// </summary>
            public string OldComment { get; set; }

            /// <summary>
            /// Gets or sets the old label color.
            /// </summary>
            public LabelColorType OldLabelColor { get; set; }

            /// <summary>
            /// Gets or sets the target.
            /// </summary>
            public FileInfo Target { get; set; }

            #endregion
        }
    }
}
